<?php
/**
 * This file adds a custom section in the customizer that recommends the installation of the Kirki plugin.
 * It can be used as-is in themes (drop-in).
 *
 * @package kirki-helpers
 */

if ( ! function_exists( 'kirki_installer_register' ) ) {
    /**
     * Registers the section, setting & control for the kirki installer.
     *
     * @param object $wp_customize The main customizer object.
     */
    function kirki_installer_register( $wp_customize ) {

        // Early exit if Kirki exists.
        if ( class_exists( 'Kirki' ) ) {
            return;
        }

        if ( class_exists( 'WP_Customize_Section' ) && ! class_exists( 'Kirki_Installer_Section' ) ) {
            /**
             * Recommend the installation of Kirki using a custom section.
             *
             * @see WP_Customize_Section
             */
            class Kirki_Installer_Section extends WP_Customize_Section {

                /**
                 * Customize section type.
                 *
                 * @access public
                 * @var string
                 */
                public $type = 'kirki_installer';

                /**
                 * Render the section.
                 *
                 * @access protected
                 */
                protected function render() {

                    // Don't proceed any further if the user has dismissed this.
                    if ( $this->is_dismissed() ) {
                        return;
                    }

                    // Determine if the plugin is not installed, or just inactive.
                    $plugins   = get_plugins();
                    $installed = false;
                    foreach ( $plugins as $plugin ) {
                        if ( 'Kirki' === $plugin['Name'] || 'Kirki Toolkit' === $plugin['Name'] ) {
                            $installed = true;
                        }
                    }
                    // Get the plugin-installation URL.
                    $plugin_install_url = add_query_arg(
                        array(
                            'action' => 'install-plugin',
                            'plugin' => 'kirki',
                        ),
                        self_admin_url( 'update.php' )
                    );
                    $plugin_install_url = wp_nonce_url( $plugin_install_url, 'install-plugin_kirki' );
                    $classes = 'cannot-expand accordion-section control-section control-section-themes control-section-' . $this->type;
                    ?>
                    <li id="accordion-section-<?php echo esc_attr( $this->id ); ?>" class="<?php echo esc_attr( $classes ); ?>" style="border-top:none;border-bottom:1px solid #ddd;padding:7px 14px 16px 14px;text-align:right;">
                        <?php if ( ! $installed ) : ?>
                            <?php $this->install_button(); ?>
                        <?php else : ?>
                            <?php $this->activate_button(); ?>
                        <?php endif; ?>
                    </li>
                    <?php
                }

                /**
                 * Check if the user has chosen to hide this.
                 *
                 * @static
                 * @access public
                 * @since 1.0.0
                 * @return bool
                 */
                public static function is_dismissed() {
                    // Get the user-meta.
                    $user_id   = get_current_user_id();
                    // @codingStandardsIgnoreLine WordPress.VIP.RestrictedFunctions.user_meta_get_user_meta)
                    $user_meta = get_user_meta( $user_id, 'dismiss-kirki-recommendation', true );

                    return ( true === $user_meta || '1' === $user_meta || 1 === $user_meta );
                }

                /**
                 * Adds the install button.
                 *
                 * @since 1.0.0
                 * @return void
                 */
                protected function install_button() {
                    ?>
                    <p style="text-align:left;margin-top:0;">
                        <?php esc_attr_e( 'Please install the Kirki plugin to take full advantage of this theme\s customizer capabilities', 'zoa' ); ?>
                    </p>
                    <a class="install-now button-primary button" data-slug="kirki" href="<?php echo esc_url_raw( $plugin_install_url ); ?>" aria-label="<?php esc_attr_e( 'Install Kirki Toolkit now', 'zoa' ); ?>" data-name="Kirki Toolkit">
                        <?php esc_html_e( 'Install Now', 'zoa' ); ?>
                    </a>
                    <?php
                }

                /**
                 * Adds the install button.
                 *
                 * @since 1.0.0
                 * @return void
                 */
                protected function activate_button() {
                    ?>
                    <p style="text-align:left;margin-top:0;">
                        <?php esc_attr_e( 'You have installed Kirki. Activate it to take advantage of this theme\'s features in the customizer.', 'zoa' ); ?>
                    </p>
                    <a class="activate-now button-primary button" data-slug="kirki" href="<?php echo esc_url_raw( self_admin_url( 'plugins.php' ) ); ?>" aria-label="<?php esc_attr_e( 'Activate Kirki Toolkit now', 'zoa' ); ?>" data-name="Kirki Toolkit">
                        <?php esc_html_e( 'Activate Now', 'zoa' ); ?>
                    </a>
                    <?php
                }
            }
        }

        // Early exit if the user has dismissed the notice.
        if ( is_callable( array( 'Kirki_Installer_Section', 'is_dismissed' ) ) && Kirki_Installer_Section::is_dismissed() ) {
            return;
        }

        $wp_customize->add_section(
            new Kirki_Installer_Section(
                $wp_customize, 'kirki_installer', array(
                    'title'      => '',
                    'capability' => 'install_plugins',
                    'priority'   => 0,
                )
            )
        );
        $wp_customize->add_setting(
            'kirki_installer_setting', array(
                'sanitize_callback' => '__return_true',
            )
        );
        $wp_customize->add_control(
            'kirki_installer_control', array(
                'section'    => 'kirki_installer',
                'settings'   => 'kirki_installer_setting',
            )
        );

    }
}
add_action( 'customize_register', 'kirki_installer_register', 999 );

if ( ! function_exists( 'kirki_installer_dismiss' ) ) {
    /**
     * Handles dismissing the plgin-install/activate recommendation.
     * If the user clicks the "Don't show this again" button, save as user-meta.
     *
     * @since 1.0.0
     * @return void
     */
    function kirki_installer_dismiss() {
        check_ajax_referer( 'dismiss-kirki-recommendation', 'security' );
        $user_id   = get_current_user_id();
        // @codingStandardsIgnoreLine WordPress.VIP.RestrictedFunctions.user_meta_update_user_meta
        if ( update_user_meta( $user_id, 'dismiss-kirki-recommendation', true ) ) {
            echo 'success! :-)';
            wp_die();
        }
        echo 'failed :-(';
        wp_die();
    }
}
add_action( 'wp_ajax_kirki_installer_dismiss', 'kirki_installer_dismiss' );