<?php
if ( !defined( 'ABSPATH' ) ) {
    die( 'No direct access allowed' );
}

if ( !function_exists( 'leap_get_current_page_id' ) ) {

    /**
     * Get current page id
     * @return integer page id
     */
    function leap_get_current_page_id() {
        $object_id = get_queried_object_id();
        if (
        !( class_exists( 'Woocommerce' ) && (is_shop() || is_tax( 'product_cat' ) || is_tax( 'product_tag' )) ) &&
        (
        (get_option( 'show_on_front' ) && get_option( 'page_for_posts' ) && is_home()) ||
        (get_option( 'page_for_posts' ) && is_archive() && !is_post_type_archive()) ||
        (get_option( 'page_for_posts' ) && is_search())
        )
        ) {
            $page_id = get_option( 'page_for_posts' );
        } else {
            if ( isset( $object_id ) ) {
                $page_id = $object_id;
            }

            if ( class_exists( 'Woocommerce' ) ) {
                //if (is_shop() || is_product_category() || is_product_tag() || is_product()) {
                if ( is_shop() || is_tax( 'product_cat' ) || is_tax( 'product_tag' ) ) {
                    $page_id = get_option( 'woocommerce_shop_page_id' );
                }
            }
        }
        return $page_id;
    }

}

if ( !function_exists( 'leap_wp_get_http' ) ) {
	
function leap_wp_get_http( $url = false, $file_path = false, $args = array() ) {

	// No need to proceed if we don't have a $url or a $file_path.
	if ( ! $url || ! $file_path ) {
		return false;
	}

	// Make sure we normalize $file_path.
	$file_path = wp_normalize_path( $file_path );

	// Include the WP_Http class if it doesn't already exist.
	if ( ! class_exists( 'WP_Http' ) ) {
		include_once wp_normalize_path( ABSPATH . WPINC . '/class-http.php' );
	}
	// Inlude the wp_remote_get function if it doesn't already exist.
	if ( ! function_exists( 'wp_remote_get' ) ) {
		include_once wp_normalize_path( ABSPATH . WPINC . '/http.php' );
	}

	$args = wp_parse_args( $args, array(
		'timeout'    => 120,
	) );
	$response = wp_remote_get( esc_url_raw( $url ), $args );
	if ( is_wp_error( $response ) ) {
		return false;
	}
	$body = wp_remote_retrieve_body( $response );

	// Try file_get_contents if body is empty.
	if ( empty( $body ) ) {
		if ( function_exists( 'ini_get' ) && ini_get( 'allow_url_fopen' ) ) {
			global $wp_filesystem;
			$body = $wp_filesystem->get_contents( $url );

		}
	}

	// Initialize the Wordpress filesystem.
	global $wp_filesystem;

	if ( ! defined( 'FS_CHMOD_DIR' ) ) {
		define( 'FS_CHMOD_DIR', ( 0755 & ~ umask() ) );
	}
	if ( ! defined( 'FS_CHMOD_FILE' ) ) {
		define( 'FS_CHMOD_FILE', ( 0644 & ~ umask() ) );
	}

	// Attempt to write the file.
	if ( ! $wp_filesystem->put_contents( $file_path, $body, FS_CHMOD_FILE ) ) {
		// If the attempt to write to the file failed, then fallback to fwrite.
		// @codingStandardsIgnoreStart
		@unlink( $file_path );	
		WP_Filesystem();
        global $wp_filesystem;

        $fp = $wp_filesystem->get_contents($file_path);

        if ( ! $wp_filesystem->put_contents( $fp, $body, 0644) ) {
            echo "There was an error saving the htaccess file. As a result no changes were made";die;
        }
		
	}

	// If all went well, then return the headers of the request.
	if ( isset( $response['headers'] ) ) {
		$response['headers']['response'] = $response['response']['code'];
		return $response['headers'];
	}

	// If all else fails, then return false.
	return false;
}
}


if ( !function_exists( 'leap_get_current_page_title' ) ) {

    /**
     * Get current page title
     * @return string page title
     */
    function leap_get_current_page_title() {
        $page_id = leap_get_current_page_id();

        $title = '';

        $leap_page_title_bar_custom_text = get_post_meta( $page_id, 'leap_page_title_bar_custom_text', true );
        if ( $leap_page_title_bar_custom_text ) {
            $title = get_post_meta( $page_id, 'leap_page_title_bar_custom_text', true );
        }

        if ( !$title ) {

            $title = get_the_title( $page_id );

            $blog_page_id = get_option( 'page_for_posts' );

            if ( is_home() ) {
                $title = esc_html( ot_get_option( 'blog_title', __( 'Blog', 'wiz' ) ) );
            }

            if ( is_singular( 'post' ) && !in_category( ot_get_option( 'blog_exclude_categories', array() ) ) && (ot_get_option( 'display_blog_title_posts', 'no' ) == 'yes') ) {
                $title = esc_html( ot_get_option( 'blog_title', __( 'Blog', 'wiz' ) ) );
            }

            if ( is_search() ) {
                $title = __( 'Search results for', 'wiz' ) . ': ' . get_search_query();
            }

            if ( is_404() ) {
                $title = __( 'Error 404 Page', 'wiz' );
            }

            if ( is_archive() ) {
                if ( is_day() ) {
                    $title = __( 'Daily Archives', 'wiz' ) . ': ' . get_the_date();
                } else if ( is_month() ) {
                    $title = __( 'Monthly Archives', 'wiz' ) . ': ' . get_the_date( _x( 'F Y', 'monthly archives date format', 'wiz' ) );
                } elseif ( is_year() ) {
                    $title = __( 'Yearly Archives', 'wiz' ) . ': ' . get_the_date( _x( 'Y', 'yearly archives date format', 'wiz' ) );
                } elseif ( is_author() ) {
                    $curauth = ( isset( $_GET[ 'author_name' ] ) ) ? get_user_by( 'slug', $_GET[ 'author_name' ] ) : get_user_by( 'id', get_the_author_meta( 'ID' ) );
                    $title   = esc_html( $curauth->nickname );
                } else {
                    $title = esc_html( single_cat_title( '', false ) );
                }
            }

            if ( class_exists( 'Woocommerce' ) && is_woocommerce() && ( is_product() || is_shop() ) && !is_search() ) {
                if ( !is_product() ) {
                    $title = esc_html( woocommerce_page_title( false ) );
                }
            }
        }

        return $title;
    }

}

if ( !function_exists( 'leap_is_valid_url' ) ) {

    /**
     * Validate given URL format
     * @param strint $url
     * @return bool
     */
    function leap_is_valid_url( $url ) {
        return (bool) parse_url( $url, PHP_URL_SCHEME );
    }

}

if ( !function_exists( 'leap_get_post_thumbnail_format' ) ) {

    /**
     * Get post thumbnail url if added or default image based on poet format
     * @param int $post_id Post id
     * @param string $size Image size
     * @return string Image URL
     */
    function leap_get_post_thumbnail_format( $post_id, $size ) {
        $attachment_image = wp_get_attachment_image_src( get_post_thumbnail_id( $post_id ), $size );
        if ( false == $attachment_image ) {
            return ( get_post_format( $post_id ) ) ? get_post_format( $post_id ) : 'standard';
        } else {
            return $attachment_image[ 0 ];
        }
    }

}

if ( !function_exists( 'leap_get_the_post_thumbnail_background' ) ) {

    /**
     * Get post thumbnail url if added or default image based on poet format
     * @param int $post_id Post id
     * @param string $size Image size
     * @return string Image URL
     */
    function leap_get_the_post_thumbnail_background( $post_id, $size ) {
        $thumbnail_format = leap_get_post_thumbnail_format( $post_id, $size );
        if ( leap_is_valid_url( $thumbnail_format ) ) {
            return '<div class="featured-image" style="background-image:url(' . $thumbnail_format . ');"></div>';
        } else {
            return '<div class="default-featured-image ' . $thumbnail_format . '"></div>';
        }
    }

}

if ( !function_exists( 'leap_get_schema_org_type' ) ) {

    /**
     * Get the schema type based on the current page type
     * @return string
     */
    function leap_get_schema_org_type() {

        $type = '';

        if ( function_exists( 'is_product' ) && is_product() ) {
            // Product
            $type = "Product";
        } elseif ( is_single() ) {
            // Single post
            $type = "Article";
        } elseif ( is_author() ) {
            // Author page
            $type = 'ProfilePage';
        } elseif ( is_search() ) {
            // Search reslts
            $type = 'SearchResultsPage';
        } else {
            // Default
            $type = 'WebPage';
        }

        return $type;
    }

}

if ( !function_exists( 'leap_print_schema_org' ) ) {

    /**
     * Print schema type by post type
     */
    function leap_print_schema_org() {
        $schema = 'http://schema.org/';

        $type = leap_get_schema_org_type();

        if ( !empty( $type ) ) {
            echo 'itemscope="itemscope" itemtype="' . $schema . $type . '"';
        }
    }

}

if ( !function_exists( 'leap_get_wrapper_class' ) ) {

    /**
     * Get wrapper class based on layout selected from theme options 
     * @return string
     */
    function leap_get_wrapper_class() {
        $layout        = ot_get_option( 'layout', 'wide' );
        $wrapper_class = '';
        if ( $layout == 'boxed' ) {
            $wrapper_class = 'container';
        } elseif ( $layout == 'framed' ) {
            $wrapper_class = '';
        }    
        return $wrapper_class;
    }

}


if ( !function_exists( 'leap_get_assigned_menu_on_theme_location' ) ) {

    /**
     * Get assigned menu for theme location
     * @param string $theme_location
     * @return integer
     */
    function leap_get_assigned_menu_on_theme_location( $theme_location = '' ) {
        $menu_locations = get_nav_menu_locations();

        $menu_id = 0;
        if ( !empty( $menu_locations[ $theme_location ] ) ) {
            $menu_id     = $menu_locations[ $theme_location ];
            $menu_object = wp_get_nav_menu_object( $menu_id );
            if ( isset( $menu_object->count ) && $menu_object->count > 0 ) {
                return $menu_id;
            }
        }

        return $menu_id;
    }

}

if ( !function_exists( 'leap_bws_col_class' ) ) {

    /**
     * Get column class if responsive or not
     * @param int $cols
     * @return string
     */
    function leap_bws_col_class( $cols = 12 ) {
        $disable_responsive = ot_get_option( 'disable_responsive', 'no' );
        $leap_bws_col_class = 'col-sm-' . $cols;
        if ( $disable_responsive == 'yes' ) {
            $leap_bws_col_class = 'col-xs-' . $cols;
        }

        return $leap_bws_col_class;
    }

}

if ( !function_exists( 'leap_main_col_class' ) ) {

    /**
     * Get Main column class
     * @param type $main_col_class
     * @return type
     */
    function leap_main_col_class() {
        $main_col_class = leap_bws_col_class( 12 );
        $page_id        = leap_get_current_page_id();

        if ( is_page() && (get_page_template_slug( $page_id ) == 'page-100-width-template.php') ) {
            $main_col_class = leap_bws_col_class( 12 );
        } elseif ( leap_sidebar_position( $page_id ) != 'full' ) {
            $main_col_class = leap_bws_col_class( 9 );
        }

        return $main_col_class;
    }

}

/**
 * Set theme images sizes
 */
function leap_theme_image_sizes() {
    set_post_thumbnail_size( 50, 50, true );
    add_image_size( 'leap-thumbnail', 50, 50, true );
    add_image_size( 'leap-related-thumbnail', 190, 145, true );
    add_image_size( 'leap-slider', 1140, 450, true );

    add_image_size( 'leap-portfolio-grid', 567, 'auto', true );
    add_image_size( 'leap-portfolio-1col', 1140, 450, true );
    add_image_size( 'leap-portfolio-2col', 567, 378, true );
    add_image_size( 'leap-portfolio-3col', 567, 378, true );
    add_image_size( 'leap-portfolio-4col', 567, 378, true );
    add_image_size( 'leap-portfolio-cat', 567, 378, true );
    add_image_size( 'leap-portfolio-single', 1140, 'auto', true );
    add_image_size( 'leap-570x570', 570, 570, true );
}

add_action( 'after_setup_theme', 'leap_theme_image_sizes' );

// Enable shortcodes in widgets
add_filter( 'widget_text', 'do_shortcode' );

/**
 * add backwards compatibility for older version to render title
 */
if ( !function_exists( '_wp_render_title_tag' ) ) {

    function leap_render_title() {
        ?>
        <title><?php wp_title( '|', true, 'right' ); ?></title>
        <?php
    }

    add_action( 'wp_head', 'leap_render_title' );
}

if ( !function_exists( 'leap_get_content_count' ) ) {

    /**
     * Get post content by maximum number of characters to take.
     */
    function leap_get_content_count( $page_id, $count = 160 ) {
        $post_content     = leap_get_the_content( $page_id );
        return wp_html_excerpt( $post_content, $count, '...' );
    }

}

if ( !function_exists( 'leap_get_the_content' ) ) {

    /**
     * 
     * @param type $id
     * @return type
     */
    function leap_get_the_content( $id = null ) {

        if ( !$id ) {
            $id = get_the_ID();
        }

        $post    = get_post( $id );
        $content = apply_filters( 'the_content', $post->post_content );
        $content = str_replace( ']]>', ']]&gt;', $content );

        return $content;
    }

}

if ( !function_exists( 'leap_get_the_excerpt' ) ) {

    /**
     * 
     * @param type $id
     * @param type $wordsreturned
     * @return string
     */
    function leap_get_the_excerpt( $id = null, $wordsreturned = null ) {

        if ( has_excerpt() ) {
            return get_the_excerpt();
        }
        
        $content = wp_strip_all_tags( leap_get_the_content( $id ), true );

        if ( !$wordsreturned ) {
            $wordsreturned = apply_filters( 'excerpt_length', 55 );
        }

        $retval = $content;  //  Just in case of a problem
        $array  = explode( " ", $content );
        /*  Already short enough, return the whole thing */
        if ( count( $array ) <= $wordsreturned ) {
            $retval = $content;
        } else { /*  Need to chop of some words */

            $excerpt_more = apply_filters( 'excerpt_more', ' ' . '[&hellip;]' );

            array_splice( $array, $wordsreturned );
            $retval = implode( " ", $array ) . $excerpt_more;
        }
        return $retval;
    }

}

/**
 * Display SEO meta data
 */
function leap_seo_meta_data() {
    $post_types          = ot_get_option( 'enable_seo', array() );
    $queried_object      = get_queried_object();
    
    if ( defined( 'WPSEO_VERSION' ) || ( isset( $queried_object->ID ) && !in_array( get_post_type( $queried_object->ID ), $post_types )) ) {
        return true;
    }

    $taxonomy_names      = get_object_taxonomies( $post_types );
    $site_tagline        = get_bloginfo( 'description', 'display' );
    $general_description = ot_get_option( 'seo_description', '' );
    $keywords            = ot_get_option( 'seo_keywords', '' );
    $description         = '';
    
    if ( is_singular( $post_types ) ) {
        $leap_seo_description  = get_post_meta( $queried_object->ID, 'leap_seo_description', TRUE );
        $leap_post_description = leap_get_content_count( $queried_object->ID );
        if ( !empty( $leap_seo_description ) ) {
            $description = $leap_seo_description;
        } elseif ( !empty( $leap_post_description ) ) {
            $description = $leap_post_description;
        } elseif ( !empty( $general_description ) ) {
            $description = $general_description;
        } elseif ( !empty( $site_tagline ) ) {
            $description = $site_tagline;
        }

        $keywords = leap_general_post_option( 'seo_keywords', 'leap_seo_keywords' );
    }

    if ( isset( $queried_object->taxonomy ) && in_array( $queried_object->taxonomy, $taxonomy_names ) ) {
        $term_des    = do_shortcode( $queried_object->description );
        $description = wp_html_excerpt( $term_des, 160, '...' );
    }

    if ( !empty( $description ) ) {
        echo "\n" . '<meta name="description" content="' . esc_attr( $description ) . '" />';
    }
    if ( !empty( $keywords ) ) {
        echo "\n" . '<meta name="keywords" content="' . esc_attr( $keywords ) . '" />';
    }
}

add_action( 'wp_head', 'leap_seo_meta_data' );

/**
 * Add Open Graph meta data to your site's head section.
 */
function leap_add_facebook_og() {
    global $post;

    $facebook_og = ot_get_option( 'facebook_og', 'on' );

    if ( ( $facebook_og == 'on') && is_singular() ) {

        if ( has_post_thumbnail( $post->ID ) ) {
            $post_thumbnail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
            $og_image       = $post_thumbnail[ 0 ];
        } else {
            $og_image = ot_get_option( 'logo', '' );
        }
        $description = leap_get_content_count( $post->ID );
        if ( empty( $description ) ) {
            $description = get_bloginfo( 'description' );
        }

        $published_time_seconds = strtotime( $post->post_date_gmt );
        $published_time         = date( 'c', $published_time_seconds );

        $updated_time_seconds = strtotime( $post->post_modified_gmt );
        $updated_time         = date( 'c', $updated_time_seconds );
        ?>

        <meta property="og:title" content="<?php echo esc_attr( get_the_title( $post->ID ) ); ?>"/>
        <meta property="og:description" content="<?php echo esc_attr( $description ); ?>"/>
        <meta property="og:type" content="article"/>
        <meta property="og:url" content="<?php echo esc_attr( get_permalink( $post->ID ) ); ?>"/>
        <meta property="og:site_name" content="<?php echo esc_attr( get_bloginfo() ); ?>"/>
        <?php if ( !empty( $og_image ) ) { ?>
            <meta property="og:image" content="<?php echo esc_attr( $og_image ); ?>"/>
        <?php } ?>
        <meta property="article:author" content="<?php echo get_author_posts_url( $post->post_author ); ?>"/>
        <meta property="article:published_time" content="<?php echo $published_time ?>" />
        <meta property="article:modified_time" content="<?php echo $updated_time ?>" />
        <meta property="og:updated_time" content="<?php echo $updated_time ?>" />

        <?php
    }
}

add_action( 'wp_head', 'leap_add_facebook_og', 10 );

/**
 * Add Twitter card meta data to your site's head section.
 */
function leap_add_twitter_card() {
    global $post;

    $twitter_card = ot_get_option( 'twitter_card', 'on' );

    if ( ( $twitter_card == 'on') && is_singular() ) {

        if ( has_post_thumbnail( $post->ID ) ) {
            $post_thumbnail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'large' );
            $og_image       = $post_thumbnail[ 0 ];
        } else {
            $og_image = ot_get_option( 'logo', '' );
        }

        $description = leap_get_content_count( $post->ID );
        if ( empty( $description ) ) {
            $description = get_bloginfo( 'description' );
        }
        ?>
        <meta name="twitter:card" content="summary_large_image"/>
        <meta name="twitter:description" content="<?php echo esc_attr( $description ); ?>"/>
        <meta name="twitter:title" content="<?php echo esc_attr( get_the_title( $post->ID ) ); ?>"/>
        <meta name="twitter:url" content="<?php echo esc_attr( get_permalink( $post->ID ) ); ?>"/>
        <meta name="twitter:domain" content="<?php echo esc_attr( get_bloginfo() ); ?>"/>
        <?php if ( !empty( $og_image ) ) { ?>
            <meta name="twitter:image:src" content="<?php echo esc_attr( $og_image ); ?>"/>
        <?php } ?>
        <?php
    }
}

add_action( 'wp_head', 'leap_add_twitter_card', 10 );

/**
 * Schema.org markup for Google+
 */
function leap_add_google_plus_schema() {
    global $post;

    if ( is_single() ) {

        $description = leap_get_content_count( $post->ID );
        if ( empty( $description ) ) {
            $description = get_bloginfo( 'description' );
        }

        $published_time_seconds = strtotime( $post->post_date_gmt );
        $published_time         = date( 'c', $published_time_seconds );
        ?>
        <meta itemprop="description" content="<?php echo esc_attr( $description ); ?>">
        <meta itemprop="datePublished" content="<?php echo esc_attr( $published_time ); ?>">
        <?php
    }
}

add_action( 'wp_head', 'leap_add_google_plus_schema', 10 );

/**
 * Add itemprop attribute to images
 */
function leap_media_image_itemprop( $html, $post_id, $post_thumbnail_id, $size,
                                    $attr ) {
    if ( 'leap-portfolio-1col' == $size ) {
        return str_replace( 'src', ' itemprop="image" src', $html );
    }
    return $html;
}

add_filter( 'post_thumbnail_html', 'leap_media_image_itemprop', 10, 5 );

/**
 * FaceBook Comments application id
 */
function leap_fb_app_id() {
    $fb_app_id = ot_get_option( 'fb_app_id', '' );
    if ( !empty( $fb_app_id ) ) {
        echo "\n" . '<meta property="fb:app_id" content="' . esc_attr( $fb_app_id ) . '"/>';
    }
}

add_action( 'wp_head', 'leap_fb_app_id' );

/**
 * Google Webmaster Tools
 */
function leap_google_webmaster_tools() {
    $google_webmaster_tools = ot_get_option( 'google_webmaster_tools', '' );
    if ( !empty( $google_webmaster_tools ) ) {
        echo "\n" . '<meta name="google-site-verification" content="' . esc_attr( $google_webmaster_tools ) . '">';
    }
}

add_action( 'wp_head', 'leap_google_webmaster_tools' );

/**
 * Bing Webmaster Tools
 */
function leap_bing_webmaster_tools() {
    $bing_webmaster_tools = ot_get_option( 'bing_webmaster_tools', '' );
    if ( !empty( $bing_webmaster_tools ) ) {
        echo "\n" . '<meta name="msvalidate.01" content="' . esc_attr( $bing_webmaster_tools ) . '" />';
    }
}

add_action( 'wp_head', 'leap_bing_webmaster_tools' );

/**
 * Alexa Verification ID
 */
function leap_alexa_verification_id() {
    $alexa_verification_id = ot_get_option( 'alexa_verification_id', '' );
    if ( !empty( $alexa_verification_id ) ) {
        echo "\n" . '<meta name="alexaVerifyID" content="' . esc_attr( $alexa_verification_id ) . '" />';
    }
}

add_action( 'wp_head', 'leap_alexa_verification_id' );

/**
 * Add Google Analytics Code to head
 */
function leap_google_analytics_code() {
    $google_analytics_ua_code = ot_get_option( 'google_analytics_ua_code', '' );
    if ( !empty( $google_analytics_ua_code ) ) {
        ?>
        <script type="text/javascript">
            ( function ( i, s, o, g, r, a, m ) {
                i['GoogleAnalyticsObject'] = r;
                i[r] = i[r] || function () {
                    ( i[r].q = i[r].q || [ ] ).push( arguments )
                }, i[r].l = 1 * new Date();
                a = s.createElement( o ),
                    m = s.getElementsByTagName( o )[0];
                a.async = 1;
                a.src = g;
                m.parentNode.insertBefore( a, m )
            } )( window, document, 'script', '//www.google-analytics.com/analytics.js', 'ga' );

            ga( 'create', '<?php echo esc_attr( $google_analytics_ua_code ); ?>', 'auto' );
            ga( 'send', 'pageview' );

        </script>

        <?php
    }
}

add_action( 'wp_head', 'leap_google_analytics_code' );

/**
 * Go to top icon
 */
function leap_go_top_icon() {
    if ( ot_get_option( 'go_top', 'yes' ) == 'yes' ) {
        ?>
        <a class="scrollup" href="#"><span>Top</span></a>
        <script type="text/javascript">
            ( function ( $ ) {
                "use strict";
                // scroll-to-top button show and hide
                jQuery( document ).ready( function () {

                    'use strict';

                    jQuery( window ).scroll( function () {
                        if ( jQuery( this ).scrollTop() > 100 ) {
                            jQuery( '.scrollup' ).fadeIn();
                        } else {
                            jQuery( '.scrollup' ).fadeOut();
                        }
                    } );
                    // scroll-to-top animate
                    jQuery( '.scrollup' ).click( function () {
                        jQuery( "html, body" ).animate( { scrollTop: 0 }, 1500, 'easeInOutExpo' );
                        return false;
                    } );
                } );
            } )( jQuery );
        </script>
        <?php
    }
}

add_action( 'wp_footer', 'leap_go_top_icon' );

/**
 * Add header code to every page
 */
function leap_head_code() {
    echo ot_get_option( 'head_code', '' );
}

add_action( 'wp_head', 'leap_head_code' );

/**
 * Add page/post custom css the head of every page
 */
function leap_page_post_custom_css() {
    $page_id         = leap_get_current_page_id();
    $leap_custom_css = get_post_meta( $page_id, 'leap_custom_css', true );
    if ( $leap_custom_css ) {
        echo '<style type="text/css">';
        echo $leap_custom_css;
        echo '</style>';
    }
}

add_action( 'wp_head', 'leap_page_post_custom_css' );

/**
 * Add Footer code to every page
 */
function leap_footer_code() {
    echo ot_get_option( 'footer_code', '' );
}

add_action( 'wp_footer', 'leap_footer_code' );


if ( !function_exists( 'leap_get_background_value' ) ) {

    /**
     * Display background option for theme options
     * @param string $bg_setting
     * @return string
     */
    function leap_get_background_value( $bg_setting ) {
        $bg_setting = ot_get_option( $bg_setting, array() );
        $style      = '';

        if ( !empty( $bg_setting[ 'background-color' ] ) && isset( $bg_setting[ 'background-color' ] ) )
            $style .= 'background-color: ' . $bg_setting[ 'background-color' ] . ';';
        if ( !empty( $bg_setting[ 'background-repeat' ] ) && isset( $bg_setting[ 'background-repeat' ] ) )
            $style .= 'background-repeat: ' . $bg_setting[ 'background-repeat' ] . ';';
        if ( !empty( $bg_setting[ 'background-attachment' ] ) && isset( $bg_setting[ 'background-attachment' ] ) )
            $style .= 'background-attachment: ' . $bg_setting[ 'background-attachment' ] . ';';
        if ( !empty( $bg_setting[ 'background-position' ] ) && isset( $bg_setting[ 'background-position' ] ) )
            $style .= 'background-position: ' . $bg_setting[ 'background-position' ] . ';';
        if ( !empty( $bg_setting[ 'background-image' ] ) && isset( $bg_setting[ 'background-image' ] ) )
            $style .= 'background-image: url("' . $bg_setting[ 'background-image' ] . '");';
        if ( !empty( $bg_setting[ 'background-size' ] ) && isset( $bg_setting[ 'background-size' ] ) )
            $style .= 'background-size: ' . $bg_setting[ 'background-size' ] . ';';

        return $style;
    }

}

if ( !function_exists( 'leap_get_typography_value' ) ) {

    /**
     * Display typography option for theme options
     * @param string $typo_setting
     * @return string
     */
    function leap_get_typography_value( $typo_setting ) {
        $typo_setting = ot_get_option( $typo_setting, array() );
        $style        = '';

        if ( !empty( $typo_setting[ 'font-color' ] ) )
            $style .= 'color: ' . $typo_setting[ 'font-color' ] . ';';


        $families = leap_all_font_families();

        if ( !empty( $typo_setting[ 'font-family' ] ) && !empty( $families[ $typo_setting[ 'font-family' ] ] ) )
            $style .= 'font-family: ' . $families[ $typo_setting[ 'font-family' ] ] . ';';
        if ( !empty( $typo_setting[ 'font-size' ] ) )
            $style .= 'font-size: ' . $typo_setting[ 'font-size' ] . ';';
        if ( !empty( $typo_setting[ 'font-style' ] ) )
            $style .= 'font-style: ' . $typo_setting[ 'font-style' ] . ';';
        if ( !empty( $typo_setting[ 'font-variant' ] ) )
            $style .= 'font-variant: ' . $typo_setting[ 'font-variant' ] . ';';
        if ( !empty( $typo_setting[ 'font-weight' ] ) )
            $style .= 'font-weight: ' . $typo_setting[ 'font-weight' ] . ';';
        if ( !empty( $typo_setting[ 'letter-spacing' ] ) )
            $style .= 'letter-spacing: ' . $typo_setting[ 'letter-spacing' ] . ';';
        if ( !empty( $typo_setting[ 'line-height' ] ) )
            $style .= 'line-height: ' . $typo_setting[ 'line-height' ] . ';';
        if ( !empty( $typo_setting[ 'text-decoration' ] ) )
            $style .= 'text-decoration: ' . $typo_setting[ 'text-decoration' ] . ';';
        if ( !empty( $typo_setting[ 'text-transform' ] ) )
            $style .= 'text-transform: ' . $typo_setting[ 'text-transform' ] . ';';

        return $style;
    }

}

if ( !function_exists( 'leap_get_post_background_value' ) ) {

    /**
     * Display background option for post and/or page
     * @param type $post_option
     * @param type $page_id
     * @return string
     */
    function leap_get_post_background_value( $post_option, $page_id = NULL ) {
        if ( !$page_id ) {
            $page_id    = leap_get_current_page_id();
            $bg_setting = get_post_meta( $page_id, $post_option, TRUE );
        } else {
            $dafault_sb_style_id = ot_get_option( 'leap_default_sidebar_style', '' );
            $default_value       = get_post_meta( $dafault_sb_style_id, $post_option, TRUE );
            $post_value          = get_post_meta( $page_id, $post_option, TRUE );
            if ( !empty( $post_value ) ) {
                $bg_setting = $post_value;
            } else {
                $bg_setting = $default_value;
            }
        }

        $style = '';

        if ( !empty( $bg_setting[ 'background-color' ] ) )
            $style .= 'background-color: ' . $bg_setting[ 'background-color' ] . ';';
        if ( !empty( $bg_setting[ 'background-repeat' ] ) )
            $style .= 'background-repeat: ' . $bg_setting[ 'background-repeat' ] . ';';
        if ( !empty( $bg_setting[ 'background-attachment' ] ) )
            $style .= 'background-attachment: ' . $bg_setting[ 'background-attachment' ] . ';';
        if ( !empty( $bg_setting[ 'background-position' ] ) )
            $style .= 'background-position: ' . $bg_setting[ 'background-position' ] . ';';
        if ( !empty( $bg_setting[ 'background-image' ] ) )
            $style .= 'background-image: url("' . $bg_setting[ 'background-image' ] . '");';
        if ( !empty( $bg_setting[ 'background-size' ] ) )
            $style .= 'background-size: ' . $bg_setting[ 'background-size' ] . ';';

        return $style;
    }

}

if ( !function_exists( 'leap_get_post_typography_value' ) ) {

    /**
     * Display typography option for post and/or page
     * @param string $post_option
     * @param int $page_id
     * @return string
     */
    function leap_get_post_typography_value( $post_option, $page_id = NULL ) {
        if ( !$page_id ) {
            $page_id      = leap_get_current_page_id();
            $typo_setting = get_post_meta( $page_id, $post_option, TRUE );
        } else {
            $dafault_sb_style_id = ot_get_option( 'leap_default_sidebar_style', '' );
            $default_value       = get_post_meta( $dafault_sb_style_id, $post_option, TRUE );
            $post_value          = get_post_meta( $page_id, $post_option, TRUE );
            if ( !empty( $post_value ) ) {
                $typo_setting = $post_value;
            } else {
                $typo_setting = $default_value;
            }
        }

        $style = '';

        $families = leap_all_font_families();

        if ( !empty( $typo_setting[ 'font-color' ] ) )
            $style .= 'color: ' . $typo_setting[ 'font-color' ] . ';';
        if ( !empty( $typo_setting[ 'font-family' ] ) )
            $style .= 'font-family: ' . $families[ $typo_setting[ 'font-family' ] ] . ';';
        if ( !empty( $typo_setting[ 'font-size' ] ) )
            $style .= 'font-size: ' . $typo_setting[ 'font-size' ] . ';';
        if ( !empty( $typo_setting[ 'font-style' ] ) )
            $style .= 'font-style: ' . $typo_setting[ 'font-style' ] . ';';
        if ( !empty( $typo_setting[ 'font-variant' ] ) )
            $style .= 'font-variant: ' . $typo_setting[ 'font-variant' ] . ';';
        if ( !empty( $typo_setting[ 'font-weight' ] ) )
            $style .= 'font-weight: ' . $typo_setting[ 'font-weight' ] . ';';
        if ( !empty( $typo_setting[ 'letter-spacing' ] ) )
            $style .= 'letter-spacing: ' . $typo_setting[ 'letter-spacing' ] . ';';
        if ( !empty( $typo_setting[ 'line-height' ] ) )
            $style .= 'line-height: ' . $typo_setting[ 'line-height' ] . ';';
        if ( !empty( $typo_setting[ 'text-decoration' ] ) )
            $style .= 'text-decoration: ' . $typo_setting[ 'text-decoration' ] . ';';
        if ( !empty( $typo_setting[ 'text-transform' ] ) )
            $style .= 'text-transform: ' . $typo_setting[ 'text-transform' ] . ';';

        return $style;
    }

}

if ( !function_exists( 'leap_get_post_option_value' ) ) {

    /**
     * Get post option value from current page or widget position style
     * @param string $post_option
     * @param int $page_id
     * @return mixed
     */
    function leap_get_post_option_value( $post_option, $page_id = NULL ) {
        if ( !$page_id ) {
            $page_id = leap_get_current_page_id();
            return get_post_meta( $page_id, $post_option, TRUE );
        } else {
            $dafault_sb_style_id = ot_get_option( 'leap_default_sidebar_style', '' );
            $default_value       = get_post_meta( $dafault_sb_style_id, $post_option, TRUE );
            $post_value          = get_post_meta( $page_id, $post_option, TRUE );
            if ( !empty( $post_value ) ) {
                return $post_value;
            }
            return $default_value;
        }
    }

}


if ( !function_exists( 'leap_get_current_sidebar_style' ) ) {

    /**
     * Get current widget positions style
     * @return int
     */
    function leap_get_current_sidebar_style() {
        $sb_style_id         = '';
        $dafault_sb_style_id = ot_get_option( 'leap_default_sidebar_style', '' );
        $curret_page_id      = leap_get_current_page_id();
        $current_sb_style_id = get_post_meta( $curret_page_id, 'leap_sidebars_style', TRUE );
        if ( !empty( $current_sb_style_id ) ) {
            $sb_style_id = $current_sb_style_id;
        } elseif ( !empty( $dafault_sb_style_id ) ) {
            $sb_style_id = $dafault_sb_style_id;
        }
        return $sb_style_id;
    }

}

if ( !function_exists( 'leap_get_sidebar_option_with_default' ) ) {

    /**
     * Get sidebar option real value if it has a default option
     * @param string $option
     * @param string $page_id
     * @return mixed
     */
    function leap_get_sidebar_option_with_default( $option, $page_id ) {
        $dafault_sb_style_id = ot_get_option( 'leap_default_sidebar_style', '' );
        $default_value       = get_post_meta( $dafault_sb_style_id, $option, TRUE );
        $post_value          = get_post_meta( $page_id, $option, TRUE );
        if ( !empty( $post_value ) && ($post_value != 'default') ) {
            return $post_value;
        } elseif ( ($post_value == 'default') && !empty( $default_value ) && ($default_value != 'default') ) {
            return $default_value;
        } else {
            return '';
        }
    }

}


if ( !function_exists( 'leap_get_option_with_default' ) ) {

    /**
     * Get option real value if it has a default option
     * @param string $general
     * @param string $custom
     * @return mixed
     */
    function leap_get_option_with_default( $general, $custom ) {
        $general_value = ot_get_option( $general );

        $page_id = leap_get_current_page_id();

        $custom_value = get_post_meta( $page_id, $custom, TRUE );

        if ( (!empty( $custom_value )) && ($custom_value != 'default') ) {
            $general_value = $custom_value;
        }
        return $general_value;
    }

}


if ( !function_exists( 'leap_general_post_option' ) ) {

    /**
     * Get option value from page if set or from theme options
     * or get the general option from theme options
     * @param string Theme option id
     * @param string Page option id
     * @return string
     */
    function leap_general_post_option( $general, $custom, $page_id = NULL ) {
        $return_value = ot_get_option( $general );
        if ( !$page_id ) {
            $page_id = leap_get_current_page_id();
        }
        $custom_value = get_post_meta( $page_id, $custom, TRUE );

        if ( $custom_value ) {
            $return_value = $custom_value;
        }

        return $return_value;
    }

}

if ( !function_exists( 'leap_general_or_custom' ) ) {

    /**
     * Decide if this option has value from the current page or it has a value from the general theme options
     * @param string $general option name from theme options
     * @param string $custom option name from page/post
     * @param int $page_id
     * @return string
     */
    function leap_general_or_custom( $general, $custom, $page_id = NULL ) {
        if ( !$page_id ) {
            $page_id = leap_get_current_page_id();
        }
        $result       = 'general';
        $custom_value = get_post_meta( $page_id, $custom, TRUE );
        if ( (!empty( $custom_value )) && ($custom_value != 'default') ) {
            $result = 'custom';
        }
        return $result;
    }

}

/**
 * Remove WordPress generator
 */
function leap_wp_generator() {
    if ( ot_get_option( 'remove_generator', 'no' ) == 'yes' ) {
        remove_action( 'wp_head', 'wp_generator' );
    }
}

add_action( 'after_setup_theme', 'leap_wp_generator' );

/**
 * Change wordpress login logo
 */
function leap_wp_login_logo() {
    $wp_login_logo = ot_get_option( 'wp_login_logo', '' );
    if ( $wp_login_logo ) {
        echo "<style>
		body.login div#login h1 a {
			background-image: url('" . esc_url( $wp_login_logo ) . "');
		}
		</style>";
    }
}

add_action( "login_enqueue_scripts", "leap_wp_login_logo" );

/**
 * Change wordpress login logo link
 * @param string $url
 * @return string
 */
function leap_wp_login_logo_link( $url ) {
    $wp_login_logo_link = ot_get_option( 'wp_login_logo_link', '' );
    if ( $wp_login_logo_link ) {
        return esc_url( $wp_login_logo_link );
    }
    return $url;
}

add_filter( 'login_headerurl', 'leap_wp_login_logo_link' );

/**
 * Change wordpress login logo link title
 * @param string $title
 * @return string
 */
function leap_wp_login_logo_link_title( $title ) {
    $wp_login_logo_link_title = ot_get_option( 'wp_login_logo_link_title', '' );
    if ( $wp_login_logo_link_title ) {
        return esc_attr( $wp_login_logo_link_title );
    }
    return $title;
}

add_filter( 'login_headertitle', 'leap_wp_login_logo_link_title' );

/**
 * Add icon to menu item
 * @param string $item_title
 * @param int $item_id
 * @return string
 */
function leap_menu_the_title( $item_title, $item_id ) {
    
    $item_icon = get_post_meta( $item_id, '_menu_item_leap_megamenu_icon', true );
    
    if(!empty($item_icon)) {
        return "<i class='{$item_icon}'></i> ".$item_title;
    }
    
    return $item_title;
}

add_filter( 'the_title', 'leap_menu_the_title', 10, 2 );

/**
 * Change excerpt length
 * @param int $length
 * @return int
 */
function leap_excerpt_length( $length ) {
    $excerpt_length = ot_get_option( 'excerpt_length', 55 );
    return $excerpt_length;
}

add_filter( 'excerpt_length', 'leap_excerpt_length', 999 );

/**
 * Replace [...] string using Filters
 * @return string
 */
function new_excerpt_more() {
    return ot_get_option( 'excerpt_more_string', '[...]' );
}

add_filter( 'excerpt_more', 'new_excerpt_more' );

/**
 * Change admin footer text
 */
function change_admin_footer_text( $text ) {
    $admin_footer_text = ot_get_option( 'change_admin_footer_text', '' );
    if ( !empty( $admin_footer_text ) ) {
        $text = $admin_footer_text;
    }
    return $text;
}

add_filter( 'admin_footer_text', 'change_admin_footer_text', 10, 1 );

/**
 * Add custom class based on wide or boxed layout
 * @param array $classes
 * @return array
 */
function leap_theme_layout( $classes ) {
    $classes[] = ot_get_option( 'layout', 'wide' );
    return $classes;
}

add_filter( 'body_class', 'leap_theme_layout' );

/**
 * Add custom class based on theme skin(light / dark)
 * @param array $classes
 * @return array
 */
function leap_theme_skin( $classes ) {
    $classes[] = ot_get_option( 'theme_skin', 'light' ) . '-skin';
    return $classes;
}

add_filter( 'body_class', 'leap_theme_skin' );

/**
 * Add custom class to body tag for vertical header
 * @param array $classes
 * @return array
 */
function leap_sticky_header_body_class( $classes ) {
    /*
    $header_layout = leap_get_option_with_default( 'header_layout', 'leap_header_layout' );
    if ( !in_array( $header_layout, array( 'header7', 'header8', 'header9' ) ) ) {
        return $classes;
    }
    */
    $sticky_header = leap_get_option_with_default( 'sticky_header', 'leap_sticky_header' );
    if ( 'yes' == $sticky_header ) {
        $classes[] = 'sticky-header';
    }
    return $classes;
}

add_filter( 'body_class', 'leap_sticky_header_body_class' );

/**
 * Add custom class to body tag for vertical header
 * @param array $classes
 * @return array
 */
function leap_vertical_header_body_classes( $classes ) {
    $header_layout = leap_get_option_with_default( 'header_layout', 'leap_header_layout' );
    $classes[]     = $header_layout . '-layout';
    if ( in_array( $header_layout, array( 'header10', 'header11', 'header12', 'header13', 'header14' ) ) ) {
        $classes[] = 'vertical-header';
    }
    return $classes;
}

add_filter( 'body_class', 'leap_vertical_header_body_classes' );

/**
 * Add custom class when the page has full width template
 * @param array $classes
 * @return array
 */
function leap_100_page_template( $classes ) {
    $page_id = leap_get_current_page_id();
    if ( get_page_template_slug( $page_id ) == 'page-100-width-template.php' ) {
        $classes[] = 'page-full-width-template';
    }
    return $classes;
}

add_filter( 'body_class', 'leap_100_page_template' );

/**
 * add custom class when the page has an absolute header
 * @param array $classes
 * @return array
 */
function leap_header_position( $classes ) {
    $header_position = leap_get_option_with_default( 'header_position', 'leap_header_position' );
    if ( 'absolute' == $header_position ) {
        $classes[] = 'absolute-header';
    }
    return $classes;
}

add_filter( 'body_class', 'leap_header_position' );

/**
 * Add custom upload mimes for custom fonts
 * @param array $existing_mimes
 * @return array
 */
function leap_custom_upload_mimes( $existing_mimes = array() ) {
    $existing_mimes[ 'eot' ]   = 'application/vnd.ms-fontobject';
    $existing_mimes[ 'woff' ]  = 'application/x-font-woff';
    $existing_mimes[ 'woff2' ] = 'application/x-font-woff';
    $existing_mimes[ 'ttf' ]   = 'application/x-font-ttf';
    $existing_mimes[ 'svg' ]   = 'image/svg+xml';
    $existing_mimes[ 'ico' ]   = 'image/x-icon';
    return $existing_mimes;
}

add_filter( 'upload_mimes', 'leap_custom_upload_mimes' );

/**
 * Install custom fonts
 */
function leap_install_custom_fonts() {
    $fonts     = ot_get_option( 'custom_fonts', array() );
    $fonts_css = '';
    if ( !empty( $fonts ) ) {
        $fonts_css = '<style type="text/css">' . "\n";
        foreach ( $fonts as $font ) {
            $fonts_css .= '@font-face {' . "\n";
            $fonts_css .= 'font-family: ' . $font[ 'title' ] . ';' . "\n";
            $fonts_css .= 'src:url("' . esc_url( $font[ 'eot_file' ] ) . '");' . "\n";
            $fonts_css .= 'src:url("' . esc_url( $font[ 'eot_file' ] ) . '?#iefix") format("embedded-opentype"),' . "\n";
            $fonts_css .= 'url("' . esc_url( $font[ 'woff2_file' ] ) . '") format("woff2"),' . "\n";
            $fonts_css .= 'url("' . esc_url( $font[ 'woff_file' ] ) . '") format("woff"),' . "\n";
            $fonts_css .= 'url("' . esc_url( $font[ 'ttf_file' ] ) . '") format("truetype"),' . "\n";
            $fonts_css .= 'url("' . esc_url( $font[ 'svg_file' ] ) . '#' . $font[ 'title' ] . '") format("svg");' . "\n";
            $fonts_css .= '}' . "\n";
        }
        $fonts_css .= '</style>' . "\n";
    }
    echo $fonts_css;
}

add_action( 'wp_head', 'leap_install_custom_fonts' );

/**
 * 
 * @param string $form
 * @return string
 */
function leap_search_form( $form ) {
    $form = '<form method="get" class="form-stacked leap-search-form" action="' . home_url( '/' ) . '" >
		<div class="form-group">
		<input class="form-control" type="text" name="s" id="search" placeholder="' . __( "Search ...", "wiz" ) . '" value="' . get_search_query() . '" />
		</div>
		<input type="submit" class="pull-right btn btn-leap" value="' . __( "Search", "wiz" ) . '" />
	</form>';

    return $form;
}

add_filter( 'get_search_form', 'leap_search_form' );

/**
 * Modify comment form fields
 * @param type $fields
 * @return string
 */
function leap_modify_comment_form_fields( $fields ) {
    $commenter = wp_get_current_commenter();
    $req       = get_option( 'require_name_email' );

    $fields[ 'author' ] = '<div class="row">'
    . '<div class="' . leap_bws_col_class( 6 ) . '">'
    . '<div class="form-group">'
    . '<label class="sr-only" for="author">' . __( "Name", "wiz" ) . '</label>'
    . '<input type="text" name="author" id="author" class="form-control" value="' . esc_attr( $commenter[ 'comment_author' ] ) . '" placeholder="' . __( "Name (required)", "wiz" ) . '" size="22" ' . ( $req ? 'aria-required="true"' : '' ) . ' />'
    . '</div>'
    . '</div>';

    $fields[ 'email' ] = '<div class="' . leap_bws_col_class( 6 ) . '">'
    . '<div class="form-group">'
    . '<label class="sr-only" for="email">' . __( "Email", "wiz" ) . '</label>'
    . '<input type="text" name="email" id="email" class="form-control" value="' . esc_attr( $commenter[ 'comment_author_email' ] ) . '" placeholder="' . __( "Email (required)", "wiz" ) . '" size="22" ' . ( $req ? 'aria-required="true"' : '' ) . ' />'
    . '</div>'
    . '</div>'
    . '</div>';

    $fields[ 'url' ] = '<div class="row">'
    . '<div class="' . leap_bws_col_class( 12 ) . '">'
    . '<div class="form-group">'
    . '<label class="sr-only" for="url">' . __( "Website", "wiz" ) . '</label>'
    . '<input type="text" name="url" id="url" value="' . esc_attr( $commenter[ 'comment_author_url' ] ) . '" placeholder="' . __( "Website", "wiz" ) . '" size="22" class="form-control" /></div>'
    . '</div>'
    . '</div>';

    return $fields;
}

add_filter( 'comment_form_default_fields', 'leap_modify_comment_form_fields' );

/**
 * Add new contact fields to user profile
 * @param array $profile_fields
 * @return array
 */
function leap_user_contact_methods( $profile_fields ) {
    $profile_fields[ 'author_facebook' ] = 'Facebook';
    $profile_fields[ 'author_twitter' ]  = 'Twitter';
    $profile_fields[ 'author_linkedin' ] = 'LinkedIn';
    $profile_fields[ 'author_dribble' ]  = 'Dribble';
    $profile_fields[ 'author_gplus' ]    = 'Google+';

    return $profile_fields;
}

add_filter( 'user_contactmethods', 'leap_user_contact_methods' );

/**
 * alter main query for home page to exclude some categories
 */
function leap_exclude_category( $query ) {
    if ( $query->is_home() && $query->is_main_query() ) {
        $query->set( 'category__not_in', ot_get_option( 'blog_exclude_categories', array() ) );
    }
}

add_action( 'pre_get_posts', 'leap_exclude_category' );

/**
 * add class to read more link
 * @param string $more
 * @return string
 */
function leap_wrap_more_link( $more ) {
    return '<div class="more">' . $more . '</div>';
}

add_filter( 'the_content_more_link', 'leap_wrap_more_link' );

/**
 * set container class based on layout type
 * @return string
 */
function leap_get_container_class( $container_class ) {
    $layout = ot_get_option( 'layout', 'wide' );
    if ( $layout == 'boxed' ) {
        $container_class = 'container';
    } else {
        $container_class = 'container-fluid';
    }
    return $container_class;
}

add_filter( 'leap_container_class', 'leap_get_container_class' );


if ( !function_exists( 'leap_get_share_buttons' ) ) {

    /**
     * Display share button by name
     * @param string $button button name
     */
    function leap_get_share_buttons( $button ) {
        $url         = urlencode( get_permalink() );
        $title       = urlencode( get_the_title() );
        $content     = urlencode( get_the_excerpt() );
        $full_image  = wp_get_attachment_image_src( get_post_thumbnail_id(), 'large' );
        $share_image = ($full_image) ? urlencode( $full_image[ 0 ] ) : '';
        switch ( $button ) {
            case 'facebook':
                echo '<a data-toggle="tooltip" data-placement="top" href="https://www.facebook.com/sharer/sharer.php?u=' . $url . '" title="Facebook"><i class="fa fa-facebook"></i></a>';
                break;

            case 'twitter':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://twitter.com/share?text=' . $title . '&amp;url=' . $url . '" title="Twitter"><i class="fa fa-twitter"></i></a>';
                break;

            case 'linkedin':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://linkedin.com/shareArticle?mini=true&amp;url=' . $url . '&amp;title=' . $title . '" title="Linkedin"><i class="fa fa-linkedin"></i></a>';
                break;

            case 'reddit':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://reddit.com/submit?url=' . $url . '&amp;title=' . $title . '" title="Reddit"><i class="fa fa-reddit"></i></a>';
                break;

            case 'tumblr':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://www.tumblr.com/share/link?url=' . $url . '&amp;name=' . $title . '&amp;description=' . $content . '" title="Tumblr"><i class="fa fa-tumblr"></i></a>';
                break;

            case 'google-plus':
                echo '<a data-toggle="tooltip" data-placement="top" href="https://plus.google.com/share?url=' . $url . '" onclick="' . esc_attr( 'javascript:window.open(this.href,"", "menubar=no,toolbar=no,resizable=yes,scrollbars=yes,height=600,width=600");return false;' ) . '" title="Google+"><i class="fa fa-google-plus"></i></a>';
                break;

            case 'pinterest':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://pinterest.com/pin/create/button/?url=' . $url . '&amp;description=' . $title . '&amp;media=' . $share_image . '" title="Pinterest"><i class="fa fa-pinterest"></i></a>';
                break;

            case 'digg':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://www.digg.com/submit?url=' . $url . '" title="Digg"><i class="fa fa-digg"></i></a>';
                break;

            case 'stumbleupon':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://www.stumbleupon.com/submit?url=' . $url . '&amp;title=' . $title . '" title="Stumbleupon"><i class="fa fa-stumbleupon"></i></a>';
                break;

            case 'delicious':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://del.icio.us/post?url=' . $url . '&amp;title=' . $title . '" title="Delicious"><i class="fa fa-delicious"></i></a>';
                break;

            case 'email':
                echo '<a data-toggle="tooltip" data-placement="top" href="mailto:?Subject=' . $title . '&amp;Body=I%20saw%20this%20and%20thought%20of%20you!%20' . $url . '" title="E-mail" ><i class="fa fa-envelope"></i></a>';
                break;
            case 'vk':
                echo '<a data-toggle="tooltip" data-placement="top" href="http://vkontakte.ru/share.php?url=' . $url . '" title="VK"><i class="fa fa-vk"></i></a>';
                break;
        }
    }

}

if ( !function_exists( 'leap_entry_meta' ) ) {

    /**
     * Display meta data for posts
     * @param array $meta_settings what meta you need to display
     */
    function leap_entry_meta( $meta_settings = array() ) {
        if ( is_sticky() && in_array( 'sticky', $meta_settings ) && is_home() && !is_paged() ) {
            echo '<small class="sticky-post small"><i class="fa fa-thumb-tack"></i> ' . __( 'Sticky', 'wiz' ) . '</small>';
        }

        // Post author
        if ( in_array( 'author', $meta_settings ) ) {
            if ( 'post' == get_post_type() ) {
                echo '<small class="entry-author small"><i class="fa fa-user"></i> <a href="' . get_author_posts_url( get_the_author_meta( 'ID' ) ) . '" title="' . esc_attr( sprintf( __( 'View all posts by %s', 'wiz' ), esc_attr( get_the_author() ) ) ) . '">' . esc_html( get_the_author() ) . '</a></small>';
            }
        }

        //Date Meta
        if ( in_array( 'date', $meta_settings ) ) {
            echo ' <small class="entry-date small"><i class="fa fa-calendar"></i> <a href="' . get_permalink() . '">' . get_the_date() . '</a></small>';
        }

        //Categories Meta
        if ( in_array( 'categories', $meta_settings ) ) {
            // Translators: used between list items, there is a space after the comma.
            $categories_list = get_the_category_list( __( ', ', 'wiz' ) );
            if ( $categories_list ) {
                echo '<small class="entry-category small"><i class="fa fa-align-left"></i> ' . $categories_list . '</small>';
            }
        }

        if ( in_array( 'comments', $meta_settings ) ) {
            if ( comments_open() ) {
                echo '<small class="entry-comments small"><i class="fa fa-comments"></i> <a href="' . get_comments_link() . '">' . get_comments_number() . '</a></small>';
            }
        }

        edit_post_link( __( 'Edit This', 'wiz' ), '<small class="edit-link small"><i class="fa fa-pencil-square-o"></i> ', '</small>' );
    }

}

if ( !function_exists( 'leap_related_projects' ) ) {

    /**
     * Get related projects by post id
     * @param int $post_id Post id
     * @return \wp_query
     */
    function leap_related_projects( $post_id ) {

        $tags = wp_get_post_terms( $post_id, 'leap-portfolio-tag', array( "fields" => "ids" ) );

        $args  = array(
            'showposts'           => -1,
            'post__not_in'        => array( $post_id ),
            'posts_per_page'      => 20,
            'ignore_sticky_posts' => 0,
            'post_type'           => 'leap-portfolio',
            'tax_query'           => array(
                array(
                    'taxonomy' => 'leap-portfolio-tag',
                    'field'    => 'id',
                    'terms'    => $tags
                )
            )
        );
        $query = new wp_query( $args );

        return $query;
    }

}

/**
 * Add customizer support for custom header image 
 */
function leap_wp_custom_header() {
    if ( !empty( get_custom_header()->url ) ) {
        ?>
        <style type="text/css">
            #header1 .navbar-inner, #header2 .navbar-inner, #header3 .navbar-inner, #header4 .logo-block, #header5 .logo-block, #header6 .logo-block, #header7, #header8 {
                background-image: url('<?php echo get_custom_header()->url; ?>');
            }
        </style>
        <?php
    }
}

add_action( 'wp_head', 'leap_wp_custom_header' );

/**
 * Add page links on list
 * @param string $link
 * @return string
 */
function leap_wp_link_pages_link( $link ) {
    return '<li>' . $link . '</li>';
}

add_filter( 'wp_link_pages_link', 'leap_wp_link_pages_link' );

/**
 * Replace logo with the retina version
 */
function leap_retina_src_images() {
    ?>

    <script type="text/javascript">
        ( function ( $ ) {
            "use strict";

            function setCookie( key, value ) {
                var expires = new Date();
                expires.setTime( expires.getTime() + ( 365 * 24 * 60 * 60 * 1000 ) );
                document.cookie = key + '=' + value + ';expires=' + expires.toUTCString();
            }

            function getCookie( key ) {
                var keyValue = document.cookie.match( '(^|;) ?' + key + '=([^;]*)(;|$)' );
                return keyValue ? keyValue[2] : null;
            }


            if ( getCookie( 'isRetina' ) == null ) {
                var isRetina = (
                    window.devicePixelRatio > 1 ||
                    ( window.matchMedia && window.matchMedia( "(-webkit-min-device-pixel-ratio: 1.5),(-moz-min-device-pixel-ratio: 1.5),(min-device-pixel-ratio: 1.5)" ).matches )
                    );

                setCookie( 'isRetina', isRetina );
            }


    <?php
    $logo_retina            = ot_get_option( 'logo_retina', '' );
    $logo_retina_dimensions = ot_get_option( 'logo_retina_dimensions', array() );
    $logo_retina_width      = (!empty( $logo_retina_dimensions[ 'width' ] )) ? $logo_retina_dimensions[ 'width' ] : '161';
    $logo_retina_height     = (!empty( $logo_retina_dimensions[ 'height' ] )) ? $logo_retina_dimensions[ 'height' ] : '63';
    ?>
            var logo_retina = '<?php echo $logo_retina; ?>';
            var logo_retina_width = '<?php echo $logo_retina_width; ?>';
            var logo_retina_height = '<?php echo $logo_retina_height; ?>';

            function retina_logo_dimensions() {
                if ( !jQuery( '.header .navbar-fixed-top' ).hasClass( "fixed-header" ) ) {
                    jQuery( '.logo img' ).css( {
                        "max-width": logo_retina_width + "px",
                        "max-height": logo_retina_height + ">px"
                    } );
                } else {
                    jQuery( '.logo img' ).css( {
                        "max-width": "",
                        "max-height": ""
                    } );
                }
            }

            jQuery( document ).ready( function ( $ ) {

                if ( logo_retina && logo_retina_width && logo_retina_height ) {
                    if ( getCookie( 'isRetina' ) === 'true' ) {

                        jQuery( '.logo img' ).attr( 'src', logo_retina );
                        retina_logo_dimensions();
                        jQuery( document ).scroll( function () {
                            retina_logo_dimensions();
                        } );

                    }
                }

            } );

        } )( jQuery );
    </script>
    <?php
}

add_action( 'wp_head', 'leap_retina_src_images' );

/**
 * Edit the password protect form
 * @return string
 */
function leap_password_form() {
    global $post;
    $label = 'pwbox-' . ( empty( $post->ID ) ? rand() : $post->ID );
    $form  = '<form action="' . esc_url( site_url( 'wp-login.php?action=postpass', 'login_post' ) ) . '" method="post">
    ' . __( "To view this protected post, enter the password below:", "wiz" ) . '
    <label for="' . $label . '">' . __( "Password:", "wiz" ) . ' </label><input name="post_password" id="' . $label . '" type="password" size="20" maxlength="20" /><div class="form-group"></div><input type="submit" name="Submit" class="btn btn-default btn-leap" value="' . esc_attr__( "Submit", "wiz" ) . '" />
    </form>
    ';
    return $form;
}

add_filter( 'the_password_form', 'leap_password_form' );

/**
 * Change dashboard portfolio menu name
 * @param string $text
 * @return string
 */
function leap_port_portfolio_name( $text ) {
    return ot_get_option( 'portfolio_menu_name', $text );
}

add_filter( 'leap_port_portfolio_name', 'leap_port_portfolio_name' );

/**
 * Change dashboard portfolio slug name
 * @param string $text
 * @return string
 */
function leap_port_portfolio_slug( $text ) {
    return ot_get_option( 'portfolio_url_slug', $text );
}

add_filter( 'leap_port_portfolio_slug', 'leap_port_portfolio_slug' );

/**
 * Change Portfolio Category Slug
 * @param string $text
 * @return string
 */
function leap_port_portfolio_category_slug( $text ) {
    return ot_get_option( 'portfolio_category_slug', $text );
}

add_filter( 'leap_port_portfolio_category_slug', 'leap_port_portfolio_category_slug' );

/**
 * Change Portfolio Tag Slug
 * @param string $text
 * @return string
 */
function leap_port_portfolio_tag_slug( $text ) {
    return ot_get_option( 'portfolio_tag_slug', $text );
}

add_filter( 'leap_port_portfolio_tag_slug', 'leap_port_portfolio_tag_slug' );


if ( !function_exists( 'leap_search_multidimensional_array' ) ) {

    /**
     * Search Multidimensional array
     * 
     * @param string $value
     * @param string $key
     * @param array $array
     * @return array
     */
    function leap_search_multidimensional_array( $value, $key, $array ) {
        $filttered_array = array();
        foreach ( $array as $k => $val ) {
            if ( $val[ $key ] == $value ) {
                $filttered_array[ $k ] = $val;
            }
        }
        return $filttered_array;
    }

}

if ( !function_exists( 'leap_portfolio_field_type' ) ) {

    /**
     * Search Multidimensional array
     * 
     * @param string $value
     * @param string $key
     * @param array $array
     * @return array
     */
    function leap_portfolio_field_type( $portfolio_field = array(), $post_id = 0 ) {
        $type                  = $portfolio_field[ 'type' ];
        $portfolio_field_value = get_post_meta( $post_id, $portfolio_field[ 'id' ], true );
        $value                 = '';

        switch ( $type ) {
            case 'leap_link':
                $link_text = (!empty( $portfolio_field_value[ "text" ] )) ? esc_attr( $portfolio_field_value[ "text" ] ) : __( "Visit Project", "wiz" );
                $link_url  = (!empty( $portfolio_field_value[ "url" ] )) ? esc_url( $portfolio_field_value[ "url" ] ) : '';

                if ( !empty( $link_url ) ) {
                    $value = '<a href="' . $link_url . '">' . $link_text . '</a>';
                }
                break;
            case 'textarea-simple':
                $value = $portfolio_field_value;
                break;
            default:
                $value = esc_attr( $portfolio_field_value );
                break;
        }
        return $value;
    }

}

/**
 * Sanitize hex color
 * @param string $color
 * @return string
 */
function leap_sanitize_hex_color( $color ) {
    if ( '' === $color )
        return '';

    // 3 or 6 hex digits, or the empty string.
    if ( preg_match( '|^#([A-Fa-f0-9]{3}){1,2}$|', $color ) )
        return $color;

    return null;
}

if ( !function_exists( 'leap_hex2rgba' ) ) {

    /**
     * Convert hexdec color string to rgb(a) string
     * @param string $color
     * @param real $opacity
     * @param bol $echo
     * @return string
     */
    function leap_hex2rgba( $color, $opacity = false, $echo = false ) {

        $default = 'rgb(0,0,0)';

        //Return default if no color provided
        if ( empty( $color ) ) {
            return $default;
        }

        //Sanitize $color if "#" is provided 
        if ( $color[ 0 ] == '#' ) {
            $color = substr( $color, 1 );
        }

        //Check if color has 6 or 3 characters and get values
        if ( strlen( $color ) == 6 ) {
            $hex = array( $color[ 0 ] . $color[ 1 ], $color[ 2 ] . $color[ 3 ], $color[ 4 ] . $color[ 5 ] );
        } elseif ( strlen( $color ) == 3 ) {
            $hex = array( $color[ 0 ] . $color[ 0 ], $color[ 1 ] . $color[ 1 ], $color[ 2 ] . $color[ 2 ] );
        } else {
            return $default;
        }

        //Convert hexadec to rgb
        $rgb = array_map( 'hexdec', $hex );

        //Check if opacity is set(rgba or rgb)
        if ( $opacity ) {
            if ( abs( $opacity ) > 1 ) {
                $opacity = 1.0;
            }
            if ( $echo ) {
                $output = 'rgba(' . implode( ",", $rgb ) . ',' . $opacity . ')';
            } else {
                $rgb[]  = $opacity;
                $output = $rgb;
            }
        } else {
            if ( $echo ) {
                $output = 'rgb(' . implode( ",", $rgb ) . ')';
            } else {
                $output = $rgb;
            }
        }

        //Return rgb(a) color string
        return $output;
    }

}

if ( !function_exists( 'leap_color_brightness' ) ) {

    /**
     * Change color brightness to be darker or lighter
     * @param string $hex color hex
     * @param float $percent brightness percent from 0 to 1
     * @param string $brightness light or dark
     * @return string the new generated color hex
     */
    function leap_color_brightness( $hex, $percent, $brightness = 'light' ) {

        if ( $brightness == 'dark' ) {
            $percent = $percent * -1;
        }

        $rgb = leap_hex2rgba( $hex );
        //// CALCULATE 
        for ( $i = 0; $i < 3; $i++ ) {
            // See if brighter or darker
            if ( $percent > 0 ) {
                // Lighter
                $rgb[ $i ] = round( $rgb[ $i ] * $percent ) + round( 255 * (1 - $percent) );
            } else {
                // Darker
                $positivePercent = $percent - ($percent * 2);
                $rgb[ $i ]       = round( $rgb[ $i ] * $positivePercent ) + round( 0 * (1 - $positivePercent) );
            }
            // In case rounding up causes us to go to 256
            if ( $rgb[ $i ] > 255 ) {
                $rgb[ $i ] = 255;
            }
        }
        //// RBG to Hex
        $new_hex = '#';
        for ( $i = 0; $i < 3; $i++ ) {
            // Convert the decimal digit to hex
            $hexDigit = dechex( $rgb[ $i ] );
            // Add a leading zero if necessary
            if ( strlen( $hexDigit ) == 1 ) {
                $hexDigit = "0" . $hexDigit;
            }
            // Append to the hex string
            $new_hex .= $hexDigit;
        }
        return $new_hex;
    }

}

function leap_opn_excludes( $excludes ) {
    if ( !empty( $excludes ) ) {
        $excludes .= ', ';
    }

    $excludes .= '.portfolio-tabs li a, a.scrollup';
    return $excludes;
}

add_filter( 'opn_excludes', 'leap_opn_excludes' );

if ( function_exists( 'set_revslider_as_theme' ) ) {

    /**
     * Set Slider Revolution as Theme
     */
    function leap_set_revslider_as_theme() {
        set_revslider_as_theme();
    }

    add_action( 'init', 'leap_set_revslider_as_theme' );
}



if ( function_exists( 'set_ess_grid_as_theme' ) ) {

    /**
     * Set Essential Grid as Theme
     */
    function leap_set_ess_grid_as_theme() {
        set_ess_grid_as_theme();
    }

    add_action( 'init', 'leap_set_ess_grid_as_theme' );
}


if ( function_exists( 'vc_set_as_theme' ) ) {

    /**
     * Set Visual Composer to work as a theme.
     */
    function leap_vc_set_as_theme() {
        vc_set_as_theme( true );
    }

    add_action( 'vc_before_init', 'leap_vc_set_as_theme' );
}

/**
 * Remove vc admin notice license activation
 */
function leap_remove_vc_admin_notice_license_activation() {
    leap_remove_anonymous_object_filter( 'admin_notices', 'Vc_License', 'adminNoticeLicenseActivation' );
}

add_filter( 'admin_notices', 'leap_remove_vc_admin_notice_license_activation', 0 );

/**
 * Disable layerslider auto-updates
 */
function leap_layerslider_disable_update() {
    $GLOBALS[ 'lsAutoUpdateBox' ] = false;
}

add_action( 'layerslider_ready', 'leap_layerslider_disable_update' );

/**
 * Remove Ultimate Addons plugin activation notices
 */
define( 'BSF_6892199_NOTICES', false );

/**
 * Remove One Page Navigator plugin activation notices
 */
define( 'BSF_10766943_NOTICES', false );

/**
 * Shortpixel Affiliate Code
 */
define('SHORTPIXEL_AFFILIATE_CODE', 'affCODE');