<?php

if ( !defined( 'ABSPATH' ) ) {
    die( 'No direct access allowed' );
}

if ( !class_exists( 'Leap_Mega_Menu_Walker' ) ) {

    /**
     * Custom Walker
     *
     * @access      public
     * @since       1.0 
     * @return      void
     */
    class Leap_Mega_Menu_Walker extends Walker_Nav_Menu {

        /**
         * @var string $menu_megamenu_status are we currently rendering a mega menu?
         */
        private $menu_megamenu_status = "";

        /**
         * @var int $num_of_columns how many columns should the mega menu have?
         */
        private $num_of_columns = 4;

        /**
         * @var int $max_num_of_columns mega menu allow for 4 columns at max
         */
        private $max_num_of_columns = 6;

        /**
         * @var string $menu_megamenu_title should a colum title be displayed?
         */
        private $menu_megamenu_title = '';

        /**
         * @var string $menu_megamenu_icon does the item have an icon?
         */
        private $menu_megamenu_icon = '';

        /**
         * @see Walker::start_lvl()
         * @since 3.0.0
         *
         * @param string $output Passed by reference. Used to append additional content.
         * @param int $depth Depth of page. Used for padding.
         */
        public function start_lvl( &$output, $depth = 0, $args = array() ) {
            $indent = str_repeat( "\t", $depth );

            if ( $depth === 0 && $this->menu_megamenu_status == "enabled" ) {

                $fullwidth = apply_filters( 'leap_container_class', 'container-fluid' ) . ' menu-fullwidth';

                $output .= "\n$indent<div class='sf-mega $fullwidth'>\n";
                $output .= "\n$indent<div class='row'>\n";
            } elseif ( $depth >= 2 && $this->menu_megamenu_status == "enabled" ) {
                $output .= "\n$indent<ul class=\"sub-menu deep-level\">\n";
            } else {
                $output .= "\n$indent{mega_section_title_end}<ul class=\"sub-menu\">\n";
            }
        }

        /**
         * @see Walker::end_lvl()
         * @since 3.0.0
         *
         * @param string $output Passed by reference. Used to append additional content.
         * @param int $depth Depth of page. Used for padding.
         */
        public function end_lvl( &$output, $depth = 0, $args = array() ) {
            $indent = str_repeat( "\t", $depth );

            if ( $depth === 0 && $this->menu_megamenu_status == "enabled" ) {
                $output .= "$indent</div>\n";
                $output .= "$indent</div>\n";
            } else {
                $output .= "$indent</ul>\n";
            }
        }

        /**
         * Start the element output.
         *
         * @see Walker::start_el()
         *
         * @since 3.0.0
         *
         * @param string $output Passed by reference. Used to append additional content.
         * @param object $item   Menu item data object.
         * @param int    $depth  Depth of menu item. Used for padding.
         * @param array  $args   An array of arguments. @see wp_nav_menu()
         * @param int    $id     Current item ID.
         */
        public function start_el( &$output, $item, $depth = 0, $args = array(),
                                  $id = 0 ) {
            $indent = ( $depth ) ? str_repeat( "\t", $depth ) : '';



            /* set some vars */
            if ( $depth === 0 ) {
                $this->menu_megamenu_status = get_post_meta( $item->ID, '_menu_item_leap_megamenu_status', true );
                $this->num_of_columns       = get_post_meta( $item->ID, '_menu_item_leap_megamenu_columns', true );
            }

            $this->menu_megamenu_title = get_post_meta( $item->ID, '_menu_item_leap_megamenu_title', true );


            $classes   = empty( $item->classes ) ? array() : (array) $item->classes;
            $classes[] = 'menu-item-' . $item->ID;


            if ( $depth == 0 && $this->menu_megamenu_status == "enabled" ) {
                $classes[] = 'menu-item-has-megamenu';
                $classes[] = 'menu-item-has-megamenu-fullwidth';
            } elseif ( $depth == 0 && $this->menu_megamenu_status != "enabled" ) {
                $classes[] = 'menu-item-no-megamenu';
            }


            /**
             * Filter the CSS class(es) applied to a menu item's <li>.
             *
             * @since 3.0.0
             *
             * @see wp_nav_menu()
             *
             * @param array  $classes The CSS classes that are applied to the menu item's <li>.
             * @param object $item    The current menu item.
             * @param array  $args    An array of wp_nav_menu() arguments.
             */
            $class_names = join( ' ', apply_filters( 'nav_menu_css_class', array_filter( $classes ), $item, $args ) );
            $class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';

            /**
             * Filter the ID applied to a menu item's <li>.
             *
             * @since 3.0.1
             *
             * @see wp_nav_menu()
             *
             * @param string $menu_id The ID that is applied to the menu item's <li>.
             * @param object $item    The current menu item.
             * @param array  $args    An array of wp_nav_menu() arguments.
             */
            $id = apply_filters( 'nav_menu_item_id', 'menu-item-' . $item->ID, $item, $args );
            $id = $id ? ' id="' . esc_attr( $id ) . '"' : '';

            // Leap edits start

            if ( $depth == 1 && $this->menu_megamenu_status == "enabled" ) {

                if ( $this->num_of_columns == 'auto' ) {
                    $num_of_columns = $this->max_num_of_columns;
                } elseif ( $this->num_of_columns < $this->max_num_of_columns ) {
                    $num_of_columns = $this->num_of_columns;
                } else {
                    $num_of_columns = $this->max_num_of_columns;
                }

                if ( $num_of_columns == 5 ) {
                    $col_span = '5th';
                } else {
                    $col_span = 12 / $num_of_columns;
                }

                $output .= $indent . '<div class="col-sm-' . $col_span . '">'; // sf-mega-section
                if ( $this->menu_megamenu_title == 'disabled' ) {
                    $output .= '<style>.leap_col_title-' . $item->ID . '.mega-section-head {display:none;}</style>';
                }
                $output .= '<span class="leap_col_title-' . $item->ID . ' mega-section-head">';
            } else {
                $output .= $indent . '<li ' . $id . $class_names . '>';
            }


            if ( $depth == 2 && $this->menu_megamenu_status == "enabled" ) {
                $output = str_replace( "{mega_section_title_end}", "</span>", $output );
            } else {
                $output = str_replace( "{mega_section_title_end}", "", $output );
            }


            // Leap edits end

            $atts             = array();
            $atts[ 'title' ]  = !empty( $item->attr_title ) ? $item->attr_title : '';
            $atts[ 'target' ] = !empty( $item->target ) ? $item->target : '';
            $atts[ 'rel' ]    = !empty( $item->xfn ) ? $item->xfn : '';
            $atts[ 'href' ]   = !empty( $item->url ) ? $item->url : '';

            /**
             * Filter the HTML attributes applied to a menu item's <a>.
             *
             * @since 3.6.0
             *
             * @see wp_nav_menu()
             *
             * @param array $atts {
             *     The HTML attributes applied to the menu item's <a>, empty strings are ignored.
             *
             *     @type string $title  Title attribute.
             *     @type string $target Target attribute.
             *     @type string $rel    The rel attribute.
             *     @type string $href   The href attribute.
             * }
             * @param object $item The current menu item.
             * @param array  $args An array of wp_nav_menu() arguments.
             */
            $atts = apply_filters( 'nav_menu_link_attributes', $atts, $item, $args );

            $attributes = '';
            foreach ( $atts as $attr => $value ) {
                if ( !empty( $value ) ) {
                    $value = ( 'href' === $attr ) ? esc_url( $value ) : esc_attr( $value );
                    $attributes .= ' ' . $attr . '="' . $value . '"';
                }
            }

            $item_output = $args->before;
            $item_output .= '<a' . $attributes . '>';

            /** This filter is documented in wp-includes/post-template.php */
            $item_output .= $args->link_before . apply_filters( 'the_title', $item->title, $item->ID ) . $args->link_after;

            if ( !empty( $item->description ) ) {
                $item_output .= '<span class="menu-item-description">' . $item->description . '</span>';
            }

            $item_output .= '</a>';
            $item_output .= $args->after;

            /**
             * Filter a menu item's starting output.
             *
             * The menu item's starting output only includes $args->before, the opening <a>,
             * the menu item's title, the closing </a>, and $args->after. Currently, there is
             * no filter for modifying the opening and closing <li> for a menu item.
             *
             * @since 3.0.0
             *
             * @see wp_nav_menu()
             *
             * @param string $item_output The menu item's starting HTML output.
             * @param object $item        Menu item data object.
             * @param int    $depth       Depth of menu item. Used for padding.
             * @param array  $args        An array of wp_nav_menu() arguments.
             */
            $output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $item, $depth, $args );
        }

        /**
         * @see Walker::end_el()
         *
         * @param string $output Passed by reference. Used to append additional content.
         * @param object $item Page data object. Not used.
         * @param int $depth Depth of page. Not Used.
         */
        function end_el( &$output, $item, $depth = 0, $args = array() ) {

            if ( $depth == 1 && $this->menu_megamenu_status == "enabled" ) {
                $output .= '</div>';
            } else {
                $output .= "</li>\n";
            }

            //$output .= "</li>\n";
        }

        /**
         * Menu Fallback
         * =============
         * If this function is assigned to the wp_nav_menu's fallback_cb variable
         * and a manu has not been assigned to the theme location in the WordPress
         * menu manager the function with display nothing to a non-logged in user,
         * and will add a link to the WordPress menu manager if logged in as an admin.
         *
         * @param array $args passed from the wp_nav_menu function.
         *
         */
        public static function fallback( $args ) {
            if ( current_user_can( 'manage_options' ) ) {

                extract( $args );

                $fb_output = null;

                /* if ( $container ) {
                  $fb_output = '<' . $container;

                  if ( $container_id )
                  $fb_output .= ' id="' . $container_id . '"';

                  if ( $container_class )
                  $fb_output .= ' class="' . $container_class . '"';

                  $fb_output .= '>';
                  }

                  $fb_output .= '<ul';

                  if ( $menu_id )
                  $fb_output .= ' id="' . $menu_id . '"';

                  if ( $menu_class )
                  $fb_output .= ' class="' . $menu_class . '"';

                  $fb_output .= '>';
                  $fb_output .= '<li><a href="' . admin_url( 'nav-menus.php' ) . '">Add a menu</a></li>';
                  $fb_output .= '</ul>';
                 */
                return $fb_output;
            }
        }

    }

}