<?php

if ( !defined( 'ABSPATH' ) ) {
    die( 'No direct access allowed' );
}

// Required: include OptionTree.
require get_template_directory() . '/functions/main/ot-loader.php';

// Leap option types
require get_template_directory() . '/functions/option-types.php';

// Theme Options
require get_template_directory() . '/functions/theme-options.php';

// Filter Theme Options
require get_template_directory() . '/functions/filter-options.php';

// Validate Theme Options
require get_template_directory() . '/functions/validate-options.php';

/**
 * Set theme version
 */
function leap_set_version() {

    if ( !current_user_can( 'switch_themes' ) ) {
        return;
    }

    $nonce = (isset( $_REQUEST[ 'leap-theme-update-nonce' ] )) ? $_REQUEST[ 'leap-theme-update-nonce' ] : '';

    if ( !wp_verify_nonce( $nonce, 'display-update-notice' ) ) {
        return;
    }

    $version = get_option( 'leap_wiz_version', '' );

    if ( version_compare( $version, THEME_VERSION ) == 0 ) {
        return;
    }

    if ( isset( $_REQUEST[ 'action' ] ) && $_REQUEST[ 'action' ] == 'leap-theme-update' ) {
        require_once get_template_directory() . '/functions/upgrade.php';
        do_action( 'leap_theme_upgrade', $version, THEME_VERSION );
        update_option( 'leap_wiz_version', THEME_VERSION );
    }
}

add_action( 'init', 'leap_set_version' );

/**
 * Enqueue LeapPanel styles
 */
function leappanel_admin_styles() {
    // LeapPanel
    wp_enqueue_style( 'leappanel', get_template_directory_uri() . '/functions/css/leappanel-admin.css', array( 'ot-admin-css' ), THEME_VERSION );

    // Font Awesome 
    wp_enqueue_style( 'leap-font-awesome', get_template_directory_uri() . '/vendor/font-awesome/css/font-awesome.min.css', array(), THEME_VERSION, 'screen' );

    // fontawesome iconpicker
    wp_enqueue_style( 'leap-fontawesome-iconpicker', get_template_directory_uri() . '/vendor/fontawesome-iconpicker/css/fontawesome-iconpicker.min.css', array(), THEME_VERSION );
}

add_action( 'admin_enqueue_scripts', 'leappanel_admin_styles' );

/**
 * Add custom code to admin haed
 */
function leap_admin_head() {
    echo '<style>
    .bg-primary {
        color: #fff;
        background-color: #337ab7;
    }
    a.bg-primary:hover,
    a.bg-primary:focus {
        background-color: #286090;
    }
  </style>';
}

add_action( 'admin_head', 'leap_admin_head' );

/**
 * Enqueue LeapPanel styles
 */
function leappanel_admin_script() {
    wp_enqueue_script( 'jquery' );
    wp_enqueue_script( 'jquery-ui-core' );
    wp_enqueue_script( 'jquery-ui-accordion' );

    if ( !empty( $_GET[ 'page' ] ) && ( $_GET[ 'page' ] == apply_filters( 'ot_theme_options_menu_slug', 'leap-theme-options' ) ) ) {
        wp_enqueue_script( 'imgpreview', get_template_directory_uri() . '/functions/js/imgpreview.full.jquery.js', array(), '0.22', true );
    }

    wp_enqueue_script( 'leap-fontawesome-iconpicker', get_template_directory_uri() . '/vendor/fontawesome-iconpicker/js/fontawesome-iconpicker.min.js', array(), THEME_VERSION, true );

    // LeapPanel
    wp_register_script( 'leappanel', get_template_directory_uri() . '/functions/js/leappanel-admin.js', array(), THEME_VERSION, true );
    // Localize the script with new data
    $translation_array = array(
        'blog_id'  => get_current_blog_id(),
        'ajax_url' => admin_url( 'admin-ajax.php' )
    );
    wp_localize_script( 'leappanel', 'translations', $translation_array );
    wp_enqueue_script( 'leappanel' );
}

add_action( 'admin_enqueue_scripts', 'leappanel_admin_script' );

/**
 * Run a filter and set to true to enable OptionTree theme mode.
 */
add_filter( 'ot_theme_mode', '__return_true' );

/**
 * Run a filter and set to false to disable OptionTree child theme mode.
 */
add_filter( 'ot_child_theme_mode', '__return_false' );

/**
 * Run a filter and set to false if you don't want to load the
 * settings & documentation pages in the admin area of WordPress.
 */
add_filter( 'ot_show_pages', '__return_false' );

/**
 * Run a filter and set to false if you want to hide the
 * Theme Options UI page in the admin area of WordPress.
 */
add_filter( 'ot_show_options_ui', '__return_false' );

/**
 * Run a filter and set to false if you want to hide the
 * Settings Import options on the Import page.
 */
add_filter( 'ot_show_settings_import', '__return_true' );

/**
 * Run a filter and set to false if you want to hide the
 * Settings Import options on the Import page.
 */
add_filter( 'ot_show_settings_export', '__return_true' );

/**
 * Run a filter and set to false if you don't want to show the
 * "New Layout" section at the top of the theme options page.
 */
add_filter( 'ot_show_new_layout', '__return_false' );

/**
 * Run a filter and set to false if you want to hide the Documentation.
 */
add_filter( 'ot_show_docs', '__return_false' );

/**
 * Run a filter and set to true if you want OptionTree 
 * to load meta boxes for post formats.
 */
add_filter( 'ot_post_formats', '__return_true' );

/**
 * Theme Options ID
 * @return string
 */
function leap_theme_options_id() {
    return 'leap_wiz_options';
}

add_filter( 'ot_options_id', 'leap_theme_options_id', 10 );

/**
 * Theme Settings ID
 * @return string
 */
function leap_theme_settings_id() {
    return 'leap_wiz_settings';
}

add_filter( 'ot_settings_id', 'leap_theme_settings_id', 10 );

/**
 * Theme Layouts ID
 * @return string
 */
function leap_theme_layouts_id() {
    return 'leap_wiz_layouts';
}

add_filter( 'ot_layouts_id', 'leap_theme_layouts_id', 10 );

/**
 * Add Online Documentation & Support links into header
 * @param string $page_id
 */
function leap_theme_option_header( $page_id ) {
    echo '<li id="option-tree-version"><span><a class="" href="' . esc_url( 'http://www.leap13.com/wiz-online-documentation/' ) . '" target="_blank">Online Documentation</a> | <a href="' . esc_url( 'https://support.leap13.com/' ) . '" target="_blank">Support</a></span></li>';
}

add_action( 'ot_header_list', 'leap_theme_option_header', 10, 1 );

/**
 * Remove version from header
 * @return string
 */
function leap_theme_version() {
    return '';
}

add_filter( 'ot_header_version_text', 'leap_theme_version', 10 );

/**
 * Remove logo from header
 * @return string
 */
function leap_theme_logo() {
    return '';
}

add_filter( 'ot_header_logo_link', 'leap_theme_logo', 10 );

/**
 * Set the default location for all radio images
 * @param string $src
 * @param string $field_id
 * @return string
 */
function leap_theme_radio_image_src( $src, $field_id ) {
    return get_template_directory_uri() . '/functions/images/' . $src;
}

add_filter( 'ot_type_radio_image_src', 'leap_theme_radio_image_src', 10, 2 );

/**
 * Set the text for the upload button used on theme options
 * @param string $text
 * @return type
 */
function leap_upload_text( $text ) {
    return __( 'Use This', 'wiz' );
}

add_filter( 'ot_upload_text', 'leap_upload_text', 10, 1 );

/**
 * Set the slug of the theme options page
 * @param string $text
 * @return string
 */
function leap_theme_options_menu_slug( $text ) {
    return 'leap-theme-options';
}

add_filter( 'ot_theme_options_menu_slug', 'leap_theme_options_menu_slug', 10, 1 );

/**
 * Replace some translated text with new one
 * @param string $translated_text
 * @param string $text
 * @param string $domain
 * @return string
 */
function leap_replace_translated_strings( $translated_text, $text, $domain ) {
    switch ( $translated_text ) {
        case 'Option Tree' :
            $translated_text = __( 'Options', 'wiz' );
            break;
    }
    return $translated_text;
}

add_filter( 'gettext', 'leap_replace_translated_strings', 20, 3 );

/**
 * Set the default theme options for the first time ever this theme installed
 */
function leap_save_options_after_first_activate() {

    // Make sure there is no options saved for this theme
    if ( false === get_option( ot_options_id() ) ) {

        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
        global $wp_filesystem;
        $default_options = $wp_filesystem->get_contents( get_template_directory() . '/functions/importer/default-options.txt' );
        $theme_options= @unserialize( call_user_func( 'base'.'64_decode', $default_options ) );
        update_option( ot_options_id(), $theme_options );

        // Fetch google fonts
        if ( is_admin() ) {
            ot_fetch_google_fonts();
            $google_fonts = ot_get_option( 'google_fonts', array() );
            ot_set_google_fonts( 'google_fonts', $google_fonts );
        }
    }
}

add_action( "after_switch_theme", "leap_save_options_after_first_activate" );

/**
 * Set Google Fonts Api key
 */
function leap_google_fonts_api_key( $key ) {
    return ot_get_option( 'google_fonts_key', 'AIzaSyBUXpOWPVF6bTpxgC2yXGFKBtViBD9BYnc' );
}

add_filter( 'ot_google_fonts_api_key', 'leap_google_fonts_api_key', 10, 1 );


if ( !function_exists( 'leap_get_setting_label_by_id' ) ) {

    /**
     * Get Option Label by field id
     * @param string $field_id
     * @return string
     */
    function leap_get_setting_label_by_id( $field_id ) {

        if ( empty( $field_id ) )
            return false;

        $settings = get_option( ot_settings_id() );

        if ( empty( $settings[ 'settings' ] ) )
            return false;

        foreach ( $settings[ 'settings' ] as $setting ) {

            if ( $setting[ 'id' ] == $field_id && isset( $setting[ 'label' ] ) ) {

                return $setting[ 'label' ];
            }
        }
    }

}

/**
 * Add custom fonts to typography fonts list
 * @param array $array
 * @param string $field_id
 * @return array
 */
function leap_add_custom_fonts_to_list( $array, $field_id ) {
    $fonts        = ot_get_option( 'custom_fonts', array() );
    $custom_fonts = array();
    foreach ( $fonts as $font ) {
        $family_value = preg_replace( "/[^a-zA-Z0-9]+/", "", $font[ 'title' ] );

        $spaces      = explode( ' ', $font[ 'title' ] );
        $font_family = count( $spaces ) > 1 ? '"' . $font[ 'title' ] . '"' : $font[ 'title' ];

        $custom_fonts[ strtolower( $family_value ) ] = $font_family;
    }
    $array = array_merge( $array, $custom_fonts );
    return $array;
}

add_filter( 'ot_recognized_font_families', 'leap_add_custom_fonts_to_list', 10, 2 );

/**
 * Get all fonts list
 * @param string $field_id
 * @return array
 */
if( !function_exists('leap_all_font_families') ) {
    /**
    * Get all fonts list
    * @param string $field_id
    * @return array
    */
   function leap_all_font_families( $field_id = '' ) {
       return apply_filters( 'ot_recognized_font_families', array(
           ''          => 'Default',
           'arial'     => 'Arial',
           'georgia'   => 'Georgia',
           'helvetica' => 'Helvetica',
           'palatino'  => 'Palatino',
           'tahoma'    => 'Tahoma',
           'times'     => '"Times New Roman", sans-serif',
           'trebuchet' => 'Trebuchet',
           'verdana'   => 'Verdana'
       ), $field_id );
   }
}

/**
 * Add Google fonts to all fonts list
 * @param array $families
 * @param string $field_id
 * @return array
 */
function leap_google_font_stack( $families, $field_id ) {

    $ot_google_fonts     = get_theme_mod( 'ot_google_fonts', array() );
    $ot_set_google_fonts = get_theme_mod( 'ot_set_google_fonts', array() );

    if ( !empty( $ot_set_google_fonts ) ) {
        foreach ( $ot_set_google_fonts as $id => $sets ) {
            foreach ( $sets as $value ) {
                $family = isset( $value[ 'family' ] ) ? $value[ 'family' ] : '';
                if ( $family && isset( $ot_google_fonts[ $family ] ) ) {
                    $spaces              = explode( ' ', $ot_google_fonts[ $family ][ 'family' ] );
                    $font_stack          = count( $spaces ) > 1 ? '"' . $ot_google_fonts[ $family ][ 'family' ] . '"' : $ot_google_fonts[ $family ][ 'family' ];
                    $families[ $family ] = apply_filters( 'ot_google_font_stack', $font_stack, $family, $field_id );
                }
            }
        }
    }

    return $families;
}

add_filter( 'ot_recognized_font_families', 'leap_google_font_stack', 10, 2 );

function leap_wpml_option_types( $types ) {
    $types[] = 'upload';
    return $types;
}

add_filter( 'ot_wpml_option_types', 'leap_wpml_option_types', 10, 1 );

/**
 * Force plugins behaviour to fit the theme
 */
function leap_force_plugins_behaviour() {
    $ultimate_css = get_option( 'ultimate_css', '' );
    $ultimate_js  = get_option( 'ultimate_js', '' );

    if ( empty( $ultimate_css ) || empty( $ultimate_js ) ) {
        update_option( 'ultimate_css', 'enable' );
        update_option( 'ultimate_js', 'enable' );
    }
}

add_action( 'admin_init', 'leap_force_plugins_behaviour' );


if ( ! function_exists( 'leap_remove_anonymous_object_filter' ) ) {
    /**
     * Leap Remove an anonymous object filter.
     *
     * @param string $tag      Hook name.
     * @param string $class    Class name
     * @param string $method   Method name
     * @param int    $priority Optional. Hook priority. Defaults to 10.
     * @return bool
     */
    function leap_remove_anonymous_object_filter($tag, $class, $method, $priority = 10) {
        if ( !isset($GLOBALS['wp_filter'][$tag][$priority]) ) {
            return false;
        }
        $filters = $GLOBALS['wp_filter'][$tag][$priority];
 
        foreach ($filters as $callback) {
            if ( is_array($callback['function'])
                && is_a( $callback['function'][0], $class )
                && $method === $callback['function'][1]
            ) {
                return remove_filter($tag, $callback['function'], $priority);
            }
        }
 
        return false;
    }
}