import times from "lodash/times";

import React, { Component } from "react";
import PropTypes from "prop-types";

import {
  View,
  Text,
  Animated,
  PanResponder,
  Image,
  StyleSheet,
  Platform,
  ViewPropTypes
} from "react-native";

// RATING IMAGES WITH STATIC BACKGROUND COLOR (white)
const STAR_IMAGE = require("./images/star.png");

const TYPES = {
  star: {
    source: STAR_IMAGE,
    color: "#f1c40f",
    backgroundColor: "white"
  }
};
const fractionsType = (props, propName, componentName) => {
  if (props[propName]) {
    const value = props[propName];
    if (typeof value === "number") {
      return value >= 0 && value <= 20
        ? null
        : new Error(
            `\`${propName}\` in \`${componentName}\` must be between 0 and 20`
          );
    }

    return new Error(
      `\`${propName}\` in \`${componentName}\` must be a number`
    );
  }
};

export default class Rating extends Component {
  static propTypes = {
    type: PropTypes.string,
    ratingColor: PropTypes.string,
    ratingBackgroundColor: PropTypes.string,
    ratingCount: PropTypes.number,
    ratingTextColor: PropTypes.string,
    imageSize: PropTypes.number,
    onStartRating: PropTypes.func,
    onFinishRating: PropTypes.func,
    showRating: PropTypes.bool,
    style: ViewPropTypes.style,
    readonly: PropTypes.bool,
    startingValue: PropTypes.number,
    fractions: fractionsType,
    minValue: PropTypes.number
  };

  static defaultProps = {
    type: "star",
    ratingColor: "#f1c40f",
    ratingBackgroundColor: "white",
    ratingCount: 5,
    imageSize: 40,
    onFinishRating: () => console.log("Attach a onFinishRating function here."),
    minValue: 0
  };

  constructor(props) {
    super(props);
    const { onStartRating, onFinishRating, fractions } = this.props;
    const position = new Animated.ValueXY();

    const panResponder = PanResponder.create({
      onStartShouldSetPanResponder: () => true,
      onPanResponderGrant: () => {
        if (typeof onStartRating === "function") {
          onStartRating();
        }
      },
      onPanResponderMove: (event, gesture) => {
        const newPosition = new Animated.ValueXY();
        newPosition.setValue({ x: gesture.dx, y: 0 });
        this.setState({ position: newPosition, value: gesture.dx });
      },
      onPanResponderRelease: event => {
        const rating = this.getCurrentRating(this.state.value);
        if (rating >= this.props.minValue) {
          if (!fractions) {
            // 'round up' to the nearest rating image
            this.setCurrentRating(rating);
          }
          onFinishRating(rating);
        }
      }
    });

    this.state = { panResponder, position, display: false };
  }

  async componentDidMount() {
    try {
      const STAR_IMAGE = await require("./images/airbnb-star.png");
      this.setState({ display: true });
    } catch (err) {
      console.log(err);
    }

    this.setCurrentRating(this.props.startingValue);
  }

  UNSAFE_componentWillReceiveProps(nextProps) {
    if (nextProps.startingValue !== this.props.startingValue) {
      this.setCurrentRating(nextProps.startingValue);
    }
  }

  getPrimaryViewStyle() {
    const { position } = this.state;
    const { imageSize, ratingCount, type, ratingColor } = this.props;

    const color = TYPES[type].color;

    const width = position.x.interpolate({
      inputRange: [
        -ratingCount * (imageSize / 2),
        0,
        ratingCount * (imageSize / 2)
      ],
      outputRange: [0, (ratingCount * imageSize) / 2, ratingCount * imageSize],
      extrapolate: "clamp"
    });

    return {
      backgroundColor: ratingColor,
      width,
      height: width ? imageSize : 0
    };
  }

  getSecondaryViewStyle() {
    const { position } = this.state;
    const { imageSize, ratingCount, type } = this.props;

    const backgroundColor = TYPES[type].backgroundColor;

    const width = position.x.interpolate({
      inputRange: [
        -ratingCount * (imageSize / 2),
        0,
        ratingCount * (imageSize / 2)
      ],
      outputRange: [ratingCount * imageSize, (ratingCount * imageSize) / 2, 0],
      extrapolate: "clamp"
    });

    return {
      backgroundColor,
      width,
      height: width ? imageSize : 0
    };
  }

  renderRatings() {
    const { imageSize, ratingCount, type, tintColor } = this.props;

    return times(ratingCount, index => (
      <View key={index} style={styles.starContainer}>
        <Image
          source={STAR_IMAGE}
          style={{ width: imageSize, height: imageSize, tintColor }}
        />
      </View>
    ));
  }

  getCurrentRating(value) {
    // const { value } = this.state;
    const { fractions, imageSize, ratingCount } = this.props;

    const startingValue = ratingCount / 2;
    let currentRating = this.props.minValue ? this.props.minValue : 0;

    if (value > (ratingCount * imageSize) / 2) {
      currentRating = ratingCount;
    } else if (value < (-ratingCount * imageSize) / 2) {
      currentRating = this.props.minValue ? this.props.minValue : 0;
    } else if (value < imageSize || value > imageSize) {
      currentRating = startingValue + value / imageSize;
      currentRating = !fractions
        ? Math.ceil(currentRating)
        : +currentRating.toFixed(fractions);
    } else {
      currentRating = !fractions
        ? Math.ceil(startingValue)
        : +startingValue.toFixed(fractions);
    }

    return currentRating;
  }

  setCurrentRating(rating) {
    const { imageSize, ratingCount } = this.props;

    // `initialRating` corresponds to `startingValue` in the getter. Naming it
    // differently here avoids confusion with `value` below.
    const initialRating = ratingCount / 2;
    let value = null;

    if (rating > ratingCount) {
      value = (ratingCount * imageSize) / 2;
    } else if (rating < 0) {
      value = (-ratingCount * imageSize) / 2;
    } else if (rating < ratingCount / 2 || rating > ratingCount / 2) {
      value = (rating - initialRating) * imageSize;
    } else {
      value = 0;
    }

    const newPosition = new Animated.ValueXY();
    newPosition.setValue({ x: value, y: 0 });
    this.setState({ position: newPosition, value });
  }

  displayCurrentRating() {
    const { ratingCount, type, readonly, ratingTextColor } = this.props;
    const color = ratingTextColor || TYPES[type].color;

    return (
      <View style={styles.showRatingView}>
        <View style={styles.ratingView}>
          <Text style={[styles.ratingText, { color }]}>Rating: </Text>
          <Text style={[styles.currentRatingText, { color }]}>
            {this.getCurrentRating(this.state.value)}
          </Text>
          <Text style={[styles.maxRatingText, { color }]}>/{ratingCount}</Text>
        </View>
      </View>
    );
  }

  render() {
    const {
      readonly,
      type,
      ratingColor,
      ratingBackgroundColor,
      style,
      showRating,
      customStyle
    } = this.props;

    return this.state.display ? (
      <View pointerEvents={readonly ? "none" : "auto"} style={style}>
        {showRating && this.displayCurrentRating()}
        <View
          style={[styles.starsWrapper]}
          {...this.state.panResponder.panHandlers}
        >
          <View style={styles.starsInsideWrapper}>
            <Animated.View style={this.getPrimaryViewStyle()} />
            <Animated.View style={this.getSecondaryViewStyle()} />
          </View>
          {this.renderRatings()}
        </View>
      </View>
    ) : null;
  }
}

const styles = StyleSheet.create({
  starsWrapper: {
    flexDirection: "row"
    // justifyContent: "center",
    // alignItems: "center"
  },
  starsInsideWrapper: {
    position: "absolute",
    top: 0,
    left: 0,
    right: 0,
    bottom: 0,
    flexDirection: "row"
    // justifyContent: "center",
    // alignItems: "center"
  },
  showRatingView: {
    flexDirection: "column",
    justifyContent: "center",
    alignItems: "center",
    paddingBottom: 5
  },
  ratingView: {
    flexDirection: "row",
    justifyContent: "center",
    alignItems: "center",
    paddingBottom: 5
  },
  ratingText: {
    fontSize: 15,
    textAlign: "center",
    color: "#34495e"
  },
  readonlyLabel: {
    justifyContent: "center",
    alignItems: "center",
    fontSize: 12,
    textAlign: "center",
    color: "#34495a"
  },
  currentRatingText: {
    fontSize: 30,
    textAlign: "center"
  },
  maxRatingText: {
    fontSize: 18,
    textAlign: "center",
    color: "#34495e"
  }
});
