<?php
/**
 * The image widget.
 *
 * @package   SimpleImageWidget
 * @copyright Copyright (c) 2015 Cedaro, LLC
 * @license   GPL-2.0+
 * @since     3.0.0
 */

/**
 * Image widget class.
 *
 * @package SimpleImageWidget
 * @since   3.0.0
 */
class Wescle_Simple_Image extends WP_Widget {

	/**
	 * Setup widget options.
	 *
	 * Child classes may override the defaults.
	 *
	 * @param string $id_base Optional Base ID for the widget, lower case, if
	 *     left empty a portion of the widget's class name will be used. Must be unique.
	 * @param string $name Name for the widget displayed on the configuration page.
	 * @param array $widget_options {
	 *     Widget options. Passed to wp_register_sidebar_widget(). Optional.
	 *
	 * @type string $description Widget description. Shown on the configuration page.
	 * @type string $classname HTML class.
	 * }
	 *
	 * @param array $control_options {
	 *     Passed to wp_register_widget_control(). Optional.
	 *
	 * @type int $width Width of the widget edit form.
	 * )
	 * @since 3.0.0
	 * @see   WP_Widget::construct()
	 *
	 */
	public function __construct( $id_base = false, $name = false, $widget_options = array(), $control_options = array() ) {
		$id_base = ( $id_base ) ? $id_base : THEME_SLUG . '_simpleimage'; // Legacy ID.
		$name    = ( $name ) ? $name : 'Wescle: ' . __( 'Баннер', 'wescle' );

		$widget_options = wp_parse_args(
			$widget_options,
			array(
				'classname'                   => 'advertising', // Legacy class name.
				'customize_selective_refresh' => true,
				'description'                 => __( 'Выводит блок с текстом/ссылкой и фоновой картинкой.', 'wescle' ),
			)
		);

		$control_options = wp_parse_args( $control_options, array() );

		parent::__construct( $id_base, $name, $widget_options, $control_options );

		// Flush widget group cache when an attachment is saved, deleted, or the theme is switched.
		add_action( 'save_post', array( $this, 'flush_group_cache' ) );
		add_action( 'delete_attachment', array( $this, 'flush_group_cache' ) );
		add_action( 'switch_theme', array( $this, 'flush_group_cache' ) );
	}

	/**
	 * Display the widget.
	 *
	 * Filters the instance data, fetches the output, displays it, then caches
	 * it. Overload or filter the render() method to modify output.
	 *
	 * @param array $args Registered sidebar arguments including before_title, after_title, before_widget, and after_widget.
	 * @param array $instance The widget instance settings.
	 *
	 * @since 3.0.0
	 *
	 */
	public function widget( $args, $instance ) {
		$cache    = (array) wp_cache_get( 'simple_image_widget', 'widget' );
		$cache_id = $this->siw_get_cache_key( $args, $instance );

		if ( ! $this->is_preview() && isset( $cache[ $cache_id ] ) ) {
			echo $cache[ $cache_id ];

			return;
		}

		// Copy the original values so they can be used in hooks.
		$instance['text_raw'] = empty( $instance['text'] ) ? '' : $instance['text'];
		$instance['text']     = apply_filters( 'widget_text', $instance['text_raw'], $instance, $this->id_base );

		// Start building the output.
		$output = '';

		// Make sure the image ID is a valid attachment.
		if ( ! empty( $instance['image_id'] ) ) {
			$image = get_post( $instance['image_id'] );
			if ( ! $image || 'attachment' != get_post_type( $image ) ) {
				$output = '<!-- Image Widget Error: Invalid Attachment ID -->';
			}
		}

		if ( empty( $output ) ) {
			$instance['link_open']  = '<div class="advertising__item">';
			$instance['link_close'] = '</div>';

			if ( ! empty ( $instance['link'] ) ) {
				$target = ( empty( $instance['new_window'] ) ) ? '' : ' target="_blank"';

				$instance['link_open']  = '<a href="' . esc_url( $instance['link'] ) . '" class="advertising__item"' . $target . '>';
				$instance['link_close'] = '</a>';
			}

			if ( $instance['image_id'] || $instance['title_text'] || $instance['text'] ) {
				$output = $this->render( $args, $instance );
			}
		}

		echo $output;

		if ( ! $this->is_preview() ) {
			$cache[ $cache_id ] = $output;
			wp_cache_set( 'simple_image_widget', array_filter( $cache ), 'widget' );
		}
	}

	/**
	 * Generate the widget output.
	 *
	 * This is typically done in the widget() method, but moving it to a
	 * separate method allows for the routine to be easily overloaded by a class
	 * extending this one without having to reimplement all the caching and
	 * filtering, or resorting to adding a filter, calling the parent method,
	 * then removing the filter.
	 *
	 * @param array $args Registered sidebar arguments including before_title, after_title, before_widget, and after_widget.
	 * @param array $instance The widget instance settings.
	 *
	 * @return string HTML output.
	 * @since 3.0.0
	 *
	 */
	public function render( $args, $instance ) {
		$output = $args['before_widget'];

		ob_start();
		set_query_var( 'template_args', [ 'instance' => $instance ] );
		get_template_part( 'template-parts/widgets/widget', 'banner' );
		$output .= ob_get_clean();

		$output .= $args['after_widget'];

		return $output;
	}

	/**
	 * Display the form to edit widget settings.
	 *
	 * @param array $instance The widget settings.
	 *
	 * @since 3.0.0
	 *
	 */
	public function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance,
			array(
				'alt'          => '', // Legacy.
				'image'        => '', // Legacy URL field.
				'image_id'     => '',
				'image_size'   => 'full',
				'link'         => '',
				'link_classes' => '',
				'link_text'    => '',
				'new_window'   => '',
				'text'         => '',
				'title_text'   => '',
				'style_text' => 'dark'
			)
		);

		$instance['image_id'] = absint( $instance['image_id'] );

		$button_class = array( 'button', 'button-hero', 'simple-image-widget-control-choose' );
		$image_id     = $instance['image_id'];

		/**
		 * The list of fields to display.
		 *
		 * The order of fields can be modified, new fields can be registered, or
		 * existing fields can be removed here. Use the widget type id to limit
		 * fields to a particular type of widget.
		 *
		 * @param array $fields List of field ids.
		 * @param string $id_base Widget type id.
		 *
		 * @since 3.0.0
		 *
		 */
		$fields = (array) apply_filters( 'simple_image_widget_fields', $this->form_fields(), $this->id_base );
		?>

        <div class="simple-image-widget-form">

			<?php
			/**
			 * Display additional information or HTML before the widget edit form.
			 *
			 * @param array $instance The widget setttings.
			 * @param string $id_base Widget type id.
			 *
			 * @since 3.0.0
			 *
			 */
			do_action( 'simple_image_widget_form_before', $instance, $this->id_base );
			?>

            <p class="simple-image-widget-control<?php echo ( $image_id ) ? ' has-image' : ''; ?>"
               data-title="<?php esc_attr_e( 'Выбрать фоновое изображение', 'wescle' ); ?>"
               data-update-text="<?php esc_attr_e( 'Обновить изображение', 'wescle' ); ?>"
               data-target=".image-id">
		        <?php
		        if ( $image_id ) {
			        echo wp_get_attachment_image( $image_id, 'medium', false );
			        unset( $button_class[ array_search( 'button-hero', $button_class ) ] );
		        }
		        ?>
                <input type="hidden" name="<?php echo esc_attr( $this->get_field_name( 'image_id' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'image_id' ) ); ?>" value="<?php echo absint( $image_id ); ?>" class="image-id simple-image-widget-control-target">
                <a href="#" class="<?php echo esc_attr( join( ' ', $button_class ) ); ?>"><?php _e( 'Выбрать фоновое изображение', 'wescle' ); ?></a>
            </p>

	        <?php
	        if ( ! empty( $fields ) ) {
		        foreach ( $fields as $field ) {
			        switch ( $field ) {
				        case 'link' :
					        ?>
                            <p class="<?php echo esc_attr( $this->siw_field_class( 'link' ) ); ?>">
                                <label for="<?php echo esc_attr( $this->get_field_id( 'link' ) ); ?>"><?php _e( 'Ссылка', 'wescle' ); ?></label>
                                <span class="simple-image-widget-input-group">
									<input type="text" name="<?php echo esc_attr( $this->get_field_name( 'link' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'link' ) ); ?>" value="<?php echo esc_url( $instance['link'] ); ?>" class="widefat">
								</span>
                            </p>
                            <p class="<?php echo esc_attr( $this->siw_field_class( 'new_window' ) ); ?>">
                                <label for="<?php echo esc_attr( $this->get_field_id( 'new_window' ) ); ?>">
                                    <input type="checkbox" name="<?php echo esc_attr( $this->get_field_name( 'new_window' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'new_window' ) ); ?>" <?php checked( $instance['new_window'] ); ?>>
							        <?php _e( 'Открывать в новой вкладке?', 'wescle' ); ?>
                                </label>
                            </p>
					        <?php
					        break;

						case 'link_text' :
							?>
                            <p class="<?php echo esc_attr( $this->siw_field_class( 'link_text' ) ); ?>">
                                <label for="<?php echo esc_attr( $this->get_field_id( 'link_text' ) ); ?>"><?php _e( 'Надпись на кнопке', 'wescle' ); ?></label>
                                <input type="text" name="<?php echo esc_attr( $this->get_field_name( 'link_text' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'link_text' ) ); ?>" value="<?php echo esc_attr( $instance['link_text'] ); ?>" class="widefat">
                            </p>
							<?php
							break;

						case 'title_text' :
							?>
                            <p class="<?php echo esc_attr( $this->siw_field_class( 'title_text' ) ); ?>">
                                <label for="<?php echo esc_attr( $this->get_field_id( 'title_text' ) ); ?>"><?php _e( 'Заголовок', 'wescle' ); ?></label>
                                <input type="text" name="<?php echo esc_attr( $this->get_field_name( 'title_text' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'title_text' ) ); ?>" value="<?php echo esc_attr( $instance['title_text'] ); ?>" class="widefat">
                            </p>
							<?php
							break;

						case 'text' :
							?>
                            <p class="<?php echo esc_attr( $this->siw_field_class( 'text' ) ); ?>">
                                <label for="<?php echo esc_attr( $this->get_field_id( 'text' ) ); ?>"><?php _e( 'Текст', 'wescle' ); ?></label>
                                <textarea name="<?php echo esc_attr( $this->get_field_name( 'text' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'text' ) ); ?>" rows="4" class="widefat"><?php echo esc_textarea( $instance['text'] ); ?></textarea>
                            </p>
							<?php
							break;

						case 'style_text' :
							?>
                            <p class="<?php echo esc_attr( $this->siw_field_class( 'style_text' ) ); ?>">
                            <div><label for="<?php echo esc_attr( $this->get_field_id( 'style_text' ) ); ?>"><?php _e( 'Стиль текста', 'wescle' ); ?></label></div>
                            <select name="<?php echo esc_attr( $this->get_field_name( 'style_text' ) ); ?>">
                                <option value="light" <?php selected( 'light', $instance['style_text'] ); ?>><?php _e( 'Светлый', 'wescle' ); ?></option>
                                <option value="dark" <?php selected( 'dark', $instance['style_text'] ); ?>><?php _e( 'Темный', 'wescle' ); ?></option>
                            </select>
                            </p>
							<?php
							break;


						default :
							/**
							 * Display a custom field.
							 *
							 * This action will fire for custom fields
							 * registered with the 'simple_image_widget_fields'
							 * filter.
							 *
							 * @param array $instance The widget setttings.
							 * @param string $widget Widget instance.
							 *
							 * @since 3.0.0
							 *
							 */
							do_action( 'simple_image_widget_field-' . sanitize_key( $field ), $instance, $this );
					}
				}
			}

			/**
			 * Display additional information or HTML after the widget edit form.
			 *
			 * @param array $instance The widget setttings.
			 * @param string $id_base Widget type id.
			 *
			 * @since 3.0.0
			 *
			 */
			do_action( 'simple_image_widget_form_after', $instance, $this->id_base );
			?>

        </div><!-- /.simple-image-widget-form -->
		<?php
	}

	/**
	 * The list of extra fields that should be shown in the widget form.
	 *
	 * Can be easily overloaded by a child class.
	 *
	 * @return string List of field ids.
	 * @since 3.0.0
	 *
	 */
	public function form_fields() {
		return array( 'title_text', 'text', 'style_text', 'link', 'link_text' );
	}

	/**
	 * Save and sanitize widget settings.
	 *
	 * @param array $new_instance New widget settings.
	 * @param array $old_instance Previous widget settings.
	 *
	 * @return array Sanitized settings.
	 * @since 3.0.0
	 *
	 */
	public function update( $new_instance, $old_instance ) {
		$instance = wp_parse_args( $new_instance, $old_instance );

		$instance = apply_filters( 'simple_image_widget_instance', $instance, $new_instance, $old_instance, $this->id_base );

		$instance['image_id']   = absint( $new_instance['image_id'] );
		$instance['new_window'] = isset( $new_instance['new_window'] );

		// Optional field that can be removed via a filter.
		foreach ( array( 'link', 'link_classes', 'link_text', 'text', 'title_text', 'style_text' ) as $key ) {
			if ( ! isset( $new_instance[ $key ] ) ) {
				continue;
			}

			switch ( $key ) {
				case 'link' :
					$instance['link'] = esc_url_raw( $new_instance['link'] );
					break;
				case 'link_classes' :
					$instance['link_classes'] = implode( ' ', array_map( 'sanitize_html_class', explode( ' ', $new_instance['link_classes'] ) ) );
					break;
				case 'link_text' :
					$instance['link_text'] = wp_kses_data( $new_instance['link_text'] );
					break;
				case 'text' :
					$instance['text'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['text'] ) ) );
					break;
				case 'style_text' :
					$instance['style_text'] = stripslashes( wp_filter_post_kses( addslashes( $new_instance['style_text'] ) ) );
					break;
			}
		}

		$this->flush_group_cache();

		return $instance;
	}

	/**
	 * Determine if the widget is being displayed in the customizer.
	 *
	 * @return bool
	 * @link  https://core.trac.wordpress.org/ticket/27538
	 *
	 * @since 4.0.1
	 */
	public function is_preview() {
		global $wp_customize;

		if ( method_exists( get_parent_class( $this ), 'is_preview' ) ) {
			return parent::is_preview();
		}

		return ( isset( $wp_customize ) && $wp_customize->is_preview() );
	}

	/**
	 * Flush the cache for all image widgets.
	 *
	 * @param int $post_id Post ID.
	 *
	 * @since 3.0.0
	 *
	 */
	public function flush_group_cache( $post_id = null ) {
		if ( 'save_post' == current_filter() && 'attachment' != get_post_type( $post_id ) ) {
			return;
		}

		wp_cache_delete( 'simple_image_widget', 'widget' );
	}

	/**
	 * Retrieve a list of templates to look up.
	 *
	 * @param array $args Registered sidebar arguments including before_title, after_title, before_widget, and after_widget.
	 * @param array $instance The widget instance settings.
	 *
	 * @return array List of template names.
	 * @since 4.0.0
	 *
	 */
	protected function get_template_names( $args, $instance ) {
		$templates = array();
		if ( ! empty( $args['id'] ) ) {
			$templates[] = $args['id'] . '_widget.php';
		}
		$templates[] = 'widget.php';

		/**
		 * List of template names to look up to render output.
		 *
		 * Child widgets should consider adding a new template using the widget type id ($this->id_base).
		 *
		 * @param array $templates List of template names.
		 * @param array $args Registered sidebar arguments including before_title, after_title, before_widget, and after_widget.
		 * @param array $instance The widget instance settings.
		 * @param string $id_base Widget type id.
		 *
		 * @since 4.0.0
		 *
		 */
		return apply_filters(
			'simple_image_widget_templates',
			$templates,
			$args,
			$instance,
			$this->id_base
		);
	}

	/**
	 * Retrieve HTML classes for a field container.
	 *
	 * @param string $id Field id.
	 *
	 * @return string
	 * @since 4.1.0
	 *
	 */
	protected function siw_field_class( $id ) {
		$classes = array( 'simple-image-widget-field', 'simple-image-widget-field-' . sanitize_html_class( $id ) );

		$hidden_fields = [];
		if ( in_array( $id, $hidden_fields ) ) {
			$classes[] = 'is-hidden';
		}

		return implode( ' ', $classes );
	}

	/**
	 * Retrieve a cache key based on a hash of passed parameters.
	 *
	 * @return string
	 * @since 4.2.0
	 *
	 */
	protected function siw_get_cache_key() {
		$data = array();
		foreach ( func_get_args() as $arg ) {
			$data = array_merge( $data, (array) $arg );
		}
		ksort( $data );

		return 'siw_' . md5( json_encode( $data ) );
	}

	/**
	 * Remove a single image widget from the cache.
	 *
	 * @since 3.0.0
	 * @deprecated 4.2.0
	 */
	public function flush_widget_cache() {
	}
}
