<?php

class Wescle_Widget_Catalog_Filter extends WP_Widget {

	function __construct() {
		parent::__construct(
			THEME_SLUG . '_widget_catalog_filter',
			'Wescle: ' . __( 'Фильтр каталога', 'wescle' ),
			array(
				'description'                 => __( 'Выводит блок с фильтрами для каталога', 'wescle' ),
				'classname'                   => 'aside-store-checkboxes wescle-catalog-filter',
				'customize_selective_refresh' => true,
				'show_instance_in_rest'       => true,
			)
		);
	}

	public function widget( $args, $instance ) {
		if ( ! is_page_template( 'templates/template-catalog_wescle.php' ) && ! is_tax( Helper::get_catalog_taxs() ) ) {
			return;
		}

		$title        = apply_filters( 'widget_title', empty( $instance['title'] ) ? '' : $instance['title'], $instance, $this->id_base );
		$taxonomy     = WsclAdminFilters::filter_taxonomy_name( $instance['filter'] );
		$query_type   = $instance['query_type'];
		$display_type = $instance['display_type'];

		if ( ! taxonomy_exists( $taxonomy ) ) {
			return;
		}

		$terms = get_terms( array(
			'taxonomy'   => $taxonomy,
			'hide_empty' => '1',
		) );

		if ( 0 === count( $terms ) ) {
			return;
		}

		ob_start();

		echo $args['before_widget'];

		if ( ! empty( $title ) ) {
			echo $args['before_title'] . $title . $args['after_title'];
		}

		if ( 'dropdown' === $display_type ) {
			$found = $this->layered_nav_dropdown( $terms, $taxonomy, $query_type );
		}
		else {
			$found = $this->filter_nav_list( $terms, $taxonomy, $query_type );
		}

		echo $args['after_widget'];

		if ( ! $found ) {
			ob_end_clean();
		}
		else {
			echo ob_get_clean(); // @codingStandardsIgnoreLine
		}
	}

	public function form( $instance ) {
		$instance = wp_parse_args(
			(array) $instance,
			array(
				'title'        => '',
				'filter'       => '',
				'query_type'   => 'and',
				'display_type' => 'list',
			)
		);

		$filter_array      = array();
		$filter_taxonomies = WsclAdminFilters::get_filter_taxonomies();

		if ( ! empty( $filter_taxonomies ) ) {
			foreach ( $filter_taxonomies as $tax ) {
				if ( taxonomy_exists( WsclAdminFilters::filter_taxonomy_name( $tax->filter_name ) ) ) {
					$filter_array[ $tax->filter_name ] = $tax->filter_label;
				}
			}
		}

		$title = sanitize_text_field( $instance['title'] );
		?>
        <p>
            <label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Заголовок', 'wescle' ); ?></label>
            <input class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>">
        </p>

        <p>
            <label for="<?php echo $this->get_field_id( 'filter' ); ?>"><?php _e( 'Фильтр', 'wescle' ); ?></label>
            <select name="<?php echo $this->get_field_name( 'filter' ); ?>" id="<?php echo $this->get_field_id( 'filter' ); ?>" class="widefat">
				<?php
				foreach ( $filter_array as $filter_name => $filter_label ) {
					echo '<option value="' . $filter_name . '" ' . selected( $filter_name, $instance['filter'], false ) . '>' . $filter_label . '</option>';
				}
				?>
            </select>
        </p>

        <p>
        <div><label for="<?php echo esc_attr( $this->get_field_id( 'query_type' ) ); ?>"><?php _e( 'Тип запроса', 'wescle' ); ?></label></div>
        <select name="<?php echo esc_attr( $this->get_field_name( 'query_type' ) ); ?>">
            <option value="and" <?php selected( 'and', $instance['query_type'] ); ?>><?php _e( 'И', 'wescle' ); ?></option>
            <option value="or" <?php selected( 'or', $instance['query_type'] ); ?>><?php _e( 'ИЛИ', 'wescle' ); ?></option>
        </select>
        </p>

        <p>
        <div><label for="<?php echo esc_attr( $this->get_field_id( 'display_type' ) ); ?>"><?php _e( 'Тип отображения', 'wescle' ); ?></label></div>
        <select name="<?php echo esc_attr( $this->get_field_name( 'display_type' ) ); ?>">
            <option value="list" <?php selected( 'list', $instance['display_type'] ); ?>><?php _e( 'Список', 'wescle' ); ?></option>
            <!--<option value="dropdown" <?php selected( 'dropdown', $instance['display_type'] ); ?>><?php _e( 'Выпадающее меню', 'wescle' ); ?></option>-->
        </select>
        </p>
		<?php
	}

	public function update( $new_instance, $old_instance ) {
		$instance                 = $old_instance;
		$instance['title']        = sanitize_text_field( $new_instance['title'] );
		$instance['filter']       = $new_instance['filter'];
		$instance['display_type'] = $new_instance['display_type'];
		$instance['query_type']   = $new_instance['query_type'];

		return $instance;
	}

	public function filter_nav_list( $terms, $taxonomy, $query_type ) {
		$term_counts    = $this->get_filtered_term_product_counts( wp_list_pluck( $terms, 'term_id' ), $taxonomy, $query_type );
		$chosen_filters = WsclAdminFilters::get_filter_nav_chosen();
		$base_link      = WsclAdminFilters::get_current_page_url();

		$current_values = isset( $chosen_filters[ $taxonomy ]['terms'] ) ? $chosen_filters[ $taxonomy ]['terms'] : array();
		$filter_slug    = str_replace( 'filter_', '', $taxonomy );
		$filter_name    = 'filters_' . $filter_slug;

		$curr_filter = isset( $_GET[ $filter_name ] ) ? explode( ',', sanitize_text_field( wp_unslash( $_GET[ $filter_name ] ) ) ) : array();
		$curr_filter = array_map( 'sanitize_title', $curr_filter );

		$ul_list = '';
		$loop    = 0;
		foreach ( $terms as $term ) {
			$option_is_set = in_array( $term->slug, $current_values, true );
			$count         = isset( $term_counts[ $term->term_id ] ) ? $term_counts[ $term->term_id ] : 0;

			// Skip the term for the current archive.
			if ( $this->get_current_term_id() === $term->term_id ) {
				continue;
			}

			// Only show options with count > 0.
			if ( 0 === $count && ! $option_is_set ) {
				continue;
			}

			$current_filter = $curr_filter;

			if ( ! in_array( $term->slug, $current_filter, true ) ) {
				$current_filter[] = $term->slug;
			}

			$link = remove_query_arg( $filter_name, $base_link );

			// Add current filters to URL.
			foreach ( $current_filter as $key => $value ) {
				// Exclude query arg for current term archive term.
				if ( $value === $this->get_current_term_slug() ) {
					unset( $current_filter[ $key ] );
				}

				// Exclude self so filter can be unset on click.
				if ( $option_is_set && $value === $term->slug ) {
					unset( $current_filter[ $key ] );
				}
			}

			if ( ! empty( $current_filter ) ) {
				$link = add_query_arg( $filter_name, implode( ',', $current_filter ), $link );

				// Add Query type Arg to URL.
				if ( 'or' === $query_type && ! ( 1 === count( $current_filter ) ) ) {
					$link = add_query_arg( 'query_type_' . $filter_slug, 'or', $link );
				}
				$link = str_replace( '%2C', ',', $link );
			}
			else {
				if ( 'or' === $query_type ) {
					$link = remove_query_arg( 'query_type_' . $filter_slug, $link );
				}
			}

			$term_check = '<div class="aside-store-checkboxes__check"><svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" style="-ms-transform: rotate(360deg); -webkit-transform: rotate(360deg); transform: rotate(360deg);" preserveaspectratio="xMidYMid meet" viewbox="0 0 16 16"><g fill="#08c"><path fill-rule="evenodd" d="M10.97 4.97a.75.75 0 0 1 1.071 1.05l-3.992 4.99a.75.75 0 0 1-1.08.02L4.324 8.384a.75.75 0 1 1 1.06-1.06l2.094 2.093l3.473-4.425a.236.236 0 0 1 .02-.022z"></path></g></svg></div>';

			$term_name  = '<span class="aside-store-checkboxes__name">' . esc_html( $term->name ) . '</span>';
			$term_count = '<span class="aside-store-checkboxes__count">(' . absint( $count ) . ')</span>';

			$term_data = $term_check . $term_name . $term_count;

			if ( $count > 0 || $option_is_set ) {
				$term_html = '<a rel="nofollow" href="' . esc_url( $link ) . '">' . $term_data . '</a>';
			}
			else {
				$term_html = '<span>' . $term_data . '</span>';
			}

			$ul_list .= '<li class="wescle-widget-filter-nav-list__item ' . ( $option_is_set ? 'chosen' : '' ) . '">';
			$ul_list .= $term_html;
			$ul_list .= '</li>';

			$loop ++;
		}

		if ( $ul_list ) {
			echo '<div class="aside-store-checkboxes__body">';
			echo '<ul class="aside-store-checkboxes__list" aria-expanded="false">';
			echo $ul_list;
			echo '</ul>';
			if ( $loop > 6 ) {
				$open_text  = Helper::get_text_label( 'show_more' );
				$close_text = Helper::get_text_label( 'close' );

				echo '<a class="view-more" href="#" aria-expanded="false" data-open-name="' . $open_text . '" data-close-name="' . $close_text . '">' . $open_text . '</a>';
			}

			if ( $current_values ) {
				$label_reset = __( 'Сбросить', 'wescle' );
				$clear_link  = remove_query_arg( [ $filter_name, 'query_type_' . $filter_slug ], $base_link );
				echo '<a rel="nofollow" class="btn btn-main btn-reset" aria-label="' . esc_attr( $label_reset ) . '" href="' . esc_url( $clear_link ) . '">' . esc_html( $label_reset ) . '</a></li>';
			}

			echo '</div>';

			return true;
		}
	}

	public function layered_nav_dropdown( $terms, $taxonomy, $query_type ) {

	}

	public function get_filtered_term_product_counts( $term_ids, $taxonomy, $query_type ) {
		global $wpdb;

		$tax_query  = WsclAdminFilters::get_filter_tax_query();
		$meta_query = [];
		if ( 'or' === $query_type ) {
			foreach ( $tax_query as $key => $query ) {
				if ( is_array( $query ) && $taxonomy === $query['taxonomy'] ) {
					unset( $tax_query[ $key ] );
				}
			}
		}

		$meta_query = new \WP_Meta_Query( $meta_query );
		$tax_query  = new \WP_Tax_Query( $tax_query );

		$query     = $this->get_filter_counts_query( $tax_query, $meta_query, $term_ids );
		$query_sql = implode( ' ', $query );

		// We have a query - let's see if cached results of this query already exist.
		$query_hash = md5( $query_sql );
		// Maybe store a transient of the count values.
		$cache = true;
		if ( true === $cache ) {
			$cached_counts = (array) get_transient( 'wescle_filter_nav_counts_' . sanitize_title( $taxonomy ) );
		}
		else {
			$cached_counts = array();
		}
		if ( ! isset( $cached_counts[ $query_hash ] ) ) {
			// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
			$results                      = $wpdb->get_results( $query_sql, ARRAY_A );
			$counts                       = array_map( 'absint', wp_list_pluck( $results, 'term_count', 'term_count_id' ) );
			$cached_counts[ $query_hash ] = $counts;
			if ( true === $cache ) {
				set_transient( 'wescle_filter_nav_counts_' . sanitize_title( $taxonomy ), $cached_counts, DAY_IN_SECONDS );
			}
		}

		return array_map( 'absint', (array) $cached_counts[ $query_hash ] );
	}

	private function get_filter_counts_query( $tax_query, $meta_query, $term_ids ) {
		global $wpdb;

		$meta_query_sql = $meta_query->get_sql( 'post', $wpdb->posts, 'ID' );
		$tax_query_sql  = $tax_query->get_sql( $wpdb->posts, 'ID' );

		// Generate query.
		$query           = array();
		$query['select'] = "SELECT COUNT( DISTINCT {$wpdb->posts}.ID ) AS term_count, terms.term_id AS term_count_id";
		$query['from']   = "FROM {$wpdb->posts}";
		$query['join']   = "
			INNER JOIN {$wpdb->term_relationships} AS term_relationships ON {$wpdb->posts}.ID = term_relationships.object_id
			INNER JOIN {$wpdb->term_taxonomy} AS term_taxonomy USING( term_taxonomy_id )
			INNER JOIN {$wpdb->terms} AS terms USING( term_id )
			" . $tax_query_sql['join'] . $meta_query_sql['join'];

		$term_ids_sql   = $this->get_term_ids_sql( $term_ids );
		$query['where'] = "
			WHERE {$wpdb->posts}.post_type IN ( 'catalog_item' )
			AND {$wpdb->posts}.post_status = 'publish'
			{$tax_query_sql['where']} {$meta_query_sql['where']}
			AND terms.term_id IN $term_ids_sql";

		$query['group_by'] = 'GROUP BY terms.term_id';

		return $query;
	}

	private function get_term_ids_sql( $term_ids ) {
		return '(' . implode( ',', array_map( 'absint', $term_ids ) ) . ')';
	}

	protected function get_current_term_id() {
		return absint( is_tax() ? get_queried_object()->term_id : 0 );
	}

	protected function get_current_term_slug() {
		return absint( is_tax() ? get_queried_object()->slug : 0 );
	}
}