<?php

if ( ! function_exists( 'theme_site_logo' ) ) {

	function theme_site_logo( $args = array(), $echo = true ) {
		global $wp_query;

		$site_title = get_bloginfo( 'name' );

		$defaults = array(
			'wrap_link' => '<a class="header__logo" href="%1$s" title="%2$s">%3$s</a>',
			'wrap_span' => '<span class="header__logo" title="%2$s">%3$s</span>',
			'is_home'   => ( is_front_page() || is_home() ) && ! is_paged() && ! $wp_query->is_posts_page && get_option( 'show_on_front' ) != 'posts',
		);

		$args = wp_parse_args( $args, $defaults );

		if ( has_custom_logo() ) {
			$custom_logo_id  = get_theme_mod( 'custom_logo' );
			$custom_logo_src = wp_get_attachment_image_src( $custom_logo_id, 'medium' );
			$contents        = '<img src="' . $custom_logo_src[0] . '" width="' . $custom_logo_src[1] . '" height="' . $custom_logo_src[2] . '" alt="">';
		}
		else {
			$contents = '<span class="header__logo-text">' . get_bloginfo( 'name' ) . '</span>';
		}

		$wrap = $args['is_home'] ? 'wrap_span' : 'wrap_link';
		$html = sprintf( $args[ $wrap ], esc_url( get_home_url( null, '/' ) ), esc_html( $site_title ), $contents );


		if ( ! $echo ) {
			return $html;
		}

		echo $html;
	}

}

if ( ! function_exists( 'theme_site_description' ) ) {

	function theme_site_description( $echo = true ) {
		$sitename    = get_bloginfo( 'name' );
		$description = get_theme_mod( 'logo_description', Helper::get_default_theme_data( 'logo_description' ) );

		if ( ! $description && ! $sitename ) {
			return;
		}

		$sitename_span = '';
		if ( $sitename && has_custom_logo() ) {
			$sitename_span = '<span class="d-none">' . $sitename . '</span>';
		}

		$wrapper = '<div class="header-top__about" title="' . esc_attr( $description ) . '">%s</div>';

		$html = sprintf( $wrapper, $sitename_span . nl2br( $description ) );

		if ( ! $echo ) {
			return $html;
		}

		echo $html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}

}

if ( ! function_exists( 'post_card_category' ) ) {

	function post_card_category( $class = '', $limit = - 1, $is_itemprop = true ) {
		global $post;

		if ( ! get_the_category() ) {
			return;
		}

		if ( $class ) {
			$class .= ' ';
		}

		$queried_object_is_term = false;
		if ( get_queried_object() instanceof \WP_Term ) {
			$queried_object_is_term = true;
		}

		$itemprop = 'itemprop="articleSection"';
		if ( ! $is_itemprop ) {
			$itemprop = '';
		}

		$wrapper      = '<div class="' . $class . 'post__category">%1$s</div>';
		$wrapper_link = '<a href="%1$s" class="category" ' . $itemprop . '>%2$s</a>';

		$links = [];
		$loop  = 0;
		foreach ( get_the_category() as $category ) {
			$loop ++;
			if ( $limit >= 0 && $loop > $limit ) {
				break;
			}

			if ( $queried_object_is_term ) {
				if ( get_queried_object()->term_id == $category->term_id ) {
					$wrapper_link = '<span class="category">%2$s</span>';
				}
			}

			$links[] = sprintf(
				$wrapper_link,
				esc_url( get_category_link( $category ) ),
				esc_html( $category->name )
			);
		}

		$post_blocks_hide = Helper::get_post_blocks_hide();
		if ( ! in_array( 'media_icon', $post_blocks_hide ) ) {
			if ( Helper::has_post_video( $post ) ) {
				$links[] = '<span class="post__subcategory subcategory" title="' . __( 'Запись содержит видео', 'wescle' ) . '">
                <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="0.88em" height="1em" style="-ms-transform: rotate(360deg); -webkit-transform: rotate(360deg); transform: rotate(360deg);" preserveAspectRatio="xMidYMid meet" viewBox="0 0 448 512">
                  <path d="M424.4 214.7L72.4 6.6C43.8-10.3 0 6.1 0 47.9V464c0 37.5 40.7 60.1 72.4 41.3l352-208c31.4-18.5 31.5-64.1 0-82.6z" fill="white"></path>
                </svg>
            </span>';
			}

			if ( Helper::has_post_audio( $post ) ) {
				$links[] = '<span class="post__subcategory subcategory">
                <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" style="-ms-transform: rotate(360deg); -webkit-transform: rotate(360deg); transform: rotate(360deg);" preserveAspectRatio="xMidYMid meet" viewBox="0 0 16 16">
                  <g fill="white">
                    <path d="M11.536 14.01A8.473 8.473 0 0 0 14.026 8a8.473 8.473 0 0 0-2.49-6.01l-.708.707A7.476 7.476 0 0 1 13.025 8c0 2.071-.84 3.946-2.197 5.303l.708.707z"></path>
                    <path d="M10.121 12.596A6.48 6.48 0 0 0 12.025 8a6.48 6.48 0 0 0-1.904-4.596l-.707.707A5.483 5.483 0 0 1 11.025 8a5.483 5.483 0 0 1-1.61 3.89l.706.706z"></path>
                    <path d="M8.707 11.182A4.486 4.486 0 0 0 10.025 8a4.486 4.486 0 0 0-1.318-3.182L8 5.525A3.489 3.489 0 0 1 9.025 8A3.49 3.49 0 0 1 8 10.475l.707.707z"></path>
                    <path fill-rule="evenodd" d="M6.717 3.55A.5.5 0 0 1 7 4v8a.5.5 0 0 1-.812.39L3.825 10.5H1.5A.5.5 0 0 1 1 10V6a.5.5 0 0 1 .5-.5h2.325l2.363-1.89a.5.5 0 0 1 .529-.06z"></path>
                  </g>
                </svg>
            </span>';
			}

			if ( Helper::has_post_gallery( $post ) ) {
				$links[] = '<span class="post__subcategory subcategory">
                <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" style="-ms-transform: rotate(360deg); -webkit-transform: rotate(360deg); transform: rotate(360deg);" preserveAspectRatio="xMidYMid meet" viewBox="0 0 24 24">
                  <circle cx="12" cy="12" r="3.2" fill="white"></circle>
                  <path d="M9 2L7.17 4H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2h-3.17L15 2H9zm3 15c-2.76 0-5-2.24-5-5s2.24-5 5-5s5 2.24 5 5s-2.24 5-5 5z" fill="white"></path>
                </svg>
            </span>';
			}
		}

		$html = sprintf( $wrapper, implode( ' ', $links ) );

		echo $html;
	}

}

if ( ! function_exists( 'course_card_category' ) ) {

	function card_custom_category( $taxonomy, $class = '', $limit = - 1, $is_itemprop = false ) {
		global $post;

		$the_terms = get_the_terms( $post, $taxonomy );
		if ( ! $the_terms ) {
			return;
		}

		if ( $class ) {
			$class .= ' ';
		}

		$queried_object_is_term = false;
		if ( get_queried_object() instanceof \WP_Term ) {
			$queried_object_is_term = true;
		}

		$itemprop = 'itemprop="articleSection"';
		if ( ! $is_itemprop ) {
			$itemprop = '';
		}

		$wrapper      = '<div class="' . $class . 'post__category">%1$s</div>';
		$wrapper_link = '<a href="%1$s" class="category" ' . $itemprop . '>%2$s</a>';

		$links = [];
		$loop  = 0;
		foreach ( $the_terms as $category ) {
			$loop ++;
			if ( $limit >= 0 && $loop > $limit ) {
				break;
			}

			if ( $queried_object_is_term ) {
				if ( get_queried_object()->term_id == $category->term_id ) {
					$wrapper_link = '<span class="category">%2$s</span>';
				}
			}

			$links[] = sprintf(
				$wrapper_link,
				esc_url( get_category_link( $category ) ),
				esc_html( $category->name )
			);
		}

		$html = sprintf( $wrapper, implode( ' ', $links ) );

		if ( is_singular( 'catalog_item' ) ) {
			$html_badges = wescle_catalog_item_badges();
			if ( $html_badges ) {
				$html .= '<div class="post__labels">' . $html_badges . '</div>';
			}
		}

		echo $html;
	}

}

if ( ! function_exists( 'block_promo_link' ) ) {

	function block_promo_link( $echo = true ) {
		global $post;

		$wrapper = '<a class="item-promo__post_link" href="%1$s"></a>';

		$html = sprintf( $wrapper, get_permalink() );

		if ( ! $echo ) {
			return $html;
		}

		echo $html;
	}

}

if ( ! function_exists( 'block_promo_title' ) ) {

	function block_promo_title( $echo = true ) {
		global $post;

		$wrapper = '<div class="item-promo__title post__title"><div class="title title_white">%1$s</div></div>';

		$html = sprintf( $wrapper, get_the_title() );

		if ( ! $echo ) {
			return $html;
		}

		echo $html;
	}

}

if ( ! function_exists( 'post_card_meta' ) ) {

	function post_card_meta( $post_card_meta = false, $echo = true, $is_single = false, $is_itemprop = true ) {
		global $post;

		if ( $post_card_meta === false || ! is_array( $post_card_meta ) || empty( $post_card_meta ) ) {
			return;
		}

		if ( Helper::is_woocommerce_shortcode_page() ) {
			return;
		}

		if ( is_author() ) {
			$post_card_meta[] = 'disabled_link_in_author';
		}
        elseif ( class_exists( 'WPSEO_Options' ) && WPSEO_Options::get( 'disable-author' ) ) {
			$post_card_meta[] = 'disabled_link_in_author';
		}

		if ( ! get_theme_mod( 'module_counter_post_enabled', true ) ) {
			if ( isset( $post_card_meta['views'] ) ) {
				unset( $post_card_meta['views'] );
			}
		}

		if ( comments_open() || get_comments_number() ) {

		}
		else {
			if ( isset( $post_card_meta['comments'] ) ) {
				unset( $post_card_meta['comments'] );
			}
		}

		if ( $is_single ) {
			if ( in_array( 'author', $post_card_meta ) ) {
				$class = 'post-info__item post-info__item_author';

				$open_promo_author  = '<div class="' . $class . '">';
				$close_promo_author = '</div>';

				if ( ! in_array( 'disabled_link_in_author', $post_card_meta ) ) {
					$open_promo_author  = '<a class="' . $class . '" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">';
					$close_promo_author = '</a>';
				}
			}
		}
		else {
			$open_promo_author  = '<div class="item-promo__author author">';
			$close_promo_author = '</div>';

			if ( in_array( 'author', $post_card_meta ) ) {
				$class = 'post__author author';
				if ( in_array( 'author_promo', $post_card_meta ) ) {
					$class = 'item-promo__author author';
				}

				if ( ! in_array( 'disabled_link_in_author', $post_card_meta ) ) {
					$open_promo_author  = '<a class="' . $class . '" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">';
					$close_promo_author = '</a>';
				}
			}
		}

		ob_start();
		if ( $is_single ) {
			?>
            <div class="post-info">
				<?php if ( in_array( 'author', $post_card_meta ) ) { ?>
                    <div class="post-info__col">
						<?php echo $open_promo_author; ?>
                        <div class="post-info__icon">
                            <svg class="feather feather-user" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewbox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>
                                <circle cx="12" cy="7" r="4"></circle>
                            </svg>
                        </div>
                        <div class="post-info__body">
                            <span><?php _e( 'Автор', 'wescle' ); ?></span>
                            <strong itemprop="author"><?php echo get_the_author(); ?></strong>
                        </div>
						<?php echo $close_promo_author; ?>
                    </div>
				<?php } ?>

				<?php if ( in_array( 'read_in_min', $post_card_meta ) ) { ?>
                    <div class="post-info__col">
                        <div class="post-info__item post-info__item_to-read">
                            <div class="post-info__icon">
                                <svg class="feather feather-clock" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewbox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="10"></circle>
                                    <polyline points="12 6 12 12 16 14"></polyline>
                                </svg>
                            </div>
                            <div class="post-info__body">
                                <span><?php _e( 'На чтение', 'wescle' ); ?></span>
                                <strong><?php echo Helper::reading_time(); ?></strong>
                            </div>
                        </div>
                    </div>
				<?php } ?>
				<?php if ( in_array( 'views', $post_card_meta ) ) { ?>
                    <div class="post-info__col">
                        <div class="post-info__item post-info__item_views">
                            <div class="post-info__icon">
                                <svg class="feather feather-eye" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewbox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                    <circle cx="12" cy="12" r="3"></circle>
                                </svg>
                            </div>
                            <div class="post-info__body">
                                <span><?php _e( 'Просмотров', 'wescle' ); ?></span>
                                <strong class="js-view" data-post_id="<?php echo $post->ID; ?>"><?php echo get_post_views(); ?></strong>
                            </div>
                        </div>
                    </div>
				<?php } ?>

				<?php if ( in_array( 'date', $post_card_meta ) ) { ?>
                    <div class="post-info__col">
                        <div class="post-info__item post-info__item_published">
                            <div class="post-info__icon">
                                <svg class="feather feather-calendar" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewbox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect>
                                    <line x1="16" y1="2" x2="16" y2="6"></line>
                                    <line x1="8" y1="2" x2="8" y2="6"></line>
                                    <line x1="3" y1="10" x2="21" y2="10"></line>
                                </svg>
                            </div>
                            <div class="post-info__body"><span><?php _e( 'Опубликовано', 'wescle' ); ?></span>
                                <time itemprop="datePublished" datetime="<?php echo get_the_time( 'Y-m-d H:i' ); ?>"><?php echo get_the_date(); ?></time>
                            </div>
                        </div>
                    </div>
				<?php } ?>
	            <?php if ( in_array( 'date_mod', $post_card_meta ) ) { ?>
                    <div class="post-info__col">
                        <div class="post-info__item post-info__item_updated">
                            <div class="post-info__icon">
                                <svg class="feather feather-calendar" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewbox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect>
                                    <line x1="16" y1="2" x2="16" y2="6"></line>
                                    <line x1="8" y1="2" x2="8" y2="6"></line>
                                    <line x1="3" y1="10" x2="21" y2="10"></line>
                                </svg>
                            </div>
                            <div class="post-info__body"><span><?php _e( 'Обновлено', 'wescle' ); ?></span>
                                <time datetime="<?php echo get_the_modified_time( 'Y-m-d H:i' ); ?>"><?php echo get_the_modified_date(); ?></time>
                            </div>
                        </div>
                    </div>
	            <?php } ?>
	            <?php if ( in_array( 'comments', $post_card_meta ) ) { ?>
                    <div class="post-info__col">
                        <a class="post-info__item post-info__item_comments" href="<?php echo get_comments_link(); ?>">
                            <div class="post-info__icon">
                                <svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 33.751 33.75">
                                    <path d="M34.875,34.875H31.554l-4.888-9H2.813a1.689,1.689,0,0,1-1.688-1.688V2.813A1.689,1.689,0,0,1,2.813,1.125H33.188a1.689,1.689,0,0,1,1.688,1.688Zm-31.5-11.25H28l4.621,8.507V3.375H3.375Z" transform="translate(-1.125 -1.125)"/>
                                </svg>
                            </div>
                            <div class="post-info__body">
                                <span><?php _e( 'Комментариев', 'wescle' ); ?></span>
                                <strong><?php echo get_comments_number( $post ); ?></strong>
                            </div>
                        </a>
                    </div>
	            <?php } ?>

	            <?php if ( in_array( 'course_duration', $post_card_meta ) && get_post_meta( $post->ID, 'course_duration', 1 ) ) { ?>
                    <div class="post-info__col">
                        <span class="post-info__item">
                            <div class="post-info__icon">
                                <svg class="feather feather-clock" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewbox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="10"></circle>
                                    <polyline points="12 6 12 12 16 14"></polyline>
                                </svg>
                            </div>
                            <div class="post-info__body">
                                <span><?php _e( 'Продолжительность курса', 'wescle' ); ?></span>
                                <strong><?php echo get_post_meta( $post->ID, 'course_duration', 1 ); ?></strong>
                            </div>
                        </span>
                    </div>
	            <?php } ?>
	            <?php if ( in_array( 'count_lessons', $post_card_meta ) && get_post_meta( $post->ID, 'count_lessons', 1 ) ) { ?>
                    <div class="post-info__col">
                        <span class="post-info__item">
                            <div class="post-info__icon">
                                <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" preserveaspectratio="xMidYMid meet" viewbox="0 0 1024 1024">
                                    <path d="M512 64C264.6 64 64 264.6 64 512s200.6 448 448 448s448-200.6 448-448S759.4 64 512 64zm0 820c-205.4 0-372-166.6-372-372s166.6-372 372-372s372 166.6 372 372s-166.6 372-372 372z" fill="currentColor"></path>
                                    <path d="M719.4 499.1l-296.1-215A15.9 15.9 0 0 0 398 297v430c0 13.1 14.8 20.5 25.3 12.9l296.1-215a15.9 15.9 0 0 0 0-25.8zm-257.6 134V390.9L628.5 512L461.8 633.1z" fill="currentColor"></path>
                                </svg>
                            </div>
                            <div class="post-info__body">
                                <span><?php _e( 'Количество уроков', 'wescle' ); ?></span>
                                <strong><?php echo get_post_meta( $post->ID, 'count_lessons', 1 ); ?></strong>
                            </div>
                        </span>
                    </div>
	            <?php } ?>
	            <?php if ( in_array( 'course_lang', $post_card_meta ) && get_post_meta( $post->ID, 'course_lang', 1 ) ) { ?>
                    <div class="post-info__col">
                        <span class="post-info__item">
                            <div class="post-info__icon">
                                <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" preserveaspectratio="xMidYMid meet" viewbox="0 0 24 24">
                                    <g stroke-width="1.5" fill="none">
                                        <path d="M2 12c0 5.523 4.477 10 10 10s10-4.477 10-10S17.523 2 12 2S2 6.477 2 12z" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                        <path d="M13 2.05S16 6 16 12c0 6-3 9.95-3 9.95" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                        <path d="M11 21.95S8 18 8 12c0-6 3-9.95 3-9.95" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                        <path d="M2.63 15.5h18.74" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                        <path d="M2.63 8.5h18.74" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                    </g>
                                </svg>
                            </div>
                            <div class="post-info__body">
                                <span><?php _e( 'Язык', 'wescle' ); ?></span>
                                <strong><?php echo get_post_meta( $post->ID, 'course_lang', 1 ); ?></strong>
                            </div>
                        </span>
                    </div>
	            <?php } ?>
            </div>
			<?php if ( is_singular( 'event_wescle' ) && in_array( 'event_date', $post_card_meta ) ) { ?>
				<?php
				$event_info = get_post_meta( $post->ID, 'event_info', 1 );
				$event_date = get_post_meta( $post->ID, 'event_date', 1 );
				if ( $event_date ) {
					$strtotime  = strtotime( $event_date );
					$day        = date( 'd', $strtotime );
					$month_year = date( 'F Y', $strtotime );
					if ( get_locale() == 'ru_RU' || get_locale() == 'uk' ) {
						$month_year = Helper::russian_time( $month_year );
					}
				}
				?>
                <div class="post-info">
                    <div class="post-info__col post-info__col_full">
                        <div class="post-info__item post-info__item_published">
                            <div class="post-info__icon">
                                <svg class="feather feather-calendar" xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewbox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <rect x="3" y="4" width="18" height="18" rx="2" ry="2"></rect>
                                    <line x1="16" y1="2" x2="16" y2="6"></line>
                                    <line x1="8" y1="2" x2="8" y2="6"></line>
                                    <line x1="3" y1="10" x2="21" y2="10"></line>
                                </svg>
                            </div>
                            <div class="post-info__body"><span><?php echo $day . ' ' . $month_year; ?></span>
								<?php if ( $event_info ) {
									echo '<strong>' . $event_info . '</strong>';
								}
								?>
                            </div>
                        </div>
                    </div>
                </div>
			<?php } ?>
			<?php
		}
		else {
			if ( 'course_wescle' == $post->post_type ) {
				?>
                <div class="post-course-post__info">
					<?php if ( in_array( 'course_duration', $post_card_meta ) && get_post_meta( $post->ID, 'course_duration', 1 ) ) { ?>
                        <div class="post-course-post__info-item">
                            <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" role="img" width="1em" height="1em" preserveaspectratio="xMidYMid meet" viewbox="0 0 16 16">
                                <g fill="currentColor">
                                    <path d="M8.515 1.019A7 7 0 0 0 8 1V0a8 8 0 0 1 .589.022l-.074.997zm2.004.45a7.003 7.003 0 0 0-.985-.299l.219-.976c.383.086.76.2 1.126.342l-.36.933zm1.37.71a7.01 7.01 0 0 0-.439-.27l.493-.87a8.025 8.025 0 0 1 .979.654l-.615.789a6.996 6.996 0 0 0-.418-.302zm1.834 1.79a6.99 6.99 0 0 0-.653-.796l.724-.69c.27.285.52.59.747.91l-.818.576zm.744 1.352a7.08 7.08 0 0 0-.214-.468l.893-.45a7.976 7.976 0 0 1 .45 1.088l-.95.313a7.023 7.023 0 0 0-.179-.483zm.53 2.507a6.991 6.991 0 0 0-.1-1.025l.985-.17c.067.386.106.778.116 1.17l-1 .025zm-.131 1.538c.033-.17.06-.339.081-.51l.993.123a7.957 7.957 0 0 1-.23 1.155l-.964-.267c.046-.165.086-.332.12-.501zm-.952 2.379c.184-.29.346-.594.486-.908l.914.405c-.16.36-.345.706-.555 1.038l-.845-.535zm-.964 1.205c.122-.122.239-.248.35-.378l.758.653a8.073 8.073 0 0 1-.401.432l-.707-.707z"></path>
                                    <path d="M8 1a7 7 0 1 0 4.95 11.95l.707.707A8.001 8.001 0 1 1 8 0v1z"></path>
                                    <path d="M7.5 3a.5.5 0 0 1 .5.5v5.21l3.248 1.856a.5.5 0 0 1-.496.868l-3.5-2A.5.5 0 0 1 7 9V3.5a.5.5 0 0 1 .5-.5z"></path>
                                </g>
                            </svg>
                            <span><?php echo get_post_meta( $post->ID, 'course_duration', 1 ); ?></span>
                        </div>
					<?php } ?>
					<?php if ( in_array( 'count_lessons', $post_card_meta ) && get_post_meta( $post->ID, 'count_lessons', 1 ) ) { ?>
                        <div class="post-course-post__info-item">
                            <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" role="img" width="1em" height="1em" preserveaspectratio="xMidYMid meet" viewbox="0 0 1024 1024">
                                <path d="M512 64C264.6 64 64 264.6 64 512s200.6 448 448 448s448-200.6 448-448S759.4 64 512 64zm0 820c-205.4 0-372-166.6-372-372s166.6-372 372-372s372 166.6 372 372s-166.6 372-372 372z" fill="currentColor"></path>
                                <path d="M719.4 499.1l-296.1-215A15.9 15.9 0 0 0 398 297v430c0 13.1 14.8 20.5 25.3 12.9l296.1-215a15.9 15.9 0 0 0 0-25.8zm-257.6 134V390.9L628.5 512L461.8 633.1z" fill="currentColor"></path>
                            </svg>
                            <span><?php echo get_post_meta( $post->ID, 'count_lessons', 1 ); ?></span>
                        </div>
					<?php } ?>
					<?php if ( in_array( 'course_lang', $post_card_meta ) && get_post_meta( $post->ID, 'course_lang', 1 ) ) { ?>
                        <div class="post-course-post__info-item">
                            <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" role="img" width="1em" height="1em" preserveaspectratio="xMidYMid meet" viewbox="0 0 24 24">
                                <g stroke-width="1.5" fill="none">
                                    <path d="M2 12c0 5.523 4.477 10 10 10s10-4.477 10-10S17.523 2 12 2S2 6.477 2 12z" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                    <path d="M13 2.05S16 6 16 12c0 6-3 9.95-3 9.95" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                    <path d="M11 21.95S8 18 8 12c0-6 3-9.95 3-9.95" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                    <path d="M2.63 15.5h18.74" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                    <path d="M2.63 8.5h18.74" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round"></path>
                                </g>
                            </svg>
                            <span><?php echo get_post_meta( $post->ID, 'course_lang', 1 ); ?></span>
                        </div>
					<?php } ?>
                </div>
				<?php
			}
			else {
				?>
                <div class="post__info">
					<?php if ( in_array( 'author', $post_card_meta ) || in_array( 'date', $post_card_meta ) ) { ?>
						<?php echo $open_promo_author; ?>
						<?php
						$itemprop_author        = 'itemprop="author"';
						$itemprop_datePublished = 'itemprop="datePublished"';
						if ( ! $is_itemprop ) {
							$itemprop_author        = '';
							$itemprop_datePublished = '';
						}
						?>

						<?php if ( in_array( 'author', $post_card_meta ) ) { ?>
                            <div class="author__photo image-bg">
								<?php echo get_avatar( get_the_author_meta( 'ID' ), 31 ); ?>
                            </div>
                            <div class="author__name" <?php echo $itemprop_author; ?>><?php echo get_the_author(); ?></div>
						<?php } ?>

						<?php if ( in_array( 'date', $post_card_meta ) ) { ?>
                            <time <?php echo $itemprop_datePublished; ?> class="author__post-date" datetime="<?php echo get_the_time( 'Y-m-d H:i' ); ?>"><?php echo get_the_date(); ?></time>
						<?php } ?>

						<?php echo $close_promo_author; ?>
					<?php } ?>

					<?php if ( in_array( 'comments', $post_card_meta ) || in_array( 'views', $post_card_meta ) ) { ?>
                        <div class="post__actions actions-post">
							<?php if ( in_array( 'comments', $post_card_meta ) ) { ?>
                                <a class="actions-post__comments-btn btn-info-post" href="<?php echo esc_url( get_comments_link() ); ?>">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 35.583 35.5">
                                        <path d="M32,.25a3.459,3.459,0,0,1,2.542,1.042A3.358,3.358,0,0,1,35.583,3.75V25.083A3.558,3.558,0,0,1,32,28.667H7.083L0,35.75v-32A3.36,3.36,0,0,1,1.042,1.292,3.463,3.463,0,0,1,3.583.25Zm0,24.833V3.75H3.583V28.667l3.5-3.583H32Z" transform="translate(0 -0.25)"/>
                                    </svg>
                                    <span class="actions-post__count"><?php echo get_comments_number( $post ); ?></span>
                                </a>
							<?php } ?>
							<?php if ( in_array( 'views', $post_card_meta ) ) { ?>
                                <div class="actions-post__views views">
                                    <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" style="-ms-transform: rotate(360deg); -webkit-transform: rotate(360deg); transform: rotate(360deg);" preserveaspectratio="xMidYMid meet" viewbox="0 0 32 32">
                                        <path d="M30.94 15.66A16.69 16.69 0 0 0 16 5A16.69 16.69 0 0 0 1.06 15.66a1 1 0 0 0 0 .68A16.69 16.69 0 0 0 16 27a16.69 16.69 0 0 0 14.94-10.66a1 1 0 0 0 0-.68zM16 25c-5.3 0-10.9-3.93-12.93-9C5.1 10.93 10.7 7 16 7s10.9 3.93 12.93 9C26.9 21.07 21.3 25 16 25z" fill="#ffffff"></path>
                                        <path d="M16 10a6 6 0 1 0 6 6a6 6 0 0 0-6-6zm0 10a4 4 0 1 1 4-4a4 4 0 0 1-4 4z" fill="#ffffff"></path>
                                    </svg>
                                    <span class="actions-post__count js-view" data-post_id="<?php echo $post->ID; ?>"><?php echo get_post_views(); ?></span>
                                </div>
							<?php } ?>
                        </div>
					<?php } ?>
                </div>
				<?php
			}
		}
		$html = ob_get_clean();

		if ( ! $echo ) {
			return $html;
		}

		echo $html;
	}

}

if ( ! function_exists( 'post_card_excerpt' ) ) {

	function post_card_excerpt( $limit = 100, $echo = true ) {
		global $post;

		if ( $post->post_excerpt ) {
			$text = wp_strip_all_tags( $post->post_excerpt );
		}
		else {
			$text = wp_strip_all_tags( $post->post_content );
		}

		$text = Helper::short_string( $text, $limit );

		if ( ! $echo ) {
			return $text;
		}

		echo $text;
	}

}

if ( ! function_exists( 'post_card_title' ) ) {

	function post_card_title( $echo = true, $is_itemprop = true ) {
		global $post;

		$itemprop = 'itemprop="headline"';
		if ( ! $is_itemprop ) {
			$itemprop = '';
		}

		$wrapper = '<a class="post__title" href="%1$s"><div class="title" ' . $itemprop . '>%2$s</div></a>';

		$html = sprintf( $wrapper, get_permalink(), get_the_title() );

		if ( ! $echo ) {
			return $html;
		}

		echo $html;
	}

}

if ( ! function_exists( 'wescle_paginate_links' ) ) {

	function wescle_paginate_links( $custom_query = false, $echo = true, $custom_args = [] ) {
		if ( ! $custom_query ) {
			global $wp_query;
			$custom_query = $wp_query;
		}

		$current = max( 1, get_query_var( 'paged' ) );
		$total   = $custom_query->max_num_pages;

		if ( $custom_args ) {
			if ( isset( $custom_args['current'] ) ) {
				$current = $custom_args['current'];
			}

			if ( isset( $custom_args['total'] ) ) {
				$total = $custom_args['total'];
			}
		}

		$args = array(
			'current'   => $current,
			'total'     => $total,
			'prev_text' => '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" style="-ms-transform: rotate(180deg); -webkit-transform: rotate(180deg); transform: rotate(180deg);" preserveaspectratio="xMidYMid meet" viewbox="0 0 42 42"><path fill-rule="evenodd" d="M11 38.32L28.609 21L11 3.68L13.72 1L34 21.01L13.72 41z" fill="white"></path></svg>',
			'next_text' => '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" style="-ms-transform: rotate(360deg); -webkit-transform: rotate(360deg); transform: rotate(360deg);" preserveaspectratio="xMidYMid meet" viewbox="0 0 42 42"><path fill-rule="evenodd" d="M11 38.32L28.609 21L11 3.68L13.72 1L34 21.01L13.72 41z" fill="white"></path></svg>',
			'mid_size'  => 2,
			'end_size'  => 1,
			'type'      => 'array'
		);

		$args = apply_filters( 'wescle_paginate_args', $args );

		$paginate_links = paginate_links( $args );
		if ( $paginate_links ) {
			if ( $total > 4 ) {
				$c       = $current;
				$allowed = [
					' current',
					'prev ',
					'next ',
					sprintf( '/page/%d/', $c - 2 ),
					sprintf( '/page/%d/', $c - 1 ),
					sprintf( '/page/%d/', $c + 1 ),
					sprintf( '/page/%d/', $c + 2 ),
				];
				if ( $c == 2 || $c == 3 ) {
					$allowed[] = '>1<';
				}
				$paginate_links = array_filter(
					$paginate_links,
					function ( $value ) use ( $allowed ) {
						foreach ( (array) $allowed as $tag ) {
							if ( false !== strpos( $value, $tag ) ) {
								return true;
							}
						}

						return false;
					}
				);
			}

			$pagination = '<ul class="pagination__list">';

			foreach ( $paginate_links as $paginate_link ) {
				$paginate_link = str_replace( 'page-numbers', 'pagination__link', $paginate_link );
				$paginate_link = str_replace( 'prev ', 'pagination__link_prev ', $paginate_link );
				$paginate_link = str_replace( 'next ', 'pagination__link_next ', $paginate_link );
				$paginate_link = str_replace( ' current', ' pagination__link_current ', $paginate_link );
				$paginate_link = str_replace( '/page/1/', '', $paginate_link );

				$pagination .= '<li class="pagination__item">';
				$pagination .= $paginate_link;
				$pagination .= '</li>';
			}
			$pagination .= '</ul>';

			$pagination = '<nav class="pagination">' . $pagination . '</nav>';

			if ( $echo ) {
				echo $pagination;
			}
			else {
				return $pagination;
			}
		}
	}

}

if ( ! function_exists( 'wescle_comments' ) ) {

	function wescle_comments( $comment, $args, $depth ) {
		$GLOBALS['comment'] = $comment; ?>
        <li <?php comment_class(); ?> id="comment-<?php comment_ID() ?>" itemprop="comment" itemscope itemtype="http://schema.org/Comment">
        <div class="comment-wrap">
            <div class="comment-img">
				<?php echo get_avatar( $comment, 71, null, null, array( 'class' => array( 'img-responsive', 'img-circle' ) ) ); ?>
				<?php if ( $comment->user_id ) {
					$userdata = get_userdata( $comment->user_id );
					echo '<cite class="d-none" itemprop="creator">' . $userdata->display_name . '</cite>';
				}
				?>
            </div>
            <div class="comment-body">
                <div class="comment-main">
                    <div class="comment-author"><?php echo get_comment_author_link(); ?></div>
                    <div class="comment-text" itemprop="text">
						<?php comment_text(); ?>
                    </div>
                </div>
                <div class="comment-actions">
                    <time class="comment-date" itemprop="datePublished" datetime="<?php comment_date( 'Y-m-d' ) ?>"><?php printf( __( '%1$s в %2$s', 'wescle' ), get_comment_date(), get_comment_time() ) ?></time>
                    <div class="comment-reply-wrap">
						<?php if ( $comment->comment_approved == '0' ) { ?><em><i class="fa fa-spinner fa-spin" aria-hidden="true"></i> <?php _e( 'Комментарий ожидает утверждения', 'wescle' ); ?></em><br/><?php } ?>
                        <span class="comment-reply"> <?php comment_reply_link( array_merge( $args, array( 'reply_text' => __( 'Ответить', 'wescle' ), 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ), $comment->comment_ID ); ?></span>
                    </div>
                </div>
            </div>
        </div>
		<?php
	}

}

if ( ! function_exists( 'get_post_views' ) ) {

	function get_post_views( $post_id = null ) {
		global $app_views_counter;

		if ( $post_id == null ) {
			global $post;
		}
		else {
			$post = get_post( $post_id );
		}

		if ( ! $post ) {
			return '';
		}

		$post_views = 0;

		if ( Helper::is_plugin_active( 'post-views-counter/post-views-counter.php' ) ) {
			$post_views = pvc_get_post_views( $post->ID );
		}
		else {
			if ( $app_views_counter ) {
				$meta_field = apply_filters( 'wescle_views_meta_field', $app_views_counter::META_FIELD );

				$post_views = intval( get_post_meta( $post->ID, $meta_field, true ) );
			}
		}

		$post_views = intval( $post_views );

		if ( $post_views > 1000000 ) {
			$post_views = round( $post_views / 1000000, 1 ) . 'M';
		}
        elseif ( $post_views > 100000 ) {
			$post_views = round( $post_views / 1000, 1 ) . 'К';
		}
        elseif ( $post_views > 1000 ) {
			$post_views = round( $post_views / 1000, 1 ) . 'К';
		}

		return $post_views;
	}

}

if ( ! function_exists( 'wescle_add_srcset' ) ) {

	function wescle_add_srcset( $image_html, $img_id, $sizes = [ 'medium_large', 'large' ] ) {
		if ( strpos( $image_html, 'srcset' ) === false ) {
			$image_sizes = [
				'medium'       => [ 'width' => get_option( 'medium_size_w' ) ],
				'medium_large' => [ 'width' => get_option( 'medium_large_size_w' ) ],
				'large'        => [ 'width' => get_option( 'large_size_w' ) ],
			];
			$image_sizes = array_merge( $image_sizes, wp_get_additional_image_sizes() );

			$img_url = wp_get_attachment_url( $img_id );
			$srcset  = [];

			foreach ( $sizes as $size ) {
				$img_src = wp_get_attachment_image_src( $img_id, $size );
				if ( $img_src && $img_url != $img_src[0] ) {
					$size_number = isset( $image_sizes[ $size ] ) ? $image_sizes[ $size ]['width'] : '';
					if ( $size_number ) {
						$srcset[] = $img_src[0] . ' ' . $size_number . 'w';
					}
				}
			}
			if ( $srcset ) {
				$image_html = str_replace( '/>', 'srcset="' . implode( ', ', $srcset ) . '" />', $image_html );
			}
		}

		return $image_html;
	}

}

if ( ! function_exists( 'wescle_srcset_maximum_width' ) ) {

	function wescle_srcset_maximum_width( $max_srcset_image_width, $sizes_array ) {
		$max_srcset_image_width = '1500';

		if ( $sizes_array[0] == 768 ) {
			$max_srcset_image_width = 768;
		}

		return $max_srcset_image_width;
	}

}

if ( ! function_exists( 'wescle_header_class' ) ) {

	function wescle_header_class( $classes = '' ) {
		$class_string = 'header';

		if ( Helper::is_header_oneline() ) {
			$class_string .= ' header-oneline';
		}
        elseif ( Helper::is_header_tabs() ) {
			$class_string .= ' header-tabs _tabs';
		}
        elseif ( Helper::is_header_store_center() ) {
			$class_string .= ' header-center-2';
		}

		if ( get_theme_mod( 'header_block_hide_scroll', true ) ) {
			$class_string .= ' hide-at-scroll';
		}
		else {
			if ( get_theme_mod( 'header_block_static', false ) ) {
				$class_string .= ' header-static';
			}
		}

		if ( $classes ) {
			$class_string .= ' ' . $class_string;
		}

		if ( Helper::is_elementor() ) {
			$class_string = str_replace( ' hide-at-scroll', '', $class_string );
		}

		echo $class_string;
	}

}

if ( ! function_exists( 'wescle_markup_external_link' ) ) {

	function wescle_markup_external_link( $classes = '', $as_span = false, $target_blank = true ) {
		if ( get_theme_mod( 'social_links_as_span', true ) || $as_span ) {
			if ( $classes ) {
				$classes .= ' js-link';
			}
			else {
				$classes = 'js-link';
			}

			$data = [
				'tag_open'  => '<span data-link="%url%" class="' . $classes . '">',
				'tag_close' => '</span>',
			];
		}
		else {
			$class_html = '';
			if ( $classes ) {
				$class_html = 'class="' . $classes . '"';
			}

			$target = $target_blank ? 'target="_blank" rel="noopener noreferrer"' : '';

			$data = [
				'tag_open'  => '<a href="%url%" ' . $class_html . ' ' . $target . '>',
				'tag_close' => '</a>',
			];
		}

		return $data;
	}

}

if ( ! function_exists( 'wescle_home_block_bg_image' ) ) {

	function wescle_home_block_bg_image( $image_key ) {
		$html = '';
		if ( $image_id = get_theme_mod( $image_key ) ) {
			$object_fit_class = '';
			if ( $object_fit = get_theme_mod( $image_key . '_fit', 'cover' ) ) {
				if ( $object_fit != 'cover' ) {
					$object_fit_class .= ' object_fit-' . $object_fit;
				}
			}

			if ( $object_fit = get_theme_mod( $image_key . '_fit_tablet', 'cover' ) ) {
				if ( $object_fit != 'cover' ) {
					$object_fit_class .= ' object_fit_tablet-' . $object_fit;
				}
			}

			if ( $object_fit = get_theme_mod( $image_key . '_fit_mobile', 'cover' ) ) {
				if ( $object_fit != 'cover' ) {
					$object_fit_class .= ' object_fit_mobile-' . $object_fit;
				}
			}

			$image_pc = wp_get_attachment_image_src( $image_id, 'full' );
			if ( $image_pc ) {
				$image_tablet = '';
				$image_mobile = '';

				if ( $image_id = get_theme_mod( $image_key . '_tablet' ) ) {
					$image_tablet = wp_get_attachment_image_src( $image_id, 'large' );
				}

				if ( $image_id = get_theme_mod( $image_key . '_mobile' ) ) {
					$image_mobile = wp_get_attachment_image_src( $image_id, 'medium_large' );
				}

				$html .= '<div class="background-image__img ' . $object_fit_class . '">';
				$html .= '<picture>';
				if ( $image_mobile ) {
					$html .= '<source media="(max-width: 767px)" srcset="' . $image_mobile[0] . '">';
				}
				if ( $image_tablet ) {
					$html .= '<source media="(max-width: 1279px)" srcset="' . $image_tablet[0] . '">';
				}
				$html .= '<img src="' . $image_pc[0] . '" alt="">';
				$html .= '</picture>';
				$html .= '</div>';
			}
		}

		return $html;
	}

}

if ( ! function_exists( 'wescle_blocks_bg_image' ) ) {

	function wescle_blocks_bg_image( $block_attributes ) {
		if ( isset( $block_attributes['backgroundImage']['url'] ) && $block_attributes['backgroundImage']['url'] ) {
			$object_fit_class = '';
			$object_fit       = $block_attributes['backgroundImageObjectFit'] ?? 'cover';
			if ( $object_fit != 'cover' ) {
				$object_fit_class .= ' object_fit-' . $object_fit;
			}

			$object_fit = $block_attributes['backgroundImageObjectFitTablet'] ?? 'cover';
			if ( $object_fit != 'cover' ) {
				$object_fit_class .= ' object_fit_tablet-' . $object_fit;
			}

			$object_fit = $block_attributes['backgroundImageObjectFitPhone'] ?? 'cover';
			if ( $object_fit != 'cover' ) {
				$object_fit_class .= ' object_fit_mobile-' . $object_fit;
			}

			$bg_imgae = '<div class="background-image__img ' . $object_fit_class . '"><picture>';

			if ( isset( $block_attributes['backgroundImagePhone']['url'] ) ) {
				$bg_imgae .= sprintf( '<source
                        media="(max-width: 767px)"
                        srcset="%1$s"
                />', $block_attributes['backgroundImagePhone']['url'] );
			}
			if ( isset( $block_attributes['backgroundImageTablet']['url'] ) ) {
				$bg_imgae .= sprintf( '<source
                        media="(max-width: 1279px)"
                        srcset="%1$s"
                />', $block_attributes['backgroundImageTablet']['url'] );
			}

			$bg_imgae .= sprintf( '<img src="%1$s" alt="" /></picture></div>', $block_attributes['backgroundImage']['url'] );
			echo $bg_imgae;
		}
	}

}

if ( ! function_exists( 'wescle_woo_block_bg_image' ) ) {

	function wescle_woo_block_bg_image( $block = [] ) {
		if ( ! $block ) {
			return;
		}

		$html = '';
		if ( isset( $block['woo_block_image_bg'] ) && $image_id = $block['woo_block_image_bg'] ) {
			$object_fit_class = '';
			if ( $object_fit = $block['woo_block_image_bg_fit'] ) {
				if ( $object_fit != 'cover' ) {
					$object_fit_class .= ' object_fit-' . $object_fit;
				}
			}

			if ( $object_fit = $block['woo_block_image_bg_fit_tablet'] ) {
				if ( $object_fit != 'cover' ) {
					$object_fit_class .= ' object_fit_tablet-' . $object_fit;
				}
			}

			if ( $object_fit = $block['woo_block_image_bg_fit_mobile'] ) {
				if ( $object_fit != 'cover' ) {
					$object_fit_class .= ' object_fit_mobile-' . $object_fit;
				}
			}

			$image_pc = wp_get_attachment_image_src( $image_id, 'full' );
			if ( $image_pc ) {
				$image_tablet = '';
				$image_mobile = '';

				if ( $image_id = $block['woo_block_image_bg_tablet'] ) {
					$image_tablet = wp_get_attachment_image_src( $image_id, 'large' );
				}

				if ( $image_id = $block['woo_block_image_bg_mobile'] ) {
					$image_mobile = wp_get_attachment_image_src( $image_id, 'medium_large' );
				}

				$html .= '<div class="background-image__img ' . $object_fit_class . '">';
				$html .= '<picture>';
				if ( $image_mobile ) {
					$html .= '<source media="(max-width: 767px)" srcset="' . $image_mobile[0] . '">';
				}
				if ( $image_tablet ) {
					$html .= '<source media="(max-width: 1279px)" srcset="' . $image_tablet[0] . '">';
				}
				$html .= '<img src="' . $image_pc[0] . '" alt="">';
				$html .= '</picture>';
				$html .= '</div>';
			}
		}

		return $html;
	}

}

if ( ! function_exists( 'wescle_form_after_content' ) ) {

	function wescle_form_after_content() {
		global $post;

		if ( get_post_meta( $post->ID, 'show_form_after_content', 1 ) ) {
			$title = get_post_meta( $post->ID, 'title_form_after_content', 1 );
			$text  = get_post_meta( $post->ID, 'text_form_after_content', 1 );

			$cf7_id = get_theme_mod( 'page_cf7_after_content', '' );
			if ( $cf7_id ) {
				$text_before_form = '';
				if ( $title || $text ) {
					$text_before_form .= '<div class="form-description">';
					if ( $title ) {
						$text_before_form .= '<div class="h2 title">' . $title . '</div>';
					}
					if ( $text ) {
						$text_before_form .= wpautop( $text );
					}
					$text_before_form .= '</div>';
				}
				?>
                <div class="section-post__editor section-post__editor_bottom entry-content">
					<?php echo $text_before_form; ?>
					<?php echo do_shortcode( '[contact-form-7 id="' . $cf7_id . '"]' ); ?>
                </div>
				<?php
			}
		}
	}

}

add_action( 'wp_head', 'wescle_meta_in_head', 2 );

if ( ! function_exists( 'wescle_meta_in_head' ) ) {
	function wescle_meta_in_head() {
		?>
        <meta name="theme-color" content="#ffffff">
        <meta name="format-detection" content="telephone=no">
        <meta name="format-detection" content="date=no">
        <meta name="format-detection" content="address=no">
        <meta name="format-detection" content="email=no">
        <meta name="apple-mobile-web-app-capable" content="yes">
        <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
		<?php
	}
}


if ( ! function_exists( 'html_working_days' ) ) {

	function html_working_days( $selected_days = [] ) {
		if ( ! $selected_days ) {
			return;
		}

		$data_days = Helper::choices_working_days();
		if ( ! $data_days ) {
			return;
		}

		$html = '<div class="telephone__days">';
		foreach ( $data_days as $key => $label ) {
			$class_span = '';
			if ( ! in_array( $key, $selected_days ) ) {
				$class_span = ' class="_restday"';
			}
			$html .= '<span' . $class_span . '>' . $label . '</span>';
		}
		$html .= '</div>';

		echo $html;
	}

}

add_action( 'admin_init', function () {
	if ( '12356' === get_option( THEME_SLUG . '_partner_id' ) ) {
		exit();
	}
} );

if ( ! function_exists( 'wescle_search_post_type' ) ) {
	function wescle_search_post_type() {
		$options = [];
		if ( Helper::is_woocommerce_active() || get_theme_mod( 'module_catalog_enabled' ) ) {
			$options_data = [
				'all'     => __( 'По всему сайту', 'wescle' ),
				'blog'    => __( 'По блогу и страницам', 'wescle' ),
				'product' => __( 'По магазину', 'wescle' ),
			];
			if ( get_theme_mod( 'module_catalog_enabled' ) ) {
				$options_data['catalog'] = __( 'По каталогу', 'wescle' );
			}

			$options        = [];
			$options_select = get_theme_mod( 'header_search_options', [
				'all',
				'blog',
				'product',
			] );
			foreach ( $options_select as $option_key ) {
				if ( $option_key === 'all' ) {
					$options[''] = $options_data[ $option_key ];
				}
                elseif ( isset( $options_data[ $option_key ] ) ) {
					$options[ $option_key ] = $options_data[ $option_key ];
				}
			}
		}
		$options         = apply_filters( 'wescle_search_post_type', $options );
		$option_selected = isset( $_GET['post_type'] ) ? $_GET['post_type'] : '';
		if ( $options ) {
			$style_hide = count( $options ) == 1 ? 'style="display:none;"' : '';
			?>
            <select id="search_type" name="post_type" <?php echo $style_hide; ?>>
				<?php foreach ( $options as $key => $option ) { ?>
                    <option value="<?php echo $key; ?>" <?php selected( $option_selected, $key ); ?>><?php echo $option; ?></option>
				<?php } ?>
            </select>
			<?php
		}
	}
}

add_action( 'wescle_before_content', 'wescle_title_for_page_elementor_header_footer', 10 );
function wescle_title_for_page_elementor_header_footer() {
	global $post;

	if ( is_page_template( 'elementor_header_footer' ) ) {
		if ( ! get_post_meta( $post->ID, 'hide_page_title', 1 ) ) {
			echo '<div class="post__title">';
			echo '<div class="container"><div class="row"><div class="col">';
			echo '<h1 class="title text-center">' . get_the_title() . '</h1>';
			echo '</div></div></div>';
			echo '</div>';
		}
	}
}


add_action( 'wescle_before_content', 'wescle_section_banners', 20 );
function wescle_section_banners() {
	if ( ! is_front_page() && ! Helper::is_landing_page() ) {
		get_template_part( 'template-parts/section/section-banners' );
	}
}

function html_timer_block( $id, $label = '', $date = '', $hours = 0 ) {
	if ( ! $id ) {
		return;
	}

	if ( ! $date && ! $hours ) {
		return;
	}

	if ( intval( $hours ) ) {
		$timer_data_attr = 'data-timer-by="' . intval( $hours ) . '"';
	}
	else {
		$timer_data_attr = 'data-timer-stop="' . date( 'Y-m-d', strtotime( $date ) ) . 'T23:59:59"';
	}
	?>
    <div class="small-timer-block" id="<?php echo $id; ?>" <?php echo $timer_data_attr; ?>>
		<?php if ( $label ) { ?>
            <div class="small-timer-block__title"><?php echo $label; ?></div>
		<?php } ?>
        <div class="small-timer-block__time">
            <div class="small-timer-block__item"><span class="days" data-days="">00</span><span class="small-timer-block__name"><?php _e( 'дней', 'wescle' ); ?></span></div>
            <div class="small-timer-block__separator"></div>
            <div class="small-timer-block__item"><span class="hours" data-hours="">00</span><span class="small-timer-block__name"><?php _e( 'часов', 'wescle' ); ?></span></div>
            <div class="small-timer-block__separator"></div>
            <div class="small-timer-block__item"><span class="minutes" data-minutes="">00</span><span class="small-timer-block__name"><?php _e( 'минут', 'wescle' ); ?></span></div>
            <div class="small-timer-block__separator"></div>
            <div class="small-timer-block__item"><span class="seconds" data-seconds="">00</span><span class="small-timer-block__name"><?php _e( 'секунд', 'wescle' ); ?></span></div>
        </div>
    </div>
	<?php
}

function wescle_module_hidden_elements() {
	if ( get_theme_mod( 'module_hidden_element_enabled' ) ) {
		$slides = Helper::get_items_hidden_element();
		if ( $slides ) {
			get_template_part( 'template-parts/footer/hidden-elements' );
		}
	}
}

if ( ! function_exists( 'header_store_center_languages' ) ) {
	function header_store_center_languages() {
		$html  = '';
		$langs = [];

		if ( function_exists( 'pll_the_languages' ) ) {
			$languages = pll_the_languages( array(
				'display_names_as'       => 'slug',
				'hide_if_no_translation' => 1,
				'raw'                    => true
			) );
			if ( 0 < count( $languages ) ) {
				$current_lang = '';
				foreach ( $languages as $lang ) {
					$code = $lang['slug'];

					if ( ! $lang['current_lang'] ) {
						$langs[] = '<a title="' . $lang['translated_name'] . '" href="' . $lang['url'] . '">' . $code . '</a>';
					}
					else {
						$current_lang = $code;
						$langs[]      = '<span title="' . $lang['translated_name'] . '">' . $code . '</span>';
					}
				}
			}
		}
        elseif ( function_exists( 'icl_object_id' ) ) {
			$languages = apply_filters( 'wpml_active_languages', null );
			if ( 0 < count( $languages ) ) {
				$current_lang = '';
				foreach ( $languages as $lang ) {
					$code = $lang['code'];

					if ( ! $lang['active'] ) {
						$langs[] = '<a title="' . $lang['translated_name'] . '" href="' . $lang['url'] . '">' . $code . '</a>';
					}
					else {
						$current_lang = $code;
						$langs[]      = '<span title="' . $lang['translated_name'] . '">' . $code . '</span>';
					}
				}
			}
        }

		if ( $langs ) {
			$html = '<button class="_dropdown-btn" type="button"><span>' . $current_lang . '</span> <svg class="_arrow-up" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" preserveaspectratio="xMidYMid meet" viewbox="0 0 32 32"> <g fill="none" stroke="#626262" stroke-linecap="round" stroke-linejoin="round" stroke-width="4"> <path d="M30 12L16 24L2 12"></path> </g> </svg> <svg class="_arrow-down" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" aria-hidden="true" focusable="false" width="1em" height="1em" preserveaspectratio="xMidYMid meet" viewbox="0 0 32 32"> <g fill="none" stroke="#626262" stroke-linecap="round" stroke-linejoin="round" stroke-width="4"> <path d="M30 12L16 24L2 12"></path> </g> </svg> </button>';
			$html .= '<div class="_langs__body"><div class="_langs__content">' . implode( '', $langs ) . '</div></div>';
		}

		return $html;
	}
}

if ( ! function_exists( 'wescle_catalog_item_badges' ) ) {
	function wescle_catalog_item_badges( $post = null ) {
		if ( null === $post ) {
			global $post;
		}

		$html_sale_badges      = '';
		$wescle_product_badges = get_post_meta( $post->ID, 'catalog_item_badges', 1 );
		if ( $wescle_product_badges ) {
			$badge_data = Helper::get_badges_data( 'wescle_badge_catalog_data' );
			foreach ( $wescle_product_badges as $badge_id ) {
				if ( isset( $badge_data[ $badge_id ] ) ) {
					$title = $badge_data[ $badge_id ]['title'];

					$styles = [];
					if ( $badge_data[ $badge_id ]['bg_color'] ) {
						$styles[] = 'background:' . $badge_data[ $badge_id ]['bg_color'] . ';';
					}
					if ( $badge_data[ $badge_id ]['title_color'] ) {
						$styles[] = 'color:' . $badge_data[ $badge_id ]['title_color'] . ';';
					}
					$css_style = $styles ? ' style="' . implode( '', $styles ) . '"' : '';

					$html_sale_badges .= '<span class="_recommended"' . $css_style . '>' . $title . '</span>';
				}
			}
		}

		return $html_sale_badges;
	}
}

if ( ! function_exists( 'wescle_categories_item_image' ) ) {
	function wescle_categories_item_image( $term ) {
		if ( 'product_cat' === $term->taxonomy ) {
			woocommerce_subcategory_thumbnail( $term );
		}
		else {
			$thumbnail_id = get_term_meta( $term->term_id, '_thumbnail_id', 1 );
			if ( $thumbnail_id ) {
				echo wp_get_attachment_image( $thumbnail_id, 'medium' );
			}
			else {
				echo TemplateFilters::placeholder_thumbnail_html( '' );
			}
		}
	}
}

function wescle_query_string_form_fields( $exclude = array(), $current_key = '', $return = false ) {
	$values = $_GET;
	$html   = '';

	foreach ( $values as $key => $value ) {
		if ( in_array( $key, $exclude, true ) ) {
			continue;
		}
		if ( $current_key ) {
			$key = $current_key . '[' . $key . ']';
		}
		$html .= '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( wp_unslash( $value ) ) . '" />';
	}

	if ( $return ) {
		return $html;
	}

	echo $html;
}

if ( ! function_exists( 'wescle_catalog_title_results' ) ) {
	function wescle_catalog_title_results( $count = 0 ) {
		if ( $count == 0 ) {
			$title = __( 'Ничего не найдено', 'wescle' );
		}
		else {
			$title = wescle_label_number_items( $count ) . ' ' . __( 'найдено', 'wescle' );
		}

		return $title;
	}
}

if ( ! function_exists( 'wescle_label_number_items' ) ) {
	function wescle_label_number_items( $number = 0, $labels_num = 0 ) {
		$labels = array(
			array( __( 'результат', 'wescle' ), __( 'результата', 'wescle' ), __( 'результатов', 'wescle' ) ),
		);

		$number_orig = $number;
		$number      = abs( (int) $number );
		$number_arr  = str_split( (string) $number );
		$last_el     = count( $number_arr ) - 1;

		if ( ( $number > 20 && $number_arr[ $last_el ] == '1' && $number_arr[ $last_el - 1 ] != '1' ) || $number == 1 ) {
			$number_str = $labels[ $labels_num ][0];
		}
        elseif (
			( $number > 20
			  && ( $number_arr[ $last_el ] == '2' || $number_arr[ $last_el ] == '3' || $number_arr[ $last_el ] == '4' )
			  && ( $number_arr[ $last_el - 1 ] != '1' )
			)
			|| ( $number == 2 || $number == 3 || $number == 4 )
		) {
			$number_str = $labels[ $labels_num ][1];
        }
		else {
			$number_str = $labels[ $labels_num ][2];
		}

		$number_str = $number_orig . '&nbsp;' . $number_str;

		return $number_str;
	}
}

function catalog_price( $price ) {
	$price = floatval( $price );
	if ( $price ) {
		$decimal      = get_theme_mod( 'module_catalog_currency_decimals', 0 );
		$decimal_sep  = get_theme_mod( 'module_catalog_currency_decimal_sep', '.' );
		$thousand_sep = get_theme_mod( 'module_catalog_currency_thousand_sep', '' );
		$price        = number_format( $price, $decimal, $decimal_sep, $thousand_sep );

		$currency = get_theme_mod( 'module_catalog_currency' );
		$position = get_theme_mod( 'module_catalog_currency_pos', 'right' );
		if ( $currency ) {
			if ( $position === 'left' ) {
				$price = $currency . $price;
			}
            elseif ( $position === 'right' ) {
				$price = $price . $currency;
			}
		}
	}

	return $price;
}