<?php

class TabProduct {

	protected $post_type = 'woo_product_tab';

	public function __construct() {
		add_action( 'init', array( $this, 'register_post_type' ) );

		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'save_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		add_filter( 'woocommerce_product_tabs', array( $this, 'add_custom_tabs' ), 20 );
		add_filter( 'woocommerce_product_tabs', array( $this, 'rename_and_reorder_tabs' ), 30 );

		add_filter( 'woocommerce_product_description_tab_title', array( $this, 'tab_title_custom' ), 10, 2 );
		add_filter( 'woocommerce_product_additional_information_tab_title', array( $this, 'tab_title_custom' ), 10, 2 );
		add_filter( 'woocommerce_product_reviews_tab_title', array( $this, 'tab_title_custom' ), 10, 2 );
		add_filter( 'woocommerce_product_description_heading', array( $this, 'description_heading_custom' ) );
		add_filter( 'woocommerce_product_additional_information_heading', array( $this, 'information_heading_custom' ) );
	}

	public function register_post_type() {
		$labels = array(
			'name'               => _x( 'Кастомные табы товара', 'Post Type General Name', 'wescle' ),
			'singular_name'      => _x( 'Кастомные табы товара', 'Post Type Singular Name', 'wescle' ),
			'menu_name'          => __( 'Кастомные табы товара', 'wescle' ),
			'name_admin_bar'     => __( 'Таб товара', 'wescle' ),
			'all_items'          => __( 'Кастомные табы', 'wescle' ),
			'add_new_item'       => __( 'Добавить новый таб', 'wescle' ),
			'add_new'            => __( 'Добавить таб', 'wescle' ),
			'new_item'           => __( 'Новый таб', 'wescle' ),
			'edit_item'          => __( 'Редактировать таб', 'wescle' ),
			'update_item'        => __( 'Обновить таб', 'wescle' ),
			'search_items'       => __( 'Найти таб', 'wescle' ),
			'not_found'          => __( 'Не найдено', 'wescle' ),
			'not_found_in_trash' => __( 'В корзине пусто', 'wescle' ),
		);

		$args = array(
			'label'             => __( 'Таб товара', 'wescle' ),
			'labels'            => $labels,
			'supports'          => array( 'title', 'editor' ),
			'show_in_rest'      => false, // Add Gutenberg Support
			'hierarchical'      => false,
			'public'            => false,
			'show_ui'           => true,
			'show_in_menu'      => 'edit.php?post_type=product',
			'show_in_admin_bar' => false,
			'show_in_nav_menus' => false,
			'can_export'        => true,
			'capability_type'   => 'post',
		);
		$args = apply_filters( 'wescle_cpt_' . $this->post_type . '_args', $args );

		register_post_type( $this->post_type, $args );
	}

	public function get_sort_tabs() {
		return get_option( 'wescle_woo_sort_tabs', [
			'description',
			'additional_information',
			'wescle_features',
			'reviews'
		] );
	}

	public function get_default_and_custom_tabs() {
		$product_tabs = [
			'description'            => __( 'Description', 'woocommerce' ),
			'additional_information' => __( 'Additional information', 'woocommerce' ),
			'wescle_features'        => 'Wescle ' . __( 'Характеристики товара', 'wescle' ),
			'reviews'                => __( 'Reviews', 'woocommerce' ),
		];

		$tab_posts = get_posts( [
			'post_type'      => $this->post_type,
			'posts_per_page' => - 1,
			'order'          => 'ASC',
		] );
		foreach ( $tab_posts as $item ) {
			$product_tabs[ $this->post_type . '-' . $item->ID ] = $item->post_title;
		}

		return $product_tabs;
	}

	public function get_title_tabs() {
		static $title_tabs;

		if ( null === $title_tabs ) {
			$description = __( 'Description', 'woocommerce' );
			$information = __( 'Additional information', 'woocommerce' );
			$reviews     = __( 'Reviews', 'woocommerce' );

			$defaults = [
				'description'            => [
					'default'     => $description,
					'title'       => $description,
					'heading'     => $description,
					'heading_tag' => 'h2',
				],
				'additional_information' => [
					'default'     => $information,
					'title'       => $information,
					'heading'     => $information,
					'heading_tag' => 'h2',
				],
				'reviews'                => [
					'default'     => $reviews,
					'title'       => $reviews,
					'heading'     => '',
					'heading_tag' => 'h2',
				]
			];

			$title_tabs = wp_parse_args( get_option( 'wescle_woo_tab_title', [] ), $defaults );
		}

		return $title_tabs;
	}

	public function html_select_heading_tags( $name, $current = '' ) {
		$options = [
			'p'  => '<p>',
			'h1' => 'H1',
			'h2' => 'H2',
			'h3' => 'H3',
			'h4' => 'H4',
			'h5' => 'H5',
			'h6' => 'H6',
		];
		echo '<select name="' . $name . '">';
		foreach ( $options as $key => $option ) {
			$selected = selected( $key, $current, false );
			echo '<option value="' . $key . '" ' . $selected . '>' . esc_html( $option ) . '</option>';
		}
		echo '</select>';
	}

	public function add_admin_menu() {
		add_submenu_page(
			'edit.php?post_type=product',
			__( 'Настройка табов', 'wescle' ),
			__( 'Настройка табов', 'wescle' ),
			'manage_options',
			'wescle_sort_tabs',
			array( $this, 'page_template' )
		);
	}

	public function page_template() {
		$wescle_woo_sort_tabs = $this->get_sort_tabs();
		$product_tabs         = $this->get_default_and_custom_tabs();
		$title_tabs           = $this->get_title_tabs();
		//
		?>
        <div class="wrap">
            <h1><?php _e( 'Настройка табов', 'wescle' ); ?></h1>
			<?php
			if ( isset( $_GET['saved'] ) ) {
				?>
                <div class="notice notice-success is-dismissible">
                    <p><strong><?php _e( 'Настройки сохранены', 'wescle' ); ?>.</strong></p>
                </div>
				<?php
			}
			?>
            <form action="?post_type=product&page=wescle_sort_tabs&saved=1" method="post">
                <h3><?php _e( 'Название стандартных табов и их заголовков', 'wescle' ); ?></h3>
                <table class="form-table" role="presentation">
                    <tbody>
                    <tr>
                        <th scope="row"><?php _e( 'Название таба', 'wescle' ); ?> "<?php echo $product_tabs['description']; ?>"</th>
                        <td>
                            <input type="text" name="wescle_woo_tab_title[description][title]" class="regular-text" value="<?php echo $title_tabs['description']['title']; ?>">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e( 'Заголовок таба', 'wescle' ); ?> "<?php echo $product_tabs['description']; ?>"</th>
                        <td>
                            <input type="text" name="wescle_woo_tab_title[description][heading]" class="regular-text" value="<?php echo $title_tabs['description']['heading']; ?>">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e( 'Тег заголовка', 'wescle' ); ?> "<?php echo $product_tabs['description']; ?>"</th>
                        <td>
		                    <?php $this->html_select_heading_tags( 'wescle_woo_tab_title[description][heading_tag]', $title_tabs['description']['heading_tag'] ); ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e( 'Название таба', 'wescle' ); ?> "<?php echo $product_tabs['additional_information']; ?>"</th>
                        <td>
                            <input type="text" name="wescle_woo_tab_title[additional_information][title]" class="regular-text" value="<?php echo $title_tabs['additional_information']['title']; ?>">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e( 'Заголовок таба', 'wescle' ); ?> "<?php echo $product_tabs['additional_information']; ?>"</th>
                        <td>
                            <input type="text" name="wescle_woo_tab_title[additional_information][heading]" class="regular-text" value="<?php echo $title_tabs['additional_information']['heading']; ?>">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e( 'Тег заголовка', 'wescle' ); ?> "<?php echo $product_tabs['additional_information']; ?>"</th>
                        <td>
		                    <?php $this->html_select_heading_tags( 'wescle_woo_tab_title[additional_information][heading_tag]', $title_tabs['additional_information']['heading_tag'] ); ?>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e( 'Название таба', 'wescle' ); ?> "<?php echo $product_tabs['reviews']; ?>"</th>
                        <td>
                            <input type="text" name="wescle_woo_tab_title[reviews][title]" class="regular-text" value="<?php echo $title_tabs['reviews']['title']; ?>">
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php _e( 'Тег заголовка', 'wescle' ); ?> "<?php echo $product_tabs['reviews']; ?>"</th>
                        <td>
		                    <?php $this->html_select_heading_tags( 'wescle_woo_tab_title[reviews][heading_tag]', $title_tabs['reviews']['heading_tag'] ); ?>
                        </td>
                    </tr>
                    </tbody>
                </table>

                <h3><?php _e( 'Сортировка табов', 'wescle' ); ?></h3>
                <div class="wescle-metabox">
                    <ul class="sortable wescle_sort_tabs repeater-fields">
			            <?php
			            foreach ( $wescle_woo_sort_tabs as $key ) {
				            $product_tab_title = $product_tabs[ $key ] ?? '';
				            if ( ! $product_tab_title ) {
								continue;
							}
							unset( $product_tabs[ $key ] );
							?>
                            <li class="kirki-sortable-item">
                                <i class="dashicons dashicons-sort"></i>
								<?php echo $product_tab_title; ?>
                                <input type="hidden" name="wescle_woo_sort_tabs[]" value="<?php echo $key; ?>">
                            </li>
						<?php } ?>

						<?php foreach ( $product_tabs as $key => $product_tab_title ) { ?>
                            <li class="kirki-sortable-item">
                                <i class="dashicons dashicons-sort"></i>
								<?php echo $product_tab_title; ?>
                                <input type="hidden" name="wescle_woo_sort_tabs[]" value="<?php echo $key; ?>">
                            </li>
						<?php } ?>
                    </ul>
                </div>
				<?php submit_button(); ?>
                <input type="hidden" name="save_wescle_woo_sort_tabs" value="1">
            </form>
        </div>
		<?php
	}

	public function save_settings() {
		if ( ! isset( $_POST['save_wescle_woo_sort_tabs'] ) ) {
			return;
		}

		$wescle_woo_sort_tabs = $_POST['wescle_woo_sort_tabs'];
		if ( ! $wescle_woo_sort_tabs ) {
			$wescle_woo_sort_tabs = [];
		}
		update_option( 'wescle_woo_sort_tabs', $wescle_woo_sort_tabs );

		update_option( 'wescle_woo_tab_title', $_POST['wescle_woo_tab_title'] );
	}

	public function enqueue_scripts() {
		$screen = get_current_screen();
		if ( 'product_page_wescle_sort_tabs' == $screen->base ) {
			wp_enqueue_script( 'jquery-ui-sortable' );
		}
	}

	public function add_custom_tabs( $tabs ) {
		$count_tabs  = count( $tabs );
		$custom_tabs = $this->get_default_and_custom_tabs();

		$loop = 0;
		foreach ( $custom_tabs as $key_tab => $title_tab ) {
			if ( strpos( $key_tab, 'woo_product_tab-' ) !== 0 ) {
				continue;
			}

			$visible_tab = true;
			$post_id     = str_replace( $this->post_type . '-', '', $key_tab );

			$tab_product_cat = get_post_meta( $post_id, 'tab_product_cat', 1 );
			$tab_product_ids = get_post_meta( $post_id, 'tab_product_ids', 1 );
			if ( $tab_product_ids ) {
				$tab_product_ids = array_map( 'trim', explode( ',', $tab_product_ids ) );
				if ( $tab_product_ids && ! in_array( get_the_ID(), $tab_product_ids ) && ! $tab_product_cat ) {
					$visible_tab = false;
				}
			}
			else {
				$tab_product_ids = [];
			}

			if ( $visible_tab && $tab_product_cat && ! in_array( get_the_ID(), $tab_product_ids ) ) {
				$cat_list = wp_get_post_terms( get_the_ID(), 'product_cat', array( 'fields' => 'ids' ) );
				if ( ! array_intersect( $cat_list, $tab_product_cat ) ) {
					$visible_tab = false;
				}
			}

			if ( $visible_tab ) {
				$loop ++;

				$priority = ( $count_tabs + $loop ) * 10;

				$tabs[ $key_tab ] = [
					'title'    => $title_tab,
					'priority' => $priority,
					'callback' => array( $this, 'custom_tab_callback' )
				];
			}
		}

		return $tabs;
	}

	public function custom_tab_callback( $key_tab, $tab ) {
		$post_id  = str_replace( $this->post_type . '-', '', $key_tab );
		$post_tab = get_post( $post_id );
		if ( $post_tab ) {
			echo apply_filters( 'the_content', $post_tab->post_content );
		}
	}

	public function rename_and_reorder_tabs( $tabs ) {
		$wescle_woo_sort_tabs = $this->get_sort_tabs();

		$new_tabs = [];
		$loop     = 0;

		foreach ( $wescle_woo_sort_tabs as $key_tab ) {
			if ( isset( $tabs[ $key_tab ] ) ) {
				$loop ++;

				$new_tabs[ $key_tab ]             = $tabs[ $key_tab ];
				$new_tabs[ $key_tab ]['priority'] = $loop * 10;

				unset( $tabs[ $key_tab ] );
			}
		}

		foreach ( $tabs as $key_tab => $tab ) {
			$loop ++;

			$tab['priority']      = $loop * 10;
			$new_tabs[ $key_tab ] = $tab;
		}

		return $new_tabs;
	}

	public function tab_title_custom( $title, $key ) {
		$title_tabs = $this->get_title_tabs();
		$new_title  = $title_tabs[ $key ]['title'];
		if ( 'reviews' == $key ) {
			$title_arr = explode( ' ', $title );
			$new_title .= ' ' . $title_arr[ count( $title_arr ) - 1 ];
		}

		return $new_title;
	}

	public function description_heading_custom( $title ) {
		$title_tabs = $this->get_title_tabs();
		$title      = $title_tabs['description']['heading'];

		return $title;
	}

	public function information_heading_custom( $title ) {
		$title_tabs = $this->get_title_tabs();
		$title      = $title_tabs['additional_information']['heading'];

		return $title;
	}
}

new TabProduct();