<?php
    /**
     * Typography control class.
     *
     * @since  1.0.0
     * @access public
     */
    class Customizer_Typo_Control_Typography extends WP_Customize_Control {
        /**
         * The type of customize control being rendered.
         *
         * @since  1.0.0
         * @access public
         * @var    string
         */
        public $type = 'typography';
        /**
         * Array
         *
         * @since  1.0.0
         * @access public
         * @var    string
         */
        public $l10n = array();
        /**
         * Set up our control.
         *
         * @since  1.0.0
         * @access public
         * @param  object  $manager
         * @param  string  $id
         * @param  array   $args
         * @return void
         */
        public function __construct( $manager, $id, $args = array() ) {
            // Let the parent class do its thing.
            parent::__construct( $manager, $id, $args );
            // Make sure we have labels.
            $this->l10n = wp_parse_args(
                $this->l10n,
                array(
                    'family'      => esc_html__( 'Font Family', 'vmagazine' ),
                    'size'        => esc_html__( 'Font Size',   'vmagazine' ),
                    'style'      => esc_html__( 'Font Weight/Style', 'vmagazine' ),
                    'line_height' => esc_html__( 'Line Height', 'vmagazine' ),
                    'text_decoration' => esc_html__( 'Text Decoration', 'vmagazine' ),
                    'text_transform' => esc_html__( 'Text Transform', 'vmagazine' ),
                    'typocolor' => esc_html__( 'Font Color label', 'vmagazine' ),
                )
            );
        }
        /**
         * Enqueue scripts/styles.
         *
         * @since  1.0.0
         * @access public
         * @return void
         */
        public function enqueue() {
            wp_enqueue_script( 'vmagazine-customize-controls' );
            wp_enqueue_style(  'vmagazine-customize-controls' );
            wp_enqueue_script( 'wp-color-picker' );
            wp_enqueue_style( 'wp-color-picker' );
        }
        /**
         * Add custom parameters to pass to the JS via JSON.
         *
         * @since  1.0.0
         * @access public
         * @return void
         */
        public function to_json() {
            parent::to_json();
            // Loop through each of the settings and set up the data for it.
            foreach ( $this->settings as $setting_key => $setting_id ) {
                $this->json[ $setting_key ] = array(
                    'link'  => $this->get_link( $setting_key ),
                    'value' => $this->value( $setting_key ),
                    'label' => isset( $this->l10n[ $setting_key ] ) ? $this->l10n[ $setting_key ] : ''
                );

                if ( 'family' === $setting_key )
                    $this->json[ $setting_key ]['choices'] = $this->get_font_families();

                elseif ( 'style' === $setting_key )
                    $this->json[ $setting_key ]['choices'] = $this->get_font_weight_choices();

                elseif ( 'text_transform' === $setting_key )
                    $this->json[ $setting_key ]['choices'] = $this->get_text_transform_choices();

                elseif ( 'text_decoration' === $setting_key )
                    $this->json[ $setting_key ]['choices'] = $this->get_text_decoration_choices();
            }
        }
        /**
         * Underscore JS template to handle the control's output.
         *
         * @since  1.0.0
         * @access public
         * @return void
         */
        public function content_template() {
            ?>
            <# if ( data.label ) { #>
                <span class="customize-control-title">{{ data.label }}</span>
            <# } #>

            <# if ( data.description ) { #>
                <span class="description customize-control-description">{{{ data.description }}}</span>
            <# } #>
            <ul>
            <# if ( data.family && data.family.choices ) { #>
                <li class="typography-font-family">
                    <# if ( data.family.label ) { #>
                        <span class="customize-control-title">{{ data.family.label }}</span>
                    <# } #>
                    <select {{{ data.family.link }}} id="{{ data.section }}" class="typography_face">
                        <# _.each( data.family.choices, function( label, choice ) { #>
                            <option value="{{ choice }}" <# if ( choice === data.family.value ) { #> selected="selected" <# } #>>{{ label }}</option>
                        <# } ) #>
                    </select>
                </li>
            <# } #>
            <# if ( data.style && data.style.choices ) { #>
                <li class="typography-font-style">
                    <# if ( data.style.label ) { #>
                        <span class="customize-control-title">{{ data.style.label }}</span>
                    <# } #>
                    <select {{{ data.style.link }}} class="typography_wt">
                        <# _.each( data.style.choices, function( label, choice ) { #>
                            <option value="{{ choice }}" <# if ( choice === data.style.value ) { #> selected="selected" <# } #>>{{ label }}</option>
                        <# } ) #>
                    </select>
                </li>
            <# } #>

            <# if ( data.text_transform && data.text_transform.choices ) { #>

                <li class="typography-text-transform">

                    <# if ( data.text_transform.label ) { #>
                        <span class="customize-control-title">{{ data.text_transform.label }}</span>
                    <# } #>

                    <select {{{ data.text_transform.link }}} class="typography_text_transform">

                        <# _.each( data.text_transform.choices, function( label, choice ) { #>

                            <option value="{{ choice }}" <# if ( choice === data.text_transform.value ) { #> selected="selected" <# } #>>{{ label }}</option>

                        <# } ) #>

                    </select>
                </li>
            <# } #>

            <# if ( data.text_decoration && data.text_decoration.choices ) { #>

                <li class="typography-text-decoration">

                    <# if ( data.text_decoration.label ) { #>
                        <span class="customize-control-title">{{ data.text_decoration.label }}</span>
                    <# } #>

                    <select {{{ data.text_decoration.link }}} class="typography_text_decoration">

                        <# _.each( data.text_decoration.choices, function( label, choice ) { #>

                            <option value="{{ choice }}" <# if ( choice === data.text_decoration.value ) { #> selected="selected" <# } #>>{{ label }}</option>

                        <# } ) #>

                    </select>
                </li>
            <# } #>

            <# if ( data.size ) { #>

                 <li class="typography-font-size">

                    <# if ( data.size.label ) { #>
                        <span class="customize-control-title">{{ data.size.label }} </span>
                    <# } #>

                    <span class="slider-value-size"></span>px
                    <div class="slider-range-size" ></div>
                    <input type="hidden" {{{ data.size.link }}} value="{{ data.size.value }}" >
                </li>
            <# } #>

            <# if ( data.line_height ) { #>

                 <li class="typography-line-height">

                    <# if ( data.line_height.label ) { #>
                        <span class="customize-control-title">{{ data.line_height.label }} (px)</span>
                    <# } #>

                    <span class="slider-value-line-height"></span>
                    <div class="slider-range-line-height" ></div>
                    <input type="hidden" {{{ data.line_height.link }}} value="{{ data.line_height.value }}" >
                </li>
            <# } #>

            <# if ( data.typocolor ) { #>

                <li class="typography-color">
                    <# if ( data.typocolor.label ) { #>
                        <span class="customize-control-title">{{{ data.typocolor.label }}}</span>
                    <# } #>

                    <div class="customize-control-content">
                        <input class="color-picker-hex" type="text" maxlength="7" placeholder="<?php esc_attr_e( 'Hex Value', 'vmagazine' ); ?>" {{{ data.typocolor.link }}} value="{{ data.typocolor.value }}"  />
                    </div>
                </li>
            <# } #>

            </ul>
        <?php }

        /**
         * Returns the available fonts.  Fonts should have available weights, styles, and subsets.
         *
         * @todo Integrate with Google fonts.
         *
         * @since  1.0.0
         * @access public
         * @return array
         */
        public function get_fonts() { return array(); }

        /**
         * Returns the available font families.
         *
         * @todo Pull families from `get_fonts()`.
         *
         * @since  1.0.0
         * @access public
         * @return array
         */
        function get_font_families() {
            /* Google font listing in to database */
        $font_decoded = 'a:708:{i:0;a:2:{s:6:"family";s:5:"Arial";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:6:"Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:1;a:2:{s:6:"family";s:7:"Verdana";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:6:"Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:2;a:2:{s:6:"family";s:9:"Trebuchet";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:6:"Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:3;a:2:{s:6:"family";s:7:"Georgia";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:6:"Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:4;a:2:{s:6:"family";s:6:"Tahoma";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:6:"Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:5;a:2:{s:6:"family";s:8:"Palatino";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:6:"Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:6;a:2:{s:6:"family";s:9:"Helvetica";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:6:"Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:7;a:2:{s:6:"family";s:7:"ABeeZee";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:8;a:2:{s:6:"family";s:4:"Abel";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:9;a:2:{s:6:"family";s:13:"Abril Fatface";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:10;a:2:{s:6:"family";s:8:"Aclonica";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:11;a:2:{s:6:"family";s:4:"Acme";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:12;a:2:{s:6:"family";s:5:"Actor";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:13;a:2:{s:6:"family";s:7:"Adamina";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:14;a:2:{s:6:"family";s:10:"Advent Pro";s:8:"variants";a:7:{i:100;s:4:"Thin";i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:15;a:2:{s:6:"family";s:15:"Aguafina Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:16;a:2:{s:6:"family";s:7:"Akronim";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:17;a:2:{s:6:"family";s:6:"Aladin";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:18;a:2:{s:6:"family";s:7:"Aldrich";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:19;a:2:{s:6:"family";s:4:"Alef";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:20;a:2:{s:6:"family";s:8:"Alegreya";s:8:"variants";a:6:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:21;a:2:{s:6:"family";s:11:"Alegreya SC";s:8:"variants";a:6:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:22;a:2:{s:6:"family";s:13:"Alegreya Sans";s:8:"variants";a:14:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:800;s:10:"Extra Bold";s:9:"800italic";s:17:"Extra Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:23;a:2:{s:6:"family";s:16:"Alegreya Sans SC";s:8:"variants";a:14:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:800;s:10:"Extra Bold";s:9:"800italic";s:17:"Extra Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:24;a:2:{s:6:"family";s:10:"Alex Brush";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:25;a:2:{s:6:"family";s:13:"Alfa Slab One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:26;a:2:{s:6:"family";s:5:"Alice";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:27;a:2:{s:6:"family";s:5:"Alike";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:28;a:2:{s:6:"family";s:13:"Alike Angular";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:29;a:2:{s:6:"family";s:5:"Allan";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:30;a:2:{s:6:"family";s:7:"Allerta";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:31;a:2:{s:6:"family";s:15:"Allerta Stencil";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:32;a:2:{s:6:"family";s:6:"Allura";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:33;a:2:{s:6:"family";s:8:"Almendra";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:34;a:2:{s:6:"family";s:16:"Almendra Display";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:35;a:2:{s:6:"family";s:11:"Almendra SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:36;a:2:{s:6:"family";s:8:"Amarante";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:37;a:2:{s:6:"family";s:8:"Amaranth";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:38;a:2:{s:6:"family";s:9:"Amatic SC";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:39;a:2:{s:6:"family";s:9:"Amethysta";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:40;a:2:{s:6:"family";s:5:"Amiri";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:41;a:2:{s:6:"family";s:7:"Anaheim";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:42;a:2:{s:6:"family";s:6:"Andada";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:43;a:2:{s:6:"family";s:6:"Andika";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:44;a:2:{s:6:"family";s:6:"Angkor";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:45;a:2:{s:6:"family";s:24:"Annie Use Your Telescope";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:46;a:2:{s:6:"family";s:13:"Anonymous Pro";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:47;a:2:{s:6:"family";s:5:"Antic";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:48;a:2:{s:6:"family";s:12:"Antic Didone";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:49;a:2:{s:6:"family";s:10:"Antic Slab";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:50;a:2:{s:6:"family";s:5:"Anton";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:51;a:2:{s:6:"family";s:6:"Arapey";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:52;a:2:{s:6:"family";s:7:"Arbutus";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:53;a:2:{s:6:"family";s:12:"Arbutus Slab";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:54;a:2:{s:6:"family";s:19:"Architects Daughter";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:55;a:2:{s:6:"family";s:13:"Archivo Black";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:56;a:2:{s:6:"family";s:14:"Archivo Narrow";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:57;a:2:{s:6:"family";s:5:"Arimo";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:58;a:2:{s:6:"family";s:8:"Arizonia";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:59;a:2:{s:6:"family";s:6:"Armata";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:60;a:2:{s:6:"family";s:8:"Artifika";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:61;a:2:{s:6:"family";s:4:"Arvo";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:62;a:2:{s:6:"family";s:4:"Asap";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:63;a:2:{s:6:"family";s:5:"Asset";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:64;a:2:{s:6:"family";s:7:"Astloch";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:65;a:2:{s:6:"family";s:4:"Asul";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:66;a:2:{s:6:"family";s:10:"Atomic Age";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:67;a:2:{s:6:"family";s:6:"Aubrey";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:68;a:2:{s:6:"family";s:9:"Audiowide";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:69;a:2:{s:6:"family";s:10:"Autour One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:70;a:2:{s:6:"family";s:7:"Average";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:71;a:2:{s:6:"family";s:12:"Average Sans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:72;a:2:{s:6:"family";s:19:"Averia Gruesa Libre";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:73;a:2:{s:6:"family";s:12:"Averia Libre";s:8:"variants";a:6:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:74;a:2:{s:6:"family";s:17:"Averia Sans Libre";s:8:"variants";a:6:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:75;a:2:{s:6:"family";s:18:"Averia Serif Libre";s:8:"variants";a:6:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:76;a:2:{s:6:"family";s:10:"Bad Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:77;a:2:{s:6:"family";s:9:"Balthazar";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:78;a:2:{s:6:"family";s:7:"Bangers";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:79;a:2:{s:6:"family";s:5:"Basic";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:80;a:2:{s:6:"family";s:10:"Battambang";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:81;a:2:{s:6:"family";s:7:"Baumans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:82;a:2:{s:6:"family";s:5:"Bayon";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:83;a:2:{s:6:"family";s:8:"Belgrano";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:84;a:2:{s:6:"family";s:7:"Belleza";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:85;a:2:{s:6:"family";s:9:"BenchNine";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:86;a:2:{s:6:"family";s:7:"Bentham";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:87;a:2:{s:6:"family";s:15:"Berkshire Swash";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:88;a:2:{s:6:"family";s:5:"Bevan";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:89;a:2:{s:6:"family";s:13:"Bigelow Rules";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:90;a:2:{s:6:"family";s:11:"Bigshot One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:91;a:2:{s:6:"family";s:5:"Bilbo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:92;a:2:{s:6:"family";s:16:"Bilbo Swash Caps";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:93;a:2:{s:6:"family";s:6:"Bitter";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:94;a:2:{s:6:"family";s:13:"Black Ops One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:95;a:2:{s:6:"family";s:5:"Bokor";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:96;a:2:{s:6:"family";s:6:"Bonbon";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:97;a:2:{s:6:"family";s:8:"Boogaloo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:98;a:2:{s:6:"family";s:10:"Bowlby One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:99;a:2:{s:6:"family";s:13:"Bowlby One SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:100;a:2:{s:6:"family";s:7:"Brawler";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:101;a:2:{s:6:"family";s:10:"Bree Serif";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:102;a:2:{s:6:"family";s:14:"Bubblegum Sans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:103;a:2:{s:6:"family";s:11:"Bubbler One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:104;a:2:{s:6:"family";s:4:"Buda";s:8:"variants";a:1:{i:300;s:5:"Light";}}i:105;a:2:{s:6:"family";s:7:"Buenard";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:106;a:2:{s:6:"family";s:10:"Butcherman";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:107;a:2:{s:6:"family";s:14:"Butterfly Kids";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:108;a:2:{s:6:"family";s:5:"Cabin";s:8:"variants";a:8:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:109;a:2:{s:6:"family";s:15:"Cabin Condensed";s:8:"variants";a:4:{i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:110;a:2:{s:6:"family";s:12:"Cabin Sketch";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:111;a:2:{s:6:"family";s:15:"Caesar Dressing";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:112;a:2:{s:6:"family";s:10:"Cagliostro";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:113;a:2:{s:6:"family";s:14:"Calligraffitti";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:114;a:2:{s:6:"family";s:6:"Cambay";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:115;a:2:{s:6:"family";s:5:"Cambo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:116;a:2:{s:6:"family";s:6:"Candal";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:117;a:2:{s:6:"family";s:9:"Cantarell";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:118;a:2:{s:6:"family";s:11:"Cantata One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:119;a:2:{s:6:"family";s:11:"Cantora One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:120;a:2:{s:6:"family";s:8:"Capriola";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:121;a:2:{s:6:"family";s:5:"Cardo";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:122;a:2:{s:6:"family";s:5:"Carme";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:123;a:2:{s:6:"family";s:14:"Carrois Gothic";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:124;a:2:{s:6:"family";s:17:"Carrois Gothic SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:125;a:2:{s:6:"family";s:10:"Carter One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:126;a:2:{s:6:"family";s:6:"Caudex";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:127;a:2:{s:6:"family";s:18:"Cedarville Cursive";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:128;a:2:{s:6:"family";s:11:"Ceviche One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:129;a:2:{s:6:"family";s:10:"Changa One";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:130;a:2:{s:6:"family";s:6:"Chango";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:131;a:2:{s:6:"family";s:18:"Chau Philomene One";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:132;a:2:{s:6:"family";s:9:"Chela One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:133;a:2:{s:6:"family";s:14:"Chelsea Market";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:134;a:2:{s:6:"family";s:6:"Chenla";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:135;a:2:{s:6:"family";s:17:"Cherry Cream Soda";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:136;a:2:{s:6:"family";s:12:"Cherry Swash";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:137;a:2:{s:6:"family";s:5:"Chewy";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:138;a:2:{s:6:"family";s:6:"Chicle";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:139;a:2:{s:6:"family";s:5:"Chivo";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:140;a:2:{s:6:"family";s:6:"Cinzel";s:8:"variants";a:3:{i:400;s:6:"Normal";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:141;a:2:{s:6:"family";s:17:"Cinzel Decorative";s:8:"variants";a:3:{i:400;s:6:"Normal";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:142;a:2:{s:6:"family";s:14:"Clicker Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:143;a:2:{s:6:"family";s:4:"Coda";s:8:"variants";a:2:{i:400;s:6:"Normal";i:800;s:10:"Extra Bold";}}i:144;a:2:{s:6:"family";s:12:"Coda Caption";s:8:"variants";a:1:{i:800;s:10:"Extra Bold";}}i:145;a:2:{s:6:"family";s:8:"Codystar";s:8:"variants";a:2:{i:300;s:5:"Light";i:400;s:6:"Normal";}}i:146;a:2:{s:6:"family";s:5:"Combo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:147;a:2:{s:6:"family";s:9:"Comfortaa";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:148;a:2:{s:6:"family";s:11:"Coming Soon";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:149;a:2:{s:6:"family";s:11:"Concert One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:150;a:2:{s:6:"family";s:9:"Condiment";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:151;a:2:{s:6:"family";s:7:"Content";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:152;a:2:{s:6:"family";s:12:"Contrail One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:153;a:2:{s:6:"family";s:11:"Convergence";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:154;a:2:{s:6:"family";s:6:"Cookie";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:155;a:2:{s:6:"family";s:5:"Copse";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:156;a:2:{s:6:"family";s:6:"Corben";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:157;a:2:{s:6:"family";s:9:"Courgette";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:158;a:2:{s:6:"family";s:7:"Cousine";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:159;a:2:{s:6:"family";s:8:"Coustard";s:8:"variants";a:2:{i:400;s:6:"Normal";i:900;s:10:"Ultra Bold";}}i:160;a:2:{s:6:"family";s:21:"Covered By Your Grace";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:161;a:2:{s:6:"family";s:12:"Crafty Girls";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:162;a:2:{s:6:"family";s:9:"Creepster";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:163;a:2:{s:6:"family";s:11:"Crete Round";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:164;a:2:{s:6:"family";s:12:"Crimson Text";s:8:"variants";a:6:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:165;a:2:{s:6:"family";s:13:"Croissant One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:166;a:2:{s:6:"family";s:7:"Crushed";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:167;a:2:{s:6:"family";s:6:"Cuprum";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:168;a:2:{s:6:"family";s:6:"Cutive";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:169;a:2:{s:6:"family";s:11:"Cutive Mono";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:170;a:2:{s:6:"family";s:6:"Damion";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:171;a:2:{s:6:"family";s:14:"Dancing Script";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:172;a:2:{s:6:"family";s:7:"Dangrek";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:173;a:2:{s:6:"family";s:20:"Dawning of a New Day";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:174;a:2:{s:6:"family";s:8:"Days One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:175;a:2:{s:6:"family";s:5:"Dekko";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:176;a:2:{s:6:"family";s:6:"Delius";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:177;a:2:{s:6:"family";s:17:"Delius Swash Caps";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:178;a:2:{s:6:"family";s:14:"Delius Unicase";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:179;a:2:{s:6:"family";s:13:"Della Respira";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:180;a:2:{s:6:"family";s:8:"Denk One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:181;a:2:{s:6:"family";s:10:"Devonshire";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:182;a:2:{s:6:"family";s:8:"Dhurjati";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:183;a:2:{s:6:"family";s:13:"Didact Gothic";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:184;a:2:{s:6:"family";s:9:"Diplomata";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:185;a:2:{s:6:"family";s:12:"Diplomata SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:186;a:2:{s:6:"family";s:6:"Domine";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:187;a:2:{s:6:"family";s:11:"Donegal One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:188;a:2:{s:6:"family";s:10:"Doppio One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:189;a:2:{s:6:"family";s:5:"Dorsa";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:190;a:2:{s:6:"family";s:5:"Dosis";s:8:"variants";a:7:{i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";i:800;s:10:"Extra Bold";}}i:191;a:2:{s:6:"family";s:11:"Dr Sugiyama";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:192;a:2:{s:6:"family";s:10:"Droid Sans";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:193;a:2:{s:6:"family";s:15:"Droid Sans Mono";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:194;a:2:{s:6:"family";s:11:"Droid Serif";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:195;a:2:{s:6:"family";s:9:"Duru Sans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:196;a:2:{s:6:"family";s:9:"Dynalight";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:197;a:2:{s:6:"family";s:11:"EB Garamond";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:198;a:2:{s:6:"family";s:10:"Eagle Lake";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:199;a:2:{s:6:"family";s:5:"Eater";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:200;a:2:{s:6:"family";s:9:"Economica";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:201;a:2:{s:6:"family";s:8:"Ek Mukta";s:8:"variants";a:7:{i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";i:800;s:10:"Extra Bold";}}i:202;a:2:{s:6:"family";s:11:"Electrolize";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:203;a:2:{s:6:"family";s:5:"Elsie";s:8:"variants";a:2:{i:400;s:6:"Normal";i:900;s:10:"Ultra Bold";}}i:204;a:2:{s:6:"family";s:16:"Elsie Swash Caps";s:8:"variants";a:2:{i:400;s:6:"Normal";i:900;s:10:"Ultra Bold";}}i:205;a:2:{s:6:"family";s:11:"Emblema One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:206;a:2:{s:6:"family";s:12:"Emilys Candy";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:207;a:2:{s:6:"family";s:10:"Engagement";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:208;a:2:{s:6:"family";s:9:"Englebert";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:209;a:2:{s:6:"family";s:9:"Enriqueta";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:210;a:2:{s:6:"family";s:9:"Erica One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:211;a:2:{s:6:"family";s:7:"Esteban";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:212;a:2:{s:6:"family";s:15:"Euphoria Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:213;a:2:{s:6:"family";s:5:"Ewert";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:214;a:2:{s:6:"family";s:3:"Exo";s:8:"variants";a:18:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:200;s:11:"Extra Light";s:9:"200italic";N;i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:800;s:10:"Extra Bold";s:9:"800italic";s:17:"Extra Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:215;a:2:{s:6:"family";s:5:"Exo 2";s:8:"variants";a:18:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:200;s:11:"Extra Light";s:9:"200italic";N;i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:800;s:10:"Extra Bold";s:9:"800italic";s:17:"Extra Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:216;a:2:{s:6:"family";s:13:"Expletus Sans";s:8:"variants";a:8:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:217;a:2:{s:6:"family";s:12:"Fanwood Text";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:218;a:2:{s:6:"family";s:9:"Fascinate";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:219;a:2:{s:6:"family";s:16:"Fascinate Inline";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:220;a:2:{s:6:"family";s:10:"Faster One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:221;a:2:{s:6:"family";s:8:"Fasthand";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:222;a:2:{s:6:"family";s:9:"Fauna One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:223;a:2:{s:6:"family";s:8:"Federant";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:224;a:2:{s:6:"family";s:6:"Federo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:225;a:2:{s:6:"family";s:6:"Felipa";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:226;a:2:{s:6:"family";s:5:"Fenix";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:227;a:2:{s:6:"family";s:12:"Finger Paint";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:228;a:2:{s:6:"family";s:9:"Fira Mono";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:229;a:2:{s:6:"family";s:9:"Fira Sans";s:8:"variants";a:8:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:230;a:2:{s:6:"family";s:10:"Fjalla One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:231;a:2:{s:6:"family";s:9:"Fjord One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:232;a:2:{s:6:"family";s:8:"Flamenco";s:8:"variants";a:2:{i:300;s:5:"Light";i:400;s:6:"Normal";}}i:233;a:2:{s:6:"family";s:7:"Flavors";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:234;a:2:{s:6:"family";s:10:"Fondamento";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:235;a:2:{s:6:"family";s:16:"Fontdiner Swanky";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:236;a:2:{s:6:"family";s:5:"Forum";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:237;a:2:{s:6:"family";s:12:"Francois One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:238;a:2:{s:6:"family";s:12:"Freckle Face";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:239;a:2:{s:6:"family";s:20:"Fredericka the Great";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:240;a:2:{s:6:"family";s:11:"Fredoka One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:241;a:2:{s:6:"family";s:8:"Freehand";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:242;a:2:{s:6:"family";s:6:"Fresca";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:243;a:2:{s:6:"family";s:7:"Frijole";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:244;a:2:{s:6:"family";s:7:"Fruktur";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:245;a:2:{s:6:"family";s:9:"Fugaz One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:246;a:2:{s:6:"family";s:9:"GFS Didot";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:247;a:2:{s:6:"family";s:15:"GFS Neohellenic";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:248;a:2:{s:6:"family";s:8:"Gabriela";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:249;a:2:{s:6:"family";s:6:"Gafata";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:250;a:2:{s:6:"family";s:8:"Galdeano";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:251;a:2:{s:6:"family";s:7:"Galindo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:252;a:2:{s:6:"family";s:13:"Gentium Basic";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:253;a:2:{s:6:"family";s:18:"Gentium Book Basic";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:254;a:2:{s:6:"family";s:3:"Geo";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:255;a:2:{s:6:"family";s:7:"Geostar";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:256;a:2:{s:6:"family";s:12:"Geostar Fill";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:257;a:2:{s:6:"family";s:12:"Germania One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:258;a:2:{s:6:"family";s:6:"Gidugu";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:259;a:2:{s:6:"family";s:13:"Gilda Display";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:260;a:2:{s:6:"family";s:14:"Give You Glory";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:261;a:2:{s:6:"family";s:13:"Glass Antiqua";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:262;a:2:{s:6:"family";s:6:"Glegoo";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:263;a:2:{s:6:"family";s:17:"Gloria Hallelujah";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:264;a:2:{s:6:"family";s:10:"Goblin One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:265;a:2:{s:6:"family";s:10:"Gochi Hand";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:266;a:2:{s:6:"family";s:8:"Gorditas";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:267;a:2:{s:6:"family";s:21:"Goudy Bookletter 1911";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:268;a:2:{s:6:"family";s:8:"Graduate";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:269;a:2:{s:6:"family";s:11:"Grand Hotel";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:270;a:2:{s:6:"family";s:12:"Gravitas One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:271;a:2:{s:6:"family";s:11:"Great Vibes";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:272;a:2:{s:6:"family";s:6:"Griffy";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:273;a:2:{s:6:"family";s:6:"Gruppo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:274;a:2:{s:6:"family";s:5:"Gudea";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:275;a:2:{s:6:"family";s:8:"Gurajada";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:276;a:2:{s:6:"family";s:6:"Habibi";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:277;a:2:{s:6:"family";s:6:"Halant";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:278;a:2:{s:6:"family";s:15:"Hammersmith One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:279;a:2:{s:6:"family";s:7:"Hanalei";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:280;a:2:{s:6:"family";s:12:"Hanalei Fill";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:281;a:2:{s:6:"family";s:7:"Handlee";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:282;a:2:{s:6:"family";s:7:"Hanuman";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:283;a:2:{s:6:"family";s:12:"Happy Monkey";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:284;a:2:{s:6:"family";s:12:"Headland One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:285;a:2:{s:6:"family";s:11:"Henny Penny";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:286;a:2:{s:6:"family";s:20:"Herr Von Muellerhoff";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:287;a:2:{s:6:"family";s:4:"Hind";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:288;a:2:{s:6:"family";s:15:"Holtwood One SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:289;a:2:{s:6:"family";s:14:"Homemade Apple";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:290;a:2:{s:6:"family";s:8:"Homenaje";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:291;a:2:{s:6:"family";s:15:"IM Fell DW Pica";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:292;a:2:{s:6:"family";s:18:"IM Fell DW Pica SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:293;a:2:{s:6:"family";s:19:"IM Fell Double Pica";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:294;a:2:{s:6:"family";s:22:"IM Fell Double Pica SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:295;a:2:{s:6:"family";s:15:"IM Fell English";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:296;a:2:{s:6:"family";s:18:"IM Fell English SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:297;a:2:{s:6:"family";s:20:"IM Fell French Canon";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:298;a:2:{s:6:"family";s:23:"IM Fell French Canon SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:299;a:2:{s:6:"family";s:20:"IM Fell Great Primer";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:300;a:2:{s:6:"family";s:23:"IM Fell Great Primer SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:301;a:2:{s:6:"family";s:7:"Iceberg";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:302;a:2:{s:6:"family";s:7:"Iceland";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:303;a:2:{s:6:"family";s:7:"Imprima";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:304;a:2:{s:6:"family";s:11:"Inconsolata";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:305;a:2:{s:6:"family";s:5:"Inder";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:306;a:2:{s:6:"family";s:12:"Indie Flower";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:307;a:2:{s:6:"family";s:5:"Inika";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:308;a:2:{s:6:"family";s:12:"Irish Grover";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:309;a:2:{s:6:"family";s:9:"Istok Web";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:310;a:2:{s:6:"family";s:8:"Italiana";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:311;a:2:{s:6:"family";s:9:"Italianno";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:312;a:2:{s:6:"family";s:16:"Jacques Francois";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:313;a:2:{s:6:"family";s:23:"Jacques Francois Shadow";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:314;a:2:{s:6:"family";s:14:"Jim Nightshade";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:315;a:2:{s:6:"family";s:10:"Jockey One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:316;a:2:{s:6:"family";s:12:"Jolly Lodger";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:317;a:2:{s:6:"family";s:12:"Josefin Sans";s:8:"variants";a:10:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:318;a:2:{s:6:"family";s:12:"Josefin Slab";s:8:"variants";a:10:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:319;a:2:{s:6:"family";s:8:"Joti One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:320;a:2:{s:6:"family";s:6:"Judson";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:321;a:2:{s:6:"family";s:5:"Julee";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:322;a:2:{s:6:"family";s:15:"Julius Sans One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:323;a:2:{s:6:"family";s:5:"Junge";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:324;a:2:{s:6:"family";s:4:"Jura";s:8:"variants";a:4:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";}}i:325;a:2:{s:6:"family";s:17:"Just Another Hand";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:326;a:2:{s:6:"family";s:23:"Just Me Again Down Here";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:327;a:2:{s:6:"family";s:5:"Kalam";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:328;a:2:{s:6:"family";s:7:"Kameron";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:329;a:2:{s:6:"family";s:9:"Kantumruy";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:330;a:2:{s:6:"family";s:5:"Karla";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:331;a:2:{s:6:"family";s:5:"Karma";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:332;a:2:{s:6:"family";s:14:"Kaushan Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:333;a:2:{s:6:"family";s:6:"Kavoon";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:334;a:2:{s:6:"family";s:10:"Kdam Thmor";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:335;a:2:{s:6:"family";s:10:"Keania One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:336;a:2:{s:6:"family";s:10:"Kelly Slab";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:337;a:2:{s:6:"family";s:5:"Kenia";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:338;a:2:{s:6:"family";s:5:"Khand";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:339;a:2:{s:6:"family";s:5:"Khmer";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:340;a:2:{s:6:"family";s:5:"Khula";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";i:800;s:10:"Extra Bold";}}i:341;a:2:{s:6:"family";s:8:"Kite One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:342;a:2:{s:6:"family";s:7:"Knewave";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:343;a:2:{s:6:"family";s:9:"Kotta One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:344;a:2:{s:6:"family";s:6:"Koulen";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:345;a:2:{s:6:"family";s:6:"Kranky";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:346;a:2:{s:6:"family";s:5:"Kreon";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:347;a:2:{s:6:"family";s:6:"Kristi";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:348;a:2:{s:6:"family";s:9:"Krona One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:349;a:2:{s:6:"family";s:15:"La Belle Aurore";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:350;a:2:{s:6:"family";s:5:"Laila";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:351;a:2:{s:6:"family";s:11:"Lakki Reddy";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:352;a:2:{s:6:"family";s:8:"Lancelot";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:353;a:2:{s:6:"family";s:6:"Lateef";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:354;a:2:{s:6:"family";s:4:"Lato";s:8:"variants";a:10:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:355;a:2:{s:6:"family";s:13:"League Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:356;a:2:{s:6:"family";s:12:"Leckerli One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:357;a:2:{s:6:"family";s:6:"Ledger";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:358;a:2:{s:6:"family";s:6:"Lekton";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:359;a:2:{s:6:"family";s:5:"Lemon";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:360;a:2:{s:6:"family";s:17:"Libre Baskerville";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:361;a:2:{s:6:"family";s:11:"Life Savers";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:362;a:2:{s:6:"family";s:10:"Lilita One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:363;a:2:{s:6:"family";s:15:"Lily Script One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:364;a:2:{s:6:"family";s:9:"Limelight";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:365;a:2:{s:6:"family";s:11:"Linden Hill";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:366;a:2:{s:6:"family";s:7:"Lobster";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:367;a:2:{s:6:"family";s:11:"Lobster Two";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:368;a:2:{s:6:"family";s:16:"Londrina Outline";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:369;a:2:{s:6:"family";s:15:"Londrina Shadow";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:370;a:2:{s:6:"family";s:15:"Londrina Sketch";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:371;a:2:{s:6:"family";s:14:"Londrina Solid";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:372;a:2:{s:6:"family";s:4:"Lora";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:373;a:2:{s:6:"family";s:21:"Love Ya Like A Sister";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:374;a:2:{s:6:"family";s:17:"Loved by the King";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:375;a:2:{s:6:"family";s:14:"Lovers Quarrel";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:376;a:2:{s:6:"family";s:12:"Luckiest Guy";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:377;a:2:{s:6:"family";s:8:"Lusitana";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:378;a:2:{s:6:"family";s:7:"Lustria";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:379;a:2:{s:6:"family";s:7:"Macondo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:380;a:2:{s:6:"family";s:18:"Macondo Swash Caps";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:381;a:2:{s:6:"family";s:5:"Magra";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:382;a:2:{s:6:"family";s:13:"Maiden Orange";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:383;a:2:{s:6:"family";s:4:"Mako";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:384;a:2:{s:6:"family";s:8:"Mallanna";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:385;a:2:{s:6:"family";s:7:"Mandali";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:386;a:2:{s:6:"family";s:9:"Marcellus";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:387;a:2:{s:6:"family";s:12:"Marcellus SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:388;a:2:{s:6:"family";s:12:"Marck Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:389;a:2:{s:6:"family";s:9:"Margarine";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:390;a:2:{s:6:"family";s:9:"Marko One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:391;a:2:{s:6:"family";s:8:"Marmelad";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:392;a:2:{s:6:"family";s:11:"Martel Sans";s:8:"variants";a:7:{i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";i:800;s:10:"Extra Bold";i:900;s:10:"Ultra Bold";}}i:393;a:2:{s:6:"family";s:6:"Marvel";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:394;a:2:{s:6:"family";s:4:"Mate";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:395;a:2:{s:6:"family";s:7:"Mate SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:396;a:2:{s:6:"family";s:9:"Maven Pro";s:8:"variants";a:4:{i:400;s:6:"Normal";i:500;s:6:"Medium";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:397;a:2:{s:6:"family";s:7:"McLaren";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:398;a:2:{s:6:"family";s:6:"Meddon";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:399;a:2:{s:6:"family";s:13:"MedievalSharp";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:400;a:2:{s:6:"family";s:10:"Medula One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:401;a:2:{s:6:"family";s:6:"Megrim";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:402;a:2:{s:6:"family";s:11:"Meie Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:403;a:2:{s:6:"family";s:8:"Merienda";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:404;a:2:{s:6:"family";s:12:"Merienda One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:405;a:2:{s:6:"family";s:12:"Merriweather";s:8:"variants";a:8:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:406;a:2:{s:6:"family";s:17:"Merriweather Sans";s:8:"variants";a:8:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:800;s:10:"Extra Bold";s:9:"800italic";s:17:"Extra Bold Italic";}}i:407;a:2:{s:6:"family";s:5:"Metal";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:408;a:2:{s:6:"family";s:11:"Metal Mania";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:409;a:2:{s:6:"family";s:12:"Metamorphous";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:410;a:2:{s:6:"family";s:11:"Metrophobic";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:411;a:2:{s:6:"family";s:8:"Michroma";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:412;a:2:{s:6:"family";s:7:"Milonga";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:413;a:2:{s:6:"family";s:9:"Miltonian";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:414;a:2:{s:6:"family";s:16:"Miltonian Tattoo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:415;a:2:{s:6:"family";s:7:"Miniver";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:416;a:2:{s:6:"family";s:14:"Miss Fajardose";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:417;a:2:{s:6:"family";s:5:"Modak";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:418;a:2:{s:6:"family";s:14:"Modern Antiqua";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:419;a:2:{s:6:"family";s:7:"Molengo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:420;a:2:{s:6:"family";s:5:"Molle";s:8:"variants";a:1:{s:9:"400italic";s:13:"Normal Italic";}}i:421;a:2:{s:6:"family";s:5:"Monda";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:422;a:2:{s:6:"family";s:8:"Monofett";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:423;a:2:{s:6:"family";s:7:"Monoton";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:424;a:2:{s:6:"family";s:20:"Monsieur La Doulaise";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:425;a:2:{s:6:"family";s:7:"Montaga";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:426;a:2:{s:6:"family";s:6:"Montez";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:427;a:2:{s:6:"family";s:10:"Montserrat";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:428;a:2:{s:6:"family";s:21:"Montserrat Alternates";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:429;a:2:{s:6:"family";s:20:"Montserrat Subrayada";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:430;a:2:{s:6:"family";s:4:"Moul";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:431;a:2:{s:6:"family";s:8:"Moulpali";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:432;a:2:{s:6:"family";s:22:"Mountains of Christmas";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:433;a:2:{s:6:"family";s:13:"Mouse Memoirs";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:434;a:2:{s:6:"family";s:10:"Mr Bedfort";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:435;a:2:{s:6:"family";s:8:"Mr Dafoe";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:436;a:2:{s:6:"family";s:14:"Mr De Haviland";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:437;a:2:{s:6:"family";s:19:"Mrs Saint Delafield";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:438;a:2:{s:6:"family";s:13:"Mrs Sheppards";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:439;a:2:{s:6:"family";s:4:"Muli";s:8:"variants";a:4:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:440;a:2:{s:6:"family";s:13:"Mystery Quest";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:441;a:2:{s:6:"family";s:3:"NTR";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:442;a:2:{s:6:"family";s:6:"Neucha";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:443;a:2:{s:6:"family";s:6:"Neuton";s:8:"variants";a:6:{i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";i:800;s:10:"Extra Bold";}}i:444;a:2:{s:6:"family";s:10:"New Rocker";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:445;a:2:{s:6:"family";s:10:"News Cycle";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:446;a:2:{s:6:"family";s:7:"Niconne";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:447;a:2:{s:6:"family";s:9:"Nixie One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:448;a:2:{s:6:"family";s:6:"Nobile";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:449;a:2:{s:6:"family";s:6:"Nokora";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:450;a:2:{s:6:"family";s:7:"Norican";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:451;a:2:{s:6:"family";s:7:"Nosifer";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:452;a:2:{s:6:"family";s:20:"Nothing You Could Do";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:453;a:2:{s:6:"family";s:12:"Noticia Text";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:454;a:2:{s:6:"family";s:9:"Noto Sans";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:455;a:2:{s:6:"family";s:10:"Noto Serif";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:456;a:2:{s:6:"family";s:8:"Nova Cut";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:457;a:2:{s:6:"family";s:9:"Nova Flat";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:458;a:2:{s:6:"family";s:9:"Nova Mono";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:459;a:2:{s:6:"family";s:9:"Nova Oval";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:460;a:2:{s:6:"family";s:10:"Nova Round";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:461;a:2:{s:6:"family";s:11:"Nova Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:462;a:2:{s:6:"family";s:9:"Nova Slim";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:463;a:2:{s:6:"family";s:11:"Nova Square";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:464;a:2:{s:6:"family";s:6:"Numans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:465;a:2:{s:6:"family";s:6:"Nunito";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:466;a:2:{s:6:"family";s:14:"Odor Mean Chey";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:467;a:2:{s:6:"family";s:7:"Offside";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:468;a:2:{s:6:"family";s:15:"Old Standard TT";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:469;a:2:{s:6:"family";s:9:"Oldenburg";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:470;a:2:{s:6:"family";s:11:"Oleo Script";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:471;a:2:{s:6:"family";s:22:"Oleo Script Swash Caps";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:472;a:2:{s:6:"family";s:9:"Open Sans";s:8:"variants";a:10:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:800;s:10:"Extra Bold";s:9:"800italic";s:17:"Extra Bold Italic";}}i:473;a:2:{s:6:"family";s:19:"Open Sans Condensed";s:8:"variants";a:3:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:700;s:4:"Bold";}}i:474;a:2:{s:6:"family";s:11:"Oranienbaum";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:475;a:2:{s:6:"family";s:8:"Orbitron";s:8:"variants";a:4:{i:400;s:6:"Normal";i:500;s:6:"Medium";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:476;a:2:{s:6:"family";s:7:"Oregano";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:477;a:2:{s:6:"family";s:7:"Orienta";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:478;a:2:{s:6:"family";s:15:"Original Surfer";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:479;a:2:{s:6:"family";s:6:"Oswald";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:480;a:2:{s:6:"family";s:16:"Over the Rainbow";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:481;a:2:{s:6:"family";s:8:"Overlock";s:8:"variants";a:6:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:482;a:2:{s:6:"family";s:11:"Overlock SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:483;a:2:{s:6:"family";s:3:"Ovo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:484;a:2:{s:6:"family";s:6:"Oxygen";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:485;a:2:{s:6:"family";s:11:"Oxygen Mono";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:486;a:2:{s:6:"family";s:7:"PT Mono";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:487;a:2:{s:6:"family";s:7:"PT Sans";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:488;a:2:{s:6:"family";s:15:"PT Sans Caption";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:489;a:2:{s:6:"family";s:14:"PT Sans Narrow";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:490;a:2:{s:6:"family";s:8:"PT Serif";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:491;a:2:{s:6:"family";s:16:"PT Serif Caption";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:492;a:2:{s:6:"family";s:8:"Pacifico";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:493;a:2:{s:6:"family";s:7:"Paprika";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:494;a:2:{s:6:"family";s:10:"Parisienne";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:495;a:2:{s:6:"family";s:11:"Passero One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:496;a:2:{s:6:"family";s:11:"Passion One";s:8:"variants";a:3:{i:400;s:6:"Normal";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:497;a:2:{s:6:"family";s:18:"Pathway Gothic One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:498;a:2:{s:6:"family";s:12:"Patrick Hand";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:499;a:2:{s:6:"family";s:15:"Patrick Hand SC";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:500;a:2:{s:6:"family";s:9:"Patua One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:501;a:2:{s:6:"family";s:11:"Paytone One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:502;a:2:{s:6:"family";s:7:"Peddana";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:503;a:2:{s:6:"family";s:7:"Peralta";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:504;a:2:{s:6:"family";s:16:"Permanent Marker";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:505;a:2:{s:6:"family";s:19:"Petit Formal Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:506;a:2:{s:6:"family";s:7:"Petrona";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:507;a:2:{s:6:"family";s:11:"Philosopher";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:508;a:2:{s:6:"family";s:6:"Piedra";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:509;a:2:{s:6:"family";s:13:"Pinyon Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:510;a:2:{s:6:"family";s:10:"Pirata One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:511;a:2:{s:6:"family";s:7:"Plaster";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:512;a:2:{s:6:"family";s:4:"Play";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:513;a:2:{s:6:"family";s:8:"Playball";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:514;a:2:{s:6:"family";s:16:"Playfair Display";s:8:"variants";a:6:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:515;a:2:{s:6:"family";s:19:"Playfair Display SC";s:8:"variants";a:6:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:516;a:2:{s:6:"family";s:7:"Podkova";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:517;a:2:{s:6:"family";s:10:"Poiret One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:518;a:2:{s:6:"family";s:10:"Poller One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:519;a:2:{s:6:"family";s:4:"Poly";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:520;a:2:{s:6:"family";s:8:"Pompiere";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:521;a:2:{s:6:"family";s:12:"Pontano Sans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:522;a:2:{s:6:"family";s:16:"Port Lligat Sans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:523;a:2:{s:6:"family";s:16:"Port Lligat Slab";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:524;a:2:{s:6:"family";s:5:"Prata";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:525;a:2:{s:6:"family";s:11:"Preahvihear";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:526;a:2:{s:6:"family";s:14:"Press Start 2P";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:527;a:2:{s:6:"family";s:14:"Princess Sofia";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:528;a:2:{s:6:"family";s:8:"Prociono";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:529;a:2:{s:6:"family";s:10:"Prosto One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:530;a:2:{s:6:"family";s:7:"Puritan";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:531;a:2:{s:6:"family";s:12:"Purple Purse";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:532;a:2:{s:6:"family";s:6:"Quando";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:533;a:2:{s:6:"family";s:8:"Quantico";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:534;a:2:{s:6:"family";s:12:"Quattrocento";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:535;a:2:{s:6:"family";s:17:"Quattrocento Sans";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:536;a:2:{s:6:"family";s:9:"Questrial";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:537;a:2:{s:6:"family";s:9:"Quicksand";s:8:"variants";a:3:{i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:538;a:2:{s:6:"family";s:14:"Quintessential";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:539;a:2:{s:6:"family";s:7:"Qwigley";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:540;a:2:{s:6:"family";s:15:"Racing Sans One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:541;a:2:{s:6:"family";s:6:"Radley";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:542;a:2:{s:6:"family";s:8:"Rajdhani";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:543;a:2:{s:6:"family";s:7:"Raleway";s:8:"variants";a:9:{i:100;s:4:"Thin";i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";i:800;s:10:"Extra Bold";i:900;s:10:"Ultra Bold";}}i:544;a:2:{s:6:"family";s:12:"Raleway Dots";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:545;a:2:{s:6:"family";s:10:"Ramabhadra";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:546;a:2:{s:6:"family";s:8:"Ramaraja";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:547;a:2:{s:6:"family";s:6:"Rambla";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:548;a:2:{s:6:"family";s:12:"Rammetto One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:549;a:2:{s:6:"family";s:8:"Ranchers";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:550;a:2:{s:6:"family";s:6:"Rancho";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:551;a:2:{s:6:"family";s:5:"Ranga";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:552;a:2:{s:6:"family";s:9:"Rationale";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:553;a:2:{s:6:"family";s:12:"Ravi Prakash";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:554;a:2:{s:6:"family";s:9:"Redressed";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:555;a:2:{s:6:"family";s:13:"Reenie Beanie";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:556;a:2:{s:6:"family";s:7:"Revalia";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:557;a:2:{s:6:"family";s:6:"Ribeye";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:558;a:2:{s:6:"family";s:13:"Ribeye Marrow";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:559;a:2:{s:6:"family";s:9:"Righteous";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:560;a:2:{s:6:"family";s:6:"Risque";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:561;a:2:{s:6:"family";s:6:"Roboto";s:8:"variants";a:12:{i:100;s:4:"Thin";s:9:"100italic";s:11:"Thin Italic";i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:562;a:2:{s:6:"family";s:16:"Roboto Condensed";s:8:"variants";a:6:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:563;a:2:{s:6:"family";s:11:"Roboto Slab";s:8:"variants";a:4:{i:100;s:4:"Thin";i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:564;a:2:{s:6:"family";s:9:"Rochester";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:565;a:2:{s:6:"family";s:9:"Rock Salt";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:566;a:2:{s:6:"family";s:7:"Rokkitt";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:567;a:2:{s:6:"family";s:9:"Romanesco";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:568;a:2:{s:6:"family";s:9:"Ropa Sans";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:569;a:2:{s:6:"family";s:7:"Rosario";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:570;a:2:{s:6:"family";s:8:"Rosarivo";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:571;a:2:{s:6:"family";s:12:"Rouge Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:572;a:2:{s:6:"family";s:9:"Rozha One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:573;a:2:{s:6:"family";s:14:"Rubik Mono One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:574;a:2:{s:6:"family";s:9:"Rubik One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:575;a:2:{s:6:"family";s:4:"Ruda";s:8:"variants";a:3:{i:400;s:6:"Normal";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:576;a:2:{s:6:"family";s:6:"Rufina";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:577;a:2:{s:6:"family";s:11:"Ruge Boogie";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:578;a:2:{s:6:"family";s:6:"Ruluko";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:579;a:2:{s:6:"family";s:10:"Rum Raisin";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:580;a:2:{s:6:"family";s:14:"Ruslan Display";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:581;a:2:{s:6:"family";s:9:"Russo One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:582;a:2:{s:6:"family";s:6:"Ruthie";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:583;a:2:{s:6:"family";s:3:"Rye";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:584;a:2:{s:6:"family";s:10:"Sacramento";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:585;a:2:{s:6:"family";s:4:"Sail";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:586;a:2:{s:6:"family";s:5:"Salsa";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:587;a:2:{s:6:"family";s:7:"Sanchez";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:588;a:2:{s:6:"family";s:8:"Sancreek";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:589;a:2:{s:6:"family";s:11:"Sansita One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:590;a:2:{s:6:"family";s:6:"Sarina";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:591;a:2:{s:6:"family";s:8:"Sarpanch";s:8:"variants";a:6:{i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";i:800;s:10:"Extra Bold";i:900;s:10:"Ultra Bold";}}i:592;a:2:{s:6:"family";s:7:"Satisfy";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:593;a:2:{s:6:"family";s:5:"Scada";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:594;a:2:{s:6:"family";s:12:"Scheherazade";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:595;a:2:{s:6:"family";s:10:"Schoolbell";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:596;a:2:{s:6:"family";s:14:"Seaweed Script";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:597;a:2:{s:6:"family";s:9:"Sevillana";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:598;a:2:{s:6:"family";s:11:"Seymour One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:599;a:2:{s:6:"family";s:18:"Shadows Into Light";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:600;a:2:{s:6:"family";s:22:"Shadows Into Light Two";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:601;a:2:{s:6:"family";s:6:"Shanti";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:602;a:2:{s:6:"family";s:5:"Share";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:603;a:2:{s:6:"family";s:10:"Share Tech";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:604;a:2:{s:6:"family";s:15:"Share Tech Mono";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:605;a:2:{s:6:"family";s:9:"Shojumaru";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:606;a:2:{s:6:"family";s:11:"Short Stack";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:607;a:2:{s:6:"family";s:8:"Siemreap";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:608;a:2:{s:6:"family";s:10:"Sigmar One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:609;a:2:{s:6:"family";s:7:"Signika";s:8:"variants";a:4:{i:300;s:5:"Light";i:400;s:6:"Normal";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:610;a:2:{s:6:"family";s:16:"Signika Negative";s:8:"variants";a:4:{i:300;s:5:"Light";i:400;s:6:"Normal";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:611;a:2:{s:6:"family";s:9:"Simonetta";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:612;a:2:{s:6:"family";s:7:"Sintony";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:613;a:2:{s:6:"family";s:13:"Sirin Stencil";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:614;a:2:{s:6:"family";s:8:"Six Caps";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:615;a:2:{s:6:"family";s:7:"Skranji";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:616;a:2:{s:6:"family";s:10:"Slabo 13px";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:617;a:2:{s:6:"family";s:10:"Slabo 27px";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:618;a:2:{s:6:"family";s:7:"Slackey";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:619;a:2:{s:6:"family";s:6:"Smokum";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:620;a:2:{s:6:"family";s:6:"Smythe";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:621;a:2:{s:6:"family";s:7:"Sniglet";s:8:"variants";a:2:{i:400;s:6:"Normal";i:800;s:10:"Extra Bold";}}i:622;a:2:{s:6:"family";s:7:"Snippet";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:623;a:2:{s:6:"family";s:13:"Snowburst One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:624;a:2:{s:6:"family";s:10:"Sofadi One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:625;a:2:{s:6:"family";s:5:"Sofia";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:626;a:2:{s:6:"family";s:10:"Sonsie One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:627;a:2:{s:6:"family";s:16:"Sorts Mill Goudy";s:8:"variants";a:2:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";}}i:628;a:2:{s:6:"family";s:15:"Source Code Pro";s:8:"variants";a:7:{i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:629;a:2:{s:6:"family";s:15:"Source Sans Pro";s:8:"variants";a:12:{i:200;s:11:"Extra Light";s:9:"200italic";N;i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";s:9:"900italic";s:17:"Ultra Bold Italic";}}i:630;a:2:{s:6:"family";s:16:"Source Serif Pro";s:8:"variants";a:3:{i:400;s:6:"Normal";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:631;a:2:{s:6:"family";s:13:"Special Elite";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:632;a:2:{s:6:"family";s:10:"Spicy Rice";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:633;a:2:{s:6:"family";s:9:"Spinnaker";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:634;a:2:{s:6:"family";s:6:"Spirax";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:635;a:2:{s:6:"family";s:10:"Squada One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:636;a:2:{s:6:"family";s:20:"Sree Krushnadevaraya";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:637;a:2:{s:6:"family";s:9:"Stalemate";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:638;a:2:{s:6:"family";s:13:"Stalinist One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:639;a:2:{s:6:"family";s:15:"Stardos Stencil";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:640;a:2:{s:6:"family";s:21:"Stint Ultra Condensed";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:641;a:2:{s:6:"family";s:20:"Stint Ultra Expanded";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:642;a:2:{s:6:"family";s:5:"Stoke";s:8:"variants";a:2:{i:300;s:5:"Light";i:400;s:6:"Normal";}}i:643;a:2:{s:6:"family";s:6:"Strait";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:644;a:2:{s:6:"family";s:19:"Sue Ellen Francisco";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:645;a:2:{s:6:"family";s:9:"Sunshiney";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:646;a:2:{s:6:"family";s:16:"Supermercado One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:647;a:2:{s:6:"family";s:7:"Suranna";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:648;a:2:{s:6:"family";s:9:"Suravaram";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:649;a:2:{s:6:"family";s:11:"Suwannaphum";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:650;a:2:{s:6:"family";s:18:"Swanky and Moo Moo";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:651;a:2:{s:6:"family";s:9:"Syncopate";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:652;a:2:{s:6:"family";s:9:"Tangerine";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:653;a:2:{s:6:"family";s:6:"Taprom";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:654;a:2:{s:6:"family";s:5:"Tauri";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:655;a:2:{s:6:"family";s:4:"Teko";s:8:"variants";a:5:{i:300;s:5:"Light";i:400;s:6:"Normal";i:500;s:6:"Medium";i:600;s:9:"Semi Bold";i:700;s:4:"Bold";}}i:656;a:2:{s:6:"family";s:5:"Telex";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:657;a:2:{s:6:"family";s:18:"Tenali Ramakrishna";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:658;a:2:{s:6:"family";s:10:"Tenor Sans";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:659;a:2:{s:6:"family";s:11:"Text Me One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:660;a:2:{s:6:"family";s:18:"The Girl Next Door";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:661;a:2:{s:6:"family";s:6:"Tienne";s:8:"variants";a:3:{i:400;s:6:"Normal";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:662;a:2:{s:6:"family";s:7:"Timmana";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:663;a:2:{s:6:"family";s:5:"Tinos";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:664;a:2:{s:6:"family";s:9:"Titan One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:665;a:2:{s:6:"family";s:13:"Titillium Web";s:8:"variants";a:11:{i:200;s:11:"Extra Light";s:9:"200italic";N;i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:600;s:9:"Semi Bold";s:9:"600italic";s:16:"Semi Bold Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";i:900;s:10:"Ultra Bold";}}i:666;a:2:{s:6:"family";s:11:"Trade Winds";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:667;a:2:{s:6:"family";s:7:"Trocchi";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:668;a:2:{s:6:"family";s:7:"Trochut";s:8:"variants";a:3:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";}}i:669;a:2:{s:6:"family";s:7:"Trykker";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:670;a:2:{s:6:"family";s:10:"Tulpen One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:671;a:2:{s:6:"family";s:6:"Ubuntu";s:8:"variants";a:8:{i:300;s:5:"Light";s:9:"300italic";s:12:"Light Italic";i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:500;s:6:"Medium";s:9:"500italic";s:13:"Medium Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:672;a:2:{s:6:"family";s:16:"Ubuntu Condensed";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:673;a:2:{s:6:"family";s:11:"Ubuntu Mono";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:674;a:2:{s:6:"family";s:5:"Ultra";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:675;a:2:{s:6:"family";s:14:"Uncial Antiqua";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:676;a:2:{s:6:"family";s:8:"Underdog";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:677;a:2:{s:6:"family";s:9:"Unica One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:678;a:2:{s:6:"family";s:14:"UnifrakturCook";s:8:"variants";a:1:{i:700;s:4:"Bold";}}i:679;a:2:{s:6:"family";s:18:"UnifrakturMaguntia";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:680;a:2:{s:6:"family";s:7:"Unkempt";s:8:"variants";a:2:{i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:681;a:2:{s:6:"family";s:6:"Unlock";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:682;a:2:{s:6:"family";s:4:"Unna";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:683;a:2:{s:6:"family";s:5:"VT323";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:684;a:2:{s:6:"family";s:11:"Vampiro One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:685;a:2:{s:6:"family";s:6:"Varela";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:686;a:2:{s:6:"family";s:12:"Varela Round";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:687;a:2:{s:6:"family";s:11:"Vast Shadow";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:688;a:2:{s:6:"family";s:12:"Vesper Libre";s:8:"variants";a:4:{i:400;s:6:"Normal";i:500;s:6:"Medium";i:700;s:4:"Bold";i:900;s:10:"Ultra Bold";}}i:689;a:2:{s:6:"family";s:5:"Vibur";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:690;a:2:{s:6:"family";s:8:"Vidaloka";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:691;a:2:{s:6:"family";s:4:"Viga";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:692;a:2:{s:6:"family";s:5:"Voces";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:693;a:2:{s:6:"family";s:7:"Volkhov";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:694;a:2:{s:6:"family";s:8:"Vollkorn";s:8:"variants";a:4:{i:400;s:6:"Normal";s:9:"400italic";s:13:"Normal Italic";i:700;s:4:"Bold";s:9:"700italic";s:11:"Bold Italic";}}i:695;a:2:{s:6:"family";s:8:"Voltaire";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:696;a:2:{s:6:"family";s:23:"Waiting for the Sunrise";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:697;a:2:{s:6:"family";s:8:"Wallpoet";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:698;a:2:{s:6:"family";s:15:"Walter Turncoat";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:699;a:2:{s:6:"family";s:6:"Warnes";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:700;a:2:{s:6:"family";s:9:"Wellfleet";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:701;a:2:{s:6:"family";s:9:"Wendy One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:702;a:2:{s:6:"family";s:8:"Wire One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:703;a:2:{s:6:"family";s:17:"Yanone Kaffeesatz";s:8:"variants";a:4:{i:200;s:11:"Extra Light";i:300;s:5:"Light";i:400;s:6:"Normal";i:700;s:4:"Bold";}}i:704;a:2:{s:6:"family";s:10:"Yellowtail";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:705;a:2:{s:6:"family";s:10:"Yeseva One";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:706;a:2:{s:6:"family";s:10:"Yesteryear";s:8:"variants";a:1:{i:400;s:6:"Normal";}}i:707;a:2:{s:6:"family";s:6:"Zeyada";s:8:"variants";a:1:{i:400;s:6:"Normal";}}}';

        $font_array = unserialize($font_decoded);
        $vmagazine_google_font = get_option('vmagazine_google_font');
            if (empty($vmagazine_google_font)) {
                update_option('vmagazine_google_font', $font_array);
                $vmagazine_google_font = get_option('vmagazine_google_font');
            }
            foreach ($vmagazine_google_font as $key => $value) {
                $ap_fonts[$value['family']] =  $value['family'] ;
            }
            return $ap_fonts;
        }
        /**
         * Returns the available font weights.
         *
         * @since  1.0.0
         * @access public
         * @return array
         */
        public function get_font_weight_choices() {
            if($this->settings['family']->id){
                $vmagazine_font_list = get_option( 'vmagazine_google_font' );

                $font_family_id = $this->settings['family']->id;
                $default_font_family = $this->settings['family']->default;
                $get_font_family = get_theme_mod( $font_family_id, $default_font_family );

                $font_array = vmagazine_search_key( $vmagazine_font_list, 'family', $get_font_family );
               
                if(!empty( $font_array) ):
                $variants_array = $font_array['0']['variants'] ;
                $options_array = array();
                foreach ($variants_array  as $key => $variants ) {
                    $options_array[$key] = $variants;
                }
                
                return $options_array;
                endif;
            }else{
            return array(
                '400' => esc_html__( 'Normal',     'vmagazine' ),
                '700' => esc_html__( 'Bold',       'vmagazine' ),
            );
            }
        }
        /**
         * Returns the available font text decoration.
         *
         * @since  1.0.0
         * @access public
         * @return array
         */
        public function get_text_decoration_choices() {
            return array(
                'none'  => esc_html__( 'None', 'vmagazine' ),
                'underline'  => esc_html__( 'Underline', 'vmagazine' ),
                'line-through' => esc_html__( 'Line-through', 'vmagazine' ),
                'overline' => esc_html__( 'Overline', 'vmagazine' )
            );
        }

        /**
         * Returns the available font text transform.
         *
         * @since  1.0.0
         * @access public
         * @return array
         */
        public function get_text_transform_choices() {
            return array(
                'none'  => esc_html__( 'None', 'vmagazine' ),
                'uppercase'  => esc_html__( 'Uppercase', 'vmagazine' ),
                'lowercase' => esc_html__( 'Lowercase', 'vmagazine' ),
                'capitalize' => esc_html__( 'Capitalize', 'vmagazine' )
            );
    }
 }
