<?php
/**
 * Vikinger USER functions
 * 
 * @since 1.0.0
 */

/**
 * Returns logged user data, or false if no user is logged in
 * 
 * @return array|bool $user           User if logged in, false otherwise.  
 */
function vikinger_get_logged_user_data() {
  $user = false;

  if (is_user_logged_in()) {
    $user_id = get_current_user_id();
    $user = vikinger_user_get_data($user_id);
  }

  return $user;
}

/**
 * Returns user data
 * 
 * @param int     $user_id      User id.
 * @return array  $user         User data.
 */
function vikinger_user_get_data($user_id) {
  $user_data = get_userdata($user_id);

  $user = [];

  if ($user_data) {
    $user = [
      'id'            => $user_data->ID,
      'name'          => $user_data->display_name,
      'mention_name'  => $user_data->user_login,
      'link'          => '',
      'media_link'    => '',
      'badges_link'   => '',
      'avatar_url'    => get_avatar_url($user_data->ID),
      'cover_url'     => vikinger_get_default_member_cover_url(),
      'badges'        => [],
      'rank'          => [
        'current' => 0,
        'total'   => 0
      ]
    ];
  
    // add GamiPress rank data if plugin is active
    if (vikinger_plugin_gamipress_is_active()) {
      $user['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $user_data->ID, 'simple');
      $user['badges'] = vikinger_gamipress_get_user_completed_achievements('badge', $user_data->ID, 'simple');
    }
  }

  return $user;
}

/**
 * Update user user metadata
 * 
 * @param int   $user_id      ID of the user to update metadata of.
 * @param array $metadata     Key value pairs of user metadata to update.
 */
function vikinger_user_meta_update($user_id, $metadata) {
  foreach ($metadata as $meta_key => $meta_value) {
    $result = update_user_meta($user_id, $meta_key, $meta_value);
  }

  return true;
}

/**
 * Get logged in user color theme
 * 
 * @return string $color_theme       Color theme (light / dark).
 */
function vikinger_logged_user_color_theme_get() {
  $color_theme = get_theme_mod('vikinger_color_presets_setting_theme_switch_default', 'light');

  if (is_user_logged_in()) {
    // user_color_theme is false if invalid user_id or '' (empty string) if no value stored
    $user_color_theme = vikinger_user_color_theme_get(get_current_user_id());

    if ($user_color_theme) {
      $color_theme = $user_color_theme;
    }
  }

  return $color_theme;
}

/**
 * Get user color theme
 * 
 * @param int $user_id      User ID.
 */
function vikinger_user_color_theme_get($user_id) {
  // clear user meta cache to force getting updated user meta
  wp_cache_delete($user_id, 'user_meta');

  return get_user_meta($user_id, 'vikinger_color_theme', true);
}

/**
 * Update logged user color theme
 * 
 * @param string    $color_theme   Color theme.
 * @return int|bool $result         Meta ID if it didn't exist, true on successful update, false on failure, if there is no logged in user, or if color_theme has the same value as stored.
 */
function vikinger_logged_user_color_theme_update($color_theme) {
  if (is_user_logged_in()) {
    return vikinger_user_color_theme_update(get_current_user_id(), $color_theme);
  }

  return false;
}

/**
 * Update user color theme
 * 
 * @param int       $user_id        User ID.
 * @param string    $color_theme   Color theme.
 * @return int|bool $result         Meta ID if it didn't exist, true on successful update, false on failure or if color_theme has the same value as stored.
 */
function vikinger_user_color_theme_update($user_id, $color_theme) {
  return update_user_meta($user_id, 'vikinger_color_theme', $color_theme);
}

/**
 * Toggle logged user color theme
 * 
 * @return int|bool $result         Meta ID if it didn't exist, true on successful update, false on failure, if there is no logged in user, or if color_theme has the same value as stored.
 */
function vikinger_logged_user_color_theme_toggle() {
  $user_color_theme = vikinger_logged_user_color_theme_get();

  if ($user_color_theme === 'light') {
    return vikinger_logged_user_color_theme_update('dark');
  } else {
    return vikinger_logged_user_color_theme_update('light');
  }
}

/**
 * Delete logged user account
 * 
 * @return bool True on success, false on failure.
 */
function vikinger_logged_user_delete_account() {
  return bp_core_delete_account();
}

?>