<?php
/**
 * Vikinger MEMBER functions
 * 
 * @since 1.0.0
 */

/**
 * Returns default member avatar URL
 * 
 * @return string
 */
function vikinger_get_default_member_avatar_url() {
  $members_default_avatar_id = get_theme_mod('vikinger_members_setting_default_avatar', false);

  if ($members_default_avatar_id) {
    $members_default_avatar_url = wp_get_attachment_image_src($members_default_avatar_id , 'full')[0];
  } else {
    $members_default_avatar_url = VIKINGER_URL . '/img/avatar/default-avatar.jpg';
  }

  return $members_default_avatar_url;
}

/**
 * Returns default member cover URL
 * 
 * @return string
 */
function vikinger_get_default_member_cover_url() {
  $members_default_cover_id = get_theme_mod('vikinger_members_setting_default_cover', false);

  if ($members_default_cover_id) {
    $members_default_cover_url = wp_get_attachment_image_src($members_default_cover_id , 'full')[0];
  } else {
    $members_default_cover_url = VIKINGER_URL . '/img/cover/default-cover.png';
  }

  return $members_default_cover_url;
}

/**
 * Returns logged user member data, or false if no user is logged in
 * 
 * @param string      $data_scope     Scope of the user data.
 * @return array|bool $user           User if logged in, false otherwise.  
 */
function vikinger_get_logged_user_member_data($data_scope = 'user-status') {
  $user = false;

  if (is_user_logged_in()) {
    $user_id = get_current_user_id();
    $user = vikinger_members_get(['include' => [$user_id], 'data_scope' => $data_scope])[0];
  }

  return $user;
}


/**
 * Returns member data or member deleted placeholder data if the member doesn't exist
 * 
 * @param int     $member_id        Member ID.
 * @param string  $data_scope       Member scope.
 * @return array  $member_data      Member data.
 */
function vikinger_members_get_fallback($member_id, $data_scope = 'user-status') {
  $member = vikinger_members_get(['include' => $member_id, 'data_scope' => $data_scope]);

  if (count($member)) {
    $member_data = $member[0];
  } else {
    $member_data = [
      'id'            => $member_id,
      'name'          => esc_html__('[deleted]', 'vikinger'),
      'mention_name'  => esc_html__('[deleted]', 'vikinger'),
      'link'          => '',
      'media_link'    => '',
      'badges_link'   => '',
      'avatar_url'    => vikinger_get_default_member_avatar_url(),
      'verified'      => false
    ];
  }

  return $member_data;
}

/**
 * Returns filtered members
 * 
 * @param array   $args       Filter for the members query.
 * @return array  $members    Filtered members.
 */
function vikinger_members_get($args = []) {
  $request = new WP_REST_Request('GET', '/buddypress/v1/members');

  $data_scope = 'user-status';

  if (array_key_exists('data_scope', $args)) {
    $data_scope = $args['data_scope'];
    unset($args['data_scope']);
  }

  $defaults = [
    'type' => 'alphabetical'
  ];

  $args = array_merge($defaults, $args);

  // set parameters
  foreach ($args as $key => $value) {
    $request->set_param($key, $value);
  }

  $bp_members = rest_do_request($request);

  $members = [];

  // if request was succesfull
  if ($bp_members->status === 200) {
    foreach ($bp_members->data as $bp_member) {
      $members[] = vikinger_member_get_data($bp_member, $data_scope);
    }
  }

  return $members;
}

/**
 * Returns base user data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_base_data       user-base scope member data.
 */
function vikinger_member_get_user_base_data($bp_member) {
  $user_base_data = [
    'id'                              => $bp_member['id'],
    'name'                            => $bp_member['name'],
    'link'                            => $bp_member['link'],
    'avatar_url'                      => $bp_member['avatar_urls']['full'],
    'rank'                            => [
      'current' => 0,
      'total'   => 0
    ],
    'verified'                        => false
  ];

  if (bp_is_active('activity')) {
    $user_base_data['mention_name'] = $bp_member['mention_name'];
  } else {
    $user = get_userdata($bp_member['id']);
    $user_base_data['mention_name'] = $user->user_nicename;
  }

  return $user_base_data;
}

/**
 * Returns user-status scope member data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_status_data     user-status scope member data.
 */
function vikinger_member_get_user_status_data($bp_member) {
  $user_status_data = [
    'media_link'                      => $bp_member['link'] . 'photos',
    'friend_requests_link'            => $bp_member['link'] . 'settings/friend-requests',
    'messages_link'                   => $bp_member['link'] . 'settings/messages',
    'notifications_link'              => $bp_member['link'] . 'settings/notifications',
    'received_group_invitations_link' => $bp_member['link'] . 'settings/received-group-invitations',
    'membership_requests_link'        => $bp_member['link'] . 'settings/membership-requests'
  ];

  // add user base data
  $user_status_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_status_data);

  // add GamiPress rank data if plugin is active
  if (vikinger_plugin_gamipress_is_active()) {
    $user_status_data['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $bp_member['id'], 'simple');
  }

  // add Verified Member for BuddyPress data if plugin is active
  if (vikinger_plugin_bpverifiedmember_is_active()) {
    $user_status_data['verified'] = !empty(get_user_meta($user_status_data['id'], 'bp_verified_member', true));
  }

  return $user_status_data;
}

/**
 * Returns user-sidebar scope member data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_status_data     user-sidebar scope member data.
 */
function vikinger_member_get_user_sidebar_data($bp_member) {
  $user_sidebar_data = [
    'badges_link'   => $bp_member['link'] . 'badges',
    'cover_url'     => vikinger_member_get_cover_url($bp_member['id']),
    'stats'         => [
      'post_count'    => bp_is_active('activity') ? vikinger_activity_get_member_post_count($bp_member['id']) : 0,
      'friend_count'  => bp_is_active('friends') ? absint(bp_get_total_friend_count($bp_member['id'])) : 0,
      'comment_count' => bp_is_active('activity') ? vikinger_activity_get_member_comment_count($bp_member['id']): 0
    ],
    'badges'        => []
  ];

  // add user base data
  $user_sidebar_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_sidebar_data);

  // add GamiPress rank data if plugin is active
  if (vikinger_plugin_gamipress_is_active()) {
    $user_sidebar_data['badges'] = vikinger_gamipress_get_user_completed_achievements('badge', $bp_member['id'], 'simple');
    $user_sidebar_data['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $bp_member['id'], 'simple');
  }

  // add Verified Member for BuddyPress data if plugin is active
  if (vikinger_plugin_bpverifiedmember_is_active()) {
    $user_sidebar_data['verified'] = !empty(get_user_meta($user_sidebar_data['id'], 'bp_verified_member', true));
  }

  return $user_sidebar_data;
}

/**
 * Returns user-preview scope member data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_preview_data    user-preview scope member data.
 */
function vikinger_member_get_user_preview_data($bp_member) {
  $user_preview_data = [
    'badges_link'   => $bp_member['link'] . 'badges',
    'cover_url'     => vikinger_member_get_cover_url($bp_member['id']),
    'stats'         => [
      'post_count'    => bp_is_active('activity') ? vikinger_activity_get_member_post_count($bp_member['id']) : 0,
      'friend_count'  => bp_is_active('friends') ? absint(bp_get_total_friend_count($bp_member['id'])) : 0,
      'comment_count' => bp_is_active('activity') ? vikinger_activity_get_member_comment_count($bp_member['id']) : 0
    ],
    'badges'        => [],
    'profile_data'  => vikinger_member_get_xprofile_data($bp_member['id'])
  ];

  // add user base data
  $user_preview_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_preview_data);

  // add GamiPress rank data if plugin is active
  if (vikinger_plugin_gamipress_is_active()) {
    $user_preview_data['badges'] = vikinger_gamipress_get_user_completed_achievements('badge', $bp_member['id'], 'simple');
    $user_preview_data['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $bp_member['id'], 'simple');
  }

  // add Verified Member for BuddyPress data if plugin is active
  if (vikinger_plugin_bpverifiedmember_is_active()) {
    $user_preview_data['verified'] = !empty(get_user_meta($user_preview_data['id'], 'bp_verified_member', true));
  }

  return $user_preview_data;
}

/**
 * Returns user-friends scope member data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_friends_data    user-friends scope member data.
 */
function vikinger_member_get_user_friends_data($bp_member) {
  $user_friends_data = [
    'friends'                   => [],
    'friend_requests_received'  => [],
    'friend_requests_sent'      => [],
    'friend_requests_link'      => $bp_member['link'] . 'settings/friend-requests',
    'messages_link'             => $bp_member['link'] . 'settings/messages',
    'notifications_link'        => $bp_member['link'] . 'settings/notifications'
  ];

  // add user base data
  $user_friends_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_friends_data);

  // add GamiPress rank data if plugin is active
  if (vikinger_plugin_gamipress_is_active()) {
    $user_friends_data['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $bp_member['id'], 'simple');
  }

  // add BuddyPress friend data if BuddyPress and the friends component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('friends')) {
    $user_friends_data['friends'] = vikinger_get_friends(['user_id' => $bp_member['id']]);
    $user_friends_data['friend_requests_received'] = vikinger_get_friend_requests_received($bp_member['id']);
    $user_friends_data['friend_requests_sent'] = vikinger_get_friend_requests_sent($bp_member['id']);
  }

  return $user_friends_data;
}

/**
 * Returns user-groups scope member data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_groups_data     user-groups scope member data.
 */
function vikinger_member_get_user_groups_data($bp_member) {
  $user_groups_data = [
    'groups'                              => [],
    'group_membership_requests_sent'      => [],
    'manage_groups_link'                  => $bp_member['link'] . 'settings/manage-groups',
    'default_group_avatar_url'            => vikinger_get_default_group_avatar_url(),
    'default_group_cover_image_url'       => vikinger_get_default_group_cover_url()
  ];

  // add user base data
  $user_groups_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_groups_data);

  // add BuddyPress group data if BuddyPress and the groups component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('groups')) {
    $user_groups_data['groups'] = vikinger_get_user_groups($bp_member['id']);

    $user_groups_data['group_membership_requests_sent'] = vikinger_group_membership_requests_get(['user_id' => $bp_member['id']]);
  }

  return $user_groups_data;
}

/**
 * Returns user-groups-preview scope member data
 * 
 * @param array   $bp_member                    BP member data.
 * @return array  $user_groups_preview_data     user-groups-preview scope member data.
 */
function vikinger_member_get_user_groups_preview_data($bp_member) {
  $user_groups_preview_data = [
    'groups'                              => [],
    'default_group_avatar_url'            => vikinger_get_default_group_avatar_url(),
    'default_group_cover_image_url'       => vikinger_get_default_group_cover_url()
  ];

  // add user base data
  $user_groups_preview_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_groups_preview_data);

  // add BuddyPress group data if BuddyPress and the groups component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('groups')) {
    $user_groups_preview_data['groups'] = vikinger_get_user_groups($bp_member['id'], 'user-groups-preview');
  }

  return $user_groups_preview_data;
}

/**
 * Returns user-activity scope member data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_activity_data   user-activity scope member data.
 */
function vikinger_member_get_user_activity_data($bp_member) {
  $user_activity_data = [
    'friends'       => [],
    'groups'        => [],
    'messages_link' => $bp_member['link'] . 'settings/messages'
  ];

  // add user base data
  $user_activity_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_activity_data);

  // add GamiPress rank data if plugin is active
  if (vikinger_plugin_gamipress_is_active()) {
    $user_activity_data['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $bp_member['id'], 'simple');
  }

  // add BuddyPress friend data if BuddyPress and the friends component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('friends')) {
    $user_activity_data['friends'] = vikinger_get_friends(['user_id' => $bp_member['id']]);
  }

  // add BuddyPress group data if BuddyPress and the groups component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('groups')) {
    $user_activity_data['groups'] = vikinger_get_user_groups($bp_member['id']);
  }

  // add Verified Member for BuddyPress data if plugin is active
  if (vikinger_plugin_bpverifiedmember_is_active()) {
    $user_activity_data['verified'] = !empty(get_user_meta($user_activity_data['id'], 'bp_verified_member', true));
  }

  return $user_activity_data;
}

/**
 * Returns user-groups-invitations scope member data
 * 
 * @param array   $bp_member                      BP member data.
 * @return array  $user_groups_invitations_data   user-groups-invitations scope member data.
 */
function vikinger_member_get_user_groups_invitations_data($bp_member) {
  $user_groups_invitations_data = [
    'friends'                     => [],
    'groups'                      => [],
    'group_invitations_sent'      => []
  ];

  // add user base data
  $user_groups_invitations_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_groups_invitations_data);

  // add BuddyPress friend data if BuddyPress and the friends component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('friends')) {
    $user_groups_invitations_data['friends'] = vikinger_get_friends(['user_id' => $bp_member['id']]);
  }

  // add BuddyPress group data if BuddyPress and the groups component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('groups')) {
    $user_groups_invitations_data['groups'] = vikinger_get_user_groups($bp_member['id'], 'user-groups-members');

    $group_invitations_sent_args = [
      'inviter_id'  => $bp_member['id']
    ];

    $user_groups_invitations_data['group_invitations_sent'] = vikinger_group_get_invitations($group_invitations_sent_args);
  }

  return $user_groups_invitations_data;
}

/**
 * Returns user-groups-received-invitations scope member data
 * 
 * @param array   $bp_member                                BP member data.
 * @return array  $user_groups_received_invitations_data    user-groups-received-invitations scope member data.
 */
function vikinger_member_get_user_groups_received_invitations_data($bp_member) {
  $user_groups_received_invitations_data = [
    'group_invitations_received'  => []
  ];

  // add user base data
  $user_groups_received_invitations_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_groups_received_invitations_data);

  // add BuddyPress group data if BuddyPress and the groups component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('groups')) {
    $group_invitations_received_args = [
      'user_id'   => $bp_member['id']
    ];

    $user_groups_received_invitations_data['group_invitations_received'] = vikinger_group_get_invitations($group_invitations_received_args);
  }

  return $user_groups_received_invitations_data;
}

/**
 * Returns user-groups-memberships scope member data
 * 
 * @param array   $bp_member                        BP member data.
 * @return array  $user_groups_memberships_data     user-groups-memberships scope member data.
 */
function vikinger_member_get_user_groups_memberships_data($bp_member) {
  $user_groups_memberships_data = [
    'groups'                              => [],
    'group_membership_requests_received'  => [],
    'group_membership_requests_sent'      => []
  ];

  // add user base data
  $user_groups_memberships_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_groups_memberships_data);

  // add BuddyPress group data if BuddyPress and the groups component are active
  if (vikinger_plugin_buddypress_is_active() && bp_is_active('groups')) {
    $user_groups_memberships_data['groups'] = vikinger_get_user_groups($bp_member['id']);

    $user_groups_memberships_data['group_membership_requests_sent'] = vikinger_group_membership_requests_get(['user_id' => $bp_member['id']]);
    
    foreach ($user_groups_memberships_data['groups'] as $user_group) {
      $user_groups_memberships_data['group_membership_requests_received'] = array_merge(vikinger_group_membership_requests_get(['group_id' => $user_group['id']]), $user_groups_memberships_data['group_membership_requests_received']);
    }
  }

  return $user_groups_memberships_data;
}

/**
 * Returns user-activity-friend scope member data
 * 
 * @param array   $bp_member                    BP member data.
 * @return array  $user_activity_friend_data    user-activity-friend scope member data.
 */
function vikinger_member_get_user_activity_friend_data($bp_member) {
  $user_activity_friend_data = [
    'messages_link' => $bp_member['link'] . 'messages',
    'cover_url'     => vikinger_member_get_cover_url($bp_member['id'])
  ];

  // add user base data
  $user_activity_friend_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_activity_friend_data);

  // add GamiPress rank data if plugin is active
  if (vikinger_plugin_gamipress_is_active()) {
    $user_activity_friend_data['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $bp_member['id'], 'simple');
  }

  // add Verified Member for BuddyPress data if plugin is active
  if (vikinger_plugin_bpverifiedmember_is_active()) {
    $user_activity_friend_data['verified'] = !empty(get_user_meta($user_activity_friend_data['id'], 'bp_verified_member', true));
  }

  return $user_activity_friend_data;
}

/**
 * Returns user-settings scope member data
 * 
 * @param array   $bp_member            BP member data.
 * @return array  $user_settings_data   user-settings scope member data.
 */
function vikinger_member_get_user_settings_data($bp_member) {
  $user_settings_data = [
    'profile_data'  => vikinger_member_get_xprofile_data($bp_member['id'])
  ];

  // add user base data
  $user_settings_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_settings_data);

  return $user_settings_data;
}

/**
 * Returns user-settings-profile scope member data
 * 
 * @param array   $bp_member                    BP member data.
 * @return array  $user_settings_profile_data   user-settings-profile scope member data.
 */
function vikinger_member_get_user_settings_profile_data($bp_member) {
  $user_settings_profile_data = [
    'cover_url'     => vikinger_member_get_cover_url($bp_member['id']),
    'profile_data'  => vikinger_member_get_xprofile_data($bp_member['id'])
  ];

  // add user base data
  $user_settings_profile_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_settings_profile_data);

  // add GamiPress rank data if plugin is active
  if (vikinger_plugin_gamipress_is_active()) {
    $user_settings_profile_data['rank'] = vikinger_gamipress_get_user_rank_priority('rank', $bp_member['id'], 'simple');
  }

  return $user_settings_profile_data;
}

/**
 * Returns user-settings-password scope member data
 * 
 * @param array   $bp_member                    BP member data.
 * @return array  $user_settings_password_data  user-settings-password scope member data.
 */
function vikinger_member_get_user_settings_password_data($bp_member) {
  $user = get_userdata($bp_member['id']);

  $user_settings_password_data = [
    'password'      => $user->user_pass
  ];

  // add user base data
  $user_settings_password_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_settings_password_data);

  return $user_settings_password_data;
}

/**
 * Returns user-settings-notification scope member data
 * 
 * @param array   $bp_member                        BP member data.
 * @return array  $user_settings_notification_data  user-settings-notification scope member data.
 */
function vikinger_member_get_user_settings_notification_data($bp_member) {
  $user_settings_notification_data = [
    'email_settings'  => vikinger_notification_get_settings($bp_member['id'])
  ];

  // add user base data
  $user_settings_notification_data = array_merge(vikinger_member_get_user_base_data($bp_member), $user_settings_notification_data);

  return $user_settings_notification_data;
}

/**
 * Returns scoped member data
 * 
 * @param array   $bp_member    BP member data.
 * @param string  $data_scope   Scope of the member data to return.        
 * @return array
 */
function vikinger_member_get_data($bp_member, $data_scope = 'user-status') {
  if ($data_scope === 'user-status') {
    return vikinger_member_get_user_status_data($bp_member);
  }

  if ($data_scope === 'user-preview') {
    return vikinger_member_get_user_preview_data($bp_member);
  }

  if ($data_scope === 'user-sidebar') {
    return vikinger_member_get_user_sidebar_data($bp_member);
  }

  if ($data_scope === 'user-friends') {
    return vikinger_member_get_user_friends_data($bp_member);
  }

  if ($data_scope === 'user-groups') {
    return vikinger_member_get_user_groups_data($bp_member);
  }

  if ($data_scope === 'user-groups-preview') {
    return vikinger_member_get_user_groups_preview_data($bp_member);
  }

  if ($data_scope === 'user-groups-invitations') {
    return vikinger_member_get_user_groups_invitations_data($bp_member);
  }

  if ($data_scope === 'user-groups-received-invitations') {
    return vikinger_member_get_user_groups_received_invitations_data($bp_member);
  }

  if ($data_scope === 'user-groups-memberships') {
    return vikinger_member_get_user_groups_memberships_data($bp_member);
  }

  if ($data_scope === 'user-activity') {
    return vikinger_member_get_user_activity_data($bp_member);
  }

  if ($data_scope === 'user-activity-friend') {
    return vikinger_member_get_user_activity_friend_data($bp_member);
  }

  if ($data_scope === 'user-settings') {
    return vikinger_member_get_user_settings_data($bp_member);
  }

  if ($data_scope === 'user-settings-profile') {
    return vikinger_member_get_user_settings_profile_data($bp_member);
  }

  if ($data_scope === 'user-settings-password') {
    return vikinger_member_get_user_settings_password_data($bp_member);
  }

  if ($data_scope === 'user-settings-notification') {
    return vikinger_member_get_user_settings_notification_data($bp_member);
  }

  return [];
}

/**
 * Returns member cover url
 * 
 * @param int     $member_id      Member id.
 * @return string $cover_url      Member cover url.
 */
function vikinger_member_get_cover_url($member_id) {
  $cover_url = bp_attachments_get_attachment('url', [
    'object_dir'  => 'members',
    'item_id'     => $member_id
  ]);

  // get default cover url if user hasn't uploaded one yet
  if (!$cover_url) {
    $cover_url = vikinger_get_default_member_cover_url();
  }

  return $cover_url;
}

/**
 * Returns member xprofile data
 * 
 * @param int     $member_id              Member id.
 * @return array  $member_xprofile_data   Structured member xprofile data.
 */
function vikinger_member_get_xprofile_data($member_id) {
  $request = new WP_REST_Request('GET', '/buddypress/v1/xprofile/groups');

  // set parameters
  $request->set_param('user_id', $member_id);
  $request->set_param('fetch_fields', true);
  $request->set_param('fetch_field_data', true);

  $bp_xprofile_groups = rest_do_request($request);

  $member_xprofile_data = [
    'group'  =>  [],
    'field'  =>  []
  ];

  // if request was succesfull
  if ($bp_xprofile_groups->status === 200) {
    foreach ($bp_xprofile_groups->data as $bp_xprofile_group) {
      $member_xprofile_data['group'][$bp_xprofile_group['name']] = [];
    
      foreach ($bp_xprofile_group['fields'] as $field) { 
        $member_xprofile_field_data = [
          'group_id'      => $bp_xprofile_group['id'],
          'id'            => $field['id'],
          'name'          => $field['name'],
          'type'          => $field['type'],
          'value'         => stripslashes($field['data']['value']['raw']),
          'field_order'   => $field['field_order'],
          'option_order'  => $field['option_order'],
          'options'       => $field['options'],
          'is_required'   => $field['is_required'],
          'meta'          => bp_xprofile_get_meta($field['id'], 'field'),
          'values'  => $field['data']
        ];

        // pass checkbox unserialized value as a comma separated string
        if ($field['type'] === 'checkbox') {
          $member_xprofile_field_data['value'] = implode(', ', $field['data']['value']['unserialized']);
        }

        $default_value_option_field_types = ['selectbox', 'radio'];

        // field type with defaultable option values
        // set default option as value if value is not yet set by user
        if ($field['data']['value']['raw'] === '') {
          if (in_array($field['type'], $default_value_option_field_types)) {
            foreach ($field['options'] as $option) {
              if ($option['is_default_option']) {
                $member_xprofile_field_data['value'] = $option['name'];
                break;
              }
            }
          }

          if ($field['type'] === 'checkbox') {
            $values = [];

            foreach ($field['options'] as $option) {
              if ($option['is_default_option']) {
                $values[] = $option['name'];
              }
            }

            $member_xprofile_field_data['value'] = implode(', ', $values);
          }
        }

        $member_xprofile_data['group'][$bp_xprofile_group['name']][] = $member_xprofile_field_data;
        $member_xprofile_data['field'][$bp_xprofile_group['name'] . '_' . $field['name']] = $member_xprofile_field_data;
      }
    }
  }

  return $member_xprofile_data;
}

/**
 * Update member xprofile data
 * 
 * @param array $fields       Fields to update (field_id => field_value)
 * @param int   $member_id    ID of the member to update the field data of
 */
function vikinger_member_update_xprofile_data($fields, $member_id) {
  foreach ($fields as $field_id => $field_value) {
    $result = xprofile_set_field_data($field_id, $member_id, $field_value);
  }

  return $result;
}

/**
 * Returns filtered members count.
 * 
 * @param array $args             Filters for the members query.
 * @return int  $members_count    Filtered members count.
 */
function vikinger_members_get_count($args = []) {
  $request = new WP_REST_Request('GET', '/buddypress/v1/members');

  $defaults = [
    'per_page' => 1
  ];

  $args = array_merge($defaults, $args);

  // set parameters
  foreach ($args as $key => $value) {
    $request->set_param($key, $value);
  }

  $bp_members = rest_do_request($request);

  $members_count = 0;

  // if request was succesfull
  if ($bp_members->status === 200) {
    if (array_key_exists('X-WP-Total', $bp_members->headers)) {
      $members_count = $bp_members->headers['X-WP-Total'];
    }
  }

  return $members_count;
}

/**
 * Returns member navigation items
 * 
 * @param array   $member           Member data
 * @param boolean $activity_single  True if activity single, false if not
 * @return array
 */
function vikinger_members_get_navigation_items($member, $activity_single) {
  $nav_items = [];


  $member_profile_page_navigation_items = [
    [
      'name'    => esc_html__('About', 'vikinger'),
      'icon'    => 'profile',
      'slug'    => 'about',
      'active'  => bp_is_current_component('about'),
      'enabled' => vikinger_settings_members_profile_page_is_enabled('about'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_about_page_order', 1)
    ],
    [
      'name'    => esc_html__('Timeline', 'vikinger'),
      'icon'    => 'timeline',
      'slug'    => 'activity',
      'active'  => bp_is_user_activity() && !$activity_single,
      'enabled' => bp_is_active('activity'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_timeline_page_order', 2)
    ],
    [
      'name'    => esc_html__('Friends', 'vikinger'),
      'icon'    => 'friend',
      'slug'    => 'friends',
      'active'  => bp_is_user_friends(),
      'enabled' => bp_is_active('friends'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_friends_page_order', 3)
    ],
    [
      'name'    => esc_html__('Groups', 'vikinger'),
      'icon'    => 'group',
      'slug'    => 'groups',
      'active'  => bp_is_user_groups(),
      'enabled' => bp_is_active('groups'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_groups_page_order', 4)
    ],
    [
      'name'    => esc_html__('Blog', 'vikinger'),
      'icon'    => 'blog-posts',
      'slug'    => 'posts',
      'active'  => bp_is_current_component('posts'),
      'enabled' => vikinger_settings_members_profile_page_is_enabled('posts'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_posts_page_order', 5)
    ],
    [
      'name'    => esc_html__('Forum', 'vikinger'),
      'icon'    => 'forum',
      'slug'    => 'forums',
      'active'  => bp_is_current_component('forums'),
      'enabled' => vikinger_plugin_bbpress_is_active(),
      'order'   => get_theme_mod('vikinger_members_setting_profile_forum_page_order', 6)
    ],
    [
      'name'    => esc_html__('Photos', 'vikinger'),
      'icon'    => 'photos',
      'slug'    => 'photos',
      'active'  => bp_is_current_component('photos'),
      'enabled' => bp_is_active('activity') && vikinger_plugin_vkmedia_is_active(),
      'order'   => get_theme_mod('vikinger_members_setting_profile_photos_page_order', 7)
    ],
    [
      'name'    => esc_html__('Credits', 'vikinger'),
      'icon'    => 'credits',
      'slug'    => 'credits',
      'active'  => bp_is_current_component('credits'),
      'enabled' => vikinger_plugin_gamipress_is_active() && vikinger_settings_members_profile_page_is_enabled('credits'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_credits_page_order', 8)
    ],
    [
      'name'    => esc_html__('Badges', 'vikinger'),
      'icon'    => 'badges',
      'slug'    => 'badges',
      'active'  => bp_is_current_component('badges'),
      'enabled' => vikinger_plugin_gamipress_is_active() && vikinger_settings_members_profile_page_is_enabled('badges'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_badges_page_order', 9)
    ],
    [
      'name'    => esc_html__('Quests', 'vikinger'),
      'icon'    => 'quests',
      'slug'    => 'quests',
      'active'  => bp_is_current_component('quests'),
      'enabled' => vikinger_plugin_gamipress_is_active() && vikinger_settings_members_profile_page_is_enabled('quests'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_quests_page_order', 10)
    ],
    [
      'name'    => esc_html__('Rank', 'vikinger'),
      'icon'    => 'ranks',
      'slug'    => 'ranks',
      'active'  => bp_is_current_component('ranks'),
      'enabled' => vikinger_plugin_gamipress_is_active() && vikinger_settings_members_profile_page_is_enabled('ranks'),
      'order'   => get_theme_mod('vikinger_members_setting_profile_ranks_page_order', 11)
    ]
  ];

  // only add enabled profile navigation items to the menu items array
  foreach ($member_profile_page_navigation_items as $member_profile_page_navigation_item) {
    if ($member_profile_page_navigation_item['enabled']) {
      $menu_items[] = $member_profile_page_navigation_item;
    }
  }

  // sort menu items by order
  usort($menu_items, function ($a, $b) {
    if ($a['order'] > $b['order']) {
      return 1;
    }

    if ($a['order'] < $b['order']) {
      return -1;
    }

    return 0;
  });

  foreach ($menu_items as $menu_item) {
    $nav_items[] = [
      'name'    => $menu_item['name'],
      'link'    => $member['link'] . $menu_item['slug'],
      'icon'    => $menu_item['icon'],
      'active'  => $menu_item['active']
    ];
  }

  return $nav_items;
}

/**
 * Returns member settings navigation items
 * 
 * @param array   $member           Member data
 * @return array
 */
function vikinger_members_get_settings_navigation_sections($member) {
  $menu_sections = [];

  // profile sections
  $profile_sections = [
    'title'       => esc_html_x('My Profile', 'Profile Settings Menu - Title', 'vikinger'),
    'description' => esc_html_x('Change your avatar &amp; cover, accept friends, read messages and more!', 'Profile Settings Menu - Description', 'vikinger'),
    'icon'        => 'profile',
    'menu_items'  => [
      [
        'name'    => esc_html_x('Profile Info', 'Profile Settings Menu - Profile Info Link', 'vikinger'),
        'slug'    => 'general',
        'link'    => $member['link'] . 'settings/general',
        'active'  => bp_is_current_action('general')
      ],
      [
        'name'    => esc_html_x('Social', 'Profile Settings Menu - Social Link', 'vikinger'),
        'slug'    => 'social',
        'link'    => $member['link'] . 'settings/social',
        'active'  => bp_is_current_action('social')
      ],
      [
        'name'    => esc_html_x('Notifications', 'Profile Settings Menu - Notifications Link', 'vikinger'),
        'slug'    => 'notifications',
        'link'    => $member['link'] . 'settings/notifications',
        'active'  => bp_is_current_action('notifications')
      ]
    ]
  ];

  // if BuddyPress messages component is active, add message related pages
  if (bp_is_active('messages') && bp_is_active('friends')) {
    $profile_sections['menu_items'][] = [
      'name'    => esc_html_x('Messages', 'Profile Settings Menu - Messages Link', 'vikinger'),
      'slug'    => 'messages',
      'link'    => $member['link'] . 'settings/messages',
      'active'  => bp_is_current_action('messages')
    ];
  }

  // if BuddyPress friends component is active, add friend related pages
  if (bp_is_active('friends')) {
    $profile_sections['menu_items'][] = [
      'name'    => esc_html_x('Friend Requests', 'Profile Settings Menu - Friend Requests Link', 'vikinger'),
      'slug'    => 'friend-requests',
      'link'    => $member['link'] . 'settings/friend-requests',
      'active'  => bp_is_current_action('friend-requests')
    ];
  }

  $account_sections = [
    'title'       => esc_html_x('Account', 'Account Settings Menu - Title', 'vikinger'),
    'description' => esc_html_x('Change your account information, password and settings!', 'Account Settings Menu - Description', 'vikinger'),
    'icon'        => 'settings',
    'menu_items'  => [
      [
        'name'    => esc_html_x('Account Info', 'Account Settings Menu - Account Info Link', 'vikinger'),
        'slug'    => 'account',
        'link'    => $member['link'] . 'settings/account',
        'active'  => bp_is_current_action('account')
      ],
      [
        'name'    => esc_html_x('Account Settings', 'Account Settings Menu - Account Settings Link', 'vikinger'),
        'slug'    => 'account-settings',
        'link'    => $member['link'] . 'settings/account-settings',
        'active'  => bp_is_current_action('account-settings')
      ],
      [
        'name'    => esc_html_x('Change Password', 'Account Settings Menu - Change Password Link', 'vikinger'),
        'slug'    => 'password',
        'link'    => $member['link'] . 'settings/password',
        'active'  => bp_is_current_action('password')
      ],
      [
        'name'    => esc_html_x('Email Settings', 'Account Settings Menu - Email Settings Link', 'vikinger'),
        'slug'    => 'email-settings',
        'link'    => $member['link'] . 'settings/email-settings',
        'active'  => bp_is_current_action('email-settings')
      ]
    ]
  ];

  $menu_sections[] = $profile_sections;
  $menu_sections[] = $account_sections;

  // if BuddyPress groups component is active, add group manage related pages
  if (bp_is_active('groups')) {
    $manage_groups_sections = [
      'title'       => esc_html_x('Groups', 'Group Settings Menu - Title', 'vikinger'),
      'description' => esc_html_x('Create new groups, manage the ones you created or accept invites!', 'Groups Settings Menu - Description', 'vikinger'),
      'icon'        => 'group',
      'menu_items'  => [
        [
          'name'    => esc_html_x('Manage Groups', 'Group Settings Menu - Manage Groups Link', 'vikinger'),
          'slug'    => 'manage-groups',
          'link'    => $member['link'] . 'settings/manage-groups',
          'active'  => bp_is_current_action('manage-groups')
        ],
        [
          'name'    => esc_html_x('Send Invitations', 'Group Settings Menu - Send Invitations Link', 'vikinger'),
          'slug'    => 'send-group-invitations',
          'link'    => $member['link'] . 'settings/send-group-invitations',
          'active'  => bp_is_current_action('send-group-invitations')
        ],
        [
          'name'    => esc_html_x('Received Invitations', 'Group Settings Menu - Received Invitations Link', 'vikinger'),
          'slug'    => 'received-group-invitations',
          'link'    => $member['link'] . 'settings/received-group-invitations',
          'active'  => bp_is_current_action('received-group-invitations')
        ],
        [
          'name'    => esc_html_x('Membership Requests', 'Group Settings Menu - Membership Requests Link', 'vikinger'),
          'slug'    => 'membership-requests',
          'link'    => $member['link'] . 'settings/membership-requests',
          'active'  => bp_is_current_action('membership-requests')
        ]
      ]
    ];
        
    $menu_sections[] = $manage_groups_sections;
  }

  return $menu_sections;
}

/**
 * Returns member xprofile social links data
 * 
 * @param array $member     Member with xprofile data
 * @return array
 */
function vikinger_members_get_xprofile_social_links($member) {
  $social_links = [];
  $valid_social_networks = [
    'facebook',
    'twitter',
    'instagram',
    'twitch',
    'discord',
    'youtube',
    'patreon',
    'artstation',
    'behance',
    'deviantart',
    'dribbble',
    'spotify',
    'snapchat',
    'reddit',
    'pinterest',
    'linkedin',
    'github',
    'vk',
    'tumblr',
    'tiktok'
  ];

  if (!array_key_exists('Social_Links', $member['profile_data']['group'])) {
    return $social_links;
  }

  foreach ($member['profile_data']['group']['Social_Links'] as $xprofile_social_link) {
    if (!empty($xprofile_social_link['value'])) {
      $social_links[] = [
        'name'  => in_array(strtolower($xprofile_social_link['name']), $valid_social_networks) ? strtolower($xprofile_social_link['name']) : 'link',
        'link'  => $xprofile_social_link['value']
      ];
    }
  }

  return $social_links;
}

/**
 * Returns xprofile field value
 * 
 * @param array   $member         Member data, contains xprofile data in 'profile_data' -> 'field'
 * @param string  $field_name     Name of the field to retrieve value from
 * @return string
 */
function vikinger_members_xprofile_get_field_value($member, $field_name) {
  if (!array_key_exists($field_name, $member['profile_data']['field'])) {
    return '';
  }

  $field = $member['profile_data']['field'][$field_name];
  
  return vikinger_members_xprofile_parse_field_value($field);
}

/**
 * Parse xprofile field data and return formatted value
 * 
 * @param array $field      xprofile field data
 * @return string
 */
function vikinger_members_xprofile_parse_field_value($field) {
  // if field has a value and its type is datebox, format date using user configuration
  if ($field['value'] !== '' && $field['type'] === 'datebox') {
    $date_format = $field['meta']['date_format'][0];

    if ($date_format === 'custom') {
      $date_format = $field['meta']['date_format_custom'][0];
    }

    $field_value = (new DateTime($field['value']))->format($date_format);

    return $field_value;
  }

  return wp_strip_all_tags($field['value']);
}

/**
 * Returns xprofile group fields
 * 
 * @param array   $member           Member data, contains xprofile data in 'profile_data' -> 'field'
 * @param string  $group_name       Name of the group to retrieve fields from
 * @param array   $exclude_fields   Array of field names to exclude from results
 * @return array
 */
function vikinger_members_xprofile_get_group_fields($member, $group_name, $exclude_fields = []) {
  $group_fields = [];

  // if group name provided doesn't exist, return
  if (!array_key_exists($group_name, $member['profile_data']['group'])) {
    return $group_fields;
  }

  // loop group name fields to get formatted field data
  foreach ($member['profile_data']['group'][$group_name] as $field) {
    // ignore field if it exists in the exclude fields array
    if (in_array($field['name'], $exclude_fields)) {
      continue;
    }

    $group_fields[] = [
      'id'    => $field['id'],
      'title' => $field['name'],
      'type'  => $field['type'],
      'value' => vikinger_members_xprofile_parse_field_value($field)
    ];
  }

  return $group_fields;
}


/**
 * MEMBER SUB PAGES
 */

if (vikinger_plugin_buddypress_is_active()) {
  if (vikinger_settings_members_profile_page_is_enabled('about')) {
    /**
     * Creates member about page navigation
     */
    function vikinger_members_about_page_setup_nav() {
      global $bp;

      bp_core_new_nav_item([
        'name'            => esc_html__('About', 'vikinger'),
        'slug'            => 'about',
        'screen_function' => 'vikinger_members_about_page'
      ]);
    }

    add_action('bp_setup_nav', 'vikinger_members_about_page_setup_nav');

    /**
     * Members post page setup
     */
    function vikinger_members_about_page() {
      add_action('bp_template_content', 'vikinger_members_about_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members post page HTML content
     */
    function vikinger_members_about_page_screen_content() {
      $member = get_query_var('member');

      ?>
      <!-- GRID -->
      <div class="grid grid-3-6-3">
        <!-- GRID COLUMN -->
        <div class="grid-column">
        <?php

          /**
           * Widget Info About
           */
          get_template_part('template-part/widget/widget-info', 'about', [
            'member' => $member
          ]);

        ?>
        </div>
        <!-- /GRID COLUMN -->

        <!-- GRID COLUMN -->
        <div class="grid-column">
        <?php

          /**
           * Widget Info Block Interests
           */
          get_template_part('template-part/widget/widget-info-block', 'interests', [
            'member' => $member
          ]);

        ?>
        </div>
        <!-- /GRID COLUMN -->

        <!-- GRID COLUMN -->
        <div class="grid-column">
        <?php

          /**
           * Widget Info Personal
           */
          get_template_part('template-part/widget/widget-info', 'personal', [
            'member' => $member
          ]);

        ?>
        </div>
        <!-- /GRID COLUMN -->
      </div>
      <!-- /GRID -->
      <?php
    }
  }

  if (vikinger_settings_members_profile_page_is_enabled('posts')) {
    /**
     * Creates member posts page navigation
     */
    function vikinger_members_posts_page_setup_nav() {
      global $bp;

      bp_core_new_nav_item([
        'name'            => esc_html__('Posts', 'vikinger'),
        'slug'            => 'posts',
        'screen_function' => 'vikinger_members_posts_page'
      ]);
    }

    add_action('bp_setup_nav', 'vikinger_members_posts_page_setup_nav');

    /**
     * Members post page setup
     */
    function vikinger_members_posts_page() {
      add_action('bp_template_content', 'vikinger_members_posts_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members post page HTML content
     */
    function vikinger_members_posts_page_screen_content() {
      $member = get_query_var('member');
      $post_count = count_user_posts($member['id']);

      /**
       * Section Header
       */
      get_template_part('template-part/section/section', 'header', [
        'section_pretitle'    => sprintf(
          esc_html_x('Browse %s', 'Section Header - Pretitle', 'vikinger'),
          $member['name']
        ),
        'section_title'       => esc_html__('Blog Posts', 'vikinger'),
        'section_text'        => $post_count
      ]);

      ?>
      <div id="post-preview-filterable-list" class="filterable-list" data-userid=<?php echo esc_attr($member['id']); ?>></div>
      <?php
    }
  }

  // add vkmedia photos page if plugin is active
  if (bp_is_active('activity') && vikinger_plugin_vkmedia_is_active()) {
    /**
     * Creates member photos page navigation
     */
    function vikinger_members_photos_page_setup_nav() {
      global $bp;

      bp_core_new_nav_item([
        'name'            => esc_html__('Photos', 'vikinger'),
        'slug'            => 'photos',
        'screen_function' => 'vikinger_members_photos_page'
      ]);
    }

    add_action('bp_setup_nav', 'vikinger_members_photos_page_setup_nav');

    /**
     * Members photos page setup
     */
    function vikinger_members_photos_page() {
      add_action('bp_template_content', 'vikinger_members_photos_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members photos page HTML content
     */
    function vikinger_members_photos_page_screen_content() {
      $member = get_query_var('member');

      ?>
      <div id="activity-media-list" data-userid=<?php echo esc_attr($member['id']); ?>></div>
      <?php
    }
  }

  // if GamiPress plugin is active, create profile pages for points, badges, quests and ranks
  if (vikinger_plugin_gamipress_is_active()) {
    if (vikinger_settings_members_profile_page_is_enabled('credits')) {
      /**
       * Creates member credits page navigation
       */
      function vikinger_members_credits_page_setup_nav() {
        global $bp;

        bp_core_new_nav_item([
          'name'            => esc_html__('Credits', 'vikinger'),
          'slug'            => 'credits',
          'screen_function' => 'vikinger_members_credits_page'
        ]);
      }

      add_action('bp_setup_nav', 'vikinger_members_credits_page_setup_nav');

      /**
       * Members credits page setup
       */
      function vikinger_members_credits_page() {
        add_action('bp_template_content', 'vikinger_members_credits_page_screen_content');
        bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
      }

      /**
       * Members credits page HTML content
       */
      function vikinger_members_credits_page_screen_content() {
        $member = get_query_var('member');
        $points = vikinger_gamipress_get_user_points($member['id']);
        $point_count = count($points);

        /**
         * Section Header
         */
        get_template_part('template-part/section/section', 'header', [
          'section_pretitle'  => sprintf(
            esc_html_x('Browse %s', 'Section Header - Pretitle', 'vikinger'),
            $member['name']
          ),
          'section_title'     => esc_html__('Credits Balance', 'vikinger')
        ]);

        if ($point_count > 0) {
        ?>
          <div class="grid grid-3-3-3-3">
          <?php

            foreach ($points as $point) {
              /**
               * Point Item Box
               */
              get_template_part('template-part/point/point-item', 'box', [
                'point' => $point
              ]);
            }
            
          ?>
          </div>
        <?php
        }
      }
    }

    if (vikinger_settings_members_profile_page_is_enabled('badges')) {
      /**
       * Creates member badges page navigation
       */
      function vikinger_members_badges_page_setup_nav() {
        global $bp;

        bp_core_new_nav_item([
          'name'            => esc_html__('Badges', 'vikinger'),
          'slug'            => 'badges',
          'screen_function' => 'vikinger_members_badges_page'
        ]);
      }

      add_action('bp_setup_nav', 'vikinger_members_badges_page_setup_nav');

      /**
       * Members badges page setup
       */
      function vikinger_members_badges_page() {
        add_action('bp_template_content', 'vikinger_members_badges_page_screen_content');
        bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
      }

      /**
       * Members badges page HTML content
       */
      function vikinger_members_badges_page_screen_content() {
        $member = get_query_var('member');
        $badges = vikinger_gamipress_get_user_completed_achievements('badge', $member['id']);
        $badges_count = count($badges);

        /**
         * Section Header
         */
        get_template_part('template-part/section/section', 'header', [
          'section_pretitle'  => sprintf(
            esc_html_x('Browse %s', 'Section Header - Pretitle', 'vikinger'),
            $member['name']
          ),
          'section_title'     => esc_html__('Badges', 'vikinger'),
          'section_text'      => $badges_count
        ]);

        if ($badges_count > 0) {
        ?>
          <div class="grid grid-3-3-3-3">
          <?php
          foreach ($badges as $badge) {
            /**
             * Achievement Item Box
             */
            get_template_part('template-part/achievement/achievement', 'item-box', [
              'achievement' => $badge
            ]);
          }
          ?>
          </div>
        <?php
        }
      }
    }

    if (vikinger_settings_members_profile_page_is_enabled('quests')) {
      /**
       * Creates member quests page navigation
       */
      function vikinger_members_quests_page_setup_nav() {
        global $bp;

        bp_core_new_nav_item([
          'name'            => esc_html__('Quests', 'vikinger'),
          'slug'            => 'quests',
          'screen_function' => 'vikinger_members_quests_page'
        ]);
      }

      add_action('bp_setup_nav', 'vikinger_members_quests_page_setup_nav');

      /**
       * Members quests page setup
       */
      function vikinger_members_quests_page() {
        add_action('bp_template_content', 'vikinger_members_quests_page_screen_content');
        bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
      }

      /**
       * Members quests page HTML content
       */
      function vikinger_members_quests_page_screen_content() {
        $member = get_query_var('member');
        $quests = vikinger_gamipress_get_user_completed_achievements('quest', $member['id']);
        $quests_count = count($quests);

        /**
         * Section Header
         */
        get_template_part('template-part/section/section', 'header', [
          'section_pretitle'  => sprintf(
            esc_html_x('Browse %s', 'Section Header - Pretitle', 'vikinger'),
            $member['name']
          ),
          'section_title'     => esc_html__('Quests', 'vikinger'),
          'section_text'      => $quests_count
        ]);

        if ($quests_count > 0) {
        ?>
          <div class="grid grid-3-3-3-3">
          <?php
          foreach ($quests as $quest) {
            /**
             * Achievement Item Box
             */
            get_template_part('template-part/achievement/achievement', 'item-box', [
              'achievement'             => $quest,
              'achievement_image_wrap'  => true
            ]);
          }
          ?>
          </div>
        <?php
        }
      }
    }

    if (vikinger_settings_members_profile_page_is_enabled('ranks')) {
      /**
       * Creates member ranks page navigation
       */
      function vikinger_members_ranks_page_setup_nav() {
        global $bp;

        bp_core_new_nav_item([
          'name'            => esc_html__('Ranks', 'vikinger'),
          'slug'            => 'ranks',
          'screen_function' => 'vikinger_members_ranks_page'
        ]);
      }

      add_action('bp_setup_nav', 'vikinger_members_ranks_page_setup_nav');

      /**
       * Members ranks page setup
       */
      function vikinger_members_ranks_page() {
        add_action('bp_template_content', 'vikinger_members_ranks_page_screen_content');
        bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
      }

      /**
       * Members ranks page HTML content
       */
      function vikinger_members_ranks_page_screen_content() {
        $member = get_query_var('member');
        $ranks = vikinger_gamipress_get_user_completed_ranks('rank', $member['id']);
        $ranks_count = count($ranks);

        /**
         * Section Header
         */
        get_template_part('template-part/section/section', 'header', [
          'section_pretitle'  => sprintf(
            esc_html_x('Browse %s', 'Section Header - Pretitle', 'vikinger'),
            $member['name']
          ),
          'section_title'     => esc_html__('Rank', 'vikinger')
        ]);

        if ($ranks_count > 0) {
        ?>
          <div class="grid grid-3-3-3-3">
          <?php
          foreach ($ranks as $rank) {
            /**
             * Achievement Item Box
             */
            get_template_part('template-part/achievement/achievement', 'item-box', [
              'achievement'       => $rank,
              'achievement_type'  => 'rank'
            ]);
          }
          ?>
          </div>
        <?php
        }
      }
    }
  }
}

// if buddypress and the settings component are active, create profile settings pages
if (vikinger_plugin_buddypress_is_active() && bp_is_active('settings')) {
  /**
   * Creates member social settings page navigation
   */
  function vikinger_members_settings_social_page_setup_nav() {
    global $bp;

    bp_core_new_subnav_item([
      'name'            => esc_html__('Social Settings', 'vikinger'),
      'slug'            => 'social',
      'parent_slug'     => bp_get_settings_slug(),
      'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
      'screen_function' => 'vikinger_members_settings_social_page',
      'user_has_access' => bp_core_can_edit_settings()
    ], 'members');
  }

  add_action('bp_setup_nav', 'vikinger_members_settings_social_page_setup_nav');

  /**
   * Members social page setup
   */
  function vikinger_members_settings_social_page() {
    add_action('bp_template_content', 'vikinger_members_settings_social_page_screen_content');
    bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
  }

  /**
   * Members social page HTML content
   */
  function vikinger_members_settings_social_page_screen_content() {
    ?>
      <div id="social-settings-screen"></div>
    </div>
    <!-- /GRID -->
    <?php
  }

  // if BuddyPress messages component is active, create message related pages
  if (bp_is_active('messages') && bp_is_active('friends')) {
    /**
     * Creates member messages settings page navigation
     */
    function vikinger_members_settings_messages_page_setup_nav() {
      global $bp;

      bp_core_new_subnav_item([
        'name'            => esc_html__('Message Settings', 'vikinger'),
        'slug'            => 'messages',
        'parent_slug'     => bp_get_settings_slug(),
        'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
        'screen_function' => 'vikinger_members_settings_messages_page',
        'user_has_access' => bp_core_can_edit_settings()
      ], 'members');
    }

    add_action('bp_setup_nav', 'vikinger_members_settings_messages_page_setup_nav');

    /**
     * Members messages page setup
     */
    function vikinger_members_settings_messages_page() {
      add_action('bp_template_content', 'vikinger_members_settings_messages_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members messages page HTML content
     */
    function vikinger_members_settings_messages_page_screen_content() {
      // if BP Better Messages plugin is active, show its messages screen
      if (vikinger_plugin_bpbettermessages_is_active()) {
      ?>
        <!-- ACCOUNT HUB CONTENT -->
        <div class="account-hub-content">
      <?php
        /**
         * Section Header
         */
        get_template_part('template-part/section/section', 'header', [
          'section_pretitle'  => esc_html__('My Profile', 'vikinger'),
          'section_title'     => esc_html__('Messages', 'vikinger')
        ]);
  
        echo BP_Better_Messages()->functions->get_page();
  
      } else {
        $user_id = isset($_GET['user_id']) && is_numeric($_GET['user_id']) ? (int) $_GET['user_id'] : false;
        $message_id = isset($_GET['message_id']) && is_numeric($_GET['message_id']) ? (int) $_GET['message_id'] : false;
  
        if ($user_id) :
  
        ?>
          <div id="messages-settings-screen" data-userid="<?php echo esc_attr($user_id); ?>"></div>
          <?php
        
        elseif ($message_id) :
  
        ?>
          <div id="messages-settings-screen" data-messageid="<?php echo esc_attr($message_id); ?>"></div>
        <?php
        
        else :
  
        ?>
          <div id="messages-settings-screen"></div>
        <?php
  
        endif;
      }
      ?>
        </div>
        <!-- /ACCOUNT HUB CONTENT -->
      </div>
      <!-- /GRID -->
      <?php
    }
  }

  // if BuddyPress friends component is active, create friend related pages
  if (bp_is_active('friends')) {
    /**
     * Creates member friend requests settings page navigation
     */
    function vikinger_members_settings_friend_requests_page_setup_nav() {
      global $bp;

      bp_core_new_subnav_item([
        'name'            => esc_html__('Friend Requests Settings', 'vikinger'),
        'slug'            => 'friend-requests',
        'parent_slug'     => bp_get_settings_slug(),
        'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
        'screen_function' => 'vikinger_members_settings_friend_requests_page',
        'user_has_access' => bp_core_can_edit_settings()
      ], 'members');
    }

    add_action('bp_setup_nav', 'vikinger_members_settings_friend_requests_page_setup_nav');

    /**
     * Members friend requests page setup
     */
    function vikinger_members_settings_friend_requests_page() {
      add_action('bp_template_content', 'vikinger_members_settings_friend_requests_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members friend requests page HTML content
     */
    function vikinger_members_settings_friend_requests_page_screen_content() {
      ?>
        <div id="friend-requests-settings-screen"></div>
      </div>
      <!-- /GRID -->
      <?php
    }
  }

  /**
   * Creates member account info settings page navigation
   */
  function vikinger_members_settings_account_info_page_setup_nav() {
    global $bp;

    bp_core_new_subnav_item([
      'name'            => esc_html__('Account Info Settings', 'vikinger'),
      'slug'            => 'account',
      'parent_slug'     => bp_get_settings_slug(),
      'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
      'screen_function' => 'vikinger_members_settings_account_info_page',
      'user_has_access' => bp_core_can_edit_settings()
    ], 'members');
  }

  add_action('bp_setup_nav', 'vikinger_members_settings_account_info_page_setup_nav');

  /**
   * Members account info page setup
   */
  function vikinger_members_settings_account_info_page() {
    add_action('bp_template_content', 'vikinger_members_settings_account_info_page_screen_content');
    bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
  }

  /**
   * Members account info page HTML content
   */
  function vikinger_members_settings_account_info_page_screen_content() {
    ?>
      <div id="account-info-settings-screen"></div>
    </div>
    <!-- /GRID -->
    <?php
  }

  /**
   * Creates member account settings settings page navigation
   */
  function vikinger_members_settings_account_settings_page_setup_nav() {
    global $bp;

    bp_core_new_subnav_item([
      'name'            => esc_html__('Account Settings', 'vikinger'),
      'slug'            => 'account-settings',
      'parent_slug'     => bp_get_settings_slug(),
      'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
      'screen_function' => 'vikinger_members_settings_account_settings_page',
      'user_has_access' => bp_core_can_edit_settings()
    ], 'members');
  }

  add_action('bp_setup_nav', 'vikinger_members_settings_account_settings_page_setup_nav');

  /**
   * Members account settings page setup
   */
  function vikinger_members_settings_account_settings_page() {
    add_action('bp_template_content', 'vikinger_members_settings_account_settings_page_screen_content');
    bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
  }

  /**
   * Members account settings page HTML content
   */
  function vikinger_members_settings_account_settings_page_screen_content() {
    $member = get_query_var('member');

  ?>
    <!-- ACCOUNT HUB CONTENT -->
    <div class="account-hub-content">
    <?php

      /**
       * Section Header
       */
      get_template_part('template-part/section/section', 'header', [
        'section_pretitle'  => esc_html__('Account', 'vikinger'),
        'section_title'     => esc_html__('Account Settings', 'vikinger')
      ]);

    ?>
      <!-- GRID COLUMN -->
      <div class="grid-column">
        <!-- WIDGET BOX -->
        <div class="widget-box">
          <!-- WIDGET BOX TITLE -->
          <p class="widget-box-title"><?php esc_html_e('Delete Account', 'vikinger'); ?></p>
          <!-- /WIDGET BOX TITLE -->

          <!-- WIDGET BOX CONTENT -->
          <div class="widget-box-content">
          <?php

            if (is_super_admin($member['id'])) {
              /**
               * Alert Line
               */
              get_template_part('template-part/alert/alert', 'line', [
                'type'  =>  'info',
                'text'  =>  sprintf(
                              esc_html__('%sDelete Account:%s Site admins cannot be deleted.', 'vikinger'),
                              '<span class="bold">',
                              '</span>'
                            )
              ]);
            } else if (bp_disable_account_deletion()) {
              /**
               * Alert Line
               */
              get_template_part('template-part/alert/alert', 'line', [
                'type'  =>  'info',
                'text'  =>  sprintf(
                              esc_html__('%sDelete Account Disabled:%s Please contact the site administrator to request account deletion.', 'vikinger'),
                              '<span class="bold">',
                              '</span>'
                            )
              ]);
            } else {
              /**
               * Alert Line
               */
              get_template_part('template-part/alert/alert', 'line', [
                'type'  =>  'error',
                'text'  =>  sprintf(
                              esc_html__('%sWarning:%s deleting your account will delete all of the content you have created. It will be completely irrecoverable.', 'vikinger'),
                              '<span class="bold">',
                              '</span>'
                            )
              ]);

          ?>

            <!-- BUTTON -->
            <div id="vk-user-account-delete" class="button tertiary button-loader">
              <span class="button-text"><?php esc_html_e('Delete My Account', 'vikinger'); ?></span>
              <span class="button-loading-text"><?php esc_html_e('Deleting...', 'vikinger'); ?></span>
              <!-- LOADER SPINNER WRAP -->
              <div class="loader-spinner-wrap small">
                <!-- LOADER SPINNER -->
                <div class="loader-spinner"></div>
                <!-- /LOADER SPINNER -->
              </div>
              <!-- /LOADER SPINNER WRAP -->
            </div>
            <!-- /BUTTON -->
          <?php

            }

          ?>
          </div>
          <!-- /WIDGET BOX CONTENT -->
        </div>
        <!-- /WIDGET BOX -->
      </div>
      <!-- /GRID COLUMN -->
    </div>
    <!-- /ACCOUNT HUB CONTENT -->
  </div>
  <!-- /GRID -->
  <?php
  }

  /**
   * Creates member password settings page navigation
   */
  function vikinger_members_settings_password_page_setup_nav() {
    global $bp;

    bp_core_new_subnav_item([
      'name'            => esc_html__('Password Settings', 'vikinger'),
      'slug'            => 'password',
      'parent_slug'     => bp_get_settings_slug(),
      'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
      'screen_function' => 'vikinger_members_settings_password_page',
      'user_has_access' => bp_core_can_edit_settings()
    ], 'members');
  }

  add_action('bp_setup_nav', 'vikinger_members_settings_password_page_setup_nav');

  /**
   * Members password page setup
   */
  function vikinger_members_settings_password_page() {
    add_action('bp_template_content', 'vikinger_members_settings_password_page_screen_content');
    bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
  }

  /**
   * Members password page HTML content
   */
  function vikinger_members_settings_password_page_screen_content() {
    ?>
      <div id="password-settings-screen"></div>
    </div>
    <!-- /GRID -->
    <?php
  }

  /**
   * Creates member email settings page navigation
   */
  function vikinger_members_settings_email_settings_page_setup_nav() {
    global $bp;

    bp_core_new_subnav_item([
      'name'            => esc_html__('Email Settings', 'vikinger'),
      'slug'            => 'email-settings',
      'parent_slug'     => bp_get_settings_slug(),
      'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
      'screen_function' => 'vikinger_members_settings_email_settings_page',
      'user_has_access' => bp_core_can_edit_settings()
    ], 'members');
  }

  add_action('bp_setup_nav', 'vikinger_members_settings_email_settings_page_setup_nav');

  /**
   * Members email settings page setup
   */
  function vikinger_members_settings_email_settings_page() {
    add_action('bp_template_content', 'vikinger_members_settings_email_settings_page_screen_content');
    bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
  }

  /**
   * Members email settings page HTML content
   */
  function vikinger_members_settings_email_settings_page_screen_content() {
    ?>
      <div id="email-settings-screen"></div>
    </div>
    <!-- /GRID -->
    <?php
  }

  // if BuddyPress groups component is active, create group manage related pages
  if (bp_is_active('groups')) {
    /**
     * Creates member manage groups settings page navigation
     */
    function vikinger_members_settings_manage_groups_page_setup_nav() {
      global $bp;

      bp_core_new_subnav_item([
        'name'            => esc_html__('Manage Groups Settings', 'vikinger'),
        'slug'            => 'manage-groups',
        'parent_slug'     => bp_get_settings_slug(),
        'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
        'screen_function' => 'vikinger_members_settings_manage_groups_page',
        'user_has_access' => bp_core_can_edit_settings()
      ], 'members');
    }

    add_action('bp_setup_nav', 'vikinger_members_settings_manage_groups_page_setup_nav');

    /**
     * Members manage groups page setup
     */
    function vikinger_members_settings_manage_groups_page() {
      add_action('bp_template_content', 'vikinger_members_settings_manage_groups_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members manage groups page HTML content
     */
    function vikinger_members_settings_manage_groups_page_screen_content() {
      ?>
        <div id="manage-groups-settings-screen"></div>
      </div>
      <!-- /GRID -->
      <?php
    }

    /**
     * Creates member send group invitations settings page navigation
     */
    function vikinger_members_settings_send_group_invitations_page_setup_nav() {
      global $bp;

      bp_core_new_subnav_item([
        'name'            => esc_html__('Send Group Invitations Settings', 'vikinger'),
        'slug'            => 'send-group-invitations',
        'parent_slug'     => bp_get_settings_slug(),
        'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
        'screen_function' => 'vikinger_members_settings_send_group_invitations_page',
        'user_has_access' => bp_core_can_edit_settings()
      ], 'members');
    }

    add_action('bp_setup_nav', 'vikinger_members_settings_send_group_invitations_page_setup_nav');

    /**
     * Members send group invitations page setup
     */
    function vikinger_members_settings_send_group_invitations_page() {
      add_action('bp_template_content', 'vikinger_members_settings_send_group_invitations_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members send group invitations page HTML content
     */
    function vikinger_members_settings_send_group_invitations_page_screen_content() {
      ?>
        <div id="send-group-invitations-settings-screen"></div>
      </div>
      <!-- /GRID -->
      <?php
    }

    /**
     * Creates member received group invitations settings page navigation
     */
    function vikinger_members_settings_received_group_invitations_page_setup_nav() {
      global $bp;

      bp_core_new_subnav_item([
        'name'            => esc_html__('Received Group Invitations Settings', 'vikinger'),
        'slug'            => 'received-group-invitations',
        'parent_slug'     => bp_get_settings_slug(),
        'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
        'screen_function' => 'vikinger_members_settings_received_group_invitations_page',
        'user_has_access' => bp_core_can_edit_settings()
      ], 'members');
    }

    add_action('bp_setup_nav', 'vikinger_members_settings_received_group_invitations_page_setup_nav');

    /**
     * Members received group invitations page setup
     */
    function vikinger_members_settings_received_group_invitations_page() {
      add_action('bp_template_content', 'vikinger_members_settings_received_group_invitations_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members received group invitations page HTML content
     */
    function vikinger_members_settings_received_group_invitations_page_screen_content() {
      ?>
        <div id="received-group-invitations-settings-screen"></div>
      </div>
      <!-- /GRID -->
      <?php
    }

    /**
     * Creates group membership requests settings page navigation
     */
    function vikinger_members_settings_group_membership_requests_page_setup_nav() {
      global $bp;

      bp_core_new_subnav_item([
        'name'            => esc_html__('Membership Requests Settings', 'vikinger'),
        'slug'            => 'membership-requests',
        'parent_slug'     => bp_get_settings_slug(),
        'parent_url'      => trailingslashit(bp_loggedin_user_domain() . bp_get_settings_slug()),
        'screen_function' => 'vikinger_members_settings_group_membership_requests_page',
        'user_has_access' => bp_core_can_edit_settings()
      ], 'members');
    }

    add_action('bp_setup_nav', 'vikinger_members_settings_group_membership_requests_page_setup_nav');

    /**
     * Members received group invitations page setup
     */
    function vikinger_members_settings_group_membership_requests_page() {
      add_action('bp_template_content', 'vikinger_members_settings_group_membership_requests_page_screen_content');
      bp_core_load_template(apply_filters('bp_core_template_plugin', 'members/single/plugins'));
    }

    /**
     * Members received group invitations page HTML content
     */
    function vikinger_members_settings_group_membership_requests_page_screen_content() {
      ?>
        <div id="group-membership-requests-settings-screen"></div>
      </div>
      <!-- /GRID -->
      <?php
    }
  }
}

/**
 * Remove unused member subnav pages
 */
function vikinger_members_remove_unused_subnav_items() {
  bp_core_remove_nav_item('profile', 'members');
  bp_core_remove_nav_item('notifications', 'members');
  bp_core_remove_nav_item('messages', 'members');

  bp_core_remove_subnav_item('activity', 'mentions', 'members');
  bp_core_remove_subnav_item('activity', 'favorites', 'members');
  bp_core_remove_subnav_item('activity', 'friends', 'members');
  bp_core_remove_subnav_item('activity', 'groups', 'members');

  bp_core_remove_subnav_item('friends', 'requests', 'members');

  bp_core_remove_subnav_item('groups', 'invites', 'members');

  bp_core_remove_subnav_item('settings', 'data', 'members');
  bp_core_remove_subnav_item('settings', 'profile', 'members');
}

add_action('bp_setup_nav', 'vikinger_members_remove_unused_subnav_items');

/**
 * Remove unused group subnav pages
 */
function vikinger_groups_remove_unused_subnav_items() {
  if (!bp_is_group()) {
    return;
  }

  $slug = bp_get_current_group_slug();

  bp_core_remove_subnav_item($slug, 'admin', 'groups');
  bp_core_remove_subnav_item($slug, 'send-invites', 'groups');

  $parent_slug = $slug . '_manage';

  $hide_tabs = array(
    'edit-details'      => 1,
    'group-settings'    => 1,
    'group-avatar'      => 1,
    'group-cover-image' => 1,
    'manage-members'    => 1,
    'delete-group'      => 1
  );
  
  foreach (array_keys($hide_tabs) as $tab) {
		bp_core_remove_subnav_item($parent_slug, $tab, 'groups');
	}

  // redirect to groups page
  if (!empty($hide_tabs[bp_action_variable(0)])) {
    bp_core_redirect(bp_get_group_permalink(groups_get_current_group()));
	}
}

add_action('bp_actions', 'vikinger_groups_remove_unused_subnav_items');

/**
 * Setup member cover preferred image size
 */
function vikinger_members_cover_image_setup($settings = []) {
  // dimension settings
  $settings['width'] = 1184;
  $settings['height'] = 300;

  return $settings;
}

add_filter('bp_before_members_cover_image_settings_parse_args', 'vikinger_members_cover_image_setup', 10, 1);

?>