<?php
/**
 * Vikinger FILE functions
 * 
 * @since 1.0.0
 */

/**
 * Upload file to the WordPress uploads directory
 * 
 * @param array $fileData
 * @param array $component
 * @return array
 */
function vikinger_upload_file($fileData, $component) {
  // jpg, jpeg, png, mp4, etc
  $fileType = strtolower(pathinfo($fileData['name'], PATHINFO_EXTENSION));
  $fileName = pathinfo($fileData['name'], PATHINFO_FILENAME);

  // remove whitespace from file name
  $fileName = preg_replace('/\s+/', '', $fileName);
  
  // allowed file types
  $allowedFileTypes = ['jpg', 'jpeg', 'png', 'gif'];

  // if uploaded file type is allowed
  if (in_array($fileType, $allowedFileTypes)) {
    // get upload path function string
    $upload_path = vikinger_get_component_upload_path($component);

    // create path if it doesn't exist
    vikinger_create_directory($upload_path);

    // create file path
    $filePath = $upload_path . '/' . $fileName . '.' . $fileType;

    // loop while a file with that name already exists,
    // add a number to the end of the filename until no file exists with that name
    $i = 0;
    while (file_exists($filePath)) {
      $fileName .= $i;
      $filePath = $upload_path . '/' . $fileName . '.' . $fileType;
      $i++;
    }

    // true if successfully moved file, false if not
    $result = move_uploaded_file($fileData['tmp_name'], $filePath);

    if ($result) {
      return [
        'name'    => $fileName,
        'type'    => $fileType,
        'link'    => vikinger_get_component_upload_url($component) . '/' . $fileName . '.' . $fileType,
        'user_id' => $component['uploader_id']
      ];
    }
  }

  return false;
}

/**
 * Delete a file
 * 
 * @param string $filepath    Complete path to the file
 * @param array $component
 * @return array
 */
function vikinger_delete_file($filepath) {
  $result = wp_delete_file($filepath);

  return $result;
}

/**
 * Delete user media directory if a user is deleted
 */
function vikinger_media_user_media_delete($user_id) {
  $result = WP_Filesystem();

  if ($result) {
    global $wp_filesystem;

    $member_uploads_path = vikinger_get_member_uploads_path($user_id);

    // recursively remove files / directories
    $wp_filesystem->delete($member_uploads_path, true);
  }
}

add_action('deleted_user', 'vikinger_media_user_media_delete');

/**
 * Delete group media directory if a group is deleted
 */
function vikinger_media_group_media_delete($group_id) {
  $result = WP_Filesystem();

  if ($result) {
    global $wp_filesystem;

    $group_uploads_path = vikinger_get_group_uploads_path($group_id);

    // recursively remove files / directories
    $wp_filesystem->delete($group_uploads_path, true);
  }
}

add_action('groups_delete_group', 'vikinger_media_group_media_delete');

/**
 * Get component upload path
 * 
 * @param array $component  Component type and id of the upload path to get
 * @return string
 */
function vikinger_get_component_upload_path($component) {
  if ($component['type'] === 'member') {
    return vikinger_get_member_uploads_path($component['id']);
  }

  if ($component['type'] === 'group') {
    return vikinger_get_group_uploads_path($component['id']);
  }
}

/**
 * Get component upload url
 * 
 * @param array $component  Component type and id of the upload url to get
 * @return string
 */
function vikinger_get_component_upload_url($component) {
  if ($component['type'] === 'member') {
    return vikinger_get_member_uploads_url($component['id']);
  }

  if ($component['type'] === 'group') {
    return vikinger_get_group_uploads_url($component['id']);
  }
}

/**
 * Get member upload path
 * 
 * @param int $member_id  ID of the member to get the path of
 * @return string
 */
function vikinger_get_member_uploads_path($member_id) {
  $path = VIKINGER_UPLOADS_PATH . '/member/' . $member_id;

  return $path;
}

/**
 * Get group upload path
 * 
 * @param int $group_id  ID of the group to get the path of
 * @return string
 */
function vikinger_get_group_uploads_path($group_id) {
  $path = VIKINGER_UPLOADS_PATH . '/group/' . $group_id;

  return $path;
}

/**
 * Get member upload url
 * 
 * @param int $member_id  ID of the member to get the url of
 * @return string
 */
function vikinger_get_member_uploads_url($member_id) {
  $url = VIKINGER_UPLOADS_URL . '/member/' . $member_id;

  return $url;
}

/**
 * Get group upload url
 * 
 * @param int $group_id  ID of the group to get the url of
 * @return string
 */
function vikinger_get_group_uploads_url($group_id) {
  $url = VIKINGER_UPLOADS_URL . '/group/' . $group_id;

  return $url;
}

/**
 * Create directory using path
 * 
 * @param string $path  Directory path to create
 * @return boolean
 */
function vikinger_create_directory($path) {
  $result = WP_Filesystem();

  if ($result) {
    global $wp_filesystem;

    if (!$wp_filesystem->exists($path)) {
      // true on success, false on failure
      $result = wp_mkdir_p($path);
    } else {
      // directory already exists
      $result = true;
    }
  }

  return $result;
}

/**
 * Converts php $_FILES form data array to regular array
 * 
 * @param array $php_files   $_FILES array
 * @return array
 */
function vikinger_php_files_to_array($php_files) {
  $files = [];

  foreach ($php_files as $key => $value) {
    for ($i = 0; $i < count($value); $i++) {
      $files[$i][$key] = $value[$i];
    }
  }

  return $files;
}

/**
 * Get file creation task
 * 
 * @param array   $args       Parameters to give to the task
 * @return Vikinger_Task
 */
function vikinger_file_create_task($args) {
  $task_execute = function ($args) {
    $result = vikinger_upload_file($args['fileData'], $args['component']);

    return $result;
  };

  $task_rewind = function ($args) {
    $filepath = vikinger_get_component_upload_path($args['component']) . '/' . $args['fileData']['name'];
    $result = vikinger_delete_file($filepath);
  };

  $task = new Vikinger_Task($task_execute, $task_rewind, $args);

  return $task;
}

/**
 * Get file media entry creation task
 * 
 * @param array   $args       Parameters to give to the task
 * @return Vikinger_Task
 */
function vikinger_file_create_media_entry_task($args = []) {
  $task_execute = function ($args, $uploaded_file_data) {
    $result = vkmedia_create_media($uploaded_file_data);

    return $result;
  };

  $task_rewind = function ($args, $media_id) {
    $result = vkmedia_delete_media($media_id);

    return $result;
  };

  $task = new Vikinger_Task($task_execute, $task_rewind, $args);

  return $task;
}

?>