<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class FollowUser
 * @package FUP
 */
class FollowUser extends Module {

	/**
	 * DB target
	 */
	const TARGET = 'user';

	/**
	 * @param bool $limit
	 * @param array $exclude
	 *
	 * @return array
	 */
	public function getFollowers( $limit = false, $exclude = [] ) {
		if ( ! $limit ) {
			$limits = fup_get_api_limits();
			$limit  = $limits['authors']['initial'];
		}

		$response = [];

		if ( ! fup_can( 'follow-user' ) ) {
			return $response;
		}

		$followers = $this->selectData( self::TARGET, true );

		foreach ( $followers as $key => $item ) {
			if ( in_array( $item->user_id, $exclude, true ) ) {
				unset( $followers[ $key ] );
			}
		}

		$followers = array_splice( $followers, 0, $limit );

		$profile_page = fup_get_profile_page();

		foreach ( $followers as $item ) {
			$user = get_user_by( 'ID', $item->user_id );

			if ( ! $user ) {
				continue;
			}

			$exclude[] = $item->user_id;

			$data              = new \stdClass();
			$data->name        = $user->display_name;
			$data->bio         = $user->description;
			$data->avatar_url  = get_avatar_url( $user->ID, [ 'size' => 120 ] );
			$data->profile_url = trailingslashit( get_permalink( $profile_page ) . $user->user_nicename );
			$data->since       = $item->since;

			$response[] = $data;
		}

		$this->update_transient( 'fup_queried_followers', $exclude );

		return $response;
	}

	/**
	 * @param bool $user_id
	 *
	 * @return array
	 */
	public function getFollowingIds( $user_id = false ) {
		$response = [];

		if ( ! fup_can( 'follow-user' ) ) {
			return $response;
		}

		$following = $this->selectData( self::TARGET, false, $user_id );
		foreach ( $following as $item ) {
			$response[] = $item->target_id;
		}

		return $response;
	}

	/**
	 * @param bool $user_id
	 * @param bool $limit
	 * @param array $exclude
	 *
	 * @return array
	 */
	public function getFollowing( $user_id = false, $limit = false, $exclude = [] ) {
		$response = [];

		$profile_page = fup_get_profile_page();

		$followings = $this->getFollowingIds( $user_id );

		if ( ! $user_id ) {
			if ( ! $limit ) {
				$limits = fup_get_api_limits();
				$limit  = $limits['authors']['initial'];
			}

			$followings = array_diff( $followings, $exclude );
			$followings = array_splice( $followings, 0, $limit );
		}

		foreach ( $followings as $item ) {
			$user = get_user_by( 'ID', $item );

			if ( ! $user ) {
				continue;
			}

			$data              = new \stdClass();
			$data->id          = $user->ID;
			$data->name        = $user->display_name;
			$data->bio         = $user->description;
			$data->avatar_url  = get_avatar_url( $user->ID, [ 'size' => 120 ] );
			$data->profile_url = trailingslashit( get_permalink( $profile_page ) . $user->user_nicename );

			$response[] = $data;
		}

		$this->update_transient( 'fup_queried_following', array_merge( $followings, $exclude ) );

		return $response;
	}

	/**
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 * @throws \Exception
	 */
	public function addFollow( \WP_REST_Request $request ) {
		if ( ! fup_can( 'follow-user' ) ) {
			return $this->response
				->error();
		}

		if ( $this->user->id !== $request['id'] ) {
			$this->insertData( self::TARGET, $request['id'] );

			return $this->response
				->message( 'user-follow' );
		}

		$this->removeTransients();

		return $this->response
			->message( __( 'You cannot follow yourself', 'front-user-profile' ) )
			->error();
	}

	/**
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 */
	public function removeFollow( \WP_REST_Request $request ) {
		if ( ! fup_can( 'follow-user' ) ) {
			return $this->response
				->error();
		}

		$this->deleteData( self::TARGET, $request['id'] );
		$this->removeTransients();

		return $this->response
			->message( 'user-unfollow' );
	}

	/**
	 * Remove transients
	 */
	private function removeTransients() {
		delete_transient( 'fup_home_feed_posts_' . $this->user->id );
		delete_transient( 'fup_queried_followers' );
		delete_transient( 'fup_queried_following' );
	}

}
