<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class FollowCategory
 * @package FUP
 */
class FollowCategory extends Module {

	/**
	 * DB target
	 */
	const TARGET = 'category';

	/**
	 * @param bool $user_id
	 *
	 * @return array
	 */
	public function getFollowingIds( $user_id = false ) {
		$response = [];

		if ( ! fup_can( 'follow-category' ) ) {
			return $response;
		}

		$following = $this->selectData( self::TARGET, false, $user_id );

		foreach ( $following as $item ) {
			$response[] = $item->target_id;
		}

		return $response;
	}

	/**
	 * @param bool $user_id
	 * @param bool $limit
	 * @param bool $merged
	 * @param array $exclude
	 * @param string $hash
	 *
	 * @return array
	 */
	public function getFollowing( $user_id = false, $limit = false, $merged = false, $exclude = [], $hash = '' ) {
		if ( ! $limit ) {
			$limits = fup_get_api_limits();
			$limit  = $limits['categories']['initial'];
		}

		$ids             = $this->getFollowingIds( $user_id );
		$remaining_liked = array_diff( $ids, $exclude );

		$settings = fup_get_settings();

		$args = [
			'include'    => $remaining_liked,
			'exclude'    => $exclude,
			'orderby'    => 'name',
			'hide_empty' => 0,
			'number'     => $limit
		];

		if ( ! $settings['functions']['show-sub-category'] ) {
			$args['parent'] = 0;
		}

		$response = get_categories( $args );

		if ( $merged ) {
			$liked = count( $response );
			if ( $limit - $liked > 0 ) {
				$exclude_existing = [];

				foreach ( $response as $item ) {
					$exclude_existing[] = $item->cat_ID;
				}

				$args2 = [
					'exclude'    => array_merge( $ids, $exclude, $exclude_existing ),
					'orderby'    => 'name',
					'hide_empty' => 0,
					'number'     => $limit - $liked
				];

				if ( ! $settings['functions']['show-sub-category'] ) {
					$args2['parent'] = 0;
				}

				$response = array_merge(
					$response,
					get_categories( $args2 )
				);
			}

			$displayed = [];
			foreach ( $response as $category ) {
				$displayed[] = $category->cat_ID;
			}

			if ( ! empty( $displayed ) ) {
				$displayed = array_merge( $displayed, $exclude );
				if ( is_user_logged_in() ) {
					$this->update_transient( 'fup_queried_categories', $displayed );
				} else {
					if ( ! $hash ) {
						$hash = uniqid( '', true ) . '_' . uniqid( '', true );
					}

					$this->update_transient( 'fup_queried_categories_' . $hash, $displayed, false );
				}
			}
		}

		return [
			'categories' => $response,
			'hash'       => $hash
		];
	}

	/**
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 * @throws \Exception
	 */
	public function addFollow( \WP_REST_Request $request ) {
		if ( ! fup_can( 'follow-category' ) ) {
			return $this->response
				->error();
		}

		$this->insertData( self::TARGET, $request['id'] );

		delete_transient( 'fup_home_feed_posts_' . $this->user->id );

		return $this->response
			->message( 'category-follow' );
	}

	/**
	 * @param $user_id
	 * @param $category_id
	 *
	 * @throws \Exception
	 */
	public function simpleAddFollow( $user_id, $category_id ) {
		if ( ! fup_can( 'follow-category' ) ) {
			return;
		}

		delete_transient( 'fup_home_feed_posts_' . $this->user->id );

		$this->insertData( self::TARGET, $category_id, $user_id );
	}

	/**
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 */
	public function removeFollow( \WP_REST_Request $request ) {
		if ( ! fup_can( 'follow-category' ) ) {
			return $this->response
				->error();
		}

		$this->deleteData( self::TARGET, $request['id'] );

		delete_transient( 'fup_home_feed_posts_' . $this->user->id );

		return $this->response
			->message( 'category-unfollow' );
	}

}
