<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Activity
 * @package FUP
 */
class Activity extends Module {

	/**
	 * @var
	 */
	private $limits;

	/**
	 * @var bool|\stdClass
	 */
	protected $user;

	/**
	 * Activity constructor.
	 */
	public function __construct() {
		parent::__construct();
		$this->limits = fup_get_api_limits();
		$this->user   = fup_get_user();
	}

	/**
	 * @param bool $limit
	 * @param array $exclude
	 *
	 * @return array|\WP_Query
	 */
	public function feed_authors( $limit = false, $exclude = [] ) {
		if ( ! $limit ) {
			$limit = $this->limits['posts']['initial'];
		}

		$response = [];

		$following_authors = fup_get_following_authors_ids();

		if ( ! empty( $following_authors ) ) {
			wp_reset_postdata();

			$response = $this->get_posts( [
				'post_status'    => 'publish',
				'author__in'     => $following_authors,
				'post__not_in'   => $exclude,
				'posts_per_page' => $limit
			] );

			foreach ( $response->posts as $post ) {
				if ( ! in_array( $post->ID, $exclude, true ) ) {
					$exclude[] = $post->ID;
				}
			}
		}

		$this->update_transient( 'fup_queried_feed_authors', $exclude );

		return $response;
	}

	/**
	 * @param bool $limit
	 * @param array $exclude
	 *
	 * @return array|\WP_Query
	 */
	public function feed_categories( $limit = false, $exclude = [] ) {
		if ( ! $limit ) {
			$limit = $this->limits['posts']['initial'];
		}

		$response = [];

		$following_categories = fup_get_following_categories_ids();

		if ( ! empty( $following_categories ) ) {
			wp_reset_postdata();

			$response = $this->get_posts( [
				'post_status'    => 'publish',
				'author__not_in' => [ $this->user->data->ID ],
				'category__in'   => $following_categories,
				'post__not_in'   => $exclude,
				'posts_per_page' => $limit
			] );

			foreach ( $response->posts as $post ) {
				if ( ! in_array( $post->ID, $exclude, true ) ) {
					$exclude[] = $post->ID;
				}
			}
		}

		$this->update_transient( 'fup_queried_feed_categories', $exclude );

		return $response;
	}

	/**
	 * @param bool $limit
	 * @param array $exclude
	 * @param string $hash
	 * @param bool $user_id
	 *
	 * @return array
	 */
	public function stories( $limit = false, $exclude = [], $hash = '', $user_id = false ) {
		if ( ! $limit ) {
			$limit = $this->limits['posts']['initial'];
		}

		wp_reset_postdata();

		$response = $this->get_posts( [
			'post_status'    => 'publish',
			'author__in'     => [ $user_id ?: $this->user->data->ID ],
			'post__not_in'   => $exclude,
			'posts_per_page' => $limit
		] );

		foreach ( $response->posts as $post ) {
			$exclude[] = $post->ID;
		}

		if ( is_user_logged_in() ) {
			$this->update_transient( 'fup_queried_stories', $exclude );
		} else {
			if ( ! $hash ) {
				$hash = uniqid( '', true ) . '_' . uniqid( '', true );
			}

			$this->update_transient( 'fup_queried_stories_' . $hash, $exclude, false );
		}

		return [
			'articles' => $response,
			'hash'     => $hash
		];
	}

	public function drafts( $limit = false, $exclude = [] ) {
		if ( ! $limit ) {
			$limit = $this->limits['posts']['initial'];
		}

		wp_reset_postdata();

		$response = $this->get_posts( [
			'post_status'    => 'draft',
			'author__in'     => [ $this->user->data->ID ],
			'post__not_in'   => $exclude,
			'posts_per_page' => $limit
		] );

		foreach ( $response->posts as $post ) {
			$exclude[] = $post->ID;
		}

		$this->update_transient( 'fup_queried_drafts', $exclude );

		return $response;
	}

}
