<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * @return mixed
 */
function fup_get_profile_page_slug() {
	return get_query_var( 'sub_page' );
}

/**
 * @return bool
 */
function fup_is_main_page() {
	return fup_get_profile_page_slug() === '';
}

/**
 * @return bool
 */
function fup_is_interests_page() {
	return fup_get_profile_page_slug() === 'interests';
}

/**
 * @return bool
 */
function fup_is_settings_page() {
	return fup_get_profile_page_slug() === 'settings';
}

/**
 * @return bool
 */
function fup_is_styles_page() {
	return fup_get_profile_page_slug() === 'styles';
}

function fup_is_publishing_admin_page() {
	$settings = fup_get_settings();

	return $settings['editor']['admin'];
}

/**
 * @return bool
 */
function fup_is_add_article_page() {
	return fup_get_profile_page_slug() === 'add-article';
}

/**
 * @return bool
 */
function fup_is_edit_article_page() {
	return fup_get_profile_page_slug() === 'edit-article';
}

/**
 * @return array|null|WP_Post
 */
function fup_get_profile_page() {
	$page = get_post( get_option( 'fup_profile_page' ) );

	if ( $page && $page->post_type === 'page' && $page->post_status === 'publish' ) {
		return $page;
	}

	return null;
}

/**
 * @return bool
 */
function fup_is_profile_page() {
	return \Fup\ProfileShortcode::instance()->is_profile_page();
}

function fup_is_current_user_profile() {
	return \Fup\ProfileShortcode::instance()->is_current_user_profile();
}

/**
 * @return array
 */
function fup_get_settings() {
	$options = get_option( 'fup_settings' );

	$defaults = [
		'functions'     => [
			'publish-post'        => false,
			'like-article'        => true,
			'like-article-unauth' => false,
			'follow-user'         => true,
			'block-user'          => false,
			'follow-category'     => true,
			'show-sub-category'   => false,
			'bookmark'            => true,
			'set-cover'           => true,
			'dark-mode'           => true,
		],
		'editor'        => [
			'admin'  => true,
			'custom' => false
		],
		'editor-custom' => [
			'add-article-shortcode'  => '',
			'edit-article-shortcode' => ''
		],
		'visibility'    => [
			'drafts'                       => true,
			'show-social'                  => true,
			'show-location'                => true,
			'show-website'                 => true,
			'show-description'             => true,
			'show-author-box'              => true,
			'show-author-box-member-since' => true,
			'show-author-box-bio'          => true,
			'show-author-box-social'       => true,
			'show-author-box-contributors' => true,
			'show-admin-as-contributor'    => true,
			'show-footer-widgets'          => false,
			'show-delete-post-btn'         => true,
			'show-edit-post-btn'           => true,
		],
		'notifications' => [
			'comments' => true,
			'digest'   => true
		],
		'email'         => [
			'logo'                   => '',
			'title'                  => '',
			'facebook'               => '',
			'twitter'                => '',
			'content_text'           => '',
			'footer_text'            => '',
			'authors_posts_limit'    => 5,
			'categories_posts_limit' => 5
		],
		'display'       => [
			'categories' => [
				'initial' => 6,
				'more'    => 6
			],
			'posts'      => [
				'initial' => 12,
				'more'    => 12
			],
			'authors'    => [
				'initial' => 12,
				'more'    => 12
			]
		],
		'render-page'   => get_option( 'fup_profile_page' )
	];

	if ( $options ) {
		foreach ( $defaults as $section => $default ) {
			if ( $section === 'render-page' ) {
				continue;
			}

			if ( isset( $options[ $section ] ) ) {
				if ( is_array( $default ) ) {
					foreach ( $default as $key => $item ) {
						if ( is_array( $item ) ) {
							foreach ( $item as $itemKey => $itemValue ) {
								if ( isset( $options[ $section ][ $key ][ $itemKey ] ) ) {
									$defaults[ $section ][ $key ][ $itemKey ] = $options[ $section ][ $key ][ $itemKey ];
								}
							}
						} else if ( isset( $options[ $section ][ $key ] ) ) {
							$defaults[ $section ][ $key ] = $options[ $section ][ $key ];
						}
					}
				} else {
					$defaults[ $section ] = $options[ $section ];
				}
			}
		}
	}

	return $defaults;
}

/**
 * @return mixed
 */
function fup_get_api_limits() {
	$settings = fup_get_settings();

	return $settings['display'];
}

/**
 * @param $type
 *
 * @return array
 */
function fup_get_api_limit( $type ) {
	$limits = fup_get_api_limits();

	if ( isset( $limits[ $type ] ) ) {
		return $limits[ $type ];
	}

	return [];
}

/**
 * @param string $action
 *
 * @return bool
 */
function fup_can( $action = '' ) {
	if ( ! $action ) {
		return false;
	}

	$settings = fup_get_settings();
	$result   = false;

	if ( isset( $settings['functions'][ $action ] ) && is_bool( $settings['functions'][ $action ] ) ) {
		$result = $settings['functions'][ $action ];
	}

	if ( $action === 'dark-mode' && $result && class_exists( '\TyperPro\Modules' ) && \TyperPro\Modules::instance()->is_active( 'darkmode-module' ) ) {
		return $result;
	}

	return $result;
}

/**
 * @param string $option
 *
 * @return bool|mixed
 */
function fup_show( $option = '' ) {
	if ( ! $option ) {
		return false;
	}

	$settings = fup_get_settings();

	if ( isset( $settings['visibility'][ $option ] ) && is_bool( $settings['visibility'][ $option ] ) ) {
		return $settings['visibility'][ $option ];
	}

	return false;
}

/**
 * @param null $id
 *
 * @return bool|stdClass
 */
function fup_get_user( $id = null ) {
	if ( $id ) {
		$wp_user = get_user_by( 'ID', $id );
	} else if ( get_query_var( 'author_slug' ) ) {
		$wp_user = get_user_by( 'slug', get_query_var( 'author_slug' ) );
	} else {
		$wp_user = wp_get_current_user();
	}

	if ( $wp_user ) {
		$profile_page = fup_get_profile_page();

		if ( ! $profile_page ) {
			return false;
		}

		$profile_url = trailingslashit( trailingslashit( get_permalink( $profile_page ) ) . $wp_user->user_nicename );

		$fup_user                    = new stdClass();
		$fup_user->data              = $wp_user;
		$fup_user->data->description = get_user_meta( $wp_user->ID, 'description', true );
		$fup_user->notifications     = '';
		$fup_user->digest            = get_user_meta( $wp_user->ID, 'weekly-digest', true );

		$fup_user->social = [
			'twitter'   => esc_url( get_user_meta( $wp_user->ID, 'twitter', true ) ),
			'facebook'  => esc_url( get_user_meta( $wp_user->ID, 'facebook', true ) ),
			'instagram' => esc_url( get_user_meta( $wp_user->ID, 'instagram', true ) ),
			'linkedin'  => esc_url( get_user_meta( $wp_user->ID, 'linkedin', true ) )
		];

		$fup_user->location = [
			'country' => get_user_meta( $wp_user->ID, 'country', true ),
			'city'    => get_user_meta( $wp_user->ID, 'city', true )
		];

		$fup_user->add_article_page  = $profile_url . 'add-article';
		$fup_user->edit_article_page = $profile_url . 'edit-article';

		$fup_user->pages = apply_filters( 'fup_profile_pages', [
			'activity'  => [
				'name'   => __( 'Activity', 'front-user-profile' ),
				'url'    => $profile_url,
				'show'   => true,
				'active' => fup_is_main_page() ? 'active' : ''
			],
			'interests' => [
				'name'   => __( 'Interests', 'front-user-profile' ),
				'url'    => $profile_url . 'interests',
				'show'   => true,
				'active' => fup_is_interests_page() ? 'active' : ''
			],
			'settings'  => [
				'name'   => __( 'Settings', 'front-user-profile' ),
				'url'    => $profile_url . 'settings',
				'show'   => true,
				'active' => fup_is_settings_page() ? 'active' : ''
			],
			'styles'    => [
				'name'   => __( 'Styles', 'front-user-profile' ),
				'url'    => $profile_url . 'styles',
				'show'   => true,
				'active' => fup_is_styles_page() ? 'active' : ''
			]
		] );

		if ( isset( $fup_user->pages['interests'] ) && ! fup_can( 'follow-category' ) && ! fup_can( 'bookmark' ) && ! fup_can( 'follow-user' ) && ! fup_can( 'block-user' ) ) {
			unset( $fup_user->pages['interests'] );
		}

		return $fup_user;
	}

	return false;
}

/**
 * @param bool $user_id
 *
 * @return mixed
 */
function fup_get_following_categories_ids( $user_id = false ) {
	return \FUP\FollowCategory::instance()->getFollowingIds( $user_id );
}

/**
 * @param bool $user_id
 * @param bool $limit
 * @param bool $merged
 *
 * @return mixed
 */
function fup_get_following_categories( $user_id = false, $limit = false, $merged = false ) {
	return \FUP\FollowCategory::instance()->getFollowing( $user_id, $limit, $merged );
}

/**
 * @return mixed
 */
function fup_get_authors_feed() {
	return \FUP\Activity::instance()->feed_authors();
}

/**
 * @return mixed
 */
function fup_get_categories_feed() {
	return \FUP\Activity::instance()->feed_categories();
}

/**
 * @return mixed
 */
function fup_get_my_stories() {
	return \FUP\Activity::instance()->stories();
}

/**
 * @return mixed
 */
function fup_get_my_drafts() {
	return \FUP\Activity::instance()->drafts();
}

/**
 * @return mixed
 */
function fup_get_bookmarked_posts_ids() {
	return \FUP\BookmarkPost::instance()->getBookmarksIds();
}

/**
 * @return mixed
 */
function fup_get_bookmarked_posts() {
	return \FUP\BookmarkPost::instance()->getBookmarks();
}

/**
 * @param bool $only_ids
 *
 * @return mixed
 */
function fup_get_liked_posts( $only_ids = false ) {
	return \FUP\LikePost::instance()->getLiked( $only_ids );
}

/**
 * @param null $post_id
 *
 * @return int
 */
function fup_get_like_count( $post_id = null ) {
	return \FUP\LikePost::instance()->getCount( $post_id );
}

/**
 * @param bool $user_id
 *
 * @return mixed
 */
function fup_get_following_authors_ids( $user_id = false ) {
	return \FUP\FollowUser::instance()->getFollowingIds( $user_id );
}

/**
 * @param bool $user_id
 *
 * @return mixed
 */
function fup_get_following_authors( $user_id = false ) {
	return \FUP\FollowUser::instance()->getFollowing( $user_id );
}

/**
 * @return mixed
 */
function fup_get_follower_authors() {
	return \FUP\FollowUser::instance()->getFollowers();
}

/**
 * @return mixed
 */
function fup_get_blocked_authors_ids() {
	return \FUP\BlockUser::instance()->getBlockedIds();
}

/**
 * @return mixed
 */
function fup_get_blocked_authors() {
	return \FUP\BlockUser::instance()->getBlocked();
}

/**
 * @param bool $user_id
 *
 * @return mixed
 */
function fup_get_blockers_ids( $user_id = false ) {
	return \FUP\BlockUser::instance()->getBlockersIds( $user_id );
}

/**
 * @return bool
 */
function fup_is_current_user() {
	$profileUser = fup_get_user();

	return get_current_user_id() === $profileUser->data->ID;
}

/**
 * @param $name
 * @param array $args
 * @param bool $require_once
 * @param bool $echo
 *
 * @return false|string|void
 */
function fup_load_template( $name, $args = [], $require_once = false, $echo = true ) {
	if ( ! $name ) {
		return;
	}

	extract( $args );
	$name = trim( $name );

	$theme_template = locate_template( 'overrides/front-profile/' . $name . '.php' );

	if ( $theme_template ) {
		load_template( $theme_template, $require_once );
	} else {
		ob_start();
		include( FUP_PATH . 'templates/' . $name . '.php' );

		if ( $echo ) {
			echo ob_get_clean();
		} else {
			return ob_get_clean();
		}
	}
}

/**
 * @param int $max
 *
 * @return mixed
 */
function fup_get_most_liked_posts( $max = 5 ) {
	return \FUP\LikePost::instance()->getMostLiked( $max );
}

/**
 * @return array
 */
function fup_colors() {
	return [
		'primary'    => '',
		'red'        => '',
		'pink'       => '',
		'purple'     => '',
		'indigo'     => '',
		'blue'       => '',
		'cyan'       => '',
		'teal'       => '',
		'green'      => '',
		'lime'       => '',
		'yellow'     => '',
		'orange'     => '',
		'brown'      => '',
		'black'      => '',
		'bottom'     => esc_html__( 'Bottom overlay', 'front-user-profile' ),
		'top-bottom' => esc_html__( 'Top & Bottom overlay', 'front-user-profile' ),
		'no-overlay' => esc_html__( 'No overlay', 'front-user-profile' )
	];
}

/**
 * @param $message
 *
 * @return mixed|string
 */
function fup_get_error_message( $message ) {
	$errors = [
		'password' => __( 'Password and password confirmation did not match.', 'front-user-profile' )
	];

	return isset( $errors[ $message ] ) ? $errors[ $message ] : '';
}

/**
 * @return false|string
 */
function fup_get_cover() {
	ob_start();
	fup_load_template( 'cover' );

	return ob_get_clean();
}

/**
 * @param $post
 * @param int $limit
 *
 * @return bool|string|string[]|null
 */
function fup_get_excerpt( $post, $limit = 90 ) {
	$excerpt = trim( get_the_excerpt( $post ) );
	$excerpt = preg_replace( " (\[.*?\])", '', $excerpt );
	$excerpt = strip_shortcodes( $excerpt );
	$excerpt = strip_tags( $excerpt );
	$excerpt = substr( $excerpt, 0, $limit );
	$excerpt = trim( preg_replace( '/\s+/', ' ', $excerpt ) );

	if ( ! $excerpt ) {
		return '';
	}

	return $excerpt;
}
