<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Response
 * @package FUP
 */
class Api extends Singleton {

	/**
	 * @var Response
	 */
	protected $response;

	/**
	 * @var mixed
	 */
	protected $limits;

	const
		MORE_CATEGORIES = 'categories',
		MORE_BOOKMARKS = 'bookmarks',
		MORE_FOLLOWING = 'following',
		MORE_FOLLOWERS = 'followers',
		MORE_BLOCKED = 'blocked',
		MORE_FEED_AUTHORS = 'feed-authors',
		MORE_FEED_CATEGORIES = 'feed-categories',
		MORE_STORIES = 'stories',
		MORE_DRAFTS = 'drafts';

	/**
	 * Api constructor.
	 */
	public function __construct() {
		parent::__construct();

		$this->response = new Response();
		$this->limits   = fup_get_api_limits();
	}

	/**
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 */
	public function more( \WP_REST_Request $request ) {
		$allowed_types = [
			self::MORE_CATEGORIES,
			self::MORE_BOOKMARKS,
			self::MORE_FOLLOWING,
			self::MORE_FOLLOWERS,
			self::MORE_BLOCKED,
			self::MORE_FEED_AUTHORS,
			self::MORE_FEED_CATEGORIES,
			self::MORE_DRAFTS
		];

		if ( ! isset( $request['type'] ) || ! in_array( $request['type'], $allowed_types, true ) ) {
			return $this->response->error();
		}

		$type = $request['type'];

		$data = '';
		switch ( $type ) {
			case self::MORE_FEED_AUTHORS:
				$data = $this->feed_authors();
				break;
			case self::MORE_FEED_CATEGORIES:
				$data = $this->feed_categories();
				break;
			case self::MORE_DRAFTS:
				$data = $this->drafts();
				break;
			case self::MORE_BOOKMARKS:
				$data = $this->bookmarks();
				break;
			case self::MORE_CATEGORIES:
				$data = $this->categories();
				break;
			case self::MORE_BLOCKED:
				$data = $this->blocked();
				break;
			case self::MORE_FOLLOWING:
				$data = $this->following();
				break;
			case self::MORE_FOLLOWERS:
				$data = $this->followers();
				break;
			default:
		}

		return $this->response->data( [ $data ] );
	}

	/**
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 */
	public function moreStories( \WP_REST_Request $request ) {
		return $this->response->data( [ $this->stories( $request['hash'], $request['user_id'] ) ] );
	}

	/**
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 */
	public function moreCategories( \WP_REST_Request $request ) {
		return $this->response->data( [ $this->categories( $request['hash'] ) ] );
	}

	/**
	 * @return string
	 */
	private function feed_authors() {
		$stories = Activity::instance()->feed_authors(
			$this->limits['posts']['more'],
			$this->find_transient( 'fup_queried_feed_authors' )
		);

		ob_start();

		if ( $stories && $stories->have_posts() ) {
			while ( $stories->have_posts() ) {
				$stories->the_post();
				fup_load_template( 'sections/article/default' );
			}
		}

		wp_reset_postdata();

		return ob_get_clean();
	}

	/**
	 * @return string
	 */
	private function feed_categories() {
		$stories = Activity::instance()->feed_categories(
			$this->limits['posts']['more'],
			$this->find_transient( 'fup_queried_feed_categories' )
		);

		ob_start();

		if ( $stories && $stories->have_posts() ) {
			while ( $stories->have_posts() ) {
				$stories->the_post();
				fup_load_template( 'sections/article/default' );
			}
		}

		wp_reset_postdata();

		return ob_get_clean();
	}

	/**
	 * @param string $hash
	 * @param bool $user_id
	 *
	 * @return false|string
	 */
	private function stories( $hash = '', $user_id = false ) {
		$transient_name = 'fup_queried_stories';

		$pass_user = true;
		if ( ! is_user_logged_in() ) {
			$pass_user      = false;
			$transient_name .= '_' . $hash;
		}

		$stories = Activity::instance()->stories(
			$this->limits['posts']['more'],
			$this->find_transient( $transient_name, $pass_user ),
			$hash,
			$user_id
		);

		$stories = $stories['articles'];

		ob_start();

		if ( $stories && $stories->have_posts() ) {
			while ( $stories->have_posts() ) {
				$stories->the_post();
				fup_load_template( 'sections/article/default' );
			}
		}

		wp_reset_postdata();

		return ob_get_clean();
	}

	/**
	 * @return string
	 */
	private function drafts() {
		$drafts = Activity::instance()->drafts(
			$this->limits['posts']['more'],
			$this->find_transient( 'fup_queried_drafts' )
		);

		ob_start();

		if ( $drafts && $drafts->have_posts() ) {
			while ( $drafts->have_posts() ) {
				$drafts->the_post();
				fup_load_template( 'sections/article/default' );
			}
		}

		wp_reset_postdata();

		return ob_get_clean();
	}

	/**
	 * @return string
	 */
	private function bookmarks() {
		$bookmarks = BookmarkPost::instance()->getBookmarks(
			$this->limits['posts']['more'],
			$this->find_transient( 'fup_queried_bookmarks' )
		);

		ob_start();

		if ( $bookmarks && $bookmarks->have_posts() ) {
			while ( $bookmarks->have_posts() ) {
				$bookmarks->the_post();
				fup_load_template( 'sections/article/default' );
			}
		}

		wp_reset_postdata();

		return ob_get_clean();
	}

	/**
	 * @param string $hash
	 *
	 * @return false|string
	 */
	private function categories( $hash = '' ) {
		$transient_name = 'fup_queried_categories';

		$pass_user = true;
		if ( ! is_user_logged_in() ) {
			$pass_user      = false;
			$transient_name .= '_' . $hash;
		}

		$categories = FollowCategory::instance()->getFollowing(
			false,
			$this->limits['categories']['more'],
			true,
			$this->find_transient( $transient_name, $pass_user ),
			$hash
		);

		ob_start();
		fup_load_template( 'sections/listing/category', [
			'categories'          => $categories['categories'],
			'followingCategories' => fup_get_following_categories_ids()
		] );

		return ob_get_clean();
	}

	/**
	 * @return string
	 */
	private function following() {
		$following = FollowUser::instance()->getFollowing(
			false, $this->limits['authors']['more'],
			$this->find_transient( 'fup_queried_following' )
		);

		ob_start();
		fup_load_template( 'sections/listing/following', [
			'following' => $following
		] );

		return ob_get_clean();
	}

	/**
	 * @return string
	 */
	private function followers() {
		$followers = FollowUser::instance()->getFollowers(
			$this->limits['authors']['more'],
			$this->find_transient( 'fup_queried_followers' )
		);

		ob_start();
		fup_load_template( 'sections/listing/followers', [
			'followers'    => $followers,
			'meta_classes' => ''
		] );

		return ob_get_clean();
	}

	/**
	 * @return string
	 */
	private function blocked() {
		$blocked = BlockUser::instance()->getBlocked(
			$this->limits['authors']['more'],
			$this->find_transient( 'fup_queried_blocked' )
		);

		ob_start();
		fup_load_template( 'sections/listing/blocked', [
			'blocked' => $blocked
		] );

		return ob_get_clean();
	}

	/**
	 * @param $name
	 * @param bool $user
	 *
	 * @return array|mixed
	 */
	private function find_transient( $name, $user = true ) {
		if ( $user && is_user_logged_in() ) {
			$name .= '_' . get_current_user_id();
		}

		$exclude = get_transient( $name );

		return $exclude === false ? [] : $exclude;
	}
}
