<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class ProfileShortcode
 * @package FUP
 */
class ProfileShortcode extends Singleton {

	/**
	 * @var string
	 */
	private $currentPage = '';

	/**
	 * @return string
	 */
	public function get_name() {
		return 'sq-front-profile';
	}

	/**
	 * @return string
	 */
	public function get() {
		return '[' . $this->get_name() . ']';
	}

	/**
	 * Init shortcode
	 */
	public function init() {
		add_action( 'wp', [ $this, 'redirect' ] );
		add_shortcode( $this->get_name(), [ $this, 'profile_render' ] );
		add_filter( 'get_post_metadata', [ $this, 'override_wpuf_form_id' ], 10, 4 );
		add_filter( 'get_edit_post_link', [ $this, 'edit_post_link' ], 10, 3 );
	}

	/**
	 * Redirect to homepage if profile page accessed with no username while not logged in
	 */
	public function redirect() {
		global $post;

		$profilePage = fup_get_profile_page();

		if ( $post && $profilePage && $profilePage->post_name === $post->post_name && ! get_query_var( 'author_slug' ) && ! is_user_logged_in() ) {
			return wp_redirect( get_home_url() );
		}

		return false;
	}

	/**
	 * Render
	 *
	 * @return false|string
	 */
	public function profile_render() {
		$this->redirect();

		if ( get_query_var( 'author_slug' ) ) {
			$user = get_user_by( 'slug', get_query_var( 'author_slug' ) );
		} else {
			$user = wp_get_current_user();
		}

		if ( ! $user ) {
			global $wp_query;
			$wp_query->set_404();
			status_header( 404 );
			get_template_part( 404 );
			exit();
		}

		fup_load_template( 'main' );

		return ob_get_clean();
	}

	/**
	 * Check if is profile page
	 *
	 * @return bool
	 */
	public function is_profile_page() {
		$this->currentPage = fup_get_profile_page_slug();

		if ( get_query_var( 'author_slug' ) ) {
			$user = get_user_by( 'slug', get_query_var( 'author_slug' ) );
		} else {
			$user = wp_get_current_user();
		}

		if ( ! $user ) {
			return false;
		}

		$profile_page = fup_get_profile_page();

		return isset( $profile_page ) && get_queried_object_id() === $profile_page->ID;
	}

	/**
	 * Check if it is the current's user profile
	 *
	 * @return bool
	 */
	public function is_current_user_profile() {
		$currentUser = wp_get_current_user();
		$profile     = get_user_by( 'slug', get_query_var( 'author_slug' ) );

		if ( ! $profile ) {
			return true;
		}

		return (int) $currentUser->ID === (int) $profile->ID;
	}

	/**
	 * @param $value
	 * @param $object_id
	 * @param $meta_key
	 * @param $single
	 *
	 * @return int
	 */
	public function override_wpuf_form_id( $value, $object_id, $meta_key, $single ) {
		if ( isset( $meta_key ) && $meta_key === '_wpuf_form_id' ) {
			$settings = fup_get_settings();

			return (int) filter_var( $settings['editor-custom']['add-article-shortcode'], FILTER_SANITIZE_NUMBER_INT );
		}

		return $value;
	}

	/**
	 * @param $link
	 * @param $post_id
	 * @param $context
	 *
	 * @return string
	 */
	public function edit_post_link( $link, $post_id, $context ) {
		if ( is_admin() ) {
			return $link;
		}

		$post      = get_post( $post_id );
		$author_id = (int) $post->post_author;

		if ( fup_is_publishing_admin_page() || $author_id !== get_current_user_id() ) {
			return $link;
		}

		$user = fup_get_user();

		if ( ! $user ) {
			return $link;
		}

		return $user->edit_article_page . '?pid=' . get_the_ID();
	}

}
