<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Settings
 * @package FUP
 */
class Settings extends Singleton {

	/**
	 * Init
	 */
	public function init() {
		add_action( 'admin_menu', [ $this, 'load_admin_menu' ] );
		add_action( 'admin_post_fup_settings', [ $this, 'handle_settings_request' ] );
		add_action( 'admin_post_fup_digest', [ $this, 'test_digest' ] );
		add_action( 'admin_notices', [ $this, 'profile_page_notice' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'load_admin_styles' ] );
		add_action( 'wp_ajax_fup_hide_profile_missing_notice', [ $this, 'fup_hide_profile_missing_notice' ] );
	}

	/**
	 * Adds the menu item in the admin panel
	 */
	public function load_admin_menu() {
		add_submenu_page(
			'options-general.php',
			'Front User Profile',
			'Front User Profile',
			'manage_options',
			'fup',
			[ $this, 'load_admin_template' ]
		);
	}

	/**
	 * Load styles
	 */
	public function load_admin_styles() {
		wp_register_style(
			'fup-admin-css',
			FUP_ASSETS_URL . 'css/settings.css',
			[],
			FUP_VERSION
		);

		wp_enqueue_style( 'fup-admin-css' );
	}

	/**
	 * Load template
	 */
	public function load_admin_template() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'You do not have sufficient permissions to access this page.' ) );
		}

		require FUP_CORE_PATH . 'admin/settings/index.php';
	}

	/**
	 * Test digest mail
	 */
	public function test_digest() {
		if ( $_SERVER['REQUEST_METHOD'] !== 'POST' || ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'You do not have sufficient permissions to access this page.' ) );
		}
		$data   = $_POST;
		$action = $data['action'];

		if ( ! empty( $action ) && $action === 'fup_digest' ) {
			Notifier::instance()->send_digest_notification( true, $data['test-email'] );
		}

		wp_redirect( admin_url( 'admin.php?page=fup' ) );
	}

	/**
	 * Save settings
	 */
	public function handle_settings_request() {
		if ( $_SERVER['REQUEST_METHOD'] !== 'POST' || ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'You do not have sufficient permissions to access this page.' ) );
		}

		$data   = $_POST;
		$action = $data['action'];

		if ( ! empty( $action ) && $action === 'fup_settings' ) {
			$settings = fup_get_settings();

			if ( ! empty( $data['render-page'] ) ) {
				$page = get_post( $data['render-page'] );

				if ( $page && $page->post_type === 'page' ) {
					$settings['render-page'] = $page->ID;
					update_option( 'fup_profile_page', $page->ID );
				}
			}

			$simple_array_settings = [
				'functions',
				'visibility',
				'notifications'
			];

			foreach ( $simple_array_settings as $item ) {
				foreach ( $settings[ $item ] as $field => $value ) {
					$settings[ $item ][ $field ] = isset( $data[ $field ] );
				}
			}

			foreach ( $settings['editor'] as $field => $value ) {
				if ( isset( $data['editor'] ) ) {
					$settings['editor'][ $field ] = $data['editor'] === $field;
				}
			}

			foreach ( $settings['email'] as $field => $value ) {
				if ( isset( $data[ $field ] ) ) {
					$settings['email'][ $field ] = $data[ $field ];
				}
			}

			foreach ( $settings['editor-custom'] as $field => $value ) {
				if ( isset( $data[ $field ] ) ) {
					$settings['editor-custom'][ $field ] = htmlentities( stripslashes( $data[ $field ] ) );
				}
			}

			foreach ( $settings['display'] as $field => $value ) {
				if ( isset( $data[ $field ] ) ) {
					if ( isset( $data[ $field ]['initial'] ) ) {
						$settings['display'][ $field ]['initial'] = (int) $data[ $field ]['initial'];
					}
					if ( isset( $data[ $field ]['more'] ) ) {
						$settings['display'][ $field ]['more'] = (int) $data[ $field ]['more'];
					}
				}
			}

			update_option( 'fup_settings', $settings );
		}

		wp_redirect( admin_url( 'admin.php?page=fup' ) );
	}

	/**
	 * Access restriction
	 */
	public function profile_page_notice() {
		if ( current_user_can( 'list_users' ) ) {
			$this->show_notice();
		}
	}

	/**
	 * Profile page not found notice
	 */
	public function show_notice() {
		$profile_page = fup_get_profile_page();

		if ( ! $profile_page ) {
			?>
            <div class="updated fup-profile-page-missing-notice">
                <p>
                    <span>
                        <?php
                        $message =
                            sprintf(
                                __( 'Your profile page is not set up. <a href="%s">Setup page</a>', 'front-user-profile' ),
                                admin_url( 'admin.php?page=fup' )
                            );
                        ?>
                        <?php echo wp_kses_post( $message ); ?>
                    </span>
                    <a href="#"
                       onclick="fup_hide_profile_missing_notice('<?php echo esc_js( wp_create_nonce( 'fup_notice' ) ); ?>');"
                       style="float:right"><?php echo esc_html__( 'Dismiss', 'front-user-profile' ); ?></a>
                </p>
            </div>
            <script type="text/javascript">
                function fup_hide_profile_missing_notice(nonce) {
                    var data = {action: 'fup_hide_profile_missing_notice', _fup_nonce: nonce};
                    jQuery.post(ajaxurl, data, function (response) {
                        response = response.trim();

                        if (response === "1")
                            jQuery('.fup-profile-page-missing-notice').remove();
                    });
                }
            </script>
			<?php
		}
	}

	/**
	 * Hide notice
	 */
	public function fup_hide_profile_missing_notice() {
		if ( check_ajax_referer( 'fup_notice', '_fup_nonce' ) && update_option( 'fup_hide_profile_missing_notice', 'hide' ) ) {
			echo '1';
		} else {
			echo '0';
		}
		die();
	}

}
