<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Routes
 * @package FUP
 */
class Routes extends Singleton {

	/*
	 * Init routes
	 */
	public function init() {
		add_action( 'rest_api_init', [ $this, 'load' ] );
	}

	/**
	 * Load routes
	 */
	public function load() {
		$postRoutesWithTarget = [
			[
				'endpoint' => '/user/follow',
				'callback' => [
					FollowUser::instance(),
					'addFollow'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/user/unfollow',
				'callback' => [
					FollowUser::instance(),
					'removeFollow'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/user/block',
				'callback' => [
					BlockUser::instance(),
					'addBlock'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/user/unblock',
				'callback' => [
					BlockUser::instance(),
					'removeBlock'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/category/follow',
				'callback' => [
					FollowCategory::instance(),
					'addFollow'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/category/unfollow',
				'callback' => [
					FollowCategory::instance(),
					'removeFollow'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/post/bookmark',
				'callback' => [
					BookmarkPost::instance(),
					'addBookmark'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/post/unbookmark',
				'callback' => [
					BookmarkPost::instance(),
					'removeBookmark'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/post/like',
				'callback' => [
					LikePost::instance(),
					'addLike'
				],
				'auth'     => false
			],
			[
				'endpoint' => '/post/unlike',
				'callback' => [
					LikePost::instance(),
					'removeLike'
				],
				'auth'     => true
			]
		];

		$postRoutesSimple = [
			[
				'endpoint' => '/notification/update-followers',
				'callback' => [
					Notifier::instance(),
					'update_follower_notifications_date'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/notification/subscribe-comments',
				'callback' => [
					Notifier::instance(),
					'subscribe_comments'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/notification/unsubscribe-comments',
				'callback' => [
					Notifier::instance(),
					'unsubscribe_comments'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/load-more',
				'callback' => [
					Api::instance(),
					'more'
				],
				'auth'     => true
			],
			[
				'endpoint' => '/load-more-stories',
				'callback' => [
					Api::instance(),
					'moreStories'
				],
				'auth'     => false
			],
			[
				'endpoint' => '/load-more-categories',
				'callback' => [
					Api::instance(),
					'moreCategories'
				],
				'auth'     => false
			]
		];

		foreach ( $postRoutesWithTarget as $route ) {
			$params = [
				'methods'  => 'POST',
				'callback' => $route['callback'],
				'args'     => [
					'id' => [
						'required' => true,
						'type'     => 'integer',
					]
				]
			];

			if ( $route['auth'] ) {
				$params['permission_callback'] = static function () {
					return is_user_logged_in();
				};
			} else {
				$params['permission_callback'] = static function () {
					return true;
				};
			}

			register_rest_route( FUP_API_NAMESPACE, $route['endpoint'], $params );
		}

		foreach ( $postRoutesSimple as $route ) {
			$params = [
				'methods'  => 'POST',
				'callback' => $route['callback'],
				'args'     => []
			];

			if ( isset( $route['auth'] ) && is_bool( $route['auth'] ) && $route['auth'] ) {
				$params['permission_callback'] = static function () {
					return is_user_logged_in();
				};
			} else {
				$params['permission_callback'] = static function () {
					return true;
				};
			}

			register_rest_route( FUP_API_NAMESPACE, $route['endpoint'], $params );
		}
	}

}
