<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Notifier
 * @package FUP
 */
class Notifier extends Singleton {

	/**
	 * Init notifications
	 */
	public function init() {
		add_action( 'comment_post', [ $this, 'process_comment_subscription' ], 10, 2 );
		add_action( 'cron_comment_notify', [ $this, 'send_comment_notification' ], 10, 2 );
		add_action( 'cron_digest_notify', [ $this, 'send_digest_notification' ] );

		add_filter( 'cron_schedules', [ $this, 'cron_weekly_recurrence' ] );
	}

	/**
	 * Get email header
	 *
	 * @return string
	 */
	public function get_email_header( $title ) {
		$settings = fup_get_settings();

		extract( $settings['email'] );

		ob_start();
		include( FUP_PATH . 'templates/email/parts/header.php' );

		return ob_get_clean();
	}

	/**
	 * Get email footer
	 *
	 * @return string
	 */
	public function get_email_footer() {
		$settings = fup_get_settings();

		extract( $settings['email'] );

		ob_start();
		include( FUP_PATH . 'templates/email/parts/footer.php' );

		return ob_get_clean();
	}

	/**
	 * Get email template for comment notification
	 *
	 * @param \WP_Post $post
	 *
	 * @return string
	 */
	public function get_comment_body( \WP_Post $post ) {
		$settings = fup_get_settings();

		extract( $settings['email'] );

		if ( isset( $content_text ) ) {
			$content_text = stripslashes( $content_text );
			$content_text = str_replace( array(
				'{{firstname}}',
				'{{lastname}}'
			), array(
				get_the_author_meta( 'first_name', $post->post_author ),
				get_the_author_meta( 'last_name', $post->post_author )
			), $content_text );
		}

		ob_start();
		include( FUP_PATH . 'templates/email/comment.php' );

		return ob_get_clean();
	}

	/**
	 * Get email template for weekly digest notification
	 *
	 * @param $posts
	 * @param $category_posts
	 * @param $user
	 *
	 * @return false|string
	 */
	public function get_digest_body( $posts, $category_posts, $user ) {
		$settings = fup_get_settings();

		extract( $settings['email'] );

		if ( isset( $content_text ) ) {
			$content_text = stripslashes( $content_text );
			$content_text = str_replace( array(
				'{{firstname}}',
				'{{lastname}}'
			), array(
				get_the_author_meta( 'first_name', $user ),
				get_the_author_meta( 'last_name', $user )
			), $content_text );
		}

		ob_start();
		include( FUP_PATH . 'templates/email/digest.php' );

		return ob_get_clean();
	}

	/**
	 * Schedule comment notification
	 *
	 * @param \WP_Post $post
	 */
	public function schedule_comment_notification( \WP_Post $post ) {
		wp_schedule_single_event( time() + 20, 'cron_comment_notify', [
			$post->ID,
			get_current_user_id()
		] );
	}

	/**
	 * Schedule comment notifications for subscribed users
	 *
	 * @param $comment_id
	 * @param $comment_approved
	 */
	public function process_comment_subscription( $comment_id, $comment_approved ) {
		if ( $comment_approved === 1 ) {
			$comment = get_comment( $comment_id );

			if ( $comment ) {
				$this->schedule_comment_notification( get_post( $comment->comment_post_ID ) );
			}
		}
	}

	/**
	 * Prepare and send comment notification
	 *
	 * @param $post_id
	 * @param $except_user_id
	 */
	public function send_comment_notification( $post_id, $except_user_id ) {
		$settings = fup_get_settings();

		if ( ! $settings['notifications']['comments'] ) {
			return;
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return;
		}

		$subscribed_users = get_post_meta( $post_id, 'comment_subscription' );

		$headers = [];

		foreach ( $subscribed_users as $user_id ) {
			if ( (int) $user_id === (int) $except_user_id || (int) $post->post_author === (int) $user_id ) {
				continue;
			}

			$user = get_userdata( $user_id );
			if ( $user ) {
				$headers[] = 'Bcc: ' . $user->user_email;
			}
		}

		if ( ! empty( $headers ) ) {
			$title   = esc_html__( 'A new comment has been added to: ', 'front-user-profile' ) . '"' . $post->post_title . '"';
			$content = $this->get_email_header( $title ) . $this->get_comment_body( $post ) . $this->get_email_footer();
			$this->send( null, $title, $content, $headers );

		}
	}

	/**
	 * Subscribe user to comments notifications
	 *
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 */
	public function subscribe_comments( \WP_REST_Request $request ) {
		$post = get_post( $request['post'] );

		$response = new Response();

		if ( ! $post ) {
			return $response->error();
		}

		update_post_meta( $post->ID, 'comment_subscription', get_current_user_id() );

		$this->schedule_comment_notification( $post );

		$post_id = $post->ID;
		ob_start();
		include( FUP_PATH . 'templates/sections/comment/unsubscribe.php' );
		$response->data( [ ob_get_clean() ] );

		return $response;
	}

	/**
	 * Unsubscribe user from comment notifications
	 *
	 * @param \WP_REST_Request $request
	 *
	 * @return Response
	 */
	public function unsubscribe_comments( \WP_REST_Request $request ) {
		$post = get_post( $request['post'] );

		$response = new Response();

		if ( ! $post ) {
			return $response->error();
		}

		delete_post_meta( $post->ID, 'comment_subscription', get_current_user_id() );

		$post_id = $post->ID;
		ob_start();
		include( FUP_PATH . 'templates/sections/comment/subscribe.php' );
		$response->data( [ ob_get_clean() ] );

		return $response;
	}

	/**
	 * Schedule digest notification
	 */
	public function schedule_digest_notification() {
		wp_schedule_event( time(), 'weekly', 'cron_digest_notify' );
	}

	/**
	 * Prepare and send digest notification for each user
	 *
	 * @param bool $preview
	 * @param string $test_email
	 */
	public function send_digest_notification( $preview = false, $test_email = '' ) {
		$settings = fup_get_settings();
		$title    = esc_html__( 'This is your weekly digest', 'front-user-profile' );

		if ( ! $settings['notifications']['digest'] && ! $preview ) {
			return;
		}

		remove_action( 'pre_get_posts', [ FupPlugin::instance(), 'filter_blocked' ] );

		if ( $preview ) {
			$posts_1 = new \WP_Query( [
				'ignore_sticky_posts' => 1,
				'post_type'           => 'post',
				'post_status'         => 'publish',
				'orderby'             => 'rand',
				'posts_per_page'      => $settings['email']['authors_posts_limit']
			] );

			$posts_2 = new \WP_Query( [
				'ignore_sticky_posts' => 1,
				'post_type'           => 'post',
				'post_status'         => 'publish',
				'orderby'             => 'rand',
				'posts_per_page'      => $settings['email']['authors_posts_limit']
			] );

			$all_categories = get_categories();

			$cat[] = [
				'category' => $all_categories[0],
				'posts'    => $posts_2->posts
			];

			$content = $this->get_email_header( $title ) . $this->get_digest_body( $posts_1->posts, $cat, fup_get_user( get_current_user_id() ) ) . $this->get_email_footer();

			if ( $test_email ) {
				$this->send( $test_email, $title, $content );
			} else {
				$content = '<div style="text-align: center; font-weight: bold; padding: 10px; background: red; color: #fff;">' . esc_html__( 'This is a preview with random posts! Only the site admins can see this!', 'front-user-profile' ) . '</div>' .
				           $content;
				echo $content;
				die();
			}
		} else {
			$users = get_users();

			foreach ( $users as $user ) {
				$user_options = fup_get_user( $user->ID );

				if ( ! $user_options->digest && ! $preview ) {
					continue;
				}

				$following_authors    = fup_get_following_authors_ids( $user->ID );
				$following_categories = fup_get_following_categories_ids( $user->ID );
				$blocked_by           = fup_get_blockers_ids( $user->ID );

				$posts          = [];
				$category_posts = [];
				$exclude_posts  = [];

				if ( ! empty( $following_authors ) || $preview ) {
					$query = new \WP_Query( [
						'author__in'          => $following_authors,
						'author__not_in'      => $blocked_by,
						'ignore_sticky_posts' => 1,
						'post_type'           => 'post',
						'post_status'         => 'publish',
						'orderby'             => 'rand',
						'posts_per_page'      => $settings['email']['authors_posts_limit'],
						'date_query'          => [
							[
								'after'     => '7 days ago',
								'inclusive' => true
							]
						],
					] );

					$posts = $query->posts;

					foreach ( $posts as $post ) {
						$exclude_posts[] = $post->ID;
					}
				}

				if ( ! empty( $following_categories ) || $preview ) {
					$limit = 1;
					foreach ( $following_categories as $following_category ) {
						if ( $limit < 1 ) {
							continue;
						}

						$category = get_category( $following_category );

						if ( $category ) {
							$query = new \WP_Query( [
								'post__not_in'        => $exclude_posts,
								'category__in'        => [ $following_category ],
								'author__not_in'      => $blocked_by,
								'ignore_sticky_posts' => 1,
								'post_type'           => 'post',
								'post_status'         => 'publish',
								'orderby'             => 'rand',
								'posts_per_page'      => $settings['email']['categories_posts_limit'],
								'date_query'          => [
									[
										'after'     => '7 days ago',
										'inclusive' => true
									]
								],
							] );

							$cat_posts = $query->posts;

							if ( ! empty( $cat_posts ) ) {
								$category_posts[] = [
									'category' => $category,
									'posts'    => $cat_posts
								];
							}

							$limit --;
						}
					}
				}

				if ( ! empty( $posts ) || ! empty( $category_posts ) ) {
					$content = $this->get_email_header( $title ) . $this->get_digest_body( $posts, $category_posts, $user_options ) . $this->get_email_footer();

					$this->send( $user->user_email, $title, $content );
				}
			}
		}
	}

	/**
	 * Sends notification email
	 *
	 * @param string $to
	 * @param $subject
	 * @param $message
	 * @param array $headers
	 */
	public function send( $to = '', $subject, $message, $headers = [] ) {
		$headers[] = 'Content-Type: text/html; charset=UTF-8';

		wp_mail( $to, $subject, $message, $headers );
	}

	/**
	 * Update last seen followers notification
	 */
	public function update_follower_notifications_date() {
		update_user_meta( get_current_user_id(), 'fup_last_followers_check', ( new \DateTime( 'now' ) )->format( 'Y-m-d H:i:s' ) );
	}

	/**
	 * Creates weekly cron recurrence
	 *
	 * @param $schedules
	 *
	 * @return mixed
	 */
	public function cron_weekly_recurrence( $schedules ) {
		$schedules['weekly'] = [
			'interval' => 604800,
			'display'  => esc_html__( 'Weekly' )
		];

		return $schedules;
	}

	/**
	 * Remove cron
	 */
	public function remove_cron_notifications() {
		wp_clear_scheduled_hook( 'cron_comment_notify' );
		wp_clear_scheduled_hook( 'cron_digest_notify' );
	}

}
