<?php

namespace FUP;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class FupPlugin
 * @package FUP
 */
class FupPlugin {

	/**
	 * @var
	 */
	public static $instance;

	/**
	 * @return FupPlugin|null
	 */
	public static function instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * FupPlugin constructor.
	 */
	public function __construct() {
		$this->include_modules();

		Notifier::instance()->init();
		Routes::instance()->init();
		Settings::instance()->init();
		Rewrites::instance()->init();
		CustomizerOptions::instance();

		register_activation_hook( FUP_FILE, [ $this, 'migrate' ] );
		register_activation_hook( FUP_FILE, [ $this, 'generate_profile_page' ] );
		register_activation_hook( FUP_FILE, [ $this, 'register_cron' ] );
		register_deactivation_hook( FUP_FILE, [ $this, 'unregister_cron' ] );

		add_filter( 'user_contactmethods', [ $this, 'user_profile_fields' ] );
		add_filter( 'get_avatar', [ $this, 'custom_avatar' ], 10, 5 );
		add_filter( 'get_avatar_url', [ $this, 'custom_avatar_url' ], 10, 2 );
		add_filter( 'the_content', [ $this, 'load_author_box' ], 13 );
		add_filter( 'the_content', [ $this, 'load_post_bookmark' ] );
		add_filter( 'typer_onboarding_display_items', [ $this, 'onboarding_items' ] );
		add_filter( 'typer_onboarding_render', [ $this, 'onboarding_render' ] );
		add_filter( 'store_user_preferences', [ $this, 'store_user_preferences' ], 10, 2 );
		add_filter( 'typer_render_general_panel', [ $this, 'remove_general_panel' ] );
		add_filter( 'typer_show_footer_widgets', [ $this, 'show_footer_widgets' ] );
		add_filter( 'typer_options_meta_items', [ $this, 'add_bookmark_options' ] );
		add_filter( 'typer_blog_options_meta_items', [ $this, 'add_bookmark_options' ] );

		add_action( 'typer_post_list_after_meta', [ $this, 'add_edit_button' ] );
		add_action( 'typer_post_list_after_meta', [ $this, 'add_delete_button' ] );

		add_filter( 'author_link', [ $this, 'redirect_profile' ], 20, 3 );
		add_filter( 'body_class', [ $this, 'profile_page_class' ] );

		add_action( 'init', [ $this, 'init' ] );
	}

	/**
	 * Include modules
	 */
	public function include_modules() {
		include FUP_CORE_PATH . 'Singleton.php';
		include FUP_CORE_PATH . 'db/db.php';
		include FUP_CORE_PATH . 'Routes.php';
		include FUP_CORE_PATH . 'Rewrites.php';
		include FUP_CORE_PATH . 'Notifier.php';
		include FUP_CORE_PATH . 'CustomizerOptions.php';

		include FUP_CORE_PATH . 'shortcodes/ProfileShortcode.php';
		include FUP_CORE_PATH . 'shortcodes/AuthorBoxShortcode.php';
		include FUP_CORE_PATH . 'shortcodes/PostBookmarkShortcode.php';
		include FUP_CORE_PATH . 'shortcodes/FollowCategoryShortcode.php';
		include FUP_CORE_PATH . 'shortcodes/BookmarksShortcode.php';
		include FUP_CORE_PATH . 'shortcodes/FollowedAuthorsShortcode.php';
		include FUP_CORE_PATH . 'shortcodes/FollowersShortcode.php';
		include FUP_CORE_PATH . 'shortcodes/BlockedAuthorsShortcode.php';

		include FUP_CORE_PATH . 'admin/Settings.php';

		include FUP_MODULES_PATH . 'Module.php';
		include FUP_MODULES_PATH . 'BlockUser.php';
		include FUP_MODULES_PATH . 'BookmarkPost.php';
		include FUP_MODULES_PATH . 'LikePost.php';
		include FUP_MODULES_PATH . 'FollowCategory.php';
		include FUP_MODULES_PATH . 'FollowUser.php';
		include FUP_MODULES_PATH . 'Activity.php';

		include FUP_HELPERS_PATH . 'Response.php';
		include FUP_HELPERS_PATH . 'CategoryImage.php';
		include FUP_HELPERS_PATH . 'FormBuilder.php';
		include FUP_HELPERS_PATH . 'Api.php';
	}

	/**
	 * Run migration
	 */
	public function migrate() {
		include FUP_CORE_PATH . 'db/migration.php';
		include FUP_CORE_PATH . 'db/upgrade_db.php';
		$migration = new Migration();
		$migration->execute();
	}

	/**
	 * Plugin init
	 */
	public function init() {
		$this->migrate();
		add_action( 'wp_enqueue_scripts', [ $this, 'load_scripts' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'load_admin_scripts' ] );
		add_action( 'admin_post_update_user', [ $this, 'handle_settings_request' ] );
		add_action( 'admin_post_update_user_style', [ $this, 'handle_settings_request' ] );
		add_action( 'pre_get_comments', [ $this, 'filter_blocked' ] );
		add_action( 'pre_get_posts', [ $this, 'filter_blocked' ] );
		add_action( 'comment_form_logged_in_after', [ $this, 'add_comment_unsubscribe_form' ] );
		add_action( 'typer_social_share_icons', [ $this, 'add_post_like_action' ] );
		add_action( 'author_meta_action_follow', [ $this, 'add_meta_button_follow' ] );
		add_action( 'author_meta_action_bookmark', [ $this, 'add_meta_button_bookmark' ] );

		ProfileShortcode::instance()->init();
		AuthorBoxShortcode::instance()->init();
		PostBookmarkShortcode::instance()->init();
		FollowCategoryShortcode::instance()->init();
		BookmarksShortcode::instance()->init();
		FollowedAuthorsShortcode::instance()->init();
		FollowersShortcode::instance()->init();
		BlockedAuthorsShortcode::instance()->init();

		CategoryImage::instance()->init();

		$this->digestPreview();
	}

	/**
	 * Load frontend scripts
	 */
	public function load_scripts() {

		wp_register_script(
			'front-user-profile',
			FUP_ASSETS_URL . 'js/fup.js',
			[
				'jquery',
			],
			FUP_VERSION
		);

		wp_localize_script( 'front-user-profile', 'fupVar', [
			'nonce'    => wp_create_nonce( 'wp_rest' ),
			'base_url' => get_option( 'siteurl' )
		] );

		wp_enqueue_script( 'front-user-profile' );
	}

	/**
	 * Load admin scripts
	 */
	public function load_admin_scripts() {
		wp_register_script(
			'fup-admin-js',
			FUP_ASSETS_URL . 'js/fup-admin.js',
			[
				'jquery'
			],
			FUP_VERSION
		);

		wp_enqueue_script( 'fup-admin-js' );
	}

	/**
	 * Register digest cron
	 */
	public function register_cron() {
		Notifier::instance()->schedule_digest_notification();
	}

	/**
	 * Remove digest cron
	 */
	public function unregister_cron() {
		Notifier::instance()->remove_cron_notifications();
	}

	/**
	 * Custom upload directory for profile assets
	 *
	 * @param $dir
	 *
	 * @return array
	 */
	public function custom_upload_dir( $dir ) {
		return array(
			       'path'   => $dir['basedir'] . '/front-user-profile',
			       'url'    => $dir['baseurl'] . '/front-user-profile',
			       'subdir' => '/front-user-profile',
		       ) + $dir;
	}

	/**
	 * Generate custom file name
	 *
	 * @param $dir
	 * @param $name
	 * @param $ext
	 *
	 * @return string
	 */
	public function custom_filename( $dir, $name, $ext ) {
		return time() . '_' . $name;
	}

	/**
	 * Process settings post request
	 */
	public function handle_settings_request() {
		$user = wp_get_current_user();

		if ( ! $user || $_SERVER['REQUEST_METHOD'] !== 'POST' ) {
			return;
		}

		$data   = $_POST;
		$action = $data['action'];

		if ( empty( $action ) ) {
			return;
		}

		if ( $action === 'update_user' ) {

			if ( ! empty( $data['password'] ) &&
			     ! empty( $data['confirm_password'] ) &&
			     $data['password'] === $data['confirm_password'] ) {
				wp_set_password( $data['password'], $user->ID );
			}

			if ( isset( $data['website'] ) ) {
				wp_update_user( (object) [ 'ID' => $user->ID, 'user_url' => esc_url_raw( $data['website'] ) ] );
			}

			if ( ! empty( $data['email'] ) && is_email( esc_attr( $data['email'] ) && email_exists( esc_attr( $data['email'] ) ) === $user->ID ) ) {
				wp_update_user( (object) [ 'ID' => $user->ID, 'user_email' => sanitize_email( $data['email'] ) ] );
			}

			if ( isset( $data['display_name'] ) && $data['display_name'] ) {
				wp_update_user( [
					'ID'           => $user->ID,
					'display_name' => sanitize_text_field( $data['display_name'] )
				] );
			}

			if ( isset( $data['firstname'] ) ) {
				update_user_meta( $user->ID, 'first_name', sanitize_text_field( $data['firstname'] ) );
			}

			if ( isset( $data['lastname'] ) ) {
				update_user_meta( $user->ID, 'last_name', sanitize_text_field( $_POST['lastname'] ) );
			}

			if ( isset( $data['bio'] ) ) {
				update_user_meta( $user->ID, 'description', sanitize_text_field( $data['bio'] ) );
			}

			if ( isset( $data['country'] ) ) {
				update_user_meta( $user->ID, 'country', sanitize_text_field( $data['country'] ) );
			}

			if ( isset( $data['city'] ) ) {
				update_user_meta( $user->ID, 'city', sanitize_text_field( $data['city'] ) );
			}

			if ( isset( $data['facebook'] ) ) {
				update_user_meta( $user->ID, 'facebook', esc_url_raw( $data['facebook'] ) );
			}

			if ( isset( $data['instagram'] ) ) {
				update_user_meta( $user->ID, 'instagram', esc_url_raw( $data['instagram'] ) );
			}

			if ( isset( $data['twitter'] ) ) {
				update_user_meta( $user->ID, 'twitter', esc_url_raw( $data['twitter'] ) );
			}

			if ( isset( $data['linkedin'] ) ) {
				update_user_meta( $user->ID, 'linkedin', esc_url_raw( $data['linkedin'] ) );
			}

			if ( isset( $data['digest'] ) ) {
				update_user_meta( $user->ID, 'weekly-digest', 1 );
			} else {
				update_user_meta( $user->ID, 'weekly-digest', 0 );
			}

			$errors = [ 'err' => [] ];

			if ( ! empty( $data['password'] ) &&
			     ! empty( $data['confirm_password'] ) &&
			     $data['password'] !== $data['confirm_password'] ) {
				$errors['err'][] = 'password';
			}

			$errors['err'] = implode( ',', $errors['err'] );

			do_action( 'edit_user_profile_update', $user->ID );
			wp_redirect( $data['redirect'] . '?' . build_query( $errors ) );

		} else if ( $action === 'update_user_style' ) {
			if ( ! empty( $_FILES ) ) {
				require_once( ABSPATH . 'wp-admin/includes/file.php' );

				foreach ( $_FILES as $name => $file ) {
					if ( ! $file['name'] || ! $file['type'] || ! $file['size'] || ! $file['tmp_name'] ) {
						continue;
					}

					add_filter( 'upload_dir', [ $this, 'custom_upload_dir' ] );

					$saved_file = wp_handle_upload( $file, [
						'test_form'                => false,
						'unique_filename_callback' => [
							$this,
							'custom_filename'
						]
					] );

					remove_filter( 'upload_dir', [ $this, 'custom_upload_dir' ] );

					if ( isset( $saved_file['error'] ) || isset( $saved_file['upload_error_handler'] ) ) {
						return false;
					}


					$file_url = $saved_file['url'];

					if ( $name === 'avatar' ) {
						$image_file = wp_get_image_editor( $saved_file['file'] );
						if ( ! is_wp_error( $image_file ) ) {
							$image_file->resize( 100, 100, true );
							$image_file->save( $saved_file['file'] );
						}
					}

					if ( $file_url ) {
						$info      = new \SplFileInfo( $file_url );
						$file_name = $info->getFilename();

						if ( $name === 'cover' ) {
							$user_cover = get_user_meta( $user->ID, 'fup_cover', true );
							if ( is_array( $user_cover ) ) {
								$user_cover[ get_current_blog_id() ] = $file_name;
								$cover_data                          = $user_cover;
							} else {
								$cover_data                          = [];
								$cover_data[ get_current_blog_id() ] = $file_name;
							}

							update_user_meta( $user->ID, 'fup_cover', $cover_data );
						} elseif ( $name === 'avatar' ) {
							$user_avatar = get_user_meta( $user->ID, 'fup_avatar', true );
							if ( is_array( $user_avatar ) ) {
								$user_avatar[ get_current_blog_id() ] = $file_name;
								$avatar_data                          = $user_avatar;
							} else {
								$avatar_data                          = [];
								$avatar_data[ get_current_blog_id() ] = $file_name;
							}

							update_user_meta( $user->ID, 'fup_avatar', $avatar_data );
						}
					}
				}
			}

			if ( isset( $data['delete-cover'] ) && (int) $data['delete-cover'] ) {
				$cover_files = get_user_meta( $user->ID, 'fup_cover', true );

				if ( is_array( $cover_files ) && isset( $cover_files[ get_current_blog_id() ] ) ) {
					$upload_dir = wp_upload_dir();
					$coverImage = $upload_dir['basedir'] . '/front-user-profile/' . $cover_files[ get_current_blog_id() ];

					@unlink( $coverImage );
					unset( $cover_files[ get_current_blog_id() ] );

					update_user_meta( $user->ID, 'fup_cover', $cover_files );
				}
			}

			if ( isset( $data['overlay'] ) && array_key_exists( $data['overlay'], fup_colors() ) ) {
				update_user_meta( $user->ID, 'fup_cover_overlay', $data['overlay'] );
			}

			do_action( 'edit_user_profile_style_update', $user->ID );
			wp_redirect( $data['redirect'] );
		}
	}

	/**
	 * @param $query
	 *
	 * @return mixed
	 */
	public function filter_blocked( $query ) {
		if ( is_admin() ) {
			return $query;
		}

		$blockers = BlockUser::instance()->getBlockersIds();

		if ( empty( $blockers ) ) {
			return $query;
		}

		if ( empty( $query->query_vars['author__not_in'] ) ) {
			$query->query_vars['author__not_in'] = $blockers;
		} else {
			if ( is_array( $query->query_vars['author__not_in'] ) ) {
				$query->query_vars['author__not_in'] = array_merge( $blockers, $query->query_vars['author__not_in'] );
			} else {
				$excluded_author                     = explode( ",", $query->query_vars['author__not_in'] );
				$query->query_vars['author__not_in'] = array_merge( $blockers, $excluded_author );
			}
		}

		return $query;
	}

	/**
	 * Subscription for comments
	 */
	public function add_comment_unsubscribe_form() {
		if ( is_user_logged_in() ) {
			$settings = fup_get_settings();

			if ( ! $settings['notifications']['comments'] ) {
				return;
			}

			$subscriptions = get_post_meta( get_the_ID(), 'comment_subscription' );

			if ( in_array( (string) get_current_user_id(), $subscriptions, true ) ) {
				ob_start();
				include( FUP_PATH . 'templates/sections/comment/unsubscribe.php' );
				echo ob_get_clean();
			} else {
				ob_start();
				include( FUP_PATH . 'templates/sections/comment/subscribe.php' );
				echo ob_get_clean();
			}
		}
	}

	/**
	 * Generate profile page
	 */
	public function generate_profile_page() {
		if ( ! get_option( 'fup_profile_page' ) ) {
			$id = wp_insert_post( [
				'post_type'      => 'page',
				'post_title'     => 'Profile',
				'post_status'    => 'publish',
				'post_content'   => '[sq-front-profile]',
				'comment_status' => 'closed'
			] );

			update_option( 'fup_profile_page', $id );
		}
	}

	/**
	 * @param $profile_fields
	 *
	 * @return mixed
	 */
	public function user_profile_fields( $profile_fields ) {
		$profile_fields['country']   = __( 'Country', 'front-user-profile' );
		$profile_fields['city']      = __( 'City', 'front-user-profile' );
		$profile_fields['twitter']   = __( 'Twitter URL', 'front-user-profile' );
		$profile_fields['facebook']  = __( 'Facebook URL', 'front-user-profile' );
		$profile_fields['instagram'] = __( 'Instagram URL', 'front-user-profile' );
		$profile_fields['linkedin']  = __( 'Linkedin URL', 'front-user-profile' );

		return $profile_fields;
	}

	/**
	 * @param $avatar
	 * @param $id_or_email
	 * @param $size
	 * @param $default
	 * @param $alt
	 *
	 * @return string
	 */
	public function custom_avatar( $avatar, $id_or_email, $size, $default, $alt ) {
		if ( isset( $GLOBALS['current_screen'], $GLOBALS['current_screen']->id ) &&
		     $GLOBALS['current_screen']->id === 'options-discussion' &&
		     is_admin() ) {
			return $avatar;
		}

		$user = false;

		if ( is_numeric( $id_or_email ) ) {
			$id   = (int) $id_or_email;
			$user = get_user_by( 'id', $id );
		} elseif ( is_object( $id_or_email ) ) {
			if ( ! empty( $id_or_email->user_id ) ) {
				$id   = (int) $id_or_email->user_id;
				$user = get_user_by( 'id', $id );
			}
		} else {
			$user = get_user_by( 'email', $id_or_email );
		}

		if ( $user && is_object( $user ) ) {
			$avatar_files = get_user_meta( $user->ID, 'fup_avatar', true );

			if ( ! is_array( $avatar_files ) || ! isset( $avatar_files[ get_current_blog_id() ] ) ) {
				return $avatar;
			}

			$upload_dir  = wp_upload_dir();
			$avatar_file = $upload_dir['baseurl'] . '/front-user-profile/' . $avatar_files[ get_current_blog_id() ];

			$avatar = sprintf( "<img alt='%s' src='%s' class='avatar avatar-%d photo avatar-img' height='%d' width='%d' />",
				esc_attr( $alt ),
				esc_url( $avatar_file ),
				(int) $size,
				(int) $size,
				(int) $size
			);
		}

		return $avatar;
	}

	/**
	 * @param $url
	 * @param $user
	 *
	 * @return false|string
	 */
	public function custom_avatar_url( $url, $user ) {
		$avatar_files = get_user_meta( $user, 'fup_avatar', true );

		if ( ! is_array( $avatar_files ) || ! isset( $avatar_files[ get_current_blog_id() ] ) ) {
			return $url;
		}

		$upload_dir = wp_upload_dir();
		$avatar_url = $upload_dir['baseurl'] . '/front-user-profile/' . $avatar_files[ get_current_blog_id() ];

		return $avatar_url;
	}

	/**
	 * @param $content
	 *
	 * @return string
	 */
	public function load_author_box( $content ) {
		if ( fup_show( 'show-author-box' ) && is_singular( 'post' ) ) {
			$box     = do_shortcode( AuthorBoxShortcode::instance()->get() );
			$content .= $box;

			remove_filter( 'the_content', [ $this, 'load_author_box' ], 20 );
		}

		return $content;
	}

	/**
	 * @param $content
	 *
	 * @return string
	 */
	public function load_post_bookmark( $content ) {
		if ( is_singular( 'post' ) ) {
			$bookmark = do_shortcode( PostBookmarkShortcode::instance()->get() );

			return $bookmark . $content;
		}

		return $content;
	}

	/**
	 * Adds like button to post sticky share
	 */
	public function add_post_like_action() {
		if ( fup_can( 'like-article' ) ) {
			fup_load_template( 'components/heart' );
		}
	}

	/**
	 * Show follow button
	 */
	public function add_meta_button_follow() {
		if ( isset( $_GET['preview'] ) && $_GET['preview'] ) {
			return;
		}

		global $post;
		$author = fup_get_user( $post->post_author );
		if ( $author ) {
			fup_load_template( 'components/meta-button-follow', [ 'author' => $author, 'post' => $post ] );
		}
	}

	/**
	 * Show bookmark button
	 */
	public function add_meta_button_bookmark() {
		if ( isset( $_GET['preview'] ) && $_GET['preview'] ) {
			return;
		}

		fup_load_template( 'components/meta-button-bookmark' );
	}

	/**
	 * Add edit button
	 */
	public function add_edit_button() {
		if ( fup_show( 'show-edit-post-btn' ) ) {
			global $post;
			$author_id = (int) $post->post_author;
			if ( $author_id !== get_current_user_id() ) {
				return;
			}

			if ( fup_is_publishing_admin_page() ) {
				$url = get_edit_post_link();
			} else {
				$user = fup_get_user();
				$url  = $user->edit_article_page . '?pid=' . get_the_ID();
			}

			fup_load_template( 'components/meta-button-edit', [ 'url' => $url ] );
		}
	}

	/**
	 * Add delete button
	 */
	public function add_delete_button() {
		if ( fup_show( 'show-delete-post-btn' ) && fup_is_profile_page() ) {
			global $post;
			$author_id = (int) $post->post_author;
			if ( $author_id !== get_current_user_id() ) {
				return;
			}

			fup_load_template( 'components/meta-button-delete', [ 'url' => get_delete_post_link( get_the_ID() ) ] );
		}
	}

	/**
	 * @param $display_items
	 *
	 * @return array
	 */
	public function onboarding_items( $display_items ) {
		$items = [
			'follow-category-box' => esc_html__( 'Follow Category Box', 'front-user-profile' )
		];

		return array_merge( $display_items, $items );
	}

	/**
	 * @param $item
	 */
	public function onboarding_render( $item ) {
		if ( $item['link_to'] === 'follow-category-box' ) {
			echo do_shortcode( FollowCategoryShortcode::instance()->get() );
		}
	}

	/**
	 * @param $user
	 * @param $categories
	 */
	public function store_user_preferences( $user, $categories ) {
		$categories = json_decode( $categories );

		foreach ( $categories as $category ) {
			FollowCategory::instance()->simpleAddFollow( $user->ID, $category );
		}
	}

	/**
	 * Remove the general panel
	 *
	 * @param $state
	 *
	 * @return bool
	 */
	public function remove_general_panel( $state ) {
		global $post;
		$profile_page = fup_get_profile_page();

		if ( $profile_page && $post && $profile_page->ID === $post->ID && $profile_page->post_type === $post->post_type ) {
			return false;
		}

		return $state;
	}

	/**
	 * Show footer widget section
	 *
	 * @param $state
	 *
	 * @return mixed
	 */
	public function show_footer_widgets( $state ) {
		if ( fup_is_profile_page() ) {
			return fup_show( 'show-footer-widgets' );
		}

		return $state;
	}

	/**
	 * Add bookmark item to Typer meta items
	 *
	 * @param $items
	 *
	 * @return array
	 */
	public function add_bookmark_options( $items ) {
		$items[] = 'bookmark';

		return $items;
	}

	/**
	 * @param $link
	 * @param $author_id
	 * @param $author_nicename
	 *
	 * @return string
	 */
	public function redirect_profile( $link, $author_id, $author_nicename ) {
		$page = fup_get_profile_page();

		if ( $page ) {
			return home_url( $page->post_name . '/' . $author_nicename );
		}

		return $link;
	}

	/**
	 * @param $classes
	 *
	 * @return array
	 */
	public function profile_page_class( $classes ) {
		$page = fup_get_profile_page();

		if ( isset( $page ) && $page->ID === get_the_ID() ) {
			$classes[] = 'svq-page-profile';
		}

		return $classes;
	}

	/**
	 * Preview weekly digest
	 */
	public function digestPreview() {
		if ( isset( $_GET['preview-digest'] ) && current_user_can( 'manage_options' ) ) {
			Notifier::instance()->send_digest_notification( true );
		}
	}

}

/**
 * Instantiate plugin
 */
FupPlugin::instance();
