<?php
/**
 * The `typer()` function.
 *
 * @package typer
 */

namespace Seventhqueen\Typer;

/**
 * Provides access to all available template tags of the theme.
 *
 * When called for the first time, the function will initialize the theme.
 *
 * @return Template_Tags Template tags instance exposing template tag methods.
 */
function typer() {
	static $theme = null;

	if ( null === $theme ) {
		$theme = new Theme();
		$theme->initialize();
	}

	return $theme->template_tags();
}

function hex_to_hsl( $color, $return = 'hsl' ) {
	if ( ! $color ) {
		return '';
	}

	$color = rbg_to_hex( $color );

	// Strip # sign is present
	$color = str_replace( "#", "", $color );

	// Make sure it's 6 digits
	if ( strlen( $color ) === 3 ) {
		$color = str_split( $color );
		$color = $color[0] . $color[0] . $color[1] . $color[1] . $color[2] . $color[2];
	}

	// Convert HEX to DEC
	$R = hexdec( $color[0] . $color[1] );
	$G = hexdec( $color[2] . $color[3] );
	$B = hexdec( $color[4] . $color[5] );

	$HSL = [];

	$var_R = ( $R / 255 );
	$var_G = ( $G / 255 );
	$var_B = ( $B / 255 );

	$var_Min = min( $var_R, $var_G, $var_B );
	$var_Max = max( $var_R, $var_G, $var_B );
	$del_Max = $var_Max - $var_Min;

	$L = ( $var_Max + $var_Min ) / 2;

	if ( $del_Max == 0 ) {
		$H = 0;
		$S = 0;
	} else {
		if ( $L < 0.5 ) {
			$S = $del_Max / ( $var_Max + $var_Min );
		} else {
			$S = $del_Max / ( 2 - $var_Max - $var_Min );
		}

		$del_R = ( ( ( $var_Max - $var_R ) / 6 ) + ( $del_Max / 2 ) ) / $del_Max;
		$del_G = ( ( ( $var_Max - $var_G ) / 6 ) + ( $del_Max / 2 ) ) / $del_Max;
		$del_B = ( ( ( $var_Max - $var_B ) / 6 ) + ( $del_Max / 2 ) ) / $del_Max;

		if ( $var_R == $var_Max ) {
			$H = $del_B - $del_G;
		} else if ( $var_G == $var_Max ) {
			$H = ( 1 / 3 ) + $del_R - $del_B;
		} else if ( $var_B == $var_Max ) {
			$H = ( 2 / 3 ) + $del_G - $del_R;
		}

		if ( $H < 0 ) {
			$H ++;
		}
		if ( $H > 1 ) {
			$H --;
		}
	}

	$HSL['H'] = strval( ceil( $H * 360 ) );
	$HSL['S'] = round( $S * 100 ) . '%';
	$HSL['L'] = round( $L * 100 ) . '%';

	if ( $return === 'h' ) {
		return $HSL['H'];
	} elseif ( $return === 's' ) {
		return $HSL['S'];
	} elseif ( $return === 'l' ) {
		return $HSL['L'];
	}

	return $HSL;
}

function color_contrast( $hexcolor, $hsl = false ) {
	$hexcolor = rbg_to_hex( $hexcolor );

	// Strip # sign is present
	$hexcolor = str_replace( "#", "", $hexcolor );

	// Make sure it's 6 digits
	if ( strlen( $hexcolor ) === 3 ) {
		$hexcolor = str_split( $hexcolor );
		$hexcolor = $hexcolor[0] . $hexcolor[0] . $hexcolor[1] . $hexcolor[1] . $hexcolor[2] . $hexcolor[2];
	}

	$r   = hexdec( substr( $hexcolor, 0, 2 ) );
	$g   = hexdec( substr( $hexcolor, 2, 2 ) );
	$b   = hexdec( substr( $hexcolor, 4, 2 ) );
	$yiq = ( ( $r * 299 ) + ( $g * 587 ) + ( $b * 114 ) ) / 1000;

	if ( $hsl ) {
		return ( $yiq >= 154 ) ? '0, 0%, 0%' : '0, 0%, 100%';
	} else {
		return ( $yiq >= 154 ) ? '#000000' : '#ffffff';
	}
}

function rbg_to_hex( $color ) {
	if ( strpos( $color, '#' ) !== false ) {
		return $color;
	}

	$pattern = "/(\d{1,3})\,?\s?(\d{1,3})\,?\s?(\d{1,3})/";
	// Only if it's RGB
	if ( preg_match( $pattern, $color, $matches ) ) {
		$r     = $matches[1];
		$g     = $matches[2];
		$b     = $matches[3];
		$color = sprintf( "#%02x%02x%02x", $r, $g, $b );
	}

	return $color;
}
