<?php
/**
 * Seventhqueen\Typer\Base_Support\Component class
 *
 * @package typer
 */

namespace Seventhqueen\Typer\Panel;

use Buddy_Builder\Core\Library\Documents\Profile;
use Seventhqueen\Typer\Component_Interface;
use function Seventhqueen\Typer\typer;


/**
 * Class for adding basic theme support, most of which is mandatory to be implemented by all themes.
 *
 */
class Component implements Component_Interface {

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug() {
		return 'panel';
	}

	private $current_slug = '';

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize() {
		$this->set_constants();
		$this->set_hooks();
		Addons_Manager::instance();

		if ( class_exists( \Seventhqueen\Typer\Core\Panel\Importer::class ) ) {
			\Seventhqueen\Typer\Core\Panel\Importer::instance();
		}

		$theme_slug         = typer()->get_config( 'slug' ) . '-';
		$current_slug       = isset( $_GET['page'] ) && strpos( $_GET['page'], $theme_slug ) === 0 ? str_replace( $theme_slug, '', esc_attr( $_GET['page'] ) ) : 'welcome';
		$this->current_slug = str_replace( '-', '_', $current_slug );
	}

	/**
	 * Defining constants
	 */
	private function set_constants() {
		if ( ! defined( 'SVQ_LIB_DIR' ) ) {
			define( 'SVQ_LIB_DIR', get_template_directory() . '/inc/Config' );
		}

		if ( ! defined( 'SVQ_PANEL_DIR' ) ) {
			define( 'SVQ_PANEL_DIR', __dir__ );
		}
		if ( ! defined( 'SVQ_PANEL_URI' ) ) {
			define( 'SVQ_PANEL_URI', get_template_directory_uri() . '/inc/Panel' );
		}
	}

	public function set_hooks() {
		add_action( 'admin_menu', [ $this, 'register_panel_page' ] );
		add_action( 'admin_init', [ $this, 'redirect_to_panel' ], 0 );

		add_action( 'typer_menu_panel_action', [ $this, 'tpl_page' ] );
		add_action( 'typer_header_right_section', [ $this, 'tpl_top_header_right_area' ] );

		add_action( 'typer_welcome_page_content', [ $this, 'tpl_main_quick_settings_content' ] );
		add_action( 'typer_welcome_page_content', [ $this, 'tpl_main_license_activation_content' ], 12 );

		add_action( 'typer_welcome_page_right_sidebar_content', [ $this, 'tpl_side_importer_section' ] );
		add_action( 'typer_welcome_page_right_sidebar_content', [ $this, 'tpl_side_docs_section' ], 12 );
		add_action( 'typer_welcome_page_right_sidebar_content', [ $this, 'tpl_side_community_section' ], 14 );

		// Add-ons
		add_filter( 'typer_addons_columns', '__return_zero' );
		add_action( 'typer_addons_page_content', [ $this, 'tpl_main_addons_page_content' ] );

		add_filter( 'http_request_args', [ $this, 'stop_update_theme' ], 5, 2 );

		add_action( 'wp_ajax_sq_theme_registration', [ $this, 'theme_registration' ] );
		add_action( 'after_switch_theme', [ $this, 'check_code_at_activation' ] );

		if ( ( isset( $_REQUEST['action'] ) && $_REQUEST['action'] === 'sq_do_plugin_action' ) || ( isset( $_GET['page'] ) && strpos( $_GET['page'], typer()->get_config( 'slug' ) ) === 0 ) ) {
			add_action( 'admin_init', [ $this, 'config_addons' ], 12 );
			add_action( 'admin_enqueue_scripts', [ $this, 'panel_scripts' ] );
		}
	}

	/**
	 * Register CSS & JS Files
	 */
	public function panel_scripts() {
		//CSS
		wp_register_style( typer()->get_config( 'slug' ) . '-panel', get_theme_file_uri( '/assets/admin/css/theme-panel.css' ), [], typer()->get_version(), 'all' );
		wp_enqueue_style( typer()->get_config( 'slug' ) . '-panel' );

		wp_register_style( typer()->get_config( 'slug' ) . '-tailwind', get_theme_file_uri( '/assets/admin/css/panel.css' ), [], typer()->get_version(), 'all' );
		wp_enqueue_style( typer()->get_config( 'slug' ) . '-tailwind' );

		//JS
		wp_enqueue_script( 'jquery-ui-tooltip' );
		wp_register_script( typer()->get_config( 'slug' ) . '-panel', get_theme_file_uri( '/assets/admin/js/theme-panel.js' ), [ 'jquery' ], typer()->get_version(), true );
		wp_enqueue_script( typer()->get_config( 'slug' ) . '-panel' );
	}

	/**
	 * Register panel page
	 */
	public function register_panel_page() {
		add_theme_page(
			sprintf( esc_html__( '%s Panel', 'typer' ), 'Typer' ),
			sprintf( esc_html__( '%s Panel', 'typer' ), 'Typer' ),
			'manage_options',
			typer()->get_config( 'slug' ),
			[ $this, 'panel_menu_callback' ]
		);

		add_theme_page(
			sprintf( esc_html__( '%s Add-ons', 'typer' ), 'Typer' ),
			sprintf( esc_html__( '%s Add-ons', 'typer' ), 'Typer' ),
			'manage_options',
			typer()->get_config( 'slug' ) . '-addons',
			[ $this, 'panel_menu_callback' ]
		);
	}

	public function panel_menu_callback() {
		$site_url      = apply_filters( 'typer_admin_site_url', 'https://' . typer()->get_config( 'slug' ) . '.seventhqueen.com' );
		$wrapper_class = apply_filters( 'typer_welcome_wrapper_class', [ $this->current_slug ] );

		?>
		<div
			class="sqp_typer_options wrap sqp-m-0 <?php echo esc_attr( implode( ' ', $wrapper_class ) ); ?>">
			<div class="sqp-bg-white sqp-py-5 sqp-mb-5 sqp-shadow">
				<div
					class="sqp-container sqp-mx-auto sqp-px-5 sqp-flex sqp-flex-wrap sqp-justify-between sqp-items-center">
					<div class="sqp-text-left">
						<a href="<?php echo esc_url( $site_url ); ?>" target="_blank" rel="noopener"
						   class="sqp-text-base sqp-flex sqp-items-center sqp-content-center sqp-no-underline">
							<img
								src="<?php echo esc_url( get_parent_theme_file_uri( 'assets/img/logo-black.png' ) ); ?>"
								class="sqp-border-0 sqp-w-32" alt="Typer">
							<span
								class="sqp-bg-gray-300 sqp-text-gray-600 sqp-ml-4 sqp-px-2 sqp-py-1 sqp-text-xs sqp-rounded sqp-no-underline"><?php echo typer()->get_version(); ?></span>

							<?php do_action( 'typer_welcome_page_header_title' ); ?>
						</a>
					</div>

					<?php do_action( 'typer_header_right_section' ); ?>

				</div>
			</div>

			<?php do_action( 'typer_menu_panel_action' ); ?>
		</div>
		<?php
	}

	/**
	 * Require panel page
	 */
	public function tpl_page() {
		$current_slug = $this->current_slug;
		require( SVQ_PANEL_DIR . '/templates/panel.php' );
	}

	public function tpl_top_header_right_area() {
		$top_links = apply_filters(
			'typer_header_top_links',
			[
				'typer-theme-info' => [
					'title' => __( 'Super Fast and Customizable WordPress theme', 'typer' ),
				],
			]
		);

		if ( empty( $top_links ) || ! is_array( $top_links ) ) {
			return;
		}

		?>
		<div class="sqp-text-right">
			<ul>
				<?php
				foreach ( $top_links as $key => $info ) {
					/* translators: %1$s: Top Link URL wrapper, %2$s: Top Link URL, %3$s: Top Link URL target attribute */
					printf(
						'<li class="sqp-m-0"><%1$s %2$s %3$s > %4$s </%1$s>',
						isset( $info['url'] ) ? 'a' : 'span',
						isset( $info['url'] ) ? 'href="' . esc_url( $info['url'] ) . '"' : '',
						isset( $info['url'] ) ? 'target="_blank" rel="noopener"' : '',
						esc_html( $info['title'] )
					);
				}
				?>
			</ul>
		</div>
		<?php
	}

	/**
	 * Main welcome content
	 *
	 * @since 1.0.0
	 */
	public function tpl_main_quick_settings_content() {

		// Quick settings.
		$quick_settings = apply_filters(
			'typer_customizer_quick_links',
			[
				'logo-favicon' => [
					'title'     => esc_html__( 'Upload Logo', 'typer' ),
					'dashicon'  => 'dashicons-format-image',
					'quick_url' => admin_url( 'customize.php?autofocus[control]=custom_logo' ),
				],
				'colors'       => [
					'title'     => esc_html__( 'Set Colors', 'typer' ),
					'dashicon'  => 'dashicons-admin-customizer',
					'quick_url' => admin_url( 'customize.php?autofocus[section]=svq_section_colors' ),
				],
				'typography'   => [
					'title'     => esc_html__( 'Typography', 'typer' ),
					'dashicon'  => 'dashicons-editor-textcolor',
					'quick_url' => admin_url( 'customize.php?autofocus[section]=svq_section_typography' ),
				],
				'layout'       => [
					'title'     => esc_html__( 'Layout Options', 'typer' ),
					'dashicon'  => 'dashicons-layout',
					'quick_url' => admin_url( 'customize.php?autofocus[section]=svq_section_layout' ),
				],
				'header'       => [
					'title'     => esc_html__( 'Header Options', 'typer' ),
					'dashicon'  => 'dashicons-align-center',
					'quick_url' => admin_url( 'customize.php?autofocus[section]=svq_section_header' ),
				],
				'footer'       => [
					'title'     => esc_html__( 'Footer Settings', 'typer' ),
					'dashicon'  => 'dashicons-admin-generic',
					'quick_url' => admin_url( 'customize.php?autofocus[section]=svq_section_footer' ),
				],
			]
		);

		$extensions = [];

		?>

		<?php if ( ! empty( $quick_settings ) && is_array( $quick_settings ) ) : ?>
			<div class="sqp_box">
				<h2 class="sqp_box_title">
					<?php esc_html_e( 'Customizer Quick Links', 'typer' ); ?>
				</h2>
				<ul class="sqp_box_list">
					<?php
					foreach ( (array) $quick_settings as $key => $link ) {
						echo '<li class="sqp_box_list_item">' .
						     '<span class="dashicons ' . esc_attr( $link['dashicon'] ) . ' sqp-ml-4"></span>' .
						     '<a class="sqp-ml-2 sqp-mr-4 sqp-no-underline focus:sqp-outline-none" href="' . esc_url( $link['quick_url'] ) . '" target="_blank" rel="noopener">' .
						     esc_html( $link['title'] ) . '</a>' .
						     '</li>';
					}
					?>
				</ul>
			</div>
		<?php endif; ?>

		<?php if ( ! empty( $extensions ) && is_array( $extensions ) ) : ?>
			<div class="sqp_box">
				<h2 class="sqp_box_title">
					<?php esc_html_e( 'More Settings from Plugins', 'typer' ); ?>
				</h2>
				<ul class="sqp_box_list">
					<?php
					foreach ( $extensions as $addon => $info ) {
						$title_url     = ( isset( $info['title_url'] ) && ! empty( $info['title_url'] ) ) ? 'href="' . esc_url( $info['title_url'] ) . '"' : '';
						$anchor_target = ( isset( $info['title_url'] ) && ! empty( $info['title_url'] ) ) ? "target='_blank' rel='noopener'" : '';

						echo '<li id="' . esc_attr( $addon ) . '"  class="' . esc_attr( $info['class'] ) . ' sqp_box_list_item">' .
						     '<a class="sqp-mx-4 sqp-no-underline focus:sqp-outline-none"' . $title_url . $anchor_target . ' >' .
						     esc_html( $info['title'] ) . '</a>' .
						     '<div class="typer-addon-link-wrapper">';

						foreach ( $info['links'] as $key => $link ) {
							printf(
								'<a class="%1$s" %2$s %3$s> %4$s </a>',
								esc_attr( $link['link_class'] ),
								isset( $link['link_url'] ) ? 'href="' . esc_url( $link['link_url'] ) . '"' : '',
								( isset( $link['target_blank'] ) && $link['target_blank'] ) ? 'target="_blank" rel="noopener"' : '',
								esc_html( $link['link_text'] )
							);
						}
						echo '</div></li>';
					}
					?>
				</ul>
			</div>

		<?php endif; ?>

		<?php
	}

	public function tpl_main_license_activation_content() {
		?>

		<div class="sqp_box">
			<h2 class="sqp_box_title">
				<?php esc_html_e( 'License Activation', 'typer' ); ?>
			</h2>
			<div class="sqp-p-4">
				<?php if ( ! typer()->is_active() ) : ?>
					<p class="sqp_box_p">
						<?php esc_html_e( 'Register your license to automatically install and update theme add-ons.', 'typer' ); ?>
					</p>

					<ul class="sqp-list-decimal sqp-list-inside">
						<li>
							<?php esc_html_e( 'Enter your Purchase code from Themeforest.', 'typer' ); ?>
							<a target="_blank" rel="nofollow"
							   href="https://my.seventhqueen.com/docs/typer/find-theme-purchase-code">
								<?php esc_html_e( 'How to get it?', 'typer' ); ?>
							</a>
						</li>
						<li><?php esc_html_e( 'Hit REGISTER button.', 'typer' ); ?></li>
					</ul>
				<?php else : ?>
					<h3 class="sqp-m-0">
						<span class="svq-status-col-icon dashicons-before dashicons-yes"></span>
						<?php esc_html_e( 'Congratulations, theme license is active on your site!', 'typer' ); ?>
					</h3>
				<?php endif; ?>

				<?php $tf_code = typer()->get_purchased_code(); ?>

				<form action="<?php echo esc_url( admin_url( 'themes.php?page=typer' ) ); ?>"
				      class="svq-panel-register-form sqp-mt-4">
					<div class="sqp-mb-4">
						<label for="tf_purchase_code"
						       class="sqp_form_label"><?php esc_html_e( 'Purchase Code', 'typer' ); ?></label>
						<input type="password" id="tf_purchase_code" name="tf_purchase_code"
						       class="sqp_form_input_big"
						       value="<?php echo esc_attr( $tf_code ); ?>"
						       placeholder="<?php esc_attr_e( 'Purchase Code', 'typer' ); ?>">
					</div>

					<div class="response-area sqp-hidden"></div>

					<?php wp_nonce_field( 'sq_theme_registration', 'sq_nonce' ); ?>

					<input type="submit"
					       class="sqp_btn_primary hover:sqp-bg-green-600 focus:sqp-outline-none"
					       value="<?php esc_attr_e( 'REGISTER', 'typer' ); ?>">
				</form>

			</div>
		</div>

		<?php
	}

	/**
	 * Include import sites sidebar section
	 */
	public function tpl_side_importer_section() {
		$page_slug   = typer()->get_config( 'slug' ) . '-import';
		$current_url = admin_url( 'themes.php?page=' . typer()->get_config( 'slug' ) );

		if ( defined( 'SVQ_CORE_VERSION' ) ) {
			$link = admin_url( 'themes.php?page=' . $page_slug );
			$text = __( '<a href="%s">Import demo sites</a> with one click and start building right away!', 'typer' );
		} else {
			$link = wp_nonce_url( $current_url . '&svq_action=activate_plugin&plugin=typer-core&redirect=' . $current_url, 'activate_plugin', 'svq_nonce' );
			$text = __( '<a href="%s">Activate Typer Core</a> to import demo sites and start building right away!', 'typer' );
		}
		?>

		<div class="sqp_box lg:sqp-ml-4">
			<h2 class="sqp_box_title">
				<span class="dashicons dashicons-admin-customizer sqp-mr-2"></span>
				<span><?php echo esc_html( apply_filters( 'typer_sites_menu_page_title', __( 'Import Demo Content', 'typer' ) ) ); ?></span>
			</h2>
			<a href="<?php echo esc_url( $link ); ?>"
			   class="sqp-block sqp-w-full">
				<img class="sqp-w-full" alt="<?php esc_html_e( 'Typer sites', 'typer' ) ?>"
				     src="<?php echo esc_url( get_parent_theme_file_uri( 'screenshot.png' ) ); ?>">
			</a>
			<div class="sqp-p-4">
				<p class="sqp-m-0 sqp-p-0">
					<?php echo wp_kses_post( sprintf( $text, $link ) ); ?>
				</p>
			</div>
		</div>

		<?php
	}

	/**
	 * Include docs content
	 *
	 * @since 1.0.0
	 */
	public function tpl_side_docs_section() {
		?>

		<div class="sqp_box lg:sqp-ml-4">
			<h2 class="sqp_box_title">
				<span class="dashicons dashicons-book sqp-mr-2"></span>
				<?php esc_html_e( 'Knowledge Base', 'typer' ); ?>
			</h2>
			<div class="sqp-p-4">
				<p class="sqp-m-0 sqp-p-0 sqp-mb-4">
					<?php esc_html_e( 'Have questions? Check out our knowledge base to learn more.', 'typer' ); ?>
				</p>
				<?php
				$link      = apply_filters( 'typer_knowledge_base_documentation_link', 'https://my.seventhqueen.com/docs/typer' );
				$link_text = apply_filters( 'typer_knowledge_base_documentation_link_text', __( 'Visit Knowledge Base &raquo;', 'typer' ) );

				printf(
				/* translators: %1$s: Typer Knowledge doc link. */
					'%1$s',
					! empty( $link ) ? '<a href=' . esc_url( $link ) . ' target="_blank" rel="noopener">' . esc_html( $link_text ) . '</a>' : esc_html( $link_text )
				);
				?>
			</div>
		</div>

		<?php
	}

	/**
	 * Include community content
	 *
	 * @since 1.0.0
	 */
	public function tpl_side_community_section() {
		?>

		<div class="sqp_box lg:sqp-ml-4">
			<h2 class="sqp_box_title">
				<span class="dashicons dashicons-groups sqp-mr-2"></span>
				<span>
						<?php
						printf(
						/* translators: %1$s: Typer Theme name. */
							esc_html__( '%1$s Community', 'typer' ),
							'Typer'
						);
						?>
				</span>
			</h2>
			<div class="sqp-p-4">
				<p class="sqp-m-0 sqp-p-0 sqp-mb-4">
					<?php
					printf(
					/* translators: %1$s: Typer Theme name. */
						esc_html__( 'Join the %1$s users. Share resources, ask questions, give feedback and help each other!', 'typer' ),
						'Typer'
					);
					?>
				</p>
				<?php
				$link      = apply_filters( 'typer_community_group_link', 'https://www.facebook.com/groups/' . typer()->get_config( 'slug' ) . 'wp' );
				$link_text = apply_filters( 'typer_community_group_link_text', __( 'Join Our Facebook Group &raquo;', 'typer' ) );

				printf(
				/* translators: %1$s: Typer Knowledge doc link. */
					'%1$s',
					! empty( $link ) ? '<a href=' . esc_url( $link ) . ' target="_blank" rel="noopener">' . esc_html( $link_text ) . '</a>' :
						esc_html( $link_text )
				);
				?>
			</div>
		</div>

		<?php
	}

	public function tpl_main_addons_page_content() {
		?>
		<div class="sqp-w-full">
			<h3 class="sqp-text-2xl sqp-font-medium"><?php esc_html_e( 'Install Add-ons', 'typer' ); ?></h3>
			<p>
				<?php esc_html_e( 'Enhance your site with our recommended extensions. They are not mandatory and you can use any plugins you like.', 'typer' ); ?>
			</p>

			<div class="sqp-flex sqp-flex-wrap sqp--mx-3 sqp-overflow-hidden">
				<?php
				foreach ( Addons_Manager::instance()->plugins as $plugin ) : ?>

					<?php
					$plugin_status = Addons_Manager::instance()->get_plugin_status( $plugin['slug'] );

					?>
					<div
						class="sqp-py-3 sqp-w-full md:sqp-w-1/2 lg:sqp-w-1/3 xl:sqp-w-1/3 sqp-overflow-hidden svq-extension <?php echo esc_attr( $plugin_status['status'] ); ?>"
						id="ext-<?php echo esc_attr( $plugin['slug'] ); ?>">
						<div
							class="sqp-bg-white sqp-border sqp-border-solid sqp-border-gray-300 sqp-text-center sqp-mx-3 sqp-h-full sqp-relative">
							<div class="sqp-px-6 sqp-py-10 sqp-pb-24">
								<h4 class="sqp-mt-0 sqp-mb-4 sqp-text-xl sqp-font-bold sqp-leading-tight"><?php echo esc_html( $plugin['name'] ); ?></h4>
								<div class="sqp-my-5">
									<span
										class="sqp_plugin_status svq-extension-status">
										<?php echo esc_html( $plugin_status['status_text'] ); ?>
									</span>
								</div>
								<p class="sqp-text-gray-700"><?php echo wp_kses_data( isset( $plugin['description'] ) ? $plugin['description'] : '' ); ?></p>
								<p class="sqp-italic sqp-text-gray-500">
									<cite><?php echo esc_html( ( isset( $plugin['required'] ) && $plugin['required'] === true ) ? __( 'REQUIRED', 'typer' ) : __( 'Recommended', 'typer' ) ); ?></cite>
								</p>
								<p class="svq-extension-ajax-text"></p>
							</div>
							<div class="sqp-w-full sqp-absolute sqp-bottom-0 sqp-left-0 svq-extension-actions">
								<?php
								echo '<a class="sqp_plugin_btn svq-extension-button"' .
								     ' data-action="' . esc_attr( $plugin_status['action'] ) . '"' .
								     ' data-status="' . esc_attr( $plugin_status['status'] ) . '"' .
								     ' data-nonce="' . esc_attr( wp_create_nonce( 'sq_plugins_nonce' ) ) . '"' .
								     ' href="#"' .
								     ' data-slug="' . esc_attr( $plugin['slug'] ) . '">' .
								     esc_html( $plugin_status['action_text'] ) .
								     '</a>';
								?>
							</div>
						</div>

					</div>
				<?php endforeach; ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Redirect to panel
	 */
	public function redirect_to_panel() {
		// Theme activation redirect
		global $pagenow;
		if ( isset( $_GET['activated'] ) && $pagenow === 'themes.php' ) {

			wp_redirect( admin_url( 'themes.php?page=' . typer()->get_config( 'slug' ) ) );
			exit;
		}
	}

	/**
	 * @param $r
	 * @param $url
	 *
	 * @return mixed
	 */
	public function stop_update_theme(
		$r, $url
	) {
		if ( 0 !== strpos( $url, 'https://api.wordpress.org/themes/update-check/1.1/' ) ) {
			return $r; // Not a theme update request. Bail immediately.
		}

		$themes = json_decode( $r['body']['themes'] );
		$theme  = get_template();
		unset( $themes->themes->$theme );
		$r['body']['themes'] = json_encode( $themes );

		return $r;
	}

	/**
	 * Register theme
	 */
	public function theme_registration() {
		if ( ! isset( $_POST['sq_nonce'] ) || ! wp_verify_nonce( $_POST['sq_nonce'], 'sq_theme_registration' ) ) {
			wp_send_json_error( [ 'error' => 'Sorry, your nonce did not verify.' ] );
		}

		$option_name = "envato_purchase_code_" . typer()->get_config( 'item_id' );
		$tf_code     = isset( $_POST['code'] ) ? $_POST['code'] : '';

		if ( ! empty( $tf_code ) ) {
			$has_purchased = typer()->is_active( $tf_code, true, true );

			if ( is_wp_error( $has_purchased ) ) {
				return wp_send_json_error( [ 'error' => $has_purchased->get_error_message() ] );
			}

			if ( $has_purchased ) {
				//Update code
				update_option( $option_name, $tf_code );

				return wp_send_json_success( [ 'message' => __( 'License successfully activated!', 'typer' ) ] );

			}

			return wp_send_json_error( [ 'error' => 'Purchase code is not valid. Please check the license provided!' ] );
		}

		return wp_send_json_error( [ 'error' => 'Please enter your purchase code.' ] );
	}

	/**
	 * Check theme activation code
	 */
	public function check_code_at_activation() {
		typer()->verify_purchase();
	}

	/**
	 * Addons config
	 */
	public function config_addons() {
		/* Move elements first */
		if ( empty( typer()->get_config( 'priority_addons' ) ) ) {
			return;
		}

		$priority_list = array_reverse( typer()->get_config( 'priority_addons' ) );
		foreach ( $priority_list as $item ) {
			if ( isset( Addons_Manager::instance()->plugins[ $item ] ) ) {
				Addons_Manager::instance()->plugins = [ $item => Addons_Manager::instance()->plugins[ $item ] ] + Addons_Manager::instance()->plugins;
			}
		}

		$prepend = [
			'typer-child' => [
				'addon_type'  => 'child_theme',
				'name'        => 'Typer Child Theme',
				'slug'        => 'typer-child',
				'source'      => 'https://updates.seventhqueen.com/check/?action=download&slug=typer-child.zip',
				'source_type' => 'bundled',
				'version'     => '1.0',
				'required'    => true,
				'description' => sprintf( __( 'Always activate the child theme to safely update Typer and for better customization. <a href="%s" target="_blank">More on Child Themes</a>.', 'typer' ), 'https://codex.wordpress.org/Child_Themes' ),
				'pro'         => false
			]
		];

		Addons_Manager::instance()->plugins = $prepend + Addons_Manager::instance()->plugins;
	}

}
