<?php

namespace Seventhqueen\Typer;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

use Seventhqueen\Typer\Elementor\AjaxPosts;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Background;

class Posts extends \Elementor\Widget_Base {

	public function get_script_depends() {
		return [ 'masonry' ];
	}

	public function get_name() {
		return 'typer-posts';
	}

	public function get_title() {
		return esc_html__( 'Posts (Typer)', 'typer' );
	}

	public function get_icon() {
		return 'eicon-posts-group';
	}

	public function get_categories() {
		return [ 'typer-elements' ];
	}

	public function get_keywords() {
		return [ 'posts', 'typer' ];
	}

	protected function _register_controls() {
		$this->start_controls_section(
			'section_groups_settings',
			[
				'label' => esc_html__( 'Settings', 'typer' ),
			]
		);

		$this->add_control(
			'title',
			[
				'label'   => esc_html__( 'Section title', 'typer' ),
				'type'    => Controls_Manager::TEXT,
				'default' => '',
			]
		);

		$listing = [
			'masonry' => esc_html__( 'Masonry', 'typer' )
		];

		if ( class_exists( \TyperPro\TyperPro::class ) ) {
			$listing['list']     = esc_html__( 'Default List', 'typer' );
			$listing['list-big'] = esc_html__( 'Big List', 'typer' );
			$listing['grid']     = esc_html__( 'Grid', 'typer' );
		}

		$this->add_control(
			'listing',
			[
				'label'   => esc_html__( 'Listing type', 'typer' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'list',
				'options' => $listing
			]
		);

		$col_options = [
			'12' => esc_html__( '1', 'typer' ),
			'6'  => esc_html__( '2', 'typer' ),
			'4'  => esc_html__( '3', 'typer' ),
			'3'  => esc_html__( '4', 'typer' ),
		];

		$this->add_control(
			'col-desktop',
			[
				'label'     => esc_html__( 'Desktop posts per row', 'typer' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '4',
				'options'   => $col_options,
				'devices'   => [ 'desktop', 'tablet', 'mobile' ],
				'condition' => [
					'listing' => 'masonry'
				],
			]
		);

		$this->add_control(
			'col-tablet',
			[
				'label'     => esc_html__( 'Tablet posts per row', 'typer' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '6',
				'options'   => $col_options,
				'devices'   => [ 'desktop', 'tablet', 'mobile' ],
				'condition' => [
					'listing' => 'masonry'
				],
			]
		);

		$this->add_control(
			'col-mobile',
			[
				'label'     => esc_html__( 'Mobile posts per row', 'typer' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '12',
				'options'   => $col_options,
				'devices'   => [ 'desktop', 'tablet', 'mobile' ],
				'condition' => [
					'listing' => 'masonry'
				],
			]
		);

		$col_options_grid = [
			'12' => esc_html__( '1', 'typer' ),
			'3'  => esc_html__( '2', 'typer' ),
		];

		$this->add_control(
			'col-grid-desktop',
			[
				'label'     => esc_html__( 'Desktop posts per row', 'typer' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '3',
				'options'   => $col_options_grid,
				'devices'   => [ 'desktop', 'tablet', 'mobile' ],
				'condition' => [
					'listing' => 'grid'
				],
			]
		);

		$this->add_control(
			'col-grid-tablet',
			[
				'label'     => esc_html__( 'Tablet posts per row', 'typer' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '12',
				'options'   => $col_options_grid,
				'devices'   => [ 'desktop', 'tablet', 'mobile' ],
				'condition' => [
					'listing' => 'grid'
				],
			]
		);

		$this->add_control(
			'col-grid-mobile',
			[
				'label'     => esc_html__( 'Mobile posts per row', 'typer' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => '12',
				'options'   => $col_options_grid,
				'devices'   => [ 'desktop', 'tablet', 'mobile' ],
				'condition' => [
					'listing' => 'grid'
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_groups_post',
			[
				'label' => esc_html__( 'Post', 'typer' ),
			]
		);

		$this->add_control(
			'image_position',
			[
				'label'     => esc_html__( 'List image position', 'typer' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'right',
				'options'   => [
					'left'  => esc_html__( 'Left', 'typer' ),
					'right' => esc_html__( 'Right', 'typer' ),
				],
				'condition' => [
					'listing' => 'list',
				],
			]
		);

		$this->add_control(
			'image_wide',
			[
				'label'        => esc_html__( 'Wide images', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0',
				'condition'    => [
					'listing' => [
						'masonry',
						'list-big'
					]
				],
			]
		);

		$this->add_control(
			'big_post',
			[
				'label'        => esc_html__( 'Make 1st post big', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0',
				'condition'    => [
					'listing' => 'masonry'
				],
			]
		);

		$this->add_control(
			'show_category',
			[
				'label'        => esc_html__( 'Show category', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '1'
			]
		);

		$this->add_control(
			'show_meta',
			[
				'label'        => esc_html__( 'Show meta', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '1'
			]
		);

		$this->add_control(
			'show_excerpt',
			[
				'label'        => esc_html__( 'Show excerpt', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '1'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_groups_query',
			[
				'label' => esc_html__( 'Query', 'typer' ),
			]
		);

		$this->add_control(
			'include_sticky_posts',
			[
				'label'        => esc_html__( 'Include sticky posts', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0'
			]
		);

		$this->add_control(
			'exclude_current_post',
			[
				'label'        => esc_html__( 'Exclude current post', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0'
			]
		);

		$this->add_control(
			'current_category',
			[
				'label'        => esc_html__( 'From current category', 'typer' ),
				'description'  => esc_html__( 'Enabling this will get the posts from current category. Use this options if you plan to use this widget within a template across all your category pages.', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0'
			]
		);

		$categoryArray = [];
		$categories    = get_categories( 'orderby=name&hide_empty=0' );

		foreach ( $categories as $category ) {
			$categoryArray[ $category->term_id ] = $category->name;
		}

		$this->add_control(
			'query_categories',
			[
				'label'       => esc_html__( 'Categories', 'typer' ),
				'label_block' => true,
				'type'        => Controls_Manager::SELECT2,
				'options'     => $categoryArray,
				'multiple'    => true,
				'conditions'  => [
					'terms' => [
						[
							'name'     => 'include_sticky_posts',
							'operator' => '!=',
							'value'    => '1'
						],
						[
							'name'     => 'current_category',
							'operator' => '!=',
							'value'    => '1'
						]
					]
				],
			]
		);

		$tagsArray = [];
		$tags      = get_tags( array(
			'hide_empty' => false
		) );

		foreach ( $tags as $tag ) {
			$tagsArray[ $tag->term_id ] = $tag->name;
		}

		$this->add_control(
			'query_tags',
			[
				'label'       => esc_html__( 'Tags', 'typer' ),
				'label_block' => true,
				'type'        => Controls_Manager::SELECT2,
				'options'     => $tagsArray,
				'multiple'    => true,
				'conditions'  => [
					'terms' => [
						[
							'name'     => 'include_sticky_posts',
							'operator' => '!=',
							'value'    => '1'
						],
						[
							'name'     => 'current_category',
							'operator' => '!=',
							'value'    => '1'
						]
					]
				],
			]
		);

		$this->add_control(
			'pagination',
			[
				'label'        => esc_html__( 'Enable pagination', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0'
			]
		);

		$this->add_control(
			'pagination_ajax',
			[
				'label'        => esc_html__( 'Ajax Pagination', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '1',
				'condition'    => [
					'pagination' => '1'
				],
			]
		);

		$this->add_control(
			'pagination_next_prev',
			[
				'label'        => esc_html__( 'Hide Next/Prev buttons', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0',
				'condition'    => [
					'pagination' => '1'
				],
			]
		);

		$this->add_control(
			'pagination_page_numbers',
			[
				'label'        => esc_html__( 'Hide page numbers', 'typer' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', 'typer' ),
				'label_off'    => esc_html__( 'No', 'typer' ),
				'return_value' => '1',
				'default'      => '0',
				'condition'    => [
					'pagination' => '1'
				],
			]
		);

		$this->add_control(
			'pagination_page_numbers_style',
			[
				'label'     => __( 'Pagination Numbers Style', 'typer' ),
				'type'      => Controls_Manager::HIDDEN,
				'default'   => '1',
				'selectors' => [
					'{{WRAPPER}} .pagination .page-numbers'      => 'display: none;',
					'{{WRAPPER}} .pagination .next.page-numbers' => 'display: block;',
					'{{WRAPPER}} .pagination .prev.page-numbers' => 'display: block;',
				],
				'condition' => [
					'pagination'              => '1',
					'pagination_page_numbers' => '1'
				],
			]
		);

		$this->add_control(
			'per_page',
			[
				'label'     => esc_html__( 'Posts per page', 'typer' ),
				'type'      => Controls_Manager::NUMBER,
				'default'   => 12,
				'condition' => [
					'pagination' => '1'
				],
			]
		);

		$this->add_control(
			'query_total',
			[
				'label'      => esc_html__( 'Total posts', 'typer' ),
				'type'       => Controls_Manager::NUMBER,
				'default'    => 12,
				'conditions' => [
					'terms' => [
						[
							'name'     => 'pagination',
							'operator' => '!=',
							'value'    => '1'
						]
					]
				],
			]
		);

		$this->add_control(
			'query_offset',
			[
				'label'      => esc_html__( 'Offset', 'typer' ),
				'type'       => Controls_Manager::NUMBER,
				'default'    => 0,
				'conditions' => [
					'terms' => [
						[
							'name'     => 'pagination',
							'operator' => '!=',
							'value'    => '1'
						]
					]
				],
			]
		);

		$this->add_control(
			'query_sort',
			[
				'label'   => esc_html__( 'Sort', 'typer' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'newest',
				'options' => [
					'newest' => esc_html__( 'Newest', 'typer' ),
					'oldest' => esc_html__( 'Oldest', 'typer' ),
					'random' => esc_html__( 'Random', 'typer' ),
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section-style-dimensions',
			[
				'label'     => esc_html__( 'Spacing', 'typer' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'listing' => [
						'list-big',
						'list'
					],
				],
			]
		);

		$this->add_responsive_control(
			'font_size', //  spacing
			[
				'label'     => esc_html__( 'Vertical', 'typer' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'devices'   => [ 'desktop', 'tablet', 'mobile' ],
				'selectors' => [
					'{{WRAPPER}} .svq-article-col:first-child' => 'margin-top: 0 !important',
					'{{WRAPPER}} .svq-article-col:last-child'  => 'margin-bottom: 0 !important',
					'{{WRAPPER}} .svq-article-col'             => 'margin-top: {{SIZE}}{{UNIT}}; margin-bottom: {{SIZE}}{{UNIT}}'
				],
				'condition' => [
					'listing' => [
						'list-big',
						'list'
					],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section-style-spacing',
			[
				'label'     => esc_html__( 'Spacing', 'typer' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'listing' => [
						'grid'
					],
				],
			]
		);

		$this->add_responsive_control(
			'outer-padding',
			[
				'label'      => __( 'Padding', 'typer' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'devices'    => [ 'desktop', 'tablet', 'mobile' ],
				'selectors'  => [
					'{{WRAPPER}} .svq-media-grid .svq-article-col' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'  => [
					'listing' => [
						'grid'
					],
				],
			]
		);

		$this->add_responsive_control(
			'inner-padding',
			[
				'label'      => __( 'Content padding', 'typer' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'devices'    => [ 'desktop', 'tablet', 'mobile' ],
				'selectors'  => [
					'{{WRAPPER}} .svq-grid-articles .svq-grid-span-2 .svq-article-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'  => [
					'listing' => [
						'grid'
					],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section-style-color',
			[
				'label' => esc_html__( 'Default Colors', 'typer' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name'      => 'box_shadow',
				'label'     => __( 'Box Shadow', 'typer' ),
				'selector'  => '{{WRAPPER}} .svq-article',
				'condition' => [
					'listing' => [
						'grid',
						'masonry'
					],
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name'      => 'background',
				'label'     => __( 'Background', 'typer' ),
				'types'     => [ 'classic' ],
				'selector'  => '{{WRAPPER}} .svq-article',
				'condition' => [
					'listing' => [
						'grid',
						'masonry'
					],
				],
			]
		);

		$this->add_control(
			'category-color',
			[
				'label'     => esc_html__( 'Category', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_4,
				],
				'selectors' => [
					'{{WRAPPER}} .meta-category .meta-category__item a' => 'color: {{VALUE}}'
				],
			]
		);

		$this->add_control(
			'title-color',
			[
				'label'     => esc_html__( 'Title', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_1,
				],
				'selectors' => [
					'{{WRAPPER}} .heading-title .heading-title-link' => 'color: {{VALUE}}'
				],
			]
		);

		$this->add_control(
			'excerpt-color',
			[
				'label'     => esc_html__( 'Excerpt', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_1,
				],
				'selectors' => [
					'{{WRAPPER}} .screen-reader-link' => 'color: {{VALUE}}'
				],
			]
		);

		$this->add_control(
			'meta-color',
			[
				'label'     => esc_html__( 'Meta', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_1,
				],
				'selectors' => [
					'{{WRAPPER}} .entry-meta .entry-meta__content'         => 'color: {{VALUE}}',
					'{{WRAPPER}} .entry-meta .entry-meta__content .author' => 'color: {{VALUE}}',
					'{{WRAPPER}} .entry-meta .entry-meta__content a'       => 'color: {{VALUE}}',
					'{{WRAPPER}} .entry-meta .entry-meta__action a'        => 'color: {{VALUE}}'
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section-style-color-big',
			[
				'label'     => esc_html__( 'Big Cards With Media - Colors', 'typer' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => [
					'listing' => [
						'grid',
						'masonry'
					],
				],
			]
		);

		$this->add_control(
			'big-category-color',
			[
				'label'     => esc_html__( 'Category', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_4,
				],
				'selectors' => [
					'{{WRAPPER}} .svq-grid-span-2 .has-post-thumbnail .meta-category .meta-category__item a'  => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-masonry-span .has-post-thumbnail .meta-category .meta-category__item a' => 'color: {{VALUE}}',
				],
				'condition' => [
					'listing' => [
						'grid',
						'masonry'
					],
				],
			]
		);

		$this->add_control(
			'big-title-color',
			[
				'label'     => esc_html__( 'Title', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_1,
				],
				'selectors' => [
					'{{WRAPPER}} .svq-grid-span-2 .has-post-thumbnail .heading-title .heading-title-link'  => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-masonry-span .has-post-thumbnail .heading-title .heading-title-link' => 'color: {{VALUE}}',
				],
				'condition' => [
					'listing' => [
						'grid',
						'masonry'
					],
				],
			]
		);

		$this->add_control(
			'big-excerpt-color',
			[
				'label'     => esc_html__( 'Excerpt', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_1,
				],
				'selectors' => [
					'{{WRAPPER}} .svq-grid-span-2 .has-post-thumbnail .screen-reader-link'  => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-masonry-span .has-post-thumbnail .screen-reader-link' => 'color: {{VALUE}}',
				],
				'condition' => [
					'listing' => [
						'grid',
						'masonry'
					],
				],
			]
		);

		$this->add_control(
			'big-meta-color',
			[
				'label'     => esc_html__( 'Meta', 'typer' ),
				'type'      => Controls_Manager::COLOR,
				'scheme'    => [
					'type'  => \Elementor\Core\Schemes\Color::get_type(),
					'value' => \Elementor\Core\Schemes\Color::COLOR_1,
				],
				'selectors' => [
					'{{WRAPPER}} .svq-grid-span-2 .has-post-thumbnail .entry-meta .entry-meta__content'          => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-grid-span-2 .has-post-thumbnail .entry-meta .entry-meta__content .author'  => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-grid-span-2 .has-post-thumbnail .entry-meta .entry-meta__content a'        => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-grid-span-2 .has-post-thumbnail .entry-meta .entry-meta__action a'         => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-masonry-span .has-post-thumbnail .entry-meta .entry-meta__content'         => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-masonry-span .has-post-thumbnail .entry-meta .entry-meta__content .author' => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-masonry-span .has-post-thumbnail .entry-meta .entry-meta__content a'       => 'color: {{VALUE}}',
					'{{WRAPPER}} .svq-masonry-span .has-post-thumbnail .entry-meta .entry-meta__action a'        => 'color: {{VALUE}}',
				],
				'condition' => [
					'listing' => [
						'grid',
						'masonry'
					],
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings();

		$params = [
			'typer_settings'      => [
				'title'                   => $settings['title'],
				'image_wide'              => $settings['image_wide'],
				'listing'                 => $settings['listing'],
				'show_category'           => $settings['show_category'],
				'show_meta'               => $settings['show_meta'],
				'show_excerpt'            => $settings['show_excerpt'],
				'image_position'          => $settings['image_position'],
				'big_post'                => $settings['big_post'],
				'pagination'              => $settings['pagination'],
				'pagination_next_prev'    => $settings['pagination_next_prev'],
				'pagination_page_numbers' => $settings['pagination_page_numbers'],
				'pagination_ajax'         => $settings['pagination_ajax'],
				'col-grid-mobile'         => $settings['col-grid-mobile'],
				'col-grid-tablet'         => $settings['col-grid-tablet'],
				'col-grid-desktop'        => $settings['col-grid-desktop'],
				'col-mobile'              => $settings['col-mobile'],
				'col-tablet'              => $settings['col-tablet'],
				'col-desktop'             => $settings['col-desktop'],
				'current_page'            => isset( $_GET['typer_page'] ) && $_GET['typer_page'] ? (int) $_GET['typer_page'] : 1
			],
			'ignore_sticky_posts' => $settings['include_sticky_posts'] ? 0 : 1,
			'post__not_in'        => $settings['exclude_current_post'] && is_single() ? [ get_the_ID() ] : [],
			'posts_per_page'      => $settings['per_page']
		];

		if ( $settings['pagination'] ) {
			$params['posts_per_page'] = $settings['per_page'];
			$params['offset']         = ( $params['typer_settings']['current_page'] - 1 ) * $settings['per_page'];
		} else {
			$params['showposts'] = $settings['query_total'];
			$params['offset']    = $settings['query_offset'];
		}

		if ( ! $settings['current_category'] ) {
			if ( ! empty( $settings['query_categories'] ) ) {
				foreach ( (array) $settings['query_categories'] as $k => $query_category ) {
					if ( ! get_category( $query_category ) || is_wp_error( get_category( $query_category ) ) ) {
						unset( $settings['query_categories'][ $k ] );
					}
				}

				$params['category__in'] = $settings['query_categories'];
			}

			if ( ! empty( $settings['query_tags'] ) ) {
				foreach ( (array) $settings['query_tags'] as $k => $query_tag ) {
					if ( ! get_tag( $query_tag ) || is_wp_error( get_tag( $query_tag ) ) ) {
						unset( $settings['query_tags'][ $k ] );
					}
				}

				$params['tag__in'] = $settings['query_tags'];
			}
		} else {
			$category = get_category( get_query_var( 'cat' ) );

			if ( ! is_wp_error( $category ) ) {
				$params['category__in'] = $category->cat_ID;
			}
		}

		if ( $settings['query_sort'] === 'oldest' ) {
			$args['order'] = 'ASC';
		} elseif ( $settings['query_sort'] === 'random' ) {
			$params['orderby'] = 'rand';
			$params['order']   = 'ASC';
		}


		// The Query
		AjaxPosts::instance()->build_template( $params );
	}

	protected function _content_template() {
	}

}
