<?php
/**
 * Handles background control class.
 *
 * @package Typer\Customizer
 *
 * @since 1.0.0
 */

namespace Seventhqueen\Typer\Customizer\Control\Group;

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Background control class.
 *
 * @since 1.0.0
 * @ignore
 * @access private
 *
 * @package Typer\Customizer
 */
class Background extends \Seventhqueen\Typer\Customizer\Base\Group_Control {

	/**
	 * Control's type.
	 *
	 * @since 1.0.0
	 *
	 * @var string
	 */
	public $type = 'sq-control-background';

	/**
	 * Set the fields for this control.
	 *
	 * @since 1.0.0
	 */
	protected function set_fields() {
		$this->add_field( 'label', [
			'type'  => 'sq-control-label',
			'label' => esc_html__( 'Background Color Type', 'typer' ),
		] );

		$this->add_field( 'type', [
			'type'     => 'sq-control-choose',
			'column'   => '4',
			'default'  => 'classic',
			'choices'  => [
				'classic'  => [
					'icon' => 'background-type-color',
				],
				'gradient' => [
					'icon' => 'background-type-gradient',
				],
			],
		] );

		$this->add_field( 'color', [
			'type'     => 'color',
			'column'   => '12',
			'icon'     => 'background-color',
			'cssClass' => 'for-classic',
			'responsive' => false,
		] );

		$this->add_field( 'image', [
			'type'     => 'sq-control-image',
			'column'   => '6',
			'label'    => esc_html__( 'Image', 'typer' ),
			'cssClass' => 'for-classic',
		] );

		$this->add_field( 'position', [
			'type'       => 'sq-control-position',
			'column'     => '6',
			'label'      => esc_html__( 'Position', 'typer' ),
			'cssClass'   => 'for-classic',
		] );

		$this->add_field( 'repeat', [
			'type'       => 'sq-control-choose',
			'column'     => '6',
			'label'      => esc_html__( 'Repeat', 'typer' ),
			'default'    => 'no-repeat',
			'choices'    => [
				'repeat' => [
					'icon' => 'repeat',
				],
				'repeat-x' => [
					'icon' => 'repeat-x',
				],
				'repeat-y' => [
					'icon' => 'repeat-y',
				],
				'no-repeat' => [
					'icon' => 'x',
				],
			],
			'cssClass'   => 'for-classic',
		] );

		$this->add_field( 'attachment', [
			'type'     => 'sq-control-choose',
			'column'   => '6',
			'text'     => esc_html__( 'Fixed', 'typer' ),
			'default'  => 'scroll',
			'choices'  => [
				'fixed' => [
					'icon' => 'check',
				],
				'scroll' => [
					'icon' => 'x',
				],
			],
			'cssClass' => 'for-classic',
		] );

		$this->add_field( 'size', [
			'type'     => 'sq-control-choose',
			'column'   => '6',
			'text'     => esc_html__( 'Cover', 'typer' ),
			'default'  => 'auto',
			'choices'  => [
				'cover' => [
					'icon' => 'check',
				],
				'auto' => [
					'icon' => 'x',
				],
			],
			'cssClass' => 'for-classic',
		] );

		$this->add_field( 'divider', [
			'type'        => 'sq-control-divider',
			'dividerType' => 'empty',
			'cssClass'    => 'for-gradient typer-divider-control-empty',
		] );

		$gradient_colors = [
			'color_from' => [
				'type'     => 'color',
				'column'   => '6',
				'icon'     => is_rtl() ? 'direction-arrow' : 'background-color',
				'cssClass' => 'for-gradient',
			],
			'color_to' => [
				'type'     => 'color',
				'column'   => '6',
				'icon'     => is_rtl() ? 'background-color' : 'direction-arrow',
				'cssClass' => 'for-gradient',
			],
		];

		if ( is_rtl() ) {
			$gradient_colors = array_reverse( $gradient_colors );
		}

		foreach ( $gradient_colors as $key => $value ) {
			$this->add_field( $key, $value );
		}

		$this->add_field( 'gradient_type', [
			'type'    => 'sq-control-choose',
			'column'  => '6',
			'default' => 'linear',
			'choices' => [
				'linear' => [
					'label' => esc_html__( 'Linear', 'typer' ),
				],
				'radial' => [
					'label' => esc_html__( 'Radial', 'typer' ),
				],
			],
			'cssClass' => 'for-gradient',
		] );

		$this->add_field( 'angle', [
			'type'       => 'sq-control-text',
			'inputType'  => 'number',
			'inputAttrs' => [
				'placeholder' => 90,
			],
			'icon'       => 'angle',
			'column'     => '6',
			'default'    => 90,
			'cssClass'   => 'for-gradient',
		] );
	}

	/**
	 * Include fields.
	 *
	 * @since 1.0.0
	 */
	protected function include_fields() {
		if ( empty( $this->include ) ) {
			return;
		}

		$include = $this->include;

		if ( in_array( 'video', $include, true ) ) {
			$this->update_field( 'type', [
				'choices'  => [
					'classic'  => [
						'icon' => 'background-type-color',
					],
					'gradient' => [
						'icon' => 'background-type-gradient',
					],
					'video' => [
						'icon' => 'background-type-video',
					],
				],
			] );

			$this->add_field( 'video_divider', [
				'type'        => 'sq-control-divider',
				'dividerType' => 'empty',
				'cssClass'    => 'for-video',
			] );

			$this->add_field( 'video_link', [
				'type'       => 'sq-control-text',
				'column'     => '8',
				'inputType'  => 'url',
				'inputAttrs' => [
					'placeholder' => esc_html__( 'Social or Self hosted video link', 'typer' ),
				],
				'label'      => esc_html__( 'Video Link', 'typer' ),
				'cssClass'   => 'for-video',
			] );

			$this->add_field( 'video_fallback', [
				'type'     => 'sq-control-image',
				'column'   => '4',
				'label'    => esc_html__( 'Video Fallback', 'typer' ),
				'cssClass' => 'for-video',
			] );
		}
	}

	/**
	 * Exclude fields.
	 *
	 * @since 1.0.0
	 */
	protected function exclude_fields() {
		if ( ! empty( $this->exclude ) && in_array( 'image', $this->exclude, true ) ) {
			$this->exclude = [
				'image',
				'position',
				'repeat',
				'attachment',
				'size',
			];
		}

		parent::exclude_fields();
	}

	/**
	 * Format CSS value from theme mod array value.
	 *
	 * @since 1.0.0
	 *
	 * @param array $value The field's value.
	 * @param array $args The field's arguments.
	 *
	 * @return array The formatted properties.
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	public static function format_properties( $value, $args ) {
		$value = wp_parse_args(
			$value,
			[
				'type'       => 'classic',
				'position'   => 'top left',
				'repeat'     => 'no-repeat',
				'attachment' => 'scroll',
				'size'       => 'auto',
			]
		);

		if ( isset( $value['image'] ) && ! empty( $value['image'] ) ) {
			$value['image'] = "url({$value['image']})";
		}

		return $value;
	}

	/**
	 * Format theme mod array value into a valid background value.
	 *
	 * @since 1.0.0
	 *
	 * @param array $value The field's value.
	 *
	 * @return string The formatted background value.
	 */
	public static function format_value( $value ) {
		$value = array_merge(
			[
				'type'          => 'classic',
				'color'         => '',
				'image'         => '',
				'repeat'        => 'no-repeat',
				'attachment'    => '',
				'position'      => 'top left',
				'gradient_type' => 'linear',
				'angle'         => '90',
				'color_from'    => 'transparent',
				'color_to'      => 'transparent',
			],
			$value
		);

		if ( 'classic' === $value['type'] ) {
			if ( ! empty( $value['image'] ) ) {
				return sprintf(
					'%1$s %2$s %3$s %4$s %5$s',
					'url(' . $value['image'] . ')',
					$value['color'],
					$value['attachment'],
					$value['repeat'],
					typer_get_direction( $value['position'] )
				);
			}

			if ( ! empty( $value['color'] ) ) {
				return $value['color'];
			}
		}

		if ( 'gradient' === $value['type'] ) {
			if ( ! is_numeric( $value['angle'] ) ) {
				$value['angle'] = '90';
			}

			$gradient = 'radial' === $value['gradient_type'] ? sprintf( 'radial-gradient(%1$s, %2$s)', $value['color_from'], $value['color_to'] ) : sprintf( 'linear-gradient(%1$sdeg, %2$s, %3$s)', $value['angle'], $value['color_from'], $value['color_to'] );

			return $gradient;
		}
	}

}
