<?php

namespace TyperPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class AdsManager
 * @package TyperPro
 */
class AdsManager {

	/**
	 * @var null
	 */
	public static $instance;

	/**
	 * @var string
	 */
	public static $module = 'ads-manager-module';

	/**
	 * @var array
	 */
	public $available_zones = [];

	/**
	 * @var array
	 */
	public $zones = [];

	/**
	 * @return AdsManager|null
	 */
	public static function instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * AdsManager constructor.
	 */
	public function __construct() {
		if ( $this->is_active() ) {
			$this->get_available_zones();
			$this->zones = get_option( 'typer_zones', [] );


			add_filter( 'typer_settings_links', [ $this, 'add_ads_quick_link' ] );
			add_action( 'typer_settings_page_content', [ $this, 'load_content' ], 11 );
			add_action( 'admin_post_typer_save_banners', [ $this, 'save_banners' ] );
			add_action( 'admin_post_typer_save_zones', [ $this, 'save_zones' ] );
			add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );

			foreach ( $this->zones as $key => $zone ) {
				if ( isset( $this->available_zones[ $key ] ) ) {
					add_action( $this->available_zones[ $key ]['action'], [
						$this,
						$this->available_zones[ $key ]['callback']
					], 9 );
				}
			}
		}
	}

	/**
	 * Get module info
	 *
	 * @return array
	 */
	public function get_info() {
		return [
			'slug'        => self::$module,
			'name'        => esc_html__( 'Advertising', 'typer-pro' ),
			'description' => esc_html__( 'Allows you to add ads and banners to the theme\'s predefined ad-content areas.' ),
			'status'      => $this->is_active()
		];
	}

	/**
	 * Add quicklink
	 *
	 * @param $quick_links
	 *
	 * @return mixed
	 */
	public function add_ads_quick_link( $quick_links ) {
		if ( ! $this->is_active() ) {
			return $quick_links;
		}

		$quick_links[ self::$module ] = [
			'title'     => esc_html__( 'Advertising', 'typer-pro' ),
			'dashicon'  => 'dashicons-chart-area',
			'quick_url' => admin_url( 'themes.php?page=' . ThemeInfo::slug() . '-settings&tab=ads' ),
			'active'    => $this->is_current_page()
		];

		return $quick_links;
	}

	/**
	 * Ads settings
	 */
	public function load_content() {
		if ( ! $this->is_current_page() ) {
			return;
		}

		$banners           = get_option( 'typer_banners', [] );
		$has_saved_banners = true;

		if ( empty( $banners ) ) {
			$banners[]         = [
				'id'      => '',
				'name'    => '',
				'content' => ''
			];
			$has_saved_banners = false;
		}

		$increment = count( $banners ) + 1;
		?>

        <div class="sqp_box">
            <h2 class="sqp_box_title">
                <span><?php esc_html_e( 'Advertising Settings', 'typer-pro' ); ?></span>
            </h2>
            <div class="sqp-p-4">
                <h3 class="sqp-mt-0 sqp-mb-3"><?php esc_html_e( 'Banners', 'typer-pro' ) ?></h3>
                <p><?php esc_html_e( 'Setup your banners and save them so you can attach them to advertisement zones.', 'typer-pro' ) ?></p>

                <form method="POST" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                    <div id="svq-banners">
						<?php foreach ( $banners as $key => $banner ) : ?>
                            <div class="sqp-border sqp-border-solid sqp-border-gray-200 sqp-rounded sqp-p-4 sqp-relative">
                                <input type="hidden" name="banner[<?php echo esc_attr( $key ) ?>][id]"
                                       value="<?php echo $banner['id'] ?>">

                                <div class="sqp-mb-4">
                                    <label for="name-<?php echo esc_attr( $key ) ?>"
                                           class="sqp-block sqp-text-gray-700 sqp-text-sm sqp-font-bold sqp-mb-2">
										<?php esc_html_e( 'Banner name', 'typer-pro' ) ?>
                                    </label>
                                    <input type="text" name="banner[<?php echo esc_attr( $key ) ?>][name]"
                                           id="name-<?php echo esc_attr( $key ) ?>"
                                           class="sqp-appearance-none sqp-border sqp-rounded sqp-w-full sqp-py-2 sqp-px-3 sqp-text-gray-700 sqp-leading-tight focus:sqp-outline-none"
                                           value="<?php echo $banner['name'] ?>" required>
                                </div>
                                <div>
                                    <label for="content-<?php echo esc_attr( $key ) ?>"
                                           class="sqp-block sqp-text-gray-700 sqp-text-sm sqp-font-bold sqp-mb-2">
										<?php esc_html_e( 'Banner content', 'typer-pro' ) ?>
                                    </label>
                                    <textarea name="banner[<?php echo esc_attr( $key ) ?>][content]"
                                              id="content-<?php echo esc_attr( $key ) ?>"
                                              rows="5"
                                              class="sqp-appearance-none sqp-border sqp-rounded sqp-w-full sqp-py-2 sqp-px-3 sqp-text-gray-700 sqp-leading-tight focus:sqp-outline-none"
                                              required><?php echo html_entity_decode( $banner['content'] ) ?></textarea>
                                </div>
                                <button type="button"
                                        class="svq-remove-button sqp-absolute sqp-top-0 sqp-right-0 sqp-mt-4 sqp-mr-4 sqp-bg-transparent sqp-border-0 sqp-p-0 sqp-cursor-pointer">
                                    <span class="dashicons dashicons-no"></span>
                                </button>
                            </div>
						<?php endforeach; ?>
                    </div>

                    <div class="sqp-my-4 sqp-text-center">
                        <button type="button" data-counter="<?php echo esc_attr( $increment ) ?>"
                                data-label-name="<?php esc_html_e( 'Banner name', 'typer-pro' ) ?>"
                                data-label-content="<?php esc_html_e( 'Banner content', 'typer-pro' ) ?>"
                                class="sqp-bg-transparent sqp-border-0 sqp-p-0 sqp-cursor-pointer svq-more-button">
                            <span class="dashicons dashicons-plus"></span>
                        </button>
                    </div>
                    <input type="hidden" name="action" value="typer_save_banners">
                    <button type="submit" class="sqp_btn sqp-btn-md">
						<?php esc_html_e( 'Save', 'typer-pro' ) ?>
                    </button>
                </form>

                <br>
                <hr>

                <h3 class="sqp-my-3"><?php esc_html_e( 'Advertisement zones', 'typer-pro' ) ?></h3>
                <p><?php esc_html_e( 'Attach banners to the available advertisement zones.', 'typer-pro' ) ?></p>

				<?php if ( $has_saved_banners ) : ?>
                    <form method="POST" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
						<?php foreach ( $this->available_zones as $key => $zone ) : ?>
                            <div class="sqp-border sqp-border-solid sqp-border-gray-200 sqp-rounded sqp-p-4 sqp-mb-4">
                                <div class="sqp-flex sqp-flex-wrap sqp-overflow-hidden">
                                    <div class="sqp-w-full md:sqp-w-2/3">
                                        <h4 class="sqp-mt-0 sqp-mb-3 sqp-text-md sqp-font-bold"><?php echo $zone['name'] ?></h4>

                                        <div class="sqp-mb-6">
                                            <h5 class="sqp-text-sm sqp-mt-0 sqp-mb-2"><?php esc_html_e( 'Select banners', 'typer-pro' ) ?></h5>
                                            <div>
												<?php foreach ( $banners as $banner ) : ?>
													<?php
													if ( ! $banner['id'] ) {
														continue;
													}
													?>
                                                    <label for="<?php echo esc_attr( $key . $banner['id'] ) ?>">
                                                        <input type="checkbox"
                                                               name="zone[<?php echo esc_attr( $key ) ?>][banners][]"
                                                               id="<?php echo esc_attr( $key . $banner['id'] ) ?>"
                                                               value="<?php echo $banner['id'] ?>" <?php checked( isset( $this->zones[ $key ] ) && in_array( $banner['id'], $this->zones[ $key ]['banners'], true ) ) ?>>

														<?php echo $banner['name'] ?>
                                                    </label>
												<?php endforeach; ?>
                                            </div>
                                        </div>

										<?php if ( $zone['allow_repeat'] ) : ?>
											<?php
											$value = isset( $this->zones[ $key ] ) ? $this->zones[ $key ]['repeat'] : 0;
											?>
                                            <div class="sqp-mb-4">
                                                <label for="repeat-<?php echo esc_attr( $key ) ?>"
                                                       class="sqp-block sqp-text-gray-700 sqp-text-sm sqp-mb-2">
													<?php esc_html_e( 'Repeat ad every X posts (0 = show only once)', 'typer-pro' ) ?>
                                                </label>
                                                <input type="number" name="zone[<?php echo esc_attr( $key ) ?>][repeat]"
                                                       id="repeat-<?php echo esc_attr( $key ) ?>" min="0"
                                                       class="sqp-appearance-none sqp-border sqp-rounded sqp-w-24 sqp-py-2 sqp-px-3 sqp-text-gray-700 sqp-leading-tight focus:sqp-outline-none"
                                                       value="<?php echo esc_attr( $value ) ?>">
                                            </div>
										<?php endif; ?>

										<?php if ( $zone['allow_random'] ) : ?>
                                            <label for="random-<?php echo esc_attr( $key ) ?>">
                                                <input type="checkbox"
                                                       name="zone[<?php echo esc_attr( $key ) ?>][random]"
                                                       id="random-<?php echo esc_attr( $key ) ?>" <?php checked( isset( $this->zones[ $key ] ) && $this->zones[ $key ]['random'] ) ?>>
												<?php esc_html_e( 'Randomize banners', 'typer-pro' ) ?>
                                            </label>
										<?php endif; ?>
                                    </div>
                                    <div class="sqp-w-full md:sqp-w-1/3 sqp-text-left md:sqp-text-right sqp-mt-6 md:sqp-mt-0">
                                        <img src="<?php echo esc_url( TYPER_PRO_ASSETS_URL . 'img/' . $zone['img'] ) ?>"
                                             style="max-width: 92px;">
                                    </div>
                                </div>
                            </div>

						<?php endforeach; ?>

                        <input type="hidden" name="action" value="typer_save_zones">
                        <button type="submit" class="sqp_btn sqp-btn-md">
							<?php esc_html_e( 'Save', 'typer-pro' ) ?>
                        </button>
                    </form>
				<?php else: ?>
                    <p><?php esc_html_e( 'No banners available. Please add some banners first.', 'typer-pro' ) ?></p>
				<?php endif; ?>
            </div>
        </div>

		<?php
	}

	/**
	 * Save banners
	 */
	public function save_banners() {
		if ( ! isset( $_POST['action'] ) || $_POST['action'] !== 'typer_save_banners' ) {
			$this->redirect();
		}

		if ( ! isset( $_POST['banner'] ) || ! is_array( $_POST['banner'] ) ) {
			delete_option( 'typer_banners' );

			$this->redirect();
		}

		$banners = [];

		foreach ( $_POST['banner'] as $banner ) {
			$banners[] = [
				'id'      => ! isset( $banner['id'] ) || ! $banner['id'] ? uniqid( '', true ) : $banner['id'],
				'name'    => $banner['name'],
				'content' => htmlentities( stripslashes( $banner['content'] ) )
			];
		}

		update_option( 'typer_banners', $banners );

		$this->redirect();
	}

	/**
	 * Save zones
	 */
	public function save_zones() {
		if ( ! isset( $_POST['action'] ) || $_POST['action'] !== 'typer_save_zones' ) {
			$this->redirect();
		}

		if ( ! isset( $_POST['zone'] ) || ! is_array( $_POST['zone'] ) ) {
			$this->redirect();
		}

		$zones = [];
		foreach ( $_POST['zone'] as $key => $data ) {
			$zones[ $key ] = [
				'banners' => isset( $data['banners'] ) && is_array( $data['banners'] ) ? $data['banners'] : [],
				'repeat'  => isset( $data['repeat'] ) && is_numeric( $data['repeat'] ) ? abs( (int) $data['repeat'] ) : 0,
				'random'  => isset( $data['random'] )
			];
		}

		update_option( 'typer_zones', $zones );

		$this->redirect();
	}

	/**
	 * Is panel page active
	 *
	 * @return bool
	 */
	private function is_current_page() {
		return isset( $_GET['tab'] ) && $_GET['tab'] === 'ads';
	}

	/**
	 * Get module status
	 *
	 * @return bool
	 */
	private function is_active() {
		return Modules::instance()->is_active( self::$module );
	}

	/**
	 * Redirect to ads
	 */
	private function redirect() {
		wp_redirect( admin_url( 'themes.php?page=typer-settings&tab=ads' ) );
	}

	/**
	 * Enqueue scripts
	 */
	public function enqueue_scripts() {
		wp_register_script(
			'typer-pro-banner-manager',
			TYPER_PRO_ASSETS_URL . 'js/banner-manager.js',
			[ 'jquery' ],
			TYPER_PRO_VERSION,
			true
		);

		wp_enqueue_script( 'typer-pro-banner-manager' );
	}

	/**
	 * Get available ad zones
	 */
	public function get_available_zones() {
//		$this->available_zones['archive_before_main'] = [
//			'name'         => esc_html__( 'Archive before main', 'typer-pro' ),
//			'action'       => 'typer_before_archive_main',
//			'callback'     => 'render_archive_before_main_ads',
//			'allow_random' => true,
//			'allow_repeat' => false
//		];
//
//		$this->available_zones['archive_after_main'] = [
//			'name'         => esc_html__( 'Archive after main', 'typer-pro' ),
//			'action'       => 'typer_after_archive_main',
//			'callback'     => 'render_archive_after_main_ads',
//			'allow_random' => true,
//			'allow_repeat' => false
//		];

		$this->available_zones['archive_before_content'] = [
			'name'         => esc_html__( 'Before content on archive page', 'typer-pro' ),
			'action'       => 'typer_before_archive_content',
			'callback'     => 'render_archive_before_content_ads',
			'allow_random' => true,
			'allow_repeat' => false,
			'img'          => 'before-content-archive.png'
		];

		$this->available_zones['archive_after_content'] = [
			'name'         => esc_html__( 'After content on archive page', 'typer-pro' ),
			'action'       => 'typer_after_archive_content',
			'callback'     => 'render_archive_after_content_ads',
			'allow_random' => true,
			'allow_repeat' => false,
			'img'          => 'after-content-archive.png'
		];

		$this->available_zones['single_before_content'] = [
			'name'         => esc_html__( 'Before content on single page', 'typer-pro' ),
			'action'       => 'typer_before_single_content',
			'callback'     => 'render_single_before_content_ads',
			'allow_random' => true,
			'allow_repeat' => false,
			'img'          => 'before-content-single.png'
		];

		$this->available_zones['single_after_content'] = [
			'name'         => esc_html__( 'After content on single page', 'typer-pro' ),
			'action'       => 'typer_after_single_content',
			'callback'     => 'render_single_after_content_ads',
			'allow_random' => true,
			'allow_repeat' => false,
			'img'          => 'after-content-single.png'
		];

		$this->available_zones['archive_cards'] = [
			'name'         => esc_html__( 'Between posts on Masonry/Grid listings', 'typer-pro' ),
			'action'       => 'typer_masonry_grid_before_box',
			'callback'     => 'render_masonry_grid_ads',
			'allow_random' => true,
			'allow_repeat' => true,
			'img'          => 'between-masonry-grid.png'
		];

		$this->available_zones['archive_list'] = [
			'name'         => esc_html__( 'Between posts on List/Big listings', 'typer-pro' ),
			'action'       => 'typer_list_before_box',
			'callback'     => 'render_list_ads',
			'allow_random' => true,
			'allow_repeat' => true,
			'img'          => 'between-list.png'
		];
	}

	/**
	 * Render ad zone
	 *
	 * @param array $zone
	 * @param string $container_class
	 * @param int $repeater
	 */
	public function render_content( $zone = [], $container_class = '', $repeater = 0 ) {
		if ( ! isset( $zone['banners'] ) || empty( $zone['banners'] ) ) {
			return;
		}

		$zone['banners'] = $this->filter_banners( $zone['banners'] );

		if ( empty( $zone['banners'] ) ) {
			return;
		}

		$key    = $zone['random'] ? array_rand( $zone['banners'] ) : 0;
		$banner = $zone['banners'][ $key ];

		if ( ! $banner['content'] ) {
			return;
		}

		echo '<div class="' . $container_class . '">' .
		     html_entity_decode( do_shortcode( $banner['content'] ) ) .
		     '</div>';
	}

	/**
	 * Render ads before archive main
	 */
	public function render_archive_before_main_ads() {
		$zone = $this->zones['archive_before_main'];

		$this->render_content( $zone, 'svq-archive-before-main' );
	}

	/**
	 * Render ads after archive main
	 */
	public function render_archive_after_main_ads() {
		$zone = $this->zones['archive_after_main'];

		$this->render_content( $zone, 'svq-archive-after-main' );
	}

	/**
	 * Render ads before main archive content
	 */
	public function render_archive_before_content_ads() {
		$zone = $this->zones['archive_before_content'];

		$this->render_content( $zone, 'svq-archive-before-content' );
	}

	/**
	 * Render ads after main archive content
	 */
	public function render_archive_after_content_ads() {
		$zone = $this->zones['archive_after_content'];

		$this->render_content( $zone, 'svq-archive-after-content' );
	}

	/**
	 * Render ads before single content
	 */
	public function render_single_before_content_ads() {
		$zone = $this->zones['single_before_content'];

		$this->render_content( $zone, 'svq-single-before-content' );
	}

	/**
	 * Render ads after single content
	 */
	public function render_single_after_content_ads() {
		$zone = $this->zones['single_after_content'];

		$this->render_content( $zone, 'svq-single-after-content' );
	}

	/**
	 * Render masonry and grid ads
	 *
	 * @param $current
	 */
	public function render_masonry_grid_ads( $current ) {
		$zone = $this->zones['archive_cards'];

		if ( empty( $zone['banners'] ) ) {
			return;
		}

		$zone['banners'] = $this->filter_banners( $zone['banners'] );

		if ( empty( $zone['banners'] ) ) {
			return;
		}

		$key    = $zone['random'] ? array_rand( $zone['banners'] ) : 0;
		$banner = $zone['banners'][ $key ];

		if ( ! $banner['content'] ) {
			return;
		}

		$show_once = false;
		if ( $zone['repeat'] === 0 && $current - 1 === 0 ) {
			$show_once = true;
		}

		if ( $show_once || ( $zone['repeat'] !== 0 && $current % $zone['repeat'] === 0 ) ) {
			echo '<div class="svq-article-col">' .
			     html_entity_decode( do_shortcode( $banner['content'] ) ) .
			     '</div>';
		}
	}

	/**
	 * Render list and big list ads
	 *
	 * @param $current
	 */
	public function render_list_ads( $current ) {
		$zone = $this->zones['archive_list'];

		if ( empty( $zone['banners'] ) ) {
			return;
		}

		$zone['banners'] = $this->filter_banners( $zone['banners'] );

		if ( empty( $zone['banners'] ) ) {
			return;
		}

		$key    = $zone['random'] ? array_rand( $zone['banners'] ) : 0;
		$banner = $zone['banners'][ $key ];

		if ( ! $banner['content'] ) {
			return;
		}

		$show_once = false;
		if ( $zone['repeat'] === 0 && $current - 1 === 0 ) {
			$show_once = true;
		}

		if ( $show_once || ( $zone['repeat'] !== 0 && $current % $zone['repeat'] === 0 ) ) {
			echo '<div class="svq-article-col">' .
			     html_entity_decode( do_shortcode( $banner['content'] ) ) .
			     '</div>';
		}
	}

	/**
	 * Find banner by ID
	 *
	 * @param $id
	 *
	 * @return bool
	 */
	public function find_banner( $id ) {
		$banners = get_option( 'typer_banners', [] );

		foreach ( $banners as $banner ) {
			if ( $banner['id'] === $id ) {
				return $banner;
			}
		}

		return false;
	}

	/**
	 * Filter banners
	 *
	 * @param $banners
	 *
	 * @return array
	 */
	public function filter_banners( $banners ) {
		$data = [];

		foreach ( $banners as $banner ) {
			$find = $this->find_banner( $banner );

			if ( $find ) {
				$data[] = $find;
			}
		}

		return $data;
	}

}

AdsManager::instance();
