<?php

namespace TyperPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Settings
 * @package TyperPro
 */
class Settings {

	/**
	 * @var null
	 */
	public static $instance;

	/**
	 * @return Settings
	 */
	public static function instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Settings constructor.
	 */
	public function __construct() {
		add_action( 'init', [ $this, 'action_init' ] );
		add_filter( 'typer_settings_columns', '__return_zero' );
		add_action( 'typer_welcome_page_content', [ $this, 'tpl_settings_page_quick_links' ], 11 );
		add_action( 'typer_settings_page_content', [ $this, 'load_settings_menu' ] );
	}

	/**
	 * Init
	 */
	public function action_init() {
		add_action( 'admin_menu', [ $this, 'register_settings_page' ] );

		global $pagenow;

		if ( $pagenow === 'themes.php' && isset( $_GET['page'] ) && $_GET['page'] === 'typer-settings' ) {
			// Redirect to modules if no other tab is selected
			if ( ! isset( $_GET['tab'] ) || ! $_GET['tab'] ) {
				wp_redirect( admin_url( 'themes.php?page=' . ThemeInfo::slug() . '-settings&tab=modules' ) );
			}
		}
	}

	/**
	 * Register settings page
	 */
	public function register_settings_page() {
		add_theme_page(
			sprintf( esc_html__( '%s Settings', 'typer-pro' ), 'Typer' ),
			sprintf( esc_html__( '%s Settings', 'typer-pro' ), 'Typer' ),
			'manage_options',
			ThemeInfo::slug() . '-settings',
			[ $this, 'settings_menu_callback' ]
		);
	}

	/**
	 * Settings page callback
	 */
	public function settings_menu_callback() {
		$site_url      = apply_filters( 'typer_admin_site_url', 'https://' . ThemeInfo::slug() . '.seventhqueen.com' );
		$wrapper_class = apply_filters( 'typer_welcome_wrapper_class', [ ThemeInfo::slug() . '_settings' ] );

		?>
        <div class="sqp_typer_options wrap sqp-m-0 <?php echo esc_attr( implode( ' ', $wrapper_class ) ); ?>">
            <div class="sqp-bg-white sqp-py-5 sqp-mb-5 sqp-shadow">
                <div class="sqp-container sqp-mx-auto sqp-px-5 sqp-flex sqp-flex-wrap sqp-justify-between sqp-items-center">
                    <div class="sqp-text-left">
                        <a href="<?php echo esc_url( $site_url ); ?>" target="_blank" rel="noopener"
                           class="sqp-text-base sqp-flex sqp-items-center sqp-content-center sqp-no-underline">
                            <img src="<?php echo esc_url( get_parent_theme_file_uri( 'assets/img/logo-black.png' ) ); ?>"
                                 class="sqp-border-0 sqp-w-32" alt="Typer">
                            <span class="sqp-bg-gray-300 sqp-text-gray-600 sqp-ml-4 sqp-px-2 sqp-py-1 sqp-text-xs sqp-rounded sqp-no-underline">
                                <?php echo ThemeInfo::version(); ?>
                            </span>

							<?php do_action( 'typer_welcome_page_header_title' ); ?>
                        </a>
                    </div>

					<?php do_action( 'typer_header_right_section' ); ?>

                </div>
            </div>

			<?php do_action( 'typer_menu_panel_action' ); ?>
        </div>
		<?php
	}

	/**
	 * Settings page quick links
	 */
	public function tpl_settings_page_quick_links() {
		$quick_settings = apply_filters(
			'typer_settings_links', []
		);

		if ( empty( $quick_settings ) || ! is_array( $quick_settings ) ) {
			return;
		}

		?>

        <div class="sqp_box">
            <h2 class="sqp_box_title">
				<?php esc_html_e( 'Settings Quick Links', 'typer' ); ?>
            </h2>
            <ul class="sqp_box_list">
				<?php
				foreach ( (array) $quick_settings as $key => $link ) {
					echo '<li class="sqp_box_list_item">' .
					     '<span class="dashicons ' . esc_attr( $link['dashicon'] ) . ' sqp-ml-4"></span>' .
					     '<a class="sqp-ml-2 sqp-mr-4 sqp-no-underline focus:sqp-outline-none" href="' . esc_url( $link['quick_url'] ) . '" target="_blank" rel="noopener">' .
					     esc_html( $link['title'] ) . '</a>' .
					     '</li>';
				}
				?>
            </ul>
        </div>

		<?php
	}

	/**
	 * Load settings menu
	 */
	public function load_settings_menu() {
		$links = [];

		$links = Modules::instance()->add_modules_quick_link( $links );
		$links = AdsManager::instance()->add_ads_quick_link( $links );

		?>

        <div class="sqp_settings_menu_container">
            <ul class="sqp-flex">
				<?php
				foreach ( $links as $key => $link ) {
					$module = $key !== 'modules-settings' ? 'data-module="' . $key . '"' : '';
					echo '<li class="sqp-mr-3 sqp-mb-0" ' . $module . '>' .
					     '<a class="sqp-inline-flex sqp-rounded sqp-items-center sqp-py-3 sqp-px-4 sqp-text-white sqp-font-medium sqp-no-underline hover:sqp-text-white hover:sqp-bg-blue-700 ' . ( isset( $link['active'] ) && $link['active'] ? 'sqp-bg-blue-700' : 'sqp-bg-blue-500' ) . '" href="' . esc_url( $link['quick_url'] ) . '" rel="noopener">' .
					     '<span class="dashicons ' . esc_attr( $link['dashicon'] ) . ' sqp-mr-3"></span>' .
					     esc_html( $link['title'] ) . '</a>' .
					     '</li>';
				}
				?>
            </ul>
        </div>
		<?php
	}

}

Settings::instance();
