<?php

namespace TyperPro;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Modules
 * @package TyperPro
 */
class Modules {

	/**
	 * @var null
	 */
	public static $instance;

	/**
	 * @var array
	 */
	public $modules = [];

	/**
	 * @return Modules|null
	 */
	public static function instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Modules constructor.
	 */
	public function __construct() {
		add_filter( 'typer_settings_links', [ $this, 'add_modules_quick_link' ] );
		add_action( 'typer_settings_page_content', [ $this, 'load_content' ], 11 );
		add_action( 'wp_ajax_typer_module_activation', [ $this, 'change_module_status' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
	}

	/**
	 * Add quicklink
	 *
	 * @param $quick_links
	 *
	 * @return mixed
	 */
	public function add_modules_quick_link( $quick_links ) {
		$quick_links['modules-settings'] = [
			'title'     => esc_html__( 'Modules', 'typer-pro' ),
			'dashicon'  => 'dashicons-admin-settings',
			'quick_url' => admin_url( 'themes.php?page=' . ThemeInfo::slug() . '-settings&tab=modules' ),
			'active'    => $this->is_current_page()
		];

		return $quick_links;
	}

	/**
	 * Modules settings
	 */
	public function load_content() {
		if ( ! $this->is_current_page() ) {
			return;
		}

		$this->populate_modules();

		?>

        <div class="sqp_box">
            <h2 class="sqp_box_title">
                <span><?php esc_html_e( 'Available modules', 'typer-pro' ); ?></span>
            </h2>
            <div class="sqp-px-4 sqp-py-2">
                <form action="<?php echo esc_url( admin_url( 'admin-ajax.php' ) ); ?>" method="POST" id="modules-form">
                    <ul class="svq-module-list">
						<?php foreach ( $this->modules as $key => $module ) : ?>
                            <li class="sqp-flex sqp-items-center sqp-justify-between sqp-mb-6">
                                <div>
                                    <div class="sqp-text-lg sqp-font-bold sqp-mb-2">
										<?php echo $module['name'] ?>
                                    </div>
                                    <div class="sqp-text-gray-600">
										<?php echo $module['description'] ?>
                                    </div>
                                </div>
                                <label class="svq-switch" for="module-label-<?php echo $key ?>">
                                    <input type="checkbox" name="<?php echo esc_attr( $module['slug'] ) ?>"
                                           id="module-label-<?php echo $key ?>"
                                           class="svq-check-input toggle-module-status" <?php checked( $module['status'] ) ?>>
                                    <span class="svq-slider"></span>
                                </label>
                            </li>
						<?php endforeach; ?>
                    </ul>

                    <input type="hidden" name="action" value="typer_module_activation">
                </form>
            </div>
        </div>

		<?php
	}

	public function change_module_status() {
		if ( ! isset( $_POST['action'] ) || $_POST['action'] !== 'typer_module_activation' ) {
			wp_send_json_error();
		}

		$options = [];

		foreach ( $_POST as $module => $status ) {
			if ( $module === 'action' ) {
				continue;
			}
			if ( $this->module_exists( $module ) ) {
				$options[ $module ] = true;
			}
		}

		update_option( 'typer_modules', $options );

		ob_start();
		Settings::instance()->load_settings_menu();
		$response = ob_get_clean();

		wp_send_json_success( $response );
	}

	/**
	 * Check if module exists
	 *
	 * @param $module_slug
	 *
	 * @return bool
	 */
	public function module_exists( $module_slug ) {
		$this->populate_modules();

		foreach ( $this->modules as $module ) {
			if ( $module['slug'] === $module_slug ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Is panel active
	 *
	 * @return bool
	 */
	private function is_current_page() {
		return isset( $_GET['tab'] ) && $_GET['tab'] === 'modules';
	}

	/**
	 * Enqueue scripts
	 */
	public function enqueue_scripts() {
		wp_register_script(
			'typer-pro-modules',
			TYPER_PRO_ASSETS_URL . 'js/modules.js',
			[ 'jquery' ],
			TYPER_PRO_VERSION,
			true
		);

		wp_enqueue_script( 'typer-pro-modules' );
	}

	/**
	 * Populate modules
	 */
	public function populate_modules() {
		$this->modules[] = AdsManager::instance()->get_info();
	}

	/**
	 * Check if module is active
	 *
	 * @param $module_slug
	 *
	 * @return bool
	 */
	public function is_active( $module_slug ) {
		$modules = get_option( 'typer_modules', [
			AdsManager::$module => false
		] );

		return isset( $modules[ $module_slug ] ) && $modules[ $module_slug ] === true;
	}

}

Modules::instance();
