<?php

namespace SQ_Elements\Widgets\Mailchimp;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Component extends \WP_Widget {

	public $api_key;

	public function __construct() {
		$widget_ops = [
			'description' => esc_html__( 'Mailchimp newsletter subscribe form.', 'typer-pro' )
		];

		parent::__construct(
			'sq_mailchimp',
			esc_html__( 'Typer Mailchimp Newsletter', 'typer-pro' ),
			$widget_ops
		);

		include_once( SQB_WIDGETS . 'Mailchimp/MailChimpAPI.php' );

		$this->api_key = get_theme_mod( 'mailchimp_api', '' );

		add_action( 'init', [ $this, 'init' ] );
	}

	public function init() {
		add_action( 'wp_ajax_mc_action', [ $this, 'mc_action' ] );
		add_action( 'wp_ajax_nopriv_mc_action', [ $this, 'mc_action' ] );
	}

	public function widget( $args, $instance ) {
		if ( ! isset( $instance['mailchimp_list'] ) && is_super_admin() ) {
			echo '<p class="alert alert-warning" role="alert">' . esc_html__( 'Warning! Your subscription form will not show up until you will select a Mailchimp list for your subscribers.', 'typer-pro' ) . '</p>';

			return;
		} elseif ( ! isset( $instance['mailchimp_list'] ) ) {
			return;
		}

		$msg_class = '';
		$msg       = '';

		// Fallback for no js
		if ( isset ( $_POST['mc_email'] ) && $instance['mailchimp_list'] === $_POST['list'] ) {
			if ( isset ( $this->api_key ) && ! empty ( $this->api_key ) ) {
				$widget_settings = $this->get_settings();
				$widget_settings = $widget_settings[ $this->number ];

				$mcAPI  = new \MailChimpAPI( $this->api_key );
				$result = $mcAPI->post( "lists/" . $_POST['list'] . "/members", [
					'email_address' => $_POST['mc_email'],
					'status'        => $widget_settings['opt_in'] === '1' ? 'pending' : 'subscribed'
				] );

				if ( is_array( $result ) && isset( $result['status'] ) ) {
					if ( $result['status'] === 'subscribed' ) {
						$msg_class = 'success';
						$msg       = esc_html__( 'Congratulations! You have subscribed!', 'typer-pro' );
					} elseif ( $result['status'] === 'pending' ) {
						$msg_class = 'success';
						$msg       = esc_html__( 'Success! Please check your inbox or spam folder for the confirmation email.', 'typer-pro' );
					} else {
						$msg_class = 'warning';
						$msg       = esc_html__( 'Error! This email was already subscribed or it is invalid.', 'typer-pro' );
					}
				}
			}
		}

		$privacy_policy_page_id = get_option( 'wp_page_for_privacy_policy' );
		$privacy_policy_page    = null;

		if ( $privacy_policy_page_id ) {
			$privacy_policy_page = get_post( $privacy_policy_page_id );
		}

		echo $args['before_widget'];

		if ( ! empty( $instance['title'] ) ) {
			echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'];
		}

		echo '<form action="' . admin_url( 'admin-ajax.php' ) . '" method="post" class="svq-newsletter-form text-center">
				<input type="hidden" id="list" class="mc_list" name="list" value="' . ( isset( $instance['mailchimp_list'] ) ? $instance['mailchimp_list'] : '' ) . '" />' .
		     ( $instance['mailchimp_title'] ? '<p class="heading-title">' . esc_html( $instance['mailchimp_title'] ) . '</p>' : '' ) .
		     ( $instance['mailchimp_description'] ? '<p class="heading-subtitle">' . esc_html( $instance['mailchimp_description'] ) . '</p>' : '' ) . '
				<div class="form-row justify-content-center main-form-container">
					<div class="col-auto">
						<input type="text" name="mc_email" class="form-control form-control-md mc_email" placeholder="' . esc_html__( "Your email", 'typer-pro' ) . '" required>
					</div>
					<div class="col-auto">
						<button class="btn btn-md btn-dark">' . esc_html__( 'Subscribe', 'typer-pro' ) . '</button>
					</div>
				</div>' .
		     ( $msg && $msg_class ? '<p class="alert alert-' . esc_attr( $msg_class ) . '" role="alert">' . esc_html( $msg ) . '</p>' : '' ) .
		     ( $privacy_policy_page && $instance['mailchimp_policy'] === '1' ? '<p class="small">' . sprintf( esc_html__( 'By subscribing you agree to our %s.', 'typer-pro' ), '<a href="' . esc_url( get_the_permalink( $privacy_policy_page ) ) . '">' . esc_html( get_the_title( $privacy_policy_page ) ) . '</a>' ) . '</p>' : '' ) . '	
			</form>';

		echo $args['after_widget'];

		$nonce = wp_create_nonce( 'mc_mail' );

		echo "<script type='text/javascript'>
			window.addEventListener('DOMContentLoaded', function() {
			  	jQuery(function($) {
			  	  	$('.svq-newsletter-form').on('submit', function(e) {
						e.preventDefault();
						let container = $(this);
						
						$.ajax({
							url: container.attr('action'),
							type: 'POST',
							data: {
								action: 'mc_action',
								mc_email: $('.mc_email', container).attr('value'),
								list: $('.mc_list', container).attr('value'),
								nonce: '" . $nonce . "'
							},
							success: function(data){
								$('.alert', container).remove();
								$('.main-form-container', container).after(data.message);
							},
							error: function() {
								$('.alert', container).remove();
								$('.main-form-container', container).after(data.message);
							}
						});
						
						return false;
					});
			  	});
			});
		</script>";
	}

	public function update( $new_instance, $old_instance ) {
		$instance['title']                 = isset( $new_instance['title'] ) ? strip_tags( stripslashes( $new_instance['title'] ) ) : '';
		$instance['mailchimp_title']       = isset( $new_instance['mailchimp_title'] ) ? stripslashes( $new_instance['mailchimp_title'] ) : '';
		$instance['mailchimp_description'] = isset( $new_instance['mailchimp_description'] ) ? stripslashes( $new_instance['mailchimp_description'] ) : '';
		$instance['mailchimp_policy']      = isset( $new_instance['mailchimp_policy'] ) ? $new_instance['mailchimp_policy'] : '';
		$instance['mailchimp_list']        = isset( $new_instance['mailchimp_list'] ) ? $new_instance['mailchimp_list'] : '';
		$instance['opt_in']                = isset( $new_instance['opt_in'] ) ? $new_instance['opt_in'] : '';

		return $instance;
	}

	public function form( $instance ) {
		$title                 = isset( $instance['title'] ) ? $instance['title'] : '';
		$mailchimp_title       = isset( $instance['mailchimp_title'] ) ? $instance['mailchimp_title'] : '';
		$mailchimp_description = isset( $instance['mailchimp_description'] ) ? $instance['mailchimp_description'] : '';
		$mailchimp_policy      = isset( $instance['mailchimp_policy'] ) ? $instance['mailchimp_policy'] : '';
		$mailchimp_list        = isset( $instance['mailchimp_list'] ) ? $instance['mailchimp_list'] : '';
		$opt_in                = isset( $instance['opt_in'] ) ? $instance['opt_in'] : '';

		if ( ! function_exists( 'curl_init' ) ) {
			?>
			<div>
				<?php esc_html_e( 'Curl is not enabled. Please contact your hosting company and ask them to enable CURL.', 'typer-pro' ) ?>
			</div>
			<?php

			return;
		}

		if ( empty ( $this->api_key ) ) {
			?>
			<div>
				<?php esc_html_e( 'You need to enter your MailChimp API_KEY into Customizer before using this widget.', 'typer-pro' ) ?>
			</div>
			<?php
			return;
		}

		$lists = [];

		if ( ! empty ( $this->api_key ) ) {
			try {
				$mcAPI = new \MailChimpAPI( $this->api_key );
				$lists = $mcAPI->get( 'lists' );
			} catch ( \Exception $e ) {

			}
		}

		?>

		<p>
			<label for="<?php echo $this->get_field_id( 'title' ); ?>">
				<?php esc_html_e( 'Title:', 'typer-pro' ); ?>
			</label>
			<input type="text" class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>"
			       name="<?php echo $this->get_field_name( 'title' ); ?>" value="<?php echo $title; ?>"/>
		</p>

		<p>
			<label for="<?php echo $this->get_field_id( 'mailchimp_list' ); ?>">
				<?php esc_html_e( 'Select List:', 'typer-pro' ); ?>
			</label>
			<select id="<?php echo $this->get_field_id( 'mailchimp_list' ); ?>"
			        name="<?php echo $this->get_field_name( 'mailchimp_list' ); ?>">
				<option value="" disabled selected><?php esc_html_e( 'Select list', 'wp-blocks' ) ?></option>

				<?php
				if ( ! empty( $lists ) && isset( $lists['lists'] ) && is_array( $lists['lists'] ) ) {
					foreach ( $lists['lists'] as $list ) {
						$selected = ( isset( $mailchimp_list ) && $mailchimp_list === $list['id'] ) ? ' selected="selected" ' : '';
						?>
						<option
							<?php echo $selected; ?>value="<?php echo $list['id']; ?>"><?php echo $list['name']; ?></option>
						<?php
					}
				}
				?>
			</select>
		</p>

		<p>
			<label for="<?php echo $this->get_field_id( 'opt_in' ); ?>">
				<?php esc_html_e( 'Double opt-in:', 'typer-pro' ); ?>
			</label>
			<select id="<?php echo $this->get_field_id( 'opt_in' ); ?>"
			        name="<?php echo $this->get_field_name( 'opt_in' ); ?>">
				<option <?php echo( isset( $opt_in ) && $opt_in === '1' ? 'selected="selected"' : '' ) ?> value="1">
					<?php esc_html_e( 'Yes', 'typer-pro' ) ?>
				</option>
				<option <?php echo( isset( $opt_in ) && $opt_in === '0' ? 'selected="selected"' : '' ) ?> value="0">
					<?php esc_html_e( 'No', 'typer-pro' ) ?>
				</option>
			</select>
		</p>

		<p>
			<label for="<?php echo $this->get_field_id( 'mailchimp_title' ); ?>">
				<?php echo esc_html__( 'Heading:', 'typer-pro' ); ?>
			</label>
			<input type="text" class="widefat" id="<?php echo $this->get_field_id( 'mailchimp_title' ); ?>"
			       name="<?php echo $this->get_field_name( 'mailchimp_title' ); ?>"
			       value="<?php echo $mailchimp_title; ?>">
		</p>

		<p>
			<label for="<?php echo $this->get_field_id( 'mailchimp_description' ); ?>">
				<?php echo esc_html__( 'Description:', 'typer-pro' ); ?>
			</label>
			<textarea class="widefat" id="<?php echo $this->get_field_id( 'mailchimp_description' ); ?>"
			          name="<?php echo $this->get_field_name( 'mailchimp_description' ); ?>"
			          rows="5"><?php echo $mailchimp_description; ?></textarea>
		</p>

		<p>
			<label for="<?php echo $this->get_field_id( 'mailchimp_policy' ); ?>">
				<?php esc_html_e( 'Show privacy notice:', 'typer-pro' ); ?>
			</label>
			<select id="<?php echo $this->get_field_id( 'mailchimp_policy' ); ?>"
			        name="<?php echo $this->get_field_name( 'mailchimp_policy' ); ?>">
				<option <?php echo( isset( $mailchimp_policy ) && $mailchimp_policy === '1' ? 'selected="selected"' : '' ) ?>
					value="1">
					<?php esc_html_e( 'Yes', 'typer-pro' ) ?>
				</option>
				<option <?php echo( isset( $mailchimp_policy ) && $mailchimp_policy === '0' ? 'selected="selected"' : '' ) ?>
					value="0">
					<?php esc_html_e( 'No', 'typer-pro' ) ?>
				</option>
			</select>
		</p>


		<?php
	}

	public function mc_action() {
		$data = [
			'message' => ''
		];

		$msg_class = '';
		$msg       = '';

		if ( isset( $_POST['mc_email'], $_POST['list'] ) && wp_verify_nonce( $_POST['nonce'], 'mc_mail' ) ) {
			if ( ! empty ( $this->api_key ) ) {
				$widget_settings = $this->get_settings();
				$widget_settings = $widget_settings[ $this->number ];

				$mcAPI  = new \MailChimpAPI( $this->api_key );
				$result = $mcAPI->post( 'lists/' . $_POST['list'] . '/members', [
					'email_address' => $_POST['mc_email'],
					'status'        => (int) $widget_settings['opt_in'] === 1 ? 'pending' : 'subscribed'
				] );

				if ( is_array( $result ) && isset( $result['status'] ) ) {
					if ( $result['status'] === 'subscribed' ) {
						$msg_class = 'success';
						$msg       = esc_html__( 'Congratulations! You have subscribed!', 'typer-pro' );
					} elseif ( $result['status'] === 'pending' ) {
						$msg_class = 'success';
						$msg       = esc_html__( 'Success! Please check your inbox or spam folder for the confirmation email.', 'typer-pro' );
					} else {
						$msg_class = 'warning';
						$msg       = esc_html__( 'Error! This email was already subscribed or it is invalid.', 'typer-pro' );

					}
				}
			} else {
				$msg_class = 'warning';
				$msg       = esc_html__( 'Something went wrong. Please try again later!', 'typer-pro' );
			}
		} else {
			$msg_class = 'warning';
			$msg       = esc_html__( 'Something went wrong. Please try again later!', 'typer-pro' );
		}

		if ( $msg && $msg_class ) {
			$data['message'] = '<p class="alert alert-' . esc_attr( $msg_class ) . '" role="alert">' . esc_html( $msg ) . '</p>';
		}

		wp_send_json( $data );
	}
}
