<?php

namespace SQ_Elements\Modules\Ajax_Auth;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use function SQ_Elements\sqe_load_template;
use function SQ_Elements\svq_option;

/**
 * Class AjaxAuth
 */
class Component {

	/**
	 * @var null
	 */
	public static $instance = null;

	/**
	 * @return Component|null
	 */
	public static function instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * AjaxAuth constructor.
	 */
	public function __construct() {
		if ( is_admin() ) {
			require_once SQB_MODULES . 'Ajax_Auth/Ajax_Walker_Nav_Menu_Checklist.php';
		}

		add_action( 'elementor/widgets/widgets_registered', [ $this, 'svq_widgets_registered' ] );
		add_filter( 'typer_theme_settings', [ $this, 'settings' ], 10, 2 );
		add_action( 'init', [ $this, 'shortcode_init' ] );

		add_action( 'load-nav-menus.php', [ $this, 'wp_nav_menu_meta_box' ] );
		add_filter( 'wp_setup_nav_menu_item', [ $this, 'setup_nav_menu_item' ], 10, 1 );

		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_styles' ] );

		add_action( 'init', [ $this, 'init' ] );
		add_action( 'init', [ $this, 'do_ajax_login' ] );

		add_action( 'wp_ajax_nopriv_svq_lost_password', [ $this, 'do_lost_password_ajax' ] );
		add_action( 'wp_ajax_nopriv_svq_register', [ $this, 'do_register_ajax' ] );

		add_filter( 'plugin_action_links_' . SQB_PLUGIN_BASE, [ $this, 'plugin_add_settings_link' ] );
	}

	/**
	 * Init shortcodes
	 */
	public function shortcode_init() {
		add_shortcode( 'svq-login-form', [ $this, 'login_form' ] );
		add_shortcode( 'svq-register-form', [ $this, 'register_form_shortcode' ] );
		add_shortcode( 'svq-lost-password-form', [ $this, 'lost_password_form' ] );
	}

	/**
	 * Return login form
	 *
	 * @return string
	 */
	public function login_form() {
		ob_start();
		if ( is_super_admin() ) {
			$this->the_admin_notification();
		} elseif ( is_user_logged_in() ) {
			return '';
		}

		$terms_page = svq_option( 'login_terms_page', '' );

		sqe_load_template( 'auth/login', compact( 'terms_page' ) );

		return ob_get_clean();
	}

	/**
	 * Registration form shortcode
	 */
	public function register_form_shortcode() {
		echo $this->register_form( true );
	}

	/**
	 * Return register form
	 *
	 * @param bool $onboarding
	 *
	 * @return false|string
	 */
	public function register_form( $onboarding = false ) {

		if ( is_super_admin() ) {
			$this->the_admin_notification();
		} elseif ( is_user_logged_in() ) {
			return '';
		}

		if ( function_exists( 'bp_is_active' ) ) {
			$link = bp_get_signup_page();
		} else {
			$link = wp_registration_url();
		}

		$terms_page = svq_option( 'login_terms_page', '' );
		$allowed    = [
			'a' => [
				'target' => [],
				'class'  => [],
				'href'   => [],
				'title'  => [],
			],
			'span'
		];

		ob_start();

		sqe_load_template( 'auth/register', compact( 'terms_page', 'allowed', 'link', 'onboarding' ) );

		return ob_get_clean();
	}

	/**
	 * Return lost password form
	 *
	 * @return string
	 */
	public function lost_password_form() {
		ob_start();

		if ( is_super_admin() ) {
			$this->the_admin_notification();
		} elseif ( is_user_logged_in() ) {
			return '';
		}

		sqe_load_template( 'auth/lost-password' );

		return ob_get_clean();
	}

	public function the_admin_notification() {
		?>
		<p class="alert alert-info">
			<?php esc_html_e( 'Even if you are logged in, you can see this form because you are a super admin. This notification is only visible to super admins!', 'typer-pro' ) ?>
		</p>
		<?php
	}

	public function svq_widgets_registered() {
		$elements = [
			'login'        => 'Login',
			'lostpassword' => 'LostPassword',
			'register'     => 'Register'
		];

		if ( defined( 'ELEMENTOR_PATH' ) && class_exists( '\Elementor\Widget_Base' ) ) {
			// get our own widgets up and running:

			if ( class_exists( '\Elementor\Plugin' ) ) {
				if ( is_callable( '\Elementor\Plugin', 'instance' ) ) {
					$elementor = \Elementor\Plugin::instance();

					if ( isset( $elementor->widgets_manager ) ) {
						if ( method_exists( $elementor->widgets_manager, 'register_widget_type' ) ) {

							foreach ( $elements as $slug => $class ) {

								sqe_load_template( 'elementor/' . $slug, [], true );

								try {
									$class_name = 'SQ_Elements\Elementor\\' . $class;
									\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new $class_name );
								} catch ( \Exception $e ) {
								}
							}
						}
					}
				}
			}
		}
	}

	public function init() {
		if ( svq_option( 'login_redirect', 'default' ) === 'custom' && svq_option( 'login_redirect_custom', '' ) !== '' ) {
			add_filter( 'login_redirect', [ $this, 'custom_after_login_redirect' ], 10, 3 );
		}
	}

	public function plugin_add_settings_link( $links ) {
		$settings_link = '<a href="' . admin_url( '/customize.php?autofocus[section]=svq_section_login' ) . '">' . __( 'Customizer', 'typer-pro' ) . '</a>';
		$links[]       = $settings_link;

		$settings_link = '<a href="' . admin_url( '/nav-menus.php' ) . '">' . __( 'Menu Items', 'typer-pro' ) . '</a>';
		$links[]       = $settings_link;

		return $links;
	}

	public function settings( $sq = [], $panel = '' ) {
		$section = 'svq_section_auth';

		$sq['sec'][ $section ] = [
			'title'    => __( 'Authentication', 'typer-pro' ),
			'panel'    => $panel,
			'priority' => 1000
		];

		$sq['set']['login_terms_page'] = [
			'section' => $section,
			'type'    => 'dropdown-pages',
			'default' => '',
			'title'   => __( 'Terms & Conditions Page', 'typer-pro' )
		];

		$sq['set']['login_redirect'] = [
			'section'     => $section,
			'type'        => 'select',
			'default'     => 'reload',
			'choices'     => [
				'default' => 'Default',
				'reload'  => 'Reload',
				'custom'  => 'Custom link'
			],
			'title'       => __( 'AJAX Login redirect', 'typer-pro' ),
			'description' => __( 'Default: WordPress default behaviour. Reload: will reload current page.', 'typer-pro' ),
		];

		$sq['set']['login_redirect_custom'] = [
			'section'         => $section,
			'type'            => 'text',
			'default'         => '',
			'title'           => __( 'Custom link redirect', 'typer-pro' ),
			'description'     => wp_kses( __( 'Set a link like http://yoursite.com/homepage for users to get redirected on login. You can use ##member_name## placeholder for current logged in username.<br> ' .
			                                  'For more complex redirect logic please set Login redirect to Default WordPress and use Peter\'s redirect plugin or similar.', 'typer-pro' ), [ 'br' => [] ] ),
			'active_callback' => [
				[
					'setting' => 'login_redirect',
					'value'   => 'custom',
					'compare' => '='
				]
			]
		];

		$sq['set']['register_after_action'] = [
			'section'     => $section,
			'type'        => 'select',
			'default'     => 'reload',
			'choices'     => [
				'default'  => 'Default',
				'popup'    => 'Close popup (if used in popup)',
				'redirect' => 'Redirect to link'
			],
			'title'       => __( 'AJAX After Register action', 'typer-pro' ),
			'description' => __( 'Default: WordPress default behaviour - show message. Close Popup: will close the popup if register form is used inside JetPopup. Redirect: will redirect to a given link.', 'typer-pro' ),
		];

		$sq['set']['register_after_redirect_action'] = [
			'section'         => $section,
			'type'            => 'text',
			'default'         => '',
			'title'           => __( 'Custom link redirect', 'typer-pro' ),
			'description'     => __( 'Set a link like http://yoursite.com/login for users to get redirected after register.', 'typer-pro' ),
			'active_callback' => [
				[
					'setting' => 'register_after_action',
					'value'   => 'redirect',
					'compare' => '='
				]
			]
		];

		return $sq;
	}

	public function enqueue_styles() {
		if ( is_user_logged_in() && ! is_super_admin() ) {
			return;
		}

		wp_register_style( 'sq-auth-style', SQB_ASSETS_URL . 'css/auth.css' );
		wp_enqueue_style( 'sq-auth-style' );
	}

	public function enqueue_scripts() {
		if ( is_user_logged_in() && ! is_super_admin() ) {
			return '';
		}

		wp_enqueue_script( 'typer-pro', SQB_ASSETS_URL . 'js/auth.js', [ 'jquery' ], SQB_VERSION, true );

		$obj_array  = [
			'loadingMessage' => esc_html__( 'Sending info, please wait..', 'typer-pro' ),
			'errorOcurred'   => esc_html__( 'Sorry, an error occurred', 'typer-pro' ),
			'loginUrl'       => site_url( 'wp-login.php', 'login_post' ),
			'ajaxUrl'        => admin_url( 'admin-ajax.php' ),
		];
		$terms_page = svq_option( 'login_terms_page', '' );

		if ( $terms_page ) {
			$obj_array['agreeTerms'] = sprintf( esc_html__( 'You must agree with %s', 'typer-pro' ), get_the_title( $terms_page ) );
		}

		$obj_array = apply_filters( 'svq_localize_app', $obj_array );

		wp_localize_script( 'typer-pro', 'SqAjaxLoginLocale', $obj_array );
	}

	public function do_ajax_login() {

		/* If not our action, bail out */
		if ( ! isset( $_POST['action'] ) || ( isset( $_POST['action'] ) && 'svq-ajax-login' !== $_POST['action'] ) ) {
			return false;
		}

		/* If user is already logged in print a specific message */
		if ( is_user_logged_in() ) {

			echo wp_json_encode( [
				'loggedin'    => true,
				'redirecturl' => null,
				'message'     => '<span class="svq-ajax-info">' . esc_html__( 'Login successful, redirecting...', 'typer-pro' ) . '</span>',
			] );
			die();
		}

		// Check the nonce, if it fails the function will break
		check_ajax_referer( 'svq-ajax-login-nonce', 'security-login' );

		// Nonce is checked, continue
		$secure_cookie = '';

		// If the user wants ssl but the session is not ssl, force a secure cookie.
		if ( ! empty( $_POST['log'] ) && ! force_ssl_admin() ) {
			$user_name = sanitize_user( $_POST['log'] );
			if ( $user = get_user_by( 'login', $user_name ) ) {
				if ( get_user_option( 'use_ssl', $user->ID ) ) {
					$secure_cookie = true;
					force_ssl_admin( true );
				}
			}
		}

		if ( isset( $_REQUEST['redirect_to'] ) ) {
			$redirect_to = $_REQUEST['redirect_to'];
			// Redirect to https if user wants ssl
			if ( $secure_cookie && false !== strpos( $redirect_to, 'wp-admin' ) ) {
				$redirect_to = preg_replace( '|^http://|', 'https://', $redirect_to );
			}
		} else {
			$redirect_to = '';
		}

		$user_signon = wp_signon( '', $secure_cookie );
		if ( is_wp_error( $user_signon ) ) {
			echo wp_json_encode( [
				'loggedin' => false,
				'message'  => '<span class="svq-ajax-warning">' . __( 'Wrong username or password. Please try again.', 'typer-pro' ) . '</span>',
			] );
		} else {
			if ( svq_option( 'login_redirect', 'default' ) === 'reload' ) {
				$redirecturl = null;
			} else {
				$requested_redirect_to = isset( $_REQUEST['redirect_to'] ) ? $_REQUEST['redirect_to'] : '';
				/**
				 * Filter the login redirect URL.
				 *
				 * @param string $redirect_to The redirect destination URL.
				 * @param string $requested_redirect_to The requested redirect destination URL passed as a parameter.
				 * @param \WP_User|\WP_Error $user WP_User object if login was successful, WP_Error object otherwise.
				 *
				 * @since 3.0.0
				 *
				 */
				$redirecturl = apply_filters( 'login_redirect', $redirect_to, $requested_redirect_to, $user_signon );
				if ( $redirecturl === false ) {
					$redirecturl = '';
				}

			}

			echo wp_json_encode( [
				'loggedin'    => true,
				'redirecturl' => $redirecturl,
				'message'     => '<span class="svq-ajax-info">' . esc_html__( 'Login successful, redirecting...', 'typer-pro' ) . '</span>',
			] );
		}

		die();
	}

	public function do_register_ajax() {
		$http_post = 'POST' === $_SERVER['REQUEST_METHOD'];

		// Check the nonce, if it fails the function will break
		check_ajax_referer( 'svq-ajax-register-nonce', 'security-register' );

		if ( $_SERVER['REQUEST_METHOD'] !== 'POST' ) {
			return false;
		}

		/*if ( is_multisite() ) {
			echo wp_json_encode( array(
				'registered'  => false,
				'redirecturl' => apply_filters( 'wp_signup_location', network_site_url( 'wp-signup.php' ) ),
				'message'     => '<span class="svq-ajax-info">' . esc_html__( 'You will be redirected.', 'typer-pro' ) . '</span>',
			) );
			exit();
		}*/

		if ( ! get_option( 'users_can_register' ) ) {
			echo wp_json_encode( [
				'registered'  => false,
				'redirecturl' => '',
				'message'     => '<span class="svq-ajax-info">' . esc_html__( 'Registration is disabled.', 'typer-pro' ) . '</span>',
			] );
			exit();
		}

		$user_login = '';
		$user_email = '';

		if ( $http_post ) {
			if ( isset( $_POST['user_login'] ) && is_string( $_POST['user_login'] ) ) {
				$user_login = $_POST['user_login'];
			}

			if ( isset( $_POST['user_email'] ) && is_string( $_POST['user_email'] ) ) {
				$user_email = wp_unslash( $_POST['user_email'] );
			}

			$user_id = register_new_user( $user_login, $user_email );

			if ( ! is_wp_error( $user_id ) ) {
				if ( isset( $_POST['category'] ) && $_POST['category'] && class_exists( '\FUP\FupPlugin' ) ) {
					$categories = @json_decode( $_POST['category'], true );
					if ( is_array( $categories ) ) {
						foreach ( $categories as $category ) {
							\FUP\FollowCategory::instance()->simpleAddFollow( $user_id, $category );
						}
					}
				}

				$redirect_url = '';
				$close_popup  = false;

				if ( svq_option( 'register_after_action', 'default' ) === 'default' ) {
					$redirect_url = $_POST['redirect_to'];
				} elseif ( svq_option( 'register_after_action', 'default' ) === 'popup' ) {
					$close_popup = true;
				} elseif ( svq_option( 'register_after_action', 'default' ) === 'redirect' ) {
					$redirect_url = svq_option( 'register_after_redirect_action', '' );
				}

				echo wp_json_encode( [
					'registered'  => true,
					'close_popup' => $close_popup,
					'redirecturl' => $redirect_url,
					'message'     => '<span class="svq-ajax-info">' . esc_html__( 'Your account was created. Please check your confirmation email.', 'typer-pro' ) . '</span>',
				] );
				exit();
			}

			$errors = '';

			foreach ( $user_id->errors as $error ) {
				$errors .= '<span class="svq-ajax-info">' . $error[0] . '</span><br>';
			}

			echo wp_json_encode( [
				'registered'  => false,
				'close_popup' => false,
				'redirecturl' => '',
				'message'     => $errors,
			] );
			exit();
		}

		echo wp_json_encode( [
			'registered'  => false,
			'redirecturl' => '',
			'message'     => '<span class="svq-ajax-info">' . esc_html__( 'Something went wrong. Please try again.', 'typer-pro' ) . '</span>',
		] );
		exit();
	}

	public function do_lost_password_ajax() {
		// Check the nonce, if it fails the function will break
		wp_verify_nonce( 'svq-ajax-lost-nonce', 'security-lost-pass' );

		$errors = new \WP_Error();

		if ( isset( $_POST ) ) {

			if ( empty( $_POST['user_login'] ) ) {
				$errors->add( 'empty_username', wp_kses_data( __( '<strong>ERROR</strong>: Enter a username or e-mail address.', 'default' ) ) );
			} elseif ( strpos( $_POST['user_login'], '@' ) ) {
				$user_data = get_user_by( 'email', trim( $_POST['user_login'] ) );
				if ( empty( $user_data ) ) {
					$errors->add( 'invalid_email', wp_kses_data( __( '<strong>ERROR</strong>: There is no user registered with that email address.', 'default' ) ) );
				}
			} else {
				$login     = trim( $_POST['user_login'] );
				$user_data = get_user_by( 'login', $login );
			}

			/**
			 * Fires before errors are returned from a password reset request.
			 *
			 * @param \WP_Error $errors A WP_Error object containing any errors generated
			 *                         by using invalid credentials.
			 *
			 * @since 4.4.0 Added the `$errors` parameter.
			 *
			 * @since 2.1.0
			 */
			do_action( 'lostpassword_post', $errors );

			if ( $errors->get_error_code() ) {
				echo '<span class="wrong-response">' . $errors->get_error_message() . '</span>';
				die();
			}

			if ( ! $user_data ) {
				$errors->add(
					'invalidcombo', wp_kses_data( __( '<strong>ERROR</strong>: Invalid username or e-mail.', 'default' ) )
				);
				echo '<span class="wrong-response">' . $errors->get_error_message() . '</span>';
				die();
			}

			// Redefining user_login ensures we return the right case in the email.
			$user_login = $user_data->user_login;
			$user_email = $user_data->user_email;
			$key        = get_password_reset_key( $user_data );

			if ( is_wp_error( $key ) ) {
				echo '<span class="wrong-response">' . $key->get_error_message() . '</span>';
				die();
			}

			$message = esc_html__( 'Someone requested that the password be reset for the following account:', 'typer-pro' ) . "\r\n\r\n";
			$message .= network_home_url( '/' ) . "\r\n\r\n";
			$message .= sprintf( esc_html__( 'Username: %s', 'default' ), $user_login ) . "\r\n\r\n";
			$message .= esc_html__( 'If this was a mistake, just ignore this email and nothing will happen.', 'typer-pro' ) . "\r\n\r\n";
			$message .= esc_html__( 'To reset your password, visit the following address:', 'typer-pro' ) . "\r\n\r\n";
			$message .= '<' . network_site_url( "wp-login.php?action=rp&key=$key&login=" . rawurlencode( $user_login ), 'login' ) . ">\r\n";

			if ( is_multisite() ) {
				$blogname = get_network()->site_name;
			} else /*
                 * The blogname option is escaped with esc_html on the way into the database
                 * in sanitize_option we want to reverse this for the plain text arena of emails.
                 */ {
				$blogname = wp_specialchars_decode( get_option( 'blogname' ), ENT_QUOTES );
			}

			$title = sprintf( esc_html__( '[%s] Password Reset' ), $blogname );

			/**
			 * Filters the subject of the password reset email.
			 *
			 * @param string $title Default email title.
			 * @param string $user_login The username for the user.
			 * @param \WP_User $user_data WP_User object.
			 *
			 * @since 4.4.0 Added the `$user_login` and `$user_data` parameters.
			 *
			 * @since 2.8.0
			 */
			$title = apply_filters( 'retrieve_password_title', $title, $user_login, $user_data );

			/**
			 * Filters the message body of the password reset mail.
			 *
			 * @param string $message Default mail message.
			 * @param string $key The activation key.
			 * @param string $user_login The username for the user.
			 * @param \WP_User $user_data WP_User object.
			 *
			 * @since 2.8.0
			 * @since 4.1.0 Added `$user_login` and `$user_data` parameters.
			 *
			 */
			$message = apply_filters( 'retrieve_password_message', $message, $key, $user_login, $user_data );


			if ( $message && ! wp_mail( $user_email, wp_specialchars_decode( $title ), $message ) ) {
				echo '<span class="wrong-response">';
				echo esc_html__( 'The e-mail could not be sent.', 'default' );
				echo '</span>';
				die();
			} else {
				echo '<span class="good-response">' . esc_html__( 'Email successfully sent!', 'typer-pro' ) . '</span>';
				die();
			}
		}
		die();
	}

	/**
	 * Redirect user after successful login.
	 *
	 * @param $redirect_to
	 * @param $requested_redirect_to
	 * @param $user
	 *
	 * @return bool|mixed|null
	 */
	public function custom_after_login_redirect( $redirect_to, $requested_redirect_to, $user ) {
		if ( ( $requested_redirect_to === '' || $requested_redirect_to !== home_url() ) && ! is_wp_error( $user ) ) {
			$redirect_to = svq_option( 'login_redirect_custom', '' );
			$redirect_to = str_replace( '##member_name##', $user->user_login, $redirect_to );
		}

		return $redirect_to;
	}

	/**
	 * Add nav menu login meta box
	 */
	public function wp_nav_menu_meta_box() {
		add_meta_box( 'add-svq-login-nav-menu', esc_html__( 'Typer Login', 'typer-pro' ), [
			$this,
			'do_wp_nav_menu_meta_box'
		], 'nav-menus', 'side', 'default' );

		add_action( 'admin_print_footer_scripts', [ $this, 'wp_nav_menu_restrict_items' ] );
	}

	/**
	 * Build and populate the SQ Login accordion on Appearance > Menus.
	 *
	 * @since 1.9.0
	 *
	 * @global $nav_menu_selected_id
	 */
	public function do_wp_nav_menu_meta_box() {
		global $nav_menu_selected_id;

		$walker = new Ajax_Walker_Nav_Menu_Checklist( false );
		$args   = [ 'walker' => $walker ];

		$post_type_name = 'svq-auth';

		$tabs = [];

		$tabs['loggedin']['label'] = esc_html__( 'Logged-In', 'typer-pro' );
		$tabs['loggedin']['pages'] = [
			(object) [
				'ID'             => - 1,
				'post_title'     => esc_html__( 'Log Out', 'typer-pro' ),
				'post_author'    => 0,
				'post_date'      => 0,
				'post_excerpt'   => 'logout',
				'post_type'      => 'page',
				'post_status'    => 'publish',
				'comment_status' => 'closed',
				'guid'           => wp_logout_url(),
			]
		];

		$tabs['loggedout']['label'] = esc_html__( 'Logged-Out', 'typer-pro' );
		$tabs['loggedout']['pages'] = [];

		$tabs['loggedout']['pages']['login'] = (object) [
			'ID'             => - 1,
			'post_title'     => esc_html__( 'Log In', 'typer-pro' ),
			'post_author'    => 0,
			'post_date'      => 0,
			'post_excerpt'   => 'login',
			'post_type'      => 'page',
			'post_status'    => 'publish',
			'comment_status' => 'closed',
			'guid'           => wp_login_url()
		];

		$title = '';
		$link  = '';

		if ( function_exists( 'bp_is_active' ) && bp_is_active() ) {
			$bp_directory_page_ids = bp_core_get_directory_page_ids();
			if ( isset( $bp_directory_page_ids['register'] ) ) {
				$register_page = get_post( $bp_directory_page_ids['register'] );
				$title         = $register_page->post_title;
				$link          = get_permalink( $register_page->ID );
			}
		}

		if ( ! $title && ! $link ) {
			$title = esc_html__( 'Register', 'typer-pro' );
			$link  = wp_registration_url();
		}

		$tabs['loggedout']['pages']['register'] = (object) [
			'ID'             => - 1,
			'post_title'     => $title,
			'post_author'    => 0,
			'post_date'      => 0,
			'post_excerpt'   => 'register',
			'post_type'      => 'page',
			'post_status'    => 'publish',
			'comment_status' => 'closed',
			'guid'           => $link,
		];
		?>

		<div id="svq-auth-menu" class="posttypediv">
			<h4><?php esc_html_e( 'Logged-Out', 'typer-pro' ) ?></h4>
			<p><?php esc_html_e( 'Logged-Out links are not visible to users who are logged in.', 'typer-pro' ) ?></p>

			<div id="tabs-panel-posttype-<?php echo $post_type_name; ?>-loggedout" class="tabs-panel tabs-panel-active">
				<ul id="svq-auth-menu-checklist-loggedout" class="categorychecklist form-no-clear">
					<?php echo walk_nav_menu_tree( array_map( 'wp_setup_nav_menu_item', $tabs['loggedout']['pages'] ), 0, (object) $args ); ?>
				</ul>
			</div>

			<h4><?php esc_html_e( 'Logged-Out', 'typer-pro' ) ?></h4>
			<p><?php esc_html_e( 'Logged-in links are visible only to users who are logged in.', 'typer-pro' ) ?></p>

			<div id="tabs-panel-posttype-<?php echo $post_type_name; ?>-loggedin" class="tabs-panel tabs-panel-active">
				<ul id="svq-auth-menu-checklist-loggedin" class="categorychecklist form-no-clear">
					<?php echo walk_nav_menu_tree( array_map( 'wp_setup_nav_menu_item', $tabs['loggedin']['pages'] ), 0, (object) $args ); ?>
				</ul>
			</div>

			<?php
			$removed_args = [
				'action',
				'customlink-tab',
				'edit-menu-item',
				'menu-item',
				'page-tab',
				'_wpnonce',
			];
			?>

			<p class="button-controls">
			<span class="list-controls">
				<a href="<?php
				echo esc_url( add_query_arg(
					[
						$post_type_name . '-tab' => 'all',
						'selectall'              => 1,
					],
					remove_query_arg( $removed_args )
				) );
				?>#svq-auth-menu" class="select-all"><?php esc_html_e( 'Select All', 'typer-pro' ); ?></a>
			</span>
				<span class="add-to-menu">
				<input
					type="submit"<?php if ( function_exists( 'wp_nav_menu_disabled_check' ) ) : wp_nav_menu_disabled_check( $nav_menu_selected_id ); endif; ?>
					class="button-secondary submit-add-to-menu right"
					value="<?php esc_attr_e( 'Add to Menu', 'typer-pro' ); ?>" name="add-custom-menu-item"
					id="submit-svq-auth-menu"/>
				<span class="spinner"></span>
			</span>
			</p>
		</div><!-- /#svq-auth-menu -->

		<?php
	}

	/**
	 * Restrict various items from view if editing a Seeko menu.
	 *
	 * If a person is editing a Seeko menu item, that person should not be able to
	 * see or edit the following fields:
	 *
	 * - CSS Classes - We use the 'seeko-login-menu' CSS class to determine if the
	 *   menu item belongs to Seeko, so we cannot allow manipulation of this field to
	 *   occur.
	 * - URL - This field is automatically generated by BP on output, so this
	 *   field is useless and can cause confusion.
	 *
	 * Note: These restrictions are only enforced if JavaScript is enabled.
	 *
	 * @since 1.0
	 */
	public function wp_nav_menu_restrict_items() {
		?>
		<script type="text/javascript">
            jQuery('#menu-to-edit').on('click', 'a.item-edit', function () {
                var settings = jQuery(this).closest('.menu-item-bar').next('.menu-item-settings');
                var css_class = settings.find('.edit-menu-item-classes');

                if (css_class.val().indexOf('svq-auth-menu') === 0) {
                    css_class.attr('readonly', 'readonly');
                    settings.find('.field-url').css('display', 'none');
                }
            });
		</script>
		<?php
	}

	/**
	 * Add Specific items to the wp_nav_menu.
	 *
	 * @param \WP_Post $menu_item The menu item.
	 *
	 * @return \WP_Post The modified WP_Post object.
	 * @since 1.9.0
	 *
	 */
	public function setup_nav_menu_item( $menu_item ) {
		if ( is_admin() ) {
			return $menu_item;
		}

		// Prevent a notice error when using the customizer.
		$menu_classes = $menu_item->classes;

		if ( is_array( $menu_classes ) ) {
			$menu_classes = implode( ' ', $menu_item->classes );
		}

		// We use information stored in the CSS class to determine what kind of
		// menu item this is, and how it should be treated.
		preg_match( '/\ssvq-auth-(.*)-nav/', $menu_classes, $matches );

		// If this isn't a SVQ menu item, we can stop here.
		if ( empty( $matches[1] ) ) {
			return $menu_item;
		}

		switch ( $matches[1] ) {
			case 'login' :
				if ( is_user_logged_in() ) {
					$menu_item->_invalid = true;
				} else {
					$menu_item->url = wp_login_url( $this->get_requested_url() );
				}

				break;


			// Don't show the Register link to logged-in users.
			case 'register' :
				if ( is_user_logged_in() ) {
					$menu_item->_invalid = true;
				}

				break;

			case 'logout' :
				if ( ! is_user_logged_in() ) {
					$menu_item->_invalid = true;
				} else {
					$menu_item->url = wp_logout_url( $this->get_requested_url() );
				}

				break;
		}

		// If component is deactivated, make sure menu item doesn't render.
		if ( empty( $menu_item->url ) ) {
			$menu_item->_invalid = true;

			// Highlight the current page.
		} else {
			$current = $this->get_requested_url();
			if ( strpos( $current, $menu_item->url ) !== false ) {
				if ( is_array( $menu_item->classes ) ) {
					$menu_item->classes[] = 'current_page_item';
					$menu_item->classes[] = 'current-menu-item';
				} else {
					$menu_item->classes = [ 'current_page_item', 'current-menu-item' ];
				}
			}
		}

		return $menu_item;
	}

	/**
	 * Return the URL as requested on the current page load by the user agent.
	 *
	 * @return string Requested URL string.
	 * @since 1.0
	 *
	 */
	public function get_requested_url() {
		$url = is_ssl() ? 'https://' : 'http://';
		$url .= $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];

		return apply_filters( 'svq_ajax_get_requested_url', esc_url( $url ) );
	}

}

Component::instance();
