<?php
/**
 * Seventhqueen\Typer\Editor\Component class
 *
 * @package typer
 */

namespace Seventhqueen\Typer\Editor;

use Seventhqueen\Typer\Component_Interface;
use function add_action;
use function add_theme_support;
use function Seventhqueen\Typer\typer;

/**
 * Class for integrating with the block editor.
 *
 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/
 */
class Component implements Component_Interface {

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug() {
		return 'editor';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize() {
		add_action( 'after_setup_theme', [ $this, 'action_add_editor_support' ] );
		add_action( 'enqueue_block_editor_assets', [ $this, 'enqueue_editor_assets' ] );
	}

	public function enqueue_editor_assets() {
		wp_enqueue_style( 'typer-root', get_theme_file_uri( 'assets/css/root-vars.css' ) , [], false );
	}

	/**
	 * Adds support for various editor features.
	 */
	public function action_add_editor_support() {

		add_theme_support( 'editor-styles' );

		// This theme styles the visual editor with editor-style.css to match the theme style.
		add_editor_style( [ 'assets/css/style-editor.css', 'assets/css/root-vars.css' ] );

		// Add support for default block styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for wide-aligned images.
		add_theme_support( 'align-wide' );

		// Add support for color palettes.
		add_theme_support( 'editor-color-palette',
			[
				[
					'name'  => esc_html__( 'Primary theme color', 'typer' ),
					'slug'  => 'primary',
					'color' => typer()->get_option( 'primary-color' ),
				],
				[
					'name'  => esc_html__( 'Pale pink', 'typer' ),
					'slug'  => 'pale-pink',
					'color' => '#f78da7',
				],
				[
					'name'  => esc_html__( 'Vivid red', 'typer' ),
					'slug'  => 'vivid-red',
					'color' => '#cf2e2e',
				],
				[
					'name'  => esc_html__( 'Luminous vivid orange', 'typer' ),
					'slug'  => 'luminous-vivid-orange',
					'color' => '#ff6900',
				],
				[
					'name'  => esc_html__( 'Luminous vivid amber', 'typer' ),
					'slug'  => 'luminous-vivid-amber',
					'color' => '#fcb900',
				],
				[
					'name'  => esc_html__( 'Light green cyan', 'typer' ),
					'slug'  => 'light-green-cyan',
					'color' => '#7bdcb5',
				],
				[
					'name'  => esc_html__( 'Vivid green cyan', 'typer' ),
					'slug'  => 'vivid-green-cyan',
					'color' => '#00d084',
				],
				[
					'name'  => esc_html__( 'Pale cyan blue', 'typer' ),
					'slug'  => 'pale-cyan-blue',
					'color' => '#8ed1fc',
				],
				[
					'name'  => esc_html__( 'Vivid cyan blue', 'typer' ),
					'slug'  => 'vivid-cyan-blue',
					'color' => '#0693e3',
				],
				[
					'name'  => esc_html__( 'Very light gray', 'typer' ),
					'slug'  => 'very-light-gray',
					'color' => '#eeeeee',
				],
				[
					'name'  => esc_html__( 'Cyan bluish gray', 'typer' ),
					'slug'  => 'cyan-bluish-gray',
					'color' => '#abb8c3',
				],
				[
					'name'  => esc_html__( 'Very dark gray', 'typer' ),
					'slug'  => 'very-dark-gray',
					'color' => '#313131',
				],
			]
		);

		/*
		 * Optional: Disable custom colors in color palettes.
		 *
		 * Uncomment the line below to disable the custom color picker in the editor.
		 *
		 * add_theme_support( 'disable-custom-colors' );
		 */

		// Add support for font sizes.
		add_theme_support(
			'editor-font-sizes',
			[
				[
					'name'      => __( 'small', 'typer' ),
					'shortName' => __( 'S', 'typer' ),
					'size'      => 16,
					'slug'      => 'small',
				],
				[
					'name'      => __( 'regular', 'typer' ),
					'shortName' => __( 'M', 'typer' ),
					'size'      => 20,
					'slug'      => 'regular',
				],
				[
					'name'      => __( 'large', 'typer' ),
					'shortName' => __( 'L', 'typer' ),
					'size'      => 36,
					'slug'      => 'large',
				],
				[
					'name'      => __( 'larger', 'typer' ),
					'shortName' => __( 'XL', 'typer' ),
					'size'      => 48,
					'slug'      => 'larger',
				],
			]
		);

		/*
		 * Optional: Disable custom font sizes in block text settings.
		 *
		 * Uncomment the line below to disable the custom custom font sizes in the editor.
		 *
		 * add_theme_support( 'disable-custom-font-sizes' );
		 */
	}
}
