<?php
/**
 * Seventhqueen\Typer\Custom_Logo\Component class
 *
 * @package typer
 */

namespace Seventhqueen\Typer\Custom_Logo;

use WP_Customize_Cropped_Image_Control;
use Seventhqueen\Typer\Component_Interface;
use Seventhqueen\Typer\Templating_Component_Interface;
use function add_action;
use function add_theme_support;
use function apply_filters;
use function Seventhqueen\Typer\typer;

/**
 * Class for adding custom logo support.
 *
 * @link https://codex.wordpress.org/Theme_Logo
 */
class Component implements Component_Interface, Templating_Component_Interface {

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug() {
		return 'custom_logo';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize() {
		add_action( 'after_setup_theme', [ $this, 'action_add_custom_logo_support' ] );
		add_action( 'customize_register', [ $this, 'logo_dark_header' ] );
		add_action( 'customize_register', [ $this, 'mobile_logo' ] );
		add_action( 'customize_register', [ $this, 'mobile_logo_dark_header' ] );
		add_filter( 'get_custom_logo', [ $this, 'get_header_logo' ] );
	}

	/**
	 * Adds support for the Custom Logo feature.
	 */
	public function action_add_custom_logo_support() {
		add_theme_support(
			'custom-logo',
			apply_filters(
				'typer_custom_logo_args',
				[
					'height'      => 80,
					'width'       => 200,
					'flex-height' => false,
					'flex-width'  => true,
					'header-text' => [ 'site-title', 'site-description' ],
				]
			)
		);
	}

	/**
	 * Dark Header logo
	 *
	 * @param $wp_customize
	 */
	public function logo_dark_header( $wp_customize ) {
		$wp_customize->add_setting(
			'logo_dark',
			[ 'sanitize_callback' => 'absint' ]
		);

		$custom_logo_args = get_theme_support( 'custom-logo' );
		$wp_customize->add_control(
			new WP_Customize_Cropped_Image_Control(
				$wp_customize,
				'logo_dark',
				[
					'label'         => esc_html__( 'Darkmode logo', 'typer' ),
					'section'       => 'title_tagline',
					'priority'      => 9,
					'height'        => $custom_logo_args[0]['height'],
					'width'         => $custom_logo_args[0]['width'],
					'flex_height'   => $custom_logo_args[0]['flex-height'],
					'flex_width'    => $custom_logo_args[0]['flex-width'],
					'settings'      => 'logo_dark',
					'button_labels' => [
						'select'       => esc_html__( 'Select logo', 'typer' ),
						'change'       => esc_html__( 'Change logo', 'typer' ),
						'remove'       => esc_html__( 'Remove', 'typer' ),
						'default'      => esc_html__( 'Default', 'typer' ),
						'placeholder'  => esc_html__( 'No logo selected', 'typer' ),
						'frame_title'  => esc_html__( 'Select logo', 'typer' ),
						'frame_button' => esc_html__( 'Choose logo', 'typer' ),
					],
				]
			)
		);
	}

	/**
	 * Mobile logo
	 *
	 * @param $wp_customize
	 */
	public function mobile_logo( $wp_customize ) {
		$wp_customize->add_setting(
			'mobile_logo',
			[ 'sanitize_callback' => 'absint' ]
		);

		$custom_logo_args = get_theme_support( 'custom-logo' );
		$wp_customize->add_control(
			new WP_Customize_Cropped_Image_Control(
				$wp_customize,
				'mobile_logo',
				[
					'label'         => esc_html__( 'Mobile logo', 'typer' ),
					'section'       => 'title_tagline',
					'priority'      => 9,
					'height'        => $custom_logo_args[0]['height'],
					'width'         => $custom_logo_args[0]['width'],
					'flex_height'   => $custom_logo_args[0]['flex-height'],
					'flex_width'    => $custom_logo_args[0]['flex-width'],
					'settings'      => 'mobile_logo',
					'button_labels' => [
						'select'       => esc_html__( 'Select logo', 'typer' ),
						'change'       => esc_html__( 'Change logo', 'typer' ),
						'remove'       => esc_html__( 'Remove', 'typer' ),
						'default'      => esc_html__( 'Default', 'typer' ),
						'placeholder'  => esc_html__( 'No logo selected', 'typer' ),
						'frame_title'  => esc_html__( 'Select logo', 'typer' ),
						'frame_button' => esc_html__( 'Choose logo', 'typer' ),
					],
				]
			)
		);
	}

	/**
	 * Dark Header Mobile logo
	 *
	 * @param $wp_customize
	 */
	public function mobile_logo_dark_header( $wp_customize ) {
		$wp_customize->add_setting(
			'mobile_logo_dark',
			[ 'sanitize_callback' => 'absint' ]
		);

		$custom_logo_args = get_theme_support( 'custom-logo' );
		$wp_customize->add_control(
			new WP_Customize_Cropped_Image_Control(
				$wp_customize,
				'mobile_logo_dark',
				[
					'label'         => esc_html__( 'Darkmode Mobile logo', 'typer' ),
					'section'       => 'title_tagline',
					'priority'      => 9,
					'height'        => $custom_logo_args[0]['height'],
					'width'         => $custom_logo_args[0]['width'],
					'flex_height'   => $custom_logo_args[0]['flex-height'],
					'flex_width'    => $custom_logo_args[0]['flex-width'],
					'settings'      => 'mobile_logo_dark',
					'button_labels' => [
						'select'       => esc_html__( 'Select logo', 'typer' ),
						'change'       => esc_html__( 'Change logo', 'typer' ),
						'remove'       => esc_html__( 'Remove', 'typer' ),
						'default'      => esc_html__( 'Default', 'typer' ),
						'placeholder'  => esc_html__( 'No logo selected', 'typer' ),
						'frame_title'  => esc_html__( 'Select logo', 'typer' ),
						'frame_button' => esc_html__( 'Choose logo', 'typer' ),
					],
				]
			)
		);
	}

	/**
	 * @param $html
	 *
	 * @return string
	 */
	public function get_header_logo( $html ) {
		$large_logo_id       = get_theme_mod( 'custom_logo' );
		$large_logo_dark_id  = get_theme_mod( 'logo_dark' );
		$mobile_logo_id      = get_theme_mod( 'mobile_logo' );
		$mobile_logo_dark_id = get_theme_mod( 'mobile_logo_dark' );

		if ( ! $large_logo_id && ! $large_logo_dark_id ) {
			return '';
		}

		$logo_dark        = '';
		$mobile_logo      = '';
		$mobile_logo_dark = '';

		$anchor_class    = 'custom-logo-link';
		$logo_class      = 'custom-logo svq-logo-default';
		$dark_logo_class = 'custom-logo svq-logo-dark';

		$logo = wp_get_attachment_image( $large_logo_id, 'full', false, [
			'class' => $logo_class,
			'alt'   => get_bloginfo( 'name', 'display' )
		] );

		if ( $large_logo_dark_id ) {
			$logo_dark = wp_get_attachment_image( $large_logo_dark_id, 'full', false, [
				'class' => $dark_logo_class,
				'alt'   => get_bloginfo( 'name', 'display' )
			] );
		}

		if ( $logo_dark !== '' ) {
			$anchor_class .= ' has-logo-dark';
		}

		if ( $mobile_logo_id ) {
			$mobile_logo = wp_get_attachment_image( $mobile_logo_id, 'full', false, [
				'class' => 'custom-logo svq-mobile-logo',
				'alt'   => get_bloginfo( 'name', 'display' )
			] );

			$anchor_class .= ' has-logo-mobile';
		}

		if ( $mobile_logo_dark_id ) {
			$mobile_logo_dark = wp_get_attachment_image( $mobile_logo_dark_id, 'full', false, [
				'class' => 'custom-logo svq-mobile-logo-dark',
				'alt'   => get_bloginfo( 'name', 'display' )
			] );

			$anchor_class .= ' has-logo-dark-mobile';
		}

		return sprintf( '<a href="%1$s" class="' . esc_attr( $anchor_class ) . '" rel="home" itemprop="url">%2$s</a>',
			esc_url( home_url( '/' ) ),
			$logo . $logo_dark . $mobile_logo . $mobile_logo_dark
		);
	}

	/**
	 * Gets template tags to expose as methods on the Template_Tags class instance, accessible through `typer()`.
	 *
	 * @return array Associative array of $method_name => $callback_info pairs. Each $callback_info must either be
	 *               a callable or an array with key 'callable'. This approach is used to reserve the possibility of
	 *               adding support for further arguments in the future.
	 */
	public function template_tags() {
		return [
			'get_logo' => [ $this, 'get_logo' ],
			'the_logo' => [ $this, 'the_logo' ],
		];
	}

	/**
	 * Outputs the logo
	 */
	public function the_logo() {
		echo wp_kses_post( $this->get_logo() );
	}

	/**
	 * Returns the logo
	 *
	 * @return string
	 */
	public function get_logo() {
		$output = '';
		$logo   = get_custom_logo( get_current_blog_id() );

		if ( function_exists( 'the_custom_logo' ) && $logo !== '' ) {
			$output .= $logo;
		} else {
			$output .= '<a href="' . home_url() . '"><span class="site-title">' . get_bloginfo() . '</span></a>';
		}

		return $output;
	}
}
