<?php
/**
 * Seventhqueen\Typer\Base_Support\Component class
 *
 * @package typer
 */

namespace Seventhqueen\Typer\Config;

use WP_Post;
use Seventhqueen\Typer\Component_Interface;
use Seventhqueen\Typer\Templating_Component_Interface;

use function Seventhqueen\Typer\typer;

/**
 * Class for adding basic theme support, most of which is mandatory to be implemented by all themes.
 *
 * Exposes template tags:
 * * `typer()->get_option( 'option_name' )`
 * * `typer()->add_setting( 'section_id' , $settings_array )`
 */
class Component implements Component_Interface, Templating_Component_Interface {

	/**
	 * Saved config options
	 *
	 * @var array
	 */
	private $config = [];

	/**
	 * Customizer defined settings
	 *
	 * @var array
	 */
	private $settings = [];

	/**
	 * Current customizer settings
	 *
	 * @var array
	 */
	private $saved_options = null;

	/**
	 * Gets the unique identifier for the theme component.
	 *
	 * @return string Component slug.
	 */
	public function get_slug() {
		return 'config';
	}

	/**
	 * Adds the action and filter hooks to integrate with WordPress.
	 */
	public function initialize() {

		$img_path = get_template_directory_uri() . '/assets/img/wp-customizer/';

		$this->config = [
			'item_id'                      => '24818607',
			'slug'                         => 'typer',
			'core_plugin'                  => 'typer-core',
			'purchase_link'                => 'https://typer.seventhqueen.com/?utm_source=wp-panel&utm_medium=wp-dash&utm_campaign=get-typer',
			'priority_addons'              => [
				'typer-core',
				'typer-pro',
				'front-user-profile',
				'stax'
			],
			'blog_panels'                  => [
				'general',
				'standard',
				'image',
				'quote',
				'audio',
				'video',
				'gallery'
			],
			'layout_options'               => [
				'no-side' => [
					'img' => $img_path . '1c.svg',
					'alt' => esc_html__( 'No sidebar', 'typer' )
				],
				'sl'      => [
					'img' => $img_path . '2cl.svg',
					'alt' => esc_html__( 'Left sidebar', 'typer' )
				],
				'sr'      => [
					'img' => $img_path . '2cr.svg',
					'alt' => esc_html__( 'Right sidebar', 'typer' )
				],
				'sll'     => [
					'img' => $img_path . '3cl.svg',
					'alt' => esc_html__( '2 left sidebars', 'typer' )
				],
				'slr'     => [
					'img' => $img_path . '3cm.svg',
					'alt' => strip_tags( __( 'Left & right sidebars', 'typer' ) )
				],
				'srr'     => [
					'img' => $img_path . '3cr.svg',
					'alt' => esc_html__( '2 right sidebars', 'typer' )
				],
			],
			'layout_width_options'         => [
				'small' => [
					'img' => $img_path . 'container-small.svg',
					'alt' => esc_html__( 'Small', 'typer' )
				],
				'large' => [
					'img' => $img_path . 'container-large.svg',
					'alt' => esc_html__( 'Large', 'typer' )
				]
			],
			'menu_position_options'        => [
				'left'   => esc_html__( 'Left', 'typer' ),
				'center' => esc_html__( 'Center', 'typer' ),
				'right'  => esc_html__( 'Right', 'typer' )
			],
			'breadcrumbs_options'          => [
				'1' => [
					'img' => $img_path . 'no-category-breadcrumb.svg',
					'alt' => esc_html__( 'Hide', 'typer' )
				],
				'2' => [
					'img' => $img_path . 'has-breadcrumb.svg',
					'alt' => esc_html__( 'Show', 'typer' )
				]
			],
			'title_alignment_page_options' => [
				'default' => [
					'img' => $img_path . 'title-align-default.svg',
					'alt' => esc_html__( 'Default', 'typer' )
				],
				'center'  => [
					'img' => $img_path . 'title-center-horizontally.svg',
					'alt' => esc_html__( 'Center horizontally', 'typer' )
				]
			],
			'title_size_options'           => [
				'default' => [
					'img' => $img_path . 'title-size-normal.svg',
					'alt' => esc_html__( 'Normal', 'typer' )
				],
				'large'   => [
					'img' => $img_path . 'title-size-large.svg',
					'alt' => esc_html__( 'Large', 'typer' )
				]
			],
			'blog_nav_options'             => [
				'normal' => [
					'img' => $img_path . 'post-nav-after-content.svg',
					'alt' => esc_html__( 'After content', 'typer' )
				],
				'sticky' => [
					'img' => $img_path . 'post-nav-sticky.svg',
					'alt' => esc_html__( 'Sticky on bottom', 'typer' )
				]
			],
			'thumbnail_width_options'      => [
				'normal' => [
					'img' => $img_path . 'post-image-normal.svg',
					'alt' => esc_html__( 'Normal', 'typer' )
				],
				'wide'   => [
					'img' => $img_path . 'post-image-wide.svg',
					'alt' => esc_html__( 'Wide', 'typer' )
				]
			],
			'switch_options'               => [
				'1' => esc_html__( 'On', 'typer' ),
				'0' => esc_html__( 'Off', 'typer' ),
			],
			'font_sizes'                   => [
				'h1' => [
					'mobile'  => [ 'size' => '1.88', 'unit' => 'rem' ],
					'tablet'  => [ 'size' => '2.63', 'unit' => 'rem' ],
					'desktop' => [ 'size' => '3', 'unit' => 'rem' ],
				],
				'h2' => [
					'mobile'  => [ 'size' => '1.5', 'unit' => 'rem' ],
					'tablet'  => [ 'size' => '2.13', 'unit' => 'rem' ],
					'desktop' => [ 'size' => '2.63', 'unit' => 'rem' ],
				],
				'h3' => [
					'mobile'  => [ 'size' => '1.31', 'unit' => 'rem' ],
					'tablet'  => [ 'size' => '1.75', 'unit' => 'rem' ],
					'desktop' => [ 'size' => '2.13', 'unit' => 'rem' ],
				],
				'h4' => [
					'mobile'  => [ 'size' => '1.25', 'unit' => 'rem' ],
					'tablet'  => [ 'size' => '1.5', 'unit' => 'rem' ],
					'desktop' => [ 'size' => '1.75', 'unit' => 'rem' ],
				],
				'h5' => [
					'mobile'  => [ 'size' => '1.13', 'unit' => 'rem' ],
					'tablet'  => [ 'size' => '1.25', 'unit' => 'rem' ],
					'desktop' => [ 'size' => '1.5', 'unit' => 'rem' ],
				],
				'h6' => [
					'mobile'  => [ 'size' => '1', 'unit' => 'rem' ],
					'tablet'  => [ 'size' => '1', 'unit' => 'rem' ],
					'desktop' => [ 'size' => '1', 'unit' => 'rem' ],
				],
			],

		];

		$sq           = [];
		$sq['panels'] = [];
		$sq['sec']    = [];
		$sq['set']    = [];

		// Add panels
		$sq['panels']['typer'] = [
			'title'       => esc_html__( 'Typer Options', 'typer' ),
			'description' => __( 'Typer Theme specific settings', 'typer' ),
			'priority'    => 10
		];

		// Add sections
		$sections = [
			'\Seventhqueen\Typer\Config\Customizer\LayoutOptions'         => 10,
			'\Seventhqueen\Typer\Config\Customizer\ColorOptions'          => 20,
			'\Seventhqueen\Typer\Config\Customizer\TypographyOptions'     => 30,
			'\Seventhqueen\Typer\Config\Customizer\HeaderOptions'         => 40,
			'\Seventhqueen\Typer\Config\Customizer\PostNavigationOptions' => 80,
			'\Seventhqueen\Typer\Config\Customizer\PageOptions'           => 90,
			'\Seventhqueen\Typer\Config\Customizer\FooterOptions'         => 120,
			'\Seventhqueen\Typer\Config\Customizer\NotFoundOptions'       => 130,
			'\Seventhqueen\Typer\Config\Customizer\MiscOptions'           => 140,
		];

		$sections = apply_filters( 'typer_theme_cutomizer_sections', $sections );

		foreach ( $sections as $section => $priority ) {

			$obj = new $section;

			// Section
			$type = 'sec';
			// Panel
			if ( isset( $obj->type ) && $obj->type == 'panel' ) {
				$type = 'panels';
			}

			$section = $obj->set_panel_name( 'typer' )->get_section( $priority );
			if ( ! empty( $section ) ) {
				$sq[ $type ][ $obj->get_section_slug() ] = $section;
			}

			$options = $obj->get_options();
			if ( isset( $options['sec'] ) ) {
				$sq['sec'] = array_merge( $sq['sec'], $options['sec'] );
				$sq['set'] = array_merge( $sq['set'], $options['set'] );
			} else {
				$sq['set'] = array_merge( $sq['set'], $options );
			}

		}

		$sq = apply_filters( 'typer_theme_settings', $sq, 'typer' );

		$this->settings = $sq;

	}

	/**
	 * Gets template tags to expose as methods on the Template_Tags class instance, accessible through `typer()`.
	 *
	 * @return array Associative array of $method_name => $callback_info pairs. Each $callback_info must either be
	 *               a callable or an array with key 'callable'. This approach is used to reserve the possibility of
	 *               adding support for further arguments in the future.
	 */
	public function template_tags() {
		return [
			'get_option'             => [ $this, 'get_option' ],
			'get_blog_option'        => [ $this, 'get_blog_option' ],
			'get_settings'           => [ $this, 'get_settings' ],
			'add_setting'            => [ $this, 'add_setting' ],
			'set_config'             => [ $this, 'set_config' ],
			'get_config'             => [ $this, 'get_config' ],
			'is_active'              => [ $this, 'is_active' ],
			'verify_purchase'        => [ $this, 'verify_purchase' ],
			'get_purchased_code'     => [ $this, 'get_saved_purchased_code' ],
			'has_media'              => [ $this, 'has_media' ],
			'get_post_data'          => [ $this, 'get_post_data' ],
			'set_post_data'          => [ $this, 'set_post_data' ],
			'get_article_components' => [ $this, 'get_article_components' ]
		];
	}

	/**
	 * Function to get options in front-end
	 *
	 * @param string|bool $option The option we need from the DB
	 * @param string|bool $default If $option doesn't exist in DB return $default value
	 * @param bool $filters
	 *
	 * @return string|array
	 */
	public function get_option( $option = false, $default = false, $filters = true ) {
		if ( $option === false ) {
			return false;
		}

		$meta_items_defaults = apply_filters( 'typer_options_meta_items', [ 'avatar', 'author', 'date' ] );

		$defaults = [
			'blog_comments_modal'              => '0',
			'blog_featured_content_status'     => '0',
			'blog_featured_content_title'      => '',
			'blog_featured_content_tag'        => '',
			'blog_featured_tag_display'        => '1',
			'css_preload'                      => 0,
			'blog_post_listing'                => 'masonry',
			'blog_post_listing_thumb_position' => 'right',
			'blog_post_thumbnail'              => 'normal',
			'blog_post_show_categories'        => '1',
			'blog_post_meta_items'             => $meta_items_defaults,
			'blog_masonry_small_items'         => [
				'desktop' => '6',
				'tablet'  => '6',
				'mobile'  => '12',
			],
			'blog_masonry_wide_items'          => [
				'desktop' => '4',
				'tablet'  => '6',
				'mobile'  => '12',
			]
		];

		if ( is_customize_preview() ) {
			$individual_overwrite = $this->post_overwrite_dependency( $option . '_individual' );

			if ( $individual_overwrite && (bool) get_post_meta( get_the_ID(), '_' . $individual_overwrite, true ) ) {
				return get_post_meta( get_the_ID(), '_' . $option . '_individual', true );
			}

			if ( $filters === true ) {
				$data = apply_filters( 'typer_option', get_theme_mod( $option, $this->get_setting( $option ) ), $option );

				if ( isset( $defaults[ $option ] ) && ! class_exists( '\TyperPro\TyperPro' ) ) {
					return $defaults[ $option ];
				}

				return $data;
			}

			$data = get_theme_mod( $option, $default );

			if ( isset( $defaults[ $option ] ) && ! class_exists( '\TyperPro\TyperPro' ) ) {
				return $defaults[ $option ];
			}

			return $data;
		}

		if ( $this->saved_options === null ) {
			$this->saved_options = get_theme_mods();
		}

		$individual_overwrite = $this->post_overwrite_dependency( $option . '_individual' );

		$is_individual_overwrite = false;

		if ( $individual_overwrite ) {
			$is_individual_overwrite = get_post_meta( get_the_ID(), '_' . $individual_overwrite, true );
		}

		if ( (bool) $is_individual_overwrite ) {
			$output_data = get_post_meta( get_the_ID(), '_' . $option . '_individual', true );
		} else if ( isset( $defaults[ $option ] ) && ! class_exists( '\TyperPro\TyperPro' ) ) {
			$output_data = $defaults[ $option ];
		} else if ( isset( $this->saved_options[ $option ] ) ) {
			$output_data = $this->saved_options[ $option ];
		} else if ( $default === false ) {
			$output_data = $this->get_setting( $option );
		} else {
			$output_data = $default;
		}

		if ( $filters === true ) {
			$output_data = apply_filters( 'typer_option', $output_data, $option );
		}

		return $output_data;
	}

	/**
	 * @param bool $option
	 * @param null $post_format
	 * @param bool $default
	 * @param bool $filters
	 *
	 * @return array|bool|mixed|string
	 */
	public function get_blog_option( $option = false, $post_format = null, $default = false, $filters = true ) {
		if ( $option === false ) {
			return false;
		}

		if ( ! $post_format ) {
			$post_format = get_post_format() ?: 'standard';
		}


		$forced_value = $this->get_post_data( $option, null );

		if ( $forced_value ) {
			return $forced_value;
		}

		$individual_overwrite = $this->post_overwrite_dependency( $option . '_individual' );

		$is_individual_overwrite = false;

		if ( $individual_overwrite ) {
			$is_individual_overwrite = get_post_meta( get_the_ID(), '_' . $individual_overwrite, true );
		}

		if ( (bool) $is_individual_overwrite ) {
			// Carbon fields fallback
			if ( $option === 'media_meta_items' ) {
				global $wpdb;
				$q = $wpdb->get_results(
					"SELECT `meta_value`
				    FROM {$wpdb->prefix}postmeta
				    WHERE `post_id` = '" . get_the_ID() . "'
				    AND `meta_key`
				    LIKE '_media_meta_items_individual%'"
				);

				$output_data = [];

				foreach ( $q as $item ) {
					if ( $item->meta_value ) {
						$output_data[] = $item->meta_value;
					}
				}
			} else {
				$output_data = get_post_meta( get_the_ID(), '_' . $option . '_individual', true );
			}
		} else {
			$meta_items_defaults = apply_filters( 'typer_blog_options_meta_items', [
				'avatar',
				'author',
				'date',
				'reading'
			] );

			$defaults = [
				'media_fullscreen'                  => '2',
				'media_max_height'                  => 800,
				'media_content_color'               => 'dark',
				'media_panel_fade_scroll'           => '1',
				'progressive_bar'                   => '0',
				'media_catbread_type'               => 'category',
				'media_catbread_animation_type'     => 'fadeInDown',
				'media_title_position'              => 'title-above',
				'media_title_over_alignment'        => 'default',
				'media_title_left_alignment'        => 'default',
				'media_title_above_below_alignment' => 'default',
				'media_title_size'                  => 'default',
				'media_title_animation_type'        => 'fadeIn',
				'media_image_enabled_general'       => '1',
				'media_image_width'                 => 'full-width',
				'media_image_overlay'               => 'none',
				'media_image_overlay_show'          => 'both',
				'media_image_background_type'       => 'cover',
				'media_video_width'                 => 'wide',
				'media_video_overlay'               => 'none',
				'media_video_overlay_show'          => 'both',
				'media_gallery_width'               => 'full-width',
				'media_gallery_item_width'          => 'portrait',
				'media_gallery_overlay'             => 'bottom',
				'media_gallery_overlay_show'        => 'single',
				'media_meta_show'                   => '1',
				'media_meta_items'                  => $meta_items_defaults,
				'media_meta_animation'              => 'fadeInUp',
				'media_shapes_show'                 => '0',
				'blog_show_social_box'              => '1',
				'blog_force_social_box_top'         => '0',
			];

			if ( isset( $defaults[ $option ] ) && ! class_exists( '\TyperPro\TyperPro' ) ) {
				$output_data = $defaults[ $option ];
			} else {
				$is_overwrite = $this->get_option( 'media_settings_overwrite_' . $post_format, $default, $filters );

				if ( $is_overwrite ) {
					$output_data = $this->get_option( $option . '_' . $post_format, $default, $filters );
				} else {
					$output_data = $this->get_option( $option . '_general', $default, $filters );
				}
			}
		}

		return $output_data;
	}

	/**
	 * Get theme options array
	 * @return array
	 */
	public function get_settings() {
		return apply_filters( 'typer_settings', $this->settings );
	}

	/**
	 * Get registered theme setting
	 *
	 * @param $id
	 *
	 * @return mixed
	 */
	public function get_setting( $id ) {
		if ( isset( $this->settings['set'][ $id ] ) ) {
			return apply_filters( 'typer_get_setting', $this->settings['set'][ $id ]['default'], $id );
		}

		return '';
	}

	/**
	 * Add option to theme options array
	 *
	 * @param string $id
	 * @param array $data
	 *
	 * @return void
	 */
	public function add_setting( $id, $data ) {
		$this->settings['set'][ $id ] = $data;
	}

	/**
	 * @return bool
	 */
	public function is_normal_header() {
		if ( ! defined( 'STAX_VERSION' ) ) {
			return true;
		}

		return false;
	}

	/**
	 * @param $name
	 *
	 * @return bool|mixed
	 */
	public function get_config( $name ) {
		if ( isset( $this->config[ $name ] ) ) {
			return $this->config[ $name ];
		}

		return false;
	}

	/**
	 * @param $name
	 * @param $value
	 */
	public function set_config( $name, $value ) {
		$this->config[ $name ] = $value;
	}

	/**
	 * Is license active
	 *
	 * @param string $code
	 * @param bool $force
	 * @param bool $with_error
	 *
	 * @return \WP_Error|bool
	 */
	public function is_active( $code = '', $force = false, $with_error = false ) {
		$data = $this->verify_purchase( $code, $force );

		if ( is_wp_error( $data ) ) {
			if ( $with_error ) {
				return $data;
			}

			return false;
		}

		return is_array( $data ) && $data['active'] === true;
	}

	public function get_saved_purchased_code() {
		if ( $code = get_option( 'envato_purchase_code_' . $this->get_config( 'item_id' ), '' ) ) {
			return $code;
		}

		return '';
	}


	/**
	 * @param string $tf_code
	 * @param bool $force
	 *
	 * @return \WP_Error|array
	 */
	public function verify_purchase( $tf_code = '', $force = false ) {
		if ( $this->get_config( 'item_id' ) === '' ) {
			return new \WP_Error( 'invalid', esc_html__( 'Something is not configured properly. Please contact theme author.', 'typer' ) );
		}

		$transient = 'svq_license_' . $this->get_config( 'item_id' );

		if ( $tf_code === '' ) {
			$tf_code = get_option( 'envato_purchase_code_' . $this->get_config( 'item_id' ), '' );
		}

		if ( $tf_code && $tf_code !== '' ) {

			if ( $force ) {
				delete_transient( $transient );
			}

			if ( $license_data = get_transient( $transient ) ) {
				return $license_data;
			}

			$data = $this->get_purchase_data( $tf_code );

			if ( is_wp_error( $data ) ) {
				return $data;
			}

			if ( isset( $data->supported_until ) ) {
				$license_data                    = [];
				$license_data['active']          = true;
				$license_data['supported_until'] = $data->supported_until;

				set_transient( $transient, $license_data, 60 * 60 * 24 );

				return $license_data;
			}

			return new \WP_Error( 'invalid', esc_html__( 'Purchase code is not valid, make sure you pasted it right.', 'typer' ) );
		}

		return new \WP_Error( 'invalid', esc_html__( 'Purchase code is empty', 'typer' ) );
	}

	/**
	 * @param $code
	 *
	 * @return array|bool|mixed|object|string
	 */
	private function get_purchase_data( $code ) {
		$theme = $this->get_config( 'item_id' );

		$purchase_get = wp_remote_get( 'https://updates.seventhqueen.com/verify-purchase/?product=' . $theme . '&code=' . $code );

		// Check for error
		if ( ! is_wp_error( $purchase_get ) ) {
			$response = wp_remote_retrieve_body( $purchase_get );

			// Check for error
			if ( $response ) {
				$response = json_decode( $response );

				return $response;
			}

		}

		return $purchase_get;
	}

	/**
	 * @param $post_id
	 *
	 * @return bool
	 */
	public function has_media( $post_id = null ) {
		$format = get_post_format() ?: 'standard';

		if ( null === $post_id ) {
			$post_id = get_the_ID();
		}

		switch ( $format ) {
			case 'video':
				$video_type = get_post_meta( $post_id, '_sq_post_video_type', true );

				$video = null;

				if ( $video_type === 'hosted' ) {
					$video = get_post_meta( $post_id, '_sq_post_video_file', true );
					if ( ! wp_get_attachment_url( $video ) ) {
						return false;
					}
				} else {
					$video = get_post_meta( $post_id, '_sq_post_video_other', true );
				}

				if ( $video ) {
					return true;
				}
				break;
			case 'gallery':
				global $wpdb;

				$q = $wpdb->get_results(
					$wpdb->prepare(
						"SELECT `meta_value` FROM {$wpdb->prefix}postmeta " .
						"WHERE `post_id` = %d " .
						"AND `meta_key` LIKE %s",
						[ get_the_ID(), '_sq_post_gallery%' ]
					) );

				$gallery = [];

				foreach ( $q as $item ) {
					if ( $item->meta_value && wp_get_attachment_url( $item->meta_value ) ) {
						$gallery[] = $item->meta_value;
					}
				}

				if ( ! empty( $gallery ) ) {
					return true;
				}
				break;
			default:
				return typer()->has_post_thumbnail();
		}

		return false;
	}

	/**
	 * @param $name
	 * @param $value
	 */
	public function set_post_data( $name, $value ) {
		global $post;
		if ( ! isset ( $post->typer ) ) {
			$post->typer = [];
		}

		$post->typer[ $name ] = $value;
	}

	/**
	 * @param $name
	 * @param bool $default
	 *
	 * @return bool
	 */
	public function get_post_data( $name, $default = false ) {
		global $post;

		if ( isset( $post->typer, $post->typer[ $name ] ) ) {
			return $post->typer[ $name ];
		}

		return $default;
	}

	/**
	 * @return array
	 */
	public function get_article_components() {
		return [
			'media',
			'categories',
			'title',
			'excerpt',
			'meta'
		];
	}

	/**
	 * @param $field
	 *
	 * @return bool|mixed
	 */
	public function post_overwrite_dependency( $field ) {
		$fields = [
			'media_fullscreen_individual'                  => 'media_panel_overwrite_individual',
			'media_content_color_individual'               => 'media_panel_overwrite_individual',
			'media_panel_fade_scroll_individual'           => 'media_panel_overwrite_individual',
			'media_image_width_individual'                 => 'media_image_overwrite_individual',
			'media_image_overlay_individual'               => 'media_image_overwrite_individual',
			'media_image_background_type_individual'       => 'media_image_overwrite_individual',
			'media_image_overlay_show_individual'          => 'media_image_overwrite_individual',
			'media_background_type_individual'             => 'media_image_overwrite_individual',
			'media_video_width_individual'                 => 'media_video_overwrite_individual',
			'media_video_overlay_individual'               => 'media_video_overwrite_individual',
			'media_video_overlay_show_individual'          => 'media_video_overwrite_individual',
			'media_gallery_width_individual'               => 'media_gallery_overwrite_individual',
			'media_gallery_item_width_individual'          => 'media_gallery_overwrite_individual',
			'media_gallery_overlay_individual'             => 'media_gallery_overwrite_individual',
			'media_gallery_overlay_show_individual'        => 'media_gallery_overwrite_individual',
			'media_catbread_type_individual'               => 'media_catbread_overwrite_individual',
			'media_catbread_animation_type_individual'     => 'media_catbread_overwrite_individual',
			'media_title_position_individual'              => 'media_title_overwrite_individual',
			'media_title_over_alignment_individual'        => 'media_title_overwrite_individual',
			'media_title_left_alignment_individual'        => 'media_title_overwrite_individual',
			'media_title_above_below_alignment_individual' => 'media_title_overwrite_individual',
			'media_title_size_individual'                  => 'media_title_overwrite_individual',
			'media_title_animation_type_individual'        => 'media_title_overwrite_individual',
			'media_meta_show_individual'                   => 'media_meta_overwrite_individual',
			'media_meta_style_individual'                  => 'media_meta_overwrite_individual',
			'media_meta_items_individual'                  => 'media_meta_overwrite_individual',
			'media_meta_animation_individual'              => 'media_meta_overwrite_individual',
			'fup_media_meta_action_individual'             => 'media_meta_overwrite_individual',
			'media_shapes_show_individual'                 => 'media_shapes_overwrite_individual',
			'blog_post_listing_thumb_position_individual'  => 'blog_post_default_listing_overwrite_individual',
			'blog_post_thumbnail_individual'               => 'blog_post_masonry_listing_overwrite_individual',
			'blog_post_thumbnail_list_big_individual'      => 'blog_post_big_listing_overwrite_individual',
			'blog_post_thumbnail_grid_individual'          => 'blog_post_grid_listing_overwrite_individual',
			'blog_show_social_box_individual'              => 'blog_social_box_overwrite_individual',
			'blog_force_social_box_top_individual'         => 'blog_social_box_overwrite_individual',
		];

		if ( isset( $fields[ $field ] ) ) {
			return $fields[ $field ];
		}

		return false;
	}

}
