<?php

/*
Plugin Name: SQ Social Share
Plugin URI: https://seventhqueen.com
Description: Social sharing for WordPress content
Author: SeventhQueen
Version: 1.4.0
Author URI: https://seventhqueen.com
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'SQSocialShare' ) ) {

	define( 'SQ_SOCIAL_VERSION', '1.4.0' );
	define( 'SQ_SOCIAL_FILE', __FILE__ );
	define( 'SQ_SOCIAL_BASE_URL', plugins_url( '/', SQ_SOCIAL_FILE ) );
	define( 'SQ_SOCIAL_BASE_PATH', plugin_dir_path( SQ_SOCIAL_FILE ) );


	/**
	 * Class SQSocialShare
	 * Main class for our sharing functionality
	 */
	class SQSocialShare {

		/**
		 * @var SQSocialShare The single instance of the class
		 * @since 1.0.0
		 */
		protected static $_instance = null;

		/**
		 * @var string Our options table entry name
		 */
		private $option_name = 'sq_social_share';

		/**
		 * @var string Our admin page slug
		 */
		private $option_slug = 'sq-social-share';

		private $first_use = false;

		private $css = [];

		public static $default_platforms = [
			'facebook'  => [
				'name' => 'Facebook'
			],
			'twitter'   => [
				'name' => 'Twitter'
			],
			'pinterest' => [
				'name' => 'Pinterest'
			],
			'linkedin'  => [
				'name' => 'LinkedIn'
			],
			'whatsapp'  => [
				'name' => 'WhatsApp'
			],
		];

		/**
		 * Private constructor so nobody else can instantiate it
		 */
		private function __construct() {
		}

		/**
		 * Main SQSocialShare Instance
		 *
		 * Ensures only one instance of SQSocialShare is loaded or can be loaded.
		 *
		 * @return SQSocialShare - Main instance
		 * @since 1.0.0
		 * @static
		 */
		public static function instance() {
			if ( self::$_instance === null ) {
				self::$_instance = new self();
				self::$_instance->load();
			}

			return self::$_instance;
		}

		/**
		 * Register actions and hooks
		 */
		public function load() {
			if ( get_option( $this->option_name ) === false ) {
				$this->first_use = true;
			}

			add_action( 'wp_enqueue_scripts', [ $this, 'load_select_share' ] );
			add_action( 'admin_menu', [ $this, 'load_admin_menu' ] );

			if ( isset( $_GET['page'] ) && $_GET['page'] === $this->option_slug ) {
				add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts_admin' ] );
			}

			if ( isset( $_POST['social_share'] ) && is_array( $_POST['social_share'] ) ) {
				add_action( 'admin_init', [ $this, 'save_options' ] );
			}

			add_action( 'typer_social_share_icons', [ $this, 'the_share_data' ], 11 );

		}

		public function get_defaults( $option = null ) {
			if ( $option === null ) {
				return false;
			}

			if ( $this->first_use === false ) {
				return [];
			}

			if ( $option === 'placement' ) {
				return [ 'after_content' ];
			}

			if ( $option === 'post_type' ) {
				return [ 'post', 'page' ];
			}

			return false;
		}

		/**
		 * Echos the share container
		 */
		public function the_share_data() {
			echo $this->get_share_data();
		}


		/**
		 * Retrieve share data to be later displayed
		 *
		 * @param array $settings
		 *
		 * @return string|mixed
		 */
		public function get_share_data( $settings = [] ) {
			if ( is_singular() && $post_types = $this->get_option( 'post_type', $this->get_defaults( 'post_type' ) ) ) {

				$current_post_type = get_post_type();
				if ( in_array( $current_post_type, $post_types, true ) ) {
					add_action( 'wp_footer', [ $this, 'front_css' ] );

					/**
					 * Array of defaults
					 */
					$defaults = array(
						'size' => $this->get_option( 'size', '24' )
					);

					/**
					 * Parse incoming $args into an array and merge it with $defaults
					 */
					extract( wp_parse_args( $settings, $defaults ) );

					$content = '';

					$using_default_platforms = false;
					$platforms               = $this->get_option( 'platforms' );
					if ( $platforms === false ) {
						$platforms               = self::$default_platforms;
						$using_default_platforms = true;
					}

					if ( ! empty( $platforms ) ) {
						//Remove the disabled platforms from the array
						foreach ( $platforms as $k => $platform ) {
							if ( ! $using_default_platforms && ! isset( $platform['enabled'] ) ) {
								unset( $platforms[ $k ] );
							}

							if ( isset( $platform['color'] ) && $platform['color'] !== '' ) {
								$this->css[ $k ] = '.social-' . $k . ' svg { fill: ' . $platform['color'] . ' }';
							}
						}

						foreach ( $platforms as $slug => $platform ) {
							ob_start();
							$this->load_platform_template( $slug );
							$content .= ob_get_clean();
						}
					}

					return $content;
				}
			}

			return '';
		}

		/**
		 * Load template for specific sharing platform
		 *
		 * @param string $platform
		 */
		public function load_platform_template( $platform ) {
			global $post;

			$data = [
				'icon'     => '',
				'url'      => '',
				'on_click' => '',
				'action'   => '',
				'extra_atts' => '',
			];

			switch ( $platform ) {
				case 'facebook':
					$data['url']      = 'http://www.facebook.com/sharer.php?u=' . get_the_permalink();
					$data['on_click'] = 'javascript:window.open(this.href,\'\', \'menubar=no,toolbar=no,resizable=yes,scrollbars=yes,height=220,width=600\');return false;';
					break;
				case 'twitter':
					$data['url']      = 'https://twitter.com/share?url=' . get_the_permalink();
					$data['on_click'] = 'javascript:window.open(this.href,\'\', \'menubar=no,toolbar=no,resizable=yes,scrollbars=yes,height=260,width=600\');return false;';
					break;
				case 'pinterest':
					$params           = [
						'media=' . ( function_exists( 'the_post_thumbnail' ) ? wp_get_attachment_url( get_post_thumbnail_id() ) : '' ),
						'description=' . strip_tags( $post->post_title )
					];
					$data['url']      = 'http://pinterest.com/pin/create/button/?url=' . get_the_permalink() . '&' . implode( '&', $params );
					$data['extra_atts'] = 'data-pin-custom="true" data-pin-do="buttonBookmark"';

					break;
				case 'linkedin':
					$data['url']      = 'https://www.linkedin.com/shareArticle?url=' . get_the_permalink();
					$data['on_click'] = 'javascript:window.open(this.href,\'\', \'menubar=no,toolbar=no,resizable=yes,scrollbars=yes,height=600,width=600\');return false;';
					break;
				case 'whatsapp':
					$url = 'https://web.whatsapp.com/send';
					if ( wp_is_mobile() ) {
						$url = 'https://api.whatsapp.com/send';
					}

					$data['url'] = $url . '?text=' . urlencode( get_the_title() . ' ' . get_the_permalink() );
					break;
				default:
					return;
			}

			$data['icon'] = $platform;
			$data         = apply_filters( 'svq_social_share_data', $data );

			if ( ! is_array( $data ) ) {
				return;
			}

			extract( $data );

			$tpl_path = $this->get_template_path( 'platform' );
			if ( $tpl_path ) {
				require $tpl_path;
			}

		}

		/**
		 * Get the template path
		 *
		 * @param $name
		 *
		 * @return string|void
		 */
		private function get_template_path( $name ) {
			if ( ! $name ) {
				return;
			}

			$name = trim( $name );

			$theme_template = locate_template( 'overrides/sq-social-share/' . $name . '.php' );

			if ( $theme_template ) {
				return $theme_template;
			}

			return trailingslashit( SQ_SOCIAL_BASE_PATH ) . 'templates/' . $name . '.php';
		}

		/**
		 * Register admin area menu entry
		 */
		function load_admin_menu() {
			add_submenu_page(
				'options-general.php',
				esc_html__( 'SQ Social Share Options', 'sq-social-share' ),
				esc_html__( 'SQ Social Share', 'sq-social-share' ),
				'manage_options',
				$this->option_slug,
				[ $this, 'settings_page' ]
			);
		}

		public function enqueue_scripts_admin() {
			wp_enqueue_style( 'wp-color-picker' );
			wp_enqueue_script( 'sq-social-share', SQ_SOCIAL_BASE_URL . 'assets/js/admin.js', array(
				'jquery',
				'jquery-ui-sortable',
				'wp-color-picker'
			), SQ_SOCIAL_VERSION, false );
		}

		/**
		 * Settings page callback
		 */
		public function settings_page() {

			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'Unauthorized user', 'sq-social-share' ) );
			}

			$value = get_option( $this->option_name );

			include SQ_SOCIAL_BASE_PATH . 'settings.php';
		}

		/**
		 * Save options page data at init hook
		 */
		public function save_options() {

			if ( ! isset( $_POST['social_nonce'] )
			     || ! wp_verify_nonce( $_POST['social_nonce'], 'social_share_options_save_action' )
			) {
				print esc_html__( 'Sorry, your nonce did not verify.', 'sq-social-share' );
				exit;
			}

			$value = $_POST['social_share'];
			update_option( $this->option_name, $value );

		}

		/**
		 * Get saved setting from options db
		 *
		 * @param string $name Option name
		 * @param string|boolean $default Default value
		 *
		 * @return bool|string|array
		 */
		private function get_option( $name, $default = false ) {
			//our array of settings
			$option = get_option( $this->option_name );

			if ( is_array( $option ) && isset( $option[ $name ] ) ) {
				return $option[ $name ];
			}

			return $default;
		}

		/**
		 * CSS for our sharing container.
		 */
		public function front_css() {
			if ( ! empty( $this->css ) ) {
				?>
                <style>
                    <?php echo implode( "\n", $this->css ) ?>
                </style>
				<?php
			}
		}

		public function load_select_share() {
			if ( $this->get_option( 'select_text' ) && get_post_type() === 'post' && is_single() ) {
				wp_register_script(
					'social-select-share',
					SQ_SOCIAL_BASE_URL . '/assets/js/social-select-share.js',
					[ 'jquery' ]
				);

				wp_enqueue_script( 'social-select-share' );

				wp_register_style(
					'social-select-share-css',
					SQ_SOCIAL_BASE_URL . '/assets/css/social-select-share.css',
					[]
				);

				wp_enqueue_style( 'social-select-share-css' );
			}
		}
	}

	//Instantiate our class
	SQSocialShare::instance();

}
