<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class CommentsMedia
 */
class CommentsMedia {

	/**
	 * @var null
	 */
	public static $instance = null;

	/**
	 * @return CommentsMedia|null
	 */
	public static function instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * CommentsMedia constructor.
	 */
	public function __construct() {
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_style' ] );
		add_action( 'plugins_loaded', [ $this, 'loaded' ] );
		add_action( 'init', [ $this, 'init' ] );
		add_action( 'admin_init', [ $this, 'admin_init' ] );
	}

	public function loaded() {
		if ( isset( $_GET['delete_attachment'] ) && ! empty( $_GET['delete_attachment'] ) ) {
			if ( isset( $_GET['c'] ) && is_numeric( $_GET['c'] ) ) {
				$attachments = get_comment_meta( $_GET['c'], 'attachments' );

				foreach ( $attachments as $attachment ) {
					if ( is_numeric( $attachment ) && ! empty( $attachment ) && $attachment === $_GET['delete_attachment'] ) {
						wp_delete_attachment( $attachment, true );
						delete_comment_meta( $_GET['c'], 'attachments', $attachment );

						add_action( 'admin_notices', static function () {
							echo "<div class='updated'><p>" . esc_html__( 'Comment attachment deleted!', 'sq-comments-media' ) . "</p></div>";
						} );
					}
				}
			}
		}
	}

	/**
	 * Init admin options
	 */
	public function admin_init() {
		add_filter( 'manage_edit-comments_columns', [ $this, 'add_comment_column' ] );
		add_filter( 'manage_comments_custom_column', [ $this, 'show_comment_attachments' ], 10, 2 );
	}

	/**
	 * Add attachments column in admin comments listing
	 *
	 * @param $columns
	 *
	 * @return mixed
	 */
	public function add_comment_column( $columns ) {
		$columns['attachments'] = esc_html__( 'Attachments' );

		return $columns;
	}

	/**
	 * Show attachments in admin panel
	 *
	 * @param $column
	 * @param $comment_id
	 */
	public function show_comment_attachments( $column, $comment_id ) {
		if ( 'attachments' === $column ) {
			$attachments = get_comment_meta( $comment_id, 'attachments' );

			$attachmentsHTML = '';

			foreach ( $attachments as $attachment ) {
				if ( is_numeric( $attachment ) && ! empty( $attachment ) ) {
					$attachmentsHTML .= '<li>' .
					                    '<div>' .
					                    wp_get_attachment_image( $attachment, [
						                    '100',
						                    '100'
					                    ], '', [ 'class' => 'svq-gallery__image' ] ) .
					                    '</div>' .
					                    '<a href="' . esc_url( get_site_url() . $_SERVER['SCRIPT_NAME'] . '?c=' . $comment_id . '&delete_attachment=' . $attachment ) . '">' .
					                    esc_html__( 'Delete', 'sq-comments-media' ) .
					                    '</a>' .
					                    '</li>';
				}
			}

			if ( $attachmentsHTML ) {
				echo '<ul class="comment-attachments">' . $attachmentsHTML . '</ul>';
			} else {
				echo esc_html__( 'No attachments for this comment.', 'sq-comments-media' );
			}

		}
	}

	/**
	 * Init plugin
	 */
	public function init() {
		add_filter( 'preprocess_comment', [ $this, 'validate_attachments' ], 10, 1 );

		add_action( 'comment_form_top', [ $this, 'add_fake_form_enctype' ] );
		add_action( 'comment_form_after_fields', [ $this, 'display_media_form' ] );
		add_action( 'comment_form_logged_in_after', [ $this, 'display_media_form' ] );
		add_filter( 'comment_text', [ $this, 'display_comment_attachment' ], 400, 3 );
		add_action( 'comment_post', [ $this, 'save_attachments' ] );
		add_action( 'delete_comment', [ $this, 'delete_comment_attachments' ] );
		add_filter( 'upload_mimes', [ $this, 'get_allowed_mime_types' ], 10, 1 );
	}

	/**
	 * Enqueue scripts
	 */
	public function enqueue_scripts() {
		wp_register_script(
			'comments-media-js', CMM_ASSETS_URL . 'js/index.js',
			[
				'jquery'
			],
			false,
			true
		);

		wp_enqueue_script( 'comments-media-js' );
	}

	/**
	 * Enqueue admin styles
	 */
	public function enqueue_admin_style() {
		wp_register_style(
			'comments-media-admin-css', CMM_ASSETS_URL . 'css/comments.admin.css'
		);

		wp_enqueue_style( 'comments-media-admin-css' );
	}

	/**
	 * Add fake enctype to the comment form
	 */
	public function add_fake_form_enctype() {
		echo '</form><form action="' . esc_url( admin_url( 'admin-ajax.php' ) ) . '" method="POST" enctype="multipart/form-data" id="sqv-comment-form-media" class="comment-form svq-form" novalidate>';
	}

	/**
	 * Display upload section in the comment form
	 */
	public function display_media_form() {
		$upload_max_size = ini_get( 'upload_max_filesize' );
		$post_max_size   = ini_get( 'post_max_size' );

		echo '<p class="svq-comment--media-notes">' . esc_html__( 'Attach images - Only PNG, JPG, JPEG and GIF are supported.', 'sq-comments-media' ) . '</p>';
		echo '<div class="comment-form-notifications svq-notice file-validation" data-max-upload="' . esc_attr( (int) $post_max_size * 1000000 ) . '" data-max-file="' . esc_attr( (int) $upload_max_size * 1000000 ) . '" data-upload-message="' . esc_attr__( "You have exceeded the max post size. Please remove a file or try to add smaller files.", "sq-comments-media" ) . '" data-file-message="' . esc_attr__( "One of your files exceeds the max file size allowed by the server.", "sq-comments-media" ) . '"></div>';
		echo '<div class="svq-comment--media comment-media-form" data-template="' . esc_attr( get_template_directory_uri() ) . '">' .
		     '<div class="svq-comment--media-el">' .
		     '<div class="svq-comment--el-preview"></div>' .

		     '<div class="svq-comment--file">' .
		     '<label class="svq-comment--file-label" for="attachments"></label>' .
		     '<input class="svq-comment--file-input svq-form-input" id="attachments" name="attachments[]" type="file" />' .
		     '</div>' .

		     '<span class="svq-comment--file-add"><span class="svq-icon icon-cloud-upload icon--x24"></span></span>' .
		     '<span class="svq-comment--file-delete svq-del-file"><span class="svq-icon icon-close icon--x18"></span></span>' .
		     '</div>' .
		     '</div>';
	}

	/**
	 * Display comment attachments in the comment
	 *
	 * @param $comment_text
	 *
	 * @return string
	 */
	public function display_comment_attachment( $comment_text ) {
		$attachments = get_comment_meta( get_comment_ID(), 'attachments' );

		if ( empty( $attachments ) ) {
			return $comment_text;
		}

		$gallery        = '';
		$before_gallery = '<div class="svq-gallery-grid svq-gallery--view-' . count( $attachments ) . '">';
		$after_gallery  = '</div>';

		foreach ( $attachments as $attachment ) {

			$attachmentLink = wp_get_attachment_url( $attachment );
			$attachmentType = get_post_mime_type( $attachment );

			$lazy = '';
			if ( wp_doing_ajax() ) {
				$lazy = ' lazy lazy-is-loaded';
			}

			if ( in_array( $attachmentType, $this->get_mime_types(), true ) ) {
				$contentInner = wp_get_attachment_image(
					$attachment,
					[
						'400',
						'400'
					],
					'',
					[
						'class'           => 'svq-gallery__image' . $lazy,
						'data-object-fit' => 'cover'
					]
				);

				$gallery .= '<div class="svq-gallery__item">' .
				            '<div class="svq-gallery__image-wrapp">' .
				            '<a href="' . $attachmentLink . '"  class="svq-gallery__image-link" data-fancybox="comment-gallery-' . get_comment_ID() . '">' .
				            $contentInner .
				            '<span class="svq-img-loader"></span>' .
				            '</a></div></div>';
			}

		}

		$comment_text .= $before_gallery . $gallery . $after_gallery;

		return $comment_text;
	}

	/**
	 * Allowed file mime types for upload
	 *
	 * @return array
	 */
	public function get_mime_types() {
		$mimes = [
			'image/jpeg',
			'image/jpg',
			'image/jp_',
			'application/jpg',
			'application/x-jpg',
			'image/pjpeg',
			'image/pipeg',
			'image/vnd.swiftview-jpeg',
			'image/x-xbitmap',
			'image/gif',
			'image/x-xbitmap',
			'image/gi_',
			'image/png',
			'application/png',
			'application/x-png'
		];

		return $mimes;
	}

	/**
	 * Allowed file extensions for upload
	 *
	 * @return array
	 */
	public function get_allowed_file_extensions() {
		return [
			'png',
			'jpg',
			'jpeg',
			'gif'
		];
	}

	/**
	 * Add mime types to wp hook
	 *
	 * @param array $existing
	 *
	 * @return array
	 */
	public function get_allowed_mime_types( $existing = [] ) {
		return $existing;
	}

	/**
	 * Save attachment using wp media
	 *
	 * @param $file_handler
	 * @param $post_id
	 *
	 * @return int|WP_Error
	 */
	public function insert_attachment( $file_handler, $post_id ) {
		if ( $_FILES[ $file_handler ]['error'] !== UPLOAD_ERR_OK ) {
			return new WP_Error( 'upload_error', esc_html__( 'Upload failed for this file', 'sq-comments-media' ) );
		}

		require_once( ABSPATH . 'wp-admin' . '/includes/image.php' );
		require_once( ABSPATH . 'wp-admin' . '/includes/file.php' );
		require_once( ABSPATH . 'wp-admin' . '/includes/media.php' );

		return media_handle_upload( $file_handler, $post_id );
	}

	/**
	 * Save attachment to comment meta
	 *
	 * @param $comment_id
	 */
	public function save_attachments( $comment_id ) {
		$files = $this->orderFiles( $_FILES['attachments'] );

		foreach ( $files as $file ) {
			if ( $file['size'] > 0 ) {
				$_FILES = [ 'attachment' => $file ];

				$attachment_id = $this->insert_attachment( 'attachment', $_POST['comment_post_ID'] );

				if ( is_numeric( $attachment_id ) && ! is_wp_error( $attachment_id ) ) {
					add_comment_meta( $comment_id, 'attachments', $attachment_id );
				}
			}
		}
	}

	/**
	 * Validate uploaded files before saving them
	 *
	 * @param $data
	 *
	 * @return mixed
	 */
	public function validate_attachments( $data ) {
		$files = $this->orderFiles( $_FILES['attachments'] );

		// todo: add json responses

		foreach ( $files as $file ) {
			$error = (int) $file['error'];

			if ( $file['size'] > 0 && $error === 0 ) {

				$fileInfo      = pathinfo( $file['name'] );
				$fileExtension = strtolower( $fileInfo['extension'] );

				if ( function_exists( 'finfo_file' ) ) {
					$fileType = finfo_file( finfo_open( FILEINFO_MIME_TYPE ), $file['tmp_name'] );
				} elseif ( function_exists( 'mime_content_type' ) ) {
					$fileType = mime_content_type( $file['tmp_name'] );
				} else {
					$fileType = $file['type'];
				}

				if ( $file['size'] > ( $this->get_maximum_upload_file_size() * 1048576 || ! in_array( $fileType, $this->get_mime_types(), true ) || ! in_array( strtolower( $fileExtension ), $this->get_allowed_file_extensions(), true ) ) ) {
					wp_die( __( '<strong>ERROR:</strong> The uploaded file exceeds the upload_max_filesize directive in php.ini.', 'sq-comments-media' ) );
				}

			} elseif ( $error === 1 ) {
				wp_die( __( '<strong>ERROR:</strong> The uploaded file exceeds the upload_max_filesize directive in php.ini.', 'sq-comments-media' ) );
			} elseif ( $error === 2 ) {
				wp_die( __( '<strong>ERROR:</strong> The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form.', 'sq-comments-media' ) );
			} elseif ( $error === 3 ) {
				wp_die( __( '<strong>ERROR:</strong> The uploaded file was only partially uploaded. Please try again later.', 'sq-comments-media' ) );
			} elseif ( $error === 6 ) {
				wp_die( __( '<strong>ERROR:</strong> Missing a temporary folder.', 'sq-comments-media' ) );
			} elseif ( $error === 7 ) {
				wp_die( __( '<strong>ERROR:</strong> Failed to write file to disk.', 'sq-comments-media' ) );
			} elseif ( $error === 7 ) {
				wp_die( __( '<strong>ERROR:</strong> A PHP extension stopped the file upload.', 'sq-comments-media' ) );
			}
		}

		return $data;
	}

	/**
	 * Check if a comment has attachments
	 *
	 * @param $comment_id
	 *
	 * @return bool
	 */
	public static function has_comment_attachments( $comment_id ) {
		$attachments = get_comment_meta( $comment_id, 'attachments' );
		foreach ( $attachments as $attachment ) {
			if ( is_numeric( $attachment ) && ! empty( $attachment ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Delete comment attachments
	 *
	 * @param $comment_id
	 */
	public function delete_comment_attachments( $comment_id ) {
		$attachments = get_comment_meta( $comment_id, 'attachments' );
		foreach ( $attachments as $attachment ) {
			if ( is_numeric( $attachment ) && ! empty( $attachment ) ) {
				wp_delete_attachment( $attachment, true );
			}
		}
	}

	/**
	 * Get max upload size from php ini
	 *
	 * @return mixed
	 */
	public function get_maximum_upload_file_size() {
		$max_upload   = (int) ( ini_get( 'upload_max_filesize' ) );
		$max_post     = (int) ( ini_get( 'post_max_size' ) );
		$memory_limit = (int) ( ini_get( 'memory_limit' ) );

		return min( $max_upload, $max_post, $memory_limit );
	}

	/**
	 * Restructure requested files
	 *
	 * @param $files
	 *
	 * @return array
	 */
	public function orderFiles( $files ) {
		$new = [];
		foreach ( $files as $key => $all ) {
			foreach ( $all as $i => $val ) {
				$new[ $i ][ $key ] = $val;
			}
		}

		return $new;
	}

}

CommentsMedia::instance();
