jQuery(document).ready(function($) {
    'use strict';

    // Currency Switcher functionality
    function initCurrencySwitcher() {
        $('[data-currency-switcher]').each(function() {
            var $container = $(this);
            var $currencyOptions = $container.find('.currency-option[data-currency]');
            
            // Handle currency option clicks
            $currencyOptions.on('click', function(e) {
                e.preventDefault();
                
                var $this = $(this);
                var currencyCode = $this.data('currency');
                
                if (currencyCode && typeof togo_currency_ajax !== 'undefined') {
                    switchCurrency(currencyCode, $container);
                }
            });
        });
    }

    // Switch currency via AJAX
    function switchCurrency(currencyCode, $container) {
        if (typeof togo_currency_ajax === 'undefined') {
            console.error('Currency AJAX data not found');
            return;
        }

        // Get manual rate data from the clicked option
        var $clickedOption = $container.find('.currency-option[data-currency="' + currencyCode + '"]');
        var exchangeRate = $clickedOption.data('exchange-rate') || 1;
        var useManualRate = true; // Always use manual rate now
        var thousandSeparator = $clickedOption.data('thousand-separator') || '.';
        var decimalSeparator = $clickedOption.data('decimal-separator') || ',';
        var rawValue = $clickedOption.data('number-of-decimals');
        var numberDecimals = rawValue !== undefined && rawValue !== null && rawValue !== '' ? parseInt(rawValue, 10) : 2;
        
        // Create formatting settings object
        var formattingSettings = {
            thousand_separator: thousandSeparator,
            decimal_separator: decimalSeparator,
            number_of_decimals: numberDecimals
        };

        // Show loading state
        $container.addClass('loading');
        
        $.ajax({
            url: togo_currency_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'togo_switch_currency',
                currency: currencyCode,
                exchange_rate: exchangeRate,
                thousand_separator: thousandSeparator,
                decimal_separator: decimalSeparator,
                number_of_decimals: numberDecimals,
                nonce: togo_currency_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    // Update current currency indicator
                    updateCurrencyIndicator($container, currencyCode);
                    
                    // Update all prices on the page with manual rate
                    updatePagePrices(currencyCode, exchangeRate, useManualRate, formattingSettings);
                    
                    // Show success message (optional)
                    showCurrencyMessage('Currency switched to ' + currencyCode, 'success');
                    
                    // Update the label if needed
                    updateCurrencyLabel($container, currencyCode);
                    window.location.reload(); // Reload the page to ensure all elements are updated
                } else {
                    showCurrencyMessage(response.data.message || 'Failed to switch currency', 'error');
                }
            },
            error: function() {
                showCurrencyMessage('Network error. Please try again.', 'error');
            },
            complete: function() {
                $container.removeClass('loading');
            }
        });
    }

    // Update currency indicator
    function updateCurrencyIndicator($container, currencyCode) {
        var $currencyOptions = $container.find('.currency-option[data-currency]');
        
        // Remove current class from all options
        $currencyOptions.removeClass('current');
        $currencyOptions.find('.current-indicator').remove();
        
        // Add current class to selected option
        var $selectedOption = $currencyOptions.filter('[data-currency="' + currencyCode + '"]');
        $selectedOption.addClass('current');
        $selectedOption.append('<span class="current-indicator">✓</span>');
    }

    // Update currency label
    function updateCurrencyLabel($container, currencyCode) {
        var $label = $container.find('.lc-button span:not(.togo-svg-icon)');
        var currentText = $label.text();
        
        // Extract the language part (before the slash) and keep it
        var parts = currentText.split(' / ');
        if (parts.length > 1) {
            $label.text(parts[0] + ' / ' + currencyCode);
        } else {
            $label.text(currencyCode);
        }
    }

    // Update all prices on the page
    function updatePagePrices(currencyCode, manualRate, useManualRate, formattingSettings) {
        // This function will be called to update all price elements
        // We'll need to implement this based on your price display structure
        
        // Example: Update elements with price data attributes
        $('[data-price]').each(function() {
            var $this = $(this);
            var originalPrice = parseFloat($this.data('price'));
            var baseCurrency = $this.data('base-currency') || 'USD';
            
            if (originalPrice && baseCurrency !== currencyCode) {
                var rate = 1;
                
                if (useManualRate && manualRate > 0) {
                    // Use manual rate
                    rate = manualRate;
                    var convertedPrice = originalPrice * rate;
                    var formattedPrice = formatPrice(convertedPrice, currencyCode, formattingSettings);
                    $this.text(formattedPrice);
                } else {
                    // Get exchange rate and convert price
                    getExchangeRate(baseCurrency, currencyCode, function(rate) {
                        if (rate > 0) {
                            var convertedPrice = originalPrice * rate;
                            var formattedPrice = formatPrice(convertedPrice, currencyCode, formattingSettings);
                            $this.text(formattedPrice);
                        }
                    });
                }
            }
        });
        
        // Update WooCommerce prices if available
        if (typeof wc_add_to_cart_params !== 'undefined') {
            updateWooCommercePrices(currencyCode, manualRate, useManualRate, formattingSettings);
        }
    }

    // Get exchange rate (fallback rates only)
    function getExchangeRate(fromCurrency, toCurrency, callback) {
        // Use fallback rates only, no API calls
        var fallbackRates = {
            'USD': {
                'EUR': 0.92, 'GBP': 0.79, 'JPY': 150.0, 'AUD': 1.52, 'CAD': 1.35,
                'CHF': 0.88, 'CNY': 7.20, 'VND': 24000.0, 'INR': 83.0, 'KRW': 1300.0,
                'SGD': 1.35, 'THB': 35.0, 'MYR': 4.70, 'IDR': 15500.0, 'PHP': 55.0,
                'BRL': 5.0, 'MXN': 17.0, 'RUB': 90.0, 'ZAR': 18.0, 'TRY': 30.0
            },
            'EUR': {
                'USD': 1.09, 'GBP': 0.86, 'JPY': 163.0, 'AUD': 1.65, 'CAD': 1.47,
                'CHF': 0.96, 'CNY': 7.85, 'VND': 26200.0, 'INR': 90.5, 'KRW': 1417.0,
                'SGD': 1.47, 'THB': 38.2, 'MYR': 5.13, 'IDR': 16900.0, 'PHP': 60.0,
                'BRL': 5.45, 'MXN': 18.5, 'RUB': 98.0, 'ZAR': 19.6, 'TRY': 32.7
            },
            'GBP': {
                'USD': 1.27, 'EUR': 1.16, 'JPY': 189.0, 'AUD': 1.92, 'CAD': 1.71,
                'CHF': 1.12, 'CNY': 9.14, 'VND': 30400.0, 'INR': 105.0, 'KRW': 1647.0,
                'SGD': 1.71, 'THB': 44.4, 'MYR': 5.96, 'IDR': 19600.0, 'PHP': 69.7,
                'BRL': 6.35, 'MXN': 21.6, 'RUB': 114.0, 'ZAR': 22.8, 'TRY': 38.0
            }
        };

        var rate = 1;
        if (fallbackRates[fromCurrency] && fallbackRates[fromCurrency][toCurrency]) {
            rate = fallbackRates[fromCurrency][toCurrency];
        } else if (fallbackRates[toCurrency] && fallbackRates[toCurrency][fromCurrency]) {
            rate = 1 / fallbackRates[toCurrency][fromCurrency];
        }
        
        callback(rate);
    }

    // Format price with currency symbol and formatting
    function formatPrice(price, currencyCode, formattingSettings) {
        var symbol = currencyCode;
        
        // Get currency symbol from WooCommerce if available
        if (typeof togo_currency_ajax !== 'undefined' && togo_currency_ajax.available_currencies) {
            // Try to get symbol from WooCommerce
            if (typeof wc_add_to_cart_params !== 'undefined' && wc_add_to_cart_params.currency_symbol) {
                symbol = wc_add_to_cart_params.currency_symbol;
            } else {
                // Use currency code as fallback
                symbol = currencyCode;
            }
        } else {
            // Fallback symbols
            var symbols = {
                'USD': '$',
                'EUR': '€',
                'GBP': '£',
                'JPY': '¥',
                'VND': '₫',
                'AUD': 'A$',
                'CAD': 'C$',
                'CHF': 'CHF',
                'CNY': '¥',
                'INR': '₹',
                'KRW': '₩',
                'SGD': 'S$',
                'THB': '฿'
            };
            symbol = symbols[currencyCode] || currencyCode;
        }
        
        // Use formatting settings if provided
        var thousandSeparator = '.';
        var decimalSeparator = ',';
        var numberOfDecimals = 2;
        
        if (formattingSettings) {
            thousandSeparator = formattingSettings.thousand_separator || '.';
            decimalSeparator = formattingSettings.decimal_separator || ',';
            numberOfDecimals = parseInt(formattingSettings.number_of_decimals) || 2;
        }
        
        // Format the number
        var formattedNumber = numberFormat(price, numberOfDecimals, decimalSeparator, thousandSeparator);
        
        return symbol + formattedNumber;
    }
    
    // Number formatting function
    function numberFormat(number, decimals, decimalSeparator, thousandSeparator) {
        var parts = number.toFixed(decimals).split('.');
        parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousandSeparator);
        return parts.join(decimalSeparator);
    }

    // Update WooCommerce prices
    function updateWooCommercePrices(currencyCode, manualRate, useManualRate, formattingSettings) {
        // This function handles WooCommerce specific price updates
        // Implementation depends on your WooCommerce setup
        
        $('.woocommerce-Price-amount').each(function() {
            var $this = $(this);
            var priceText = $this.text();
            var price = parseFloat(priceText.replace(/[^\d.-]/g, ''));
            
            if (price) {
                if (useManualRate && manualRate > 0) {
                    // Use manual rate
                    var convertedPrice = price * manualRate;
                    var formattedPrice = formatPrice(convertedPrice, currencyCode, formattingSettings);
                    $this.text(formattedPrice);
                } else {
                    // Use auto rate
                    getExchangeRate('USD', currencyCode, function(rate) {
                        if (rate > 0) {
                            var convertedPrice = price * rate;
                            var formattedPrice = formatPrice(convertedPrice, currencyCode, formattingSettings);
                            $this.text(formattedPrice);
                        }
                    });
                }
            }
        });
    }

    // Show currency message
    function showCurrencyMessage(message, type) {
        // You can customize this to show notifications
        console.log(type + ': ' + message);
        
        // Example: Show a simple alert or notification
        if (type === 'error') {
            alert('Error: ' + message);
        }
    }

    // Initialize currency switcher
    initCurrencySwitcher();

    // Re-initialize when new content is loaded (for AJAX loaded content)
    $(document).on('elementor/popup/show', function() {
        initCurrencySwitcher();
    });
});
