<?php
/**
 * License Status Banner Component
 * 
 * Reusable component that displays license status messages at the top of dashboard pages.
 * Shows one of three states: No License, Expired Support, or Expiring Soon.
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

// Get dashboard setup
$gbt_dashboard_setup = GBT_Dashboard_Setup::init();
$theme_slug = $gbt_dashboard_setup->get_theme_slug();

// Don't show for block-shop theme
if ($theme_slug === 'block-shop') {
	return;
}

// Check if initialization period is complete
if (class_exists('Theme_LI') && Theme_LI::is_init_period_completed() !== true) {
	return;
}

// Get license manager instance
$license_manager = GBT_License_Manager::get_instance();
$theme_name_gbt_dash = $gbt_dashboard_setup->get_theme_name();
$theme_version_gbt_dash = $gbt_dashboard_setup->get_theme_version();
$license_page_url = admin_url('admin.php?page=getbowtied-license');

// Get license status
$is_license_active = $license_manager->is_license_active();
$is_support_active = $license_manager->is_support_active();

// Display appropriate banner based on license status
if (!$is_license_active) {
	// No license detected
	?>
	<div class="relative overflow-hidden bg-white border border-[var(--color-wp-red)] border-l-4 border-l-[var(--color-wp-red)] shadow-sm px-8 py-6 mb-8">
		<div class="w-full">
			<div class="flex items-start gap-6">
				<div class="flex-shrink-0">
					<div class="flex h-12 w-12 items-center justify-center rounded-full bg-[var(--color-wp-red)]/10">
						<svg class="h-7 w-7 text-[var(--color-wp-red)]" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
							<path stroke-linecap="round" stroke-linejoin="round" d="M12 9v3.75m-9.303 3.376c-.866 1.5.217 3.374 1.948 3.374h14.71c1.73 0 2.813-1.874 1.948-3.374L13.949 3.378c-.866-1.5-3.032-1.5-3.898 0L2.697 16.126zM12 15.75h.007v.008H12v-.008z" />
						</svg>
					</div>
				</div>
				<div class="flex-1 min-w-0">
					<div class="flex flex-wrap items-center gap-2 mb-2">
						<span class="inline-flex items-center rounded-md bg-[var(--color-wp-red)]/10 px-2.5 py-1 text-xs font-semibold text-[var(--color-wp-red)] ring-1 ring-inset ring-[var(--color-wp-red)]/20">
							No License
						</span>
						<span class="text-gray-400">•</span>
						<span class="text-sm text-gray-600"><?php echo esc_html($theme_name_gbt_dash); ?> v<?php echo esc_html($theme_version_gbt_dash); ?></span>
					</div>
					<h3 class="text-xl font-semibold text-gray-900 mb-2">
						A license key is required to use <?php echo esc_html($theme_name_gbt_dash); ?> theme
					</h3>
					<div class="flex items-center gap-4 mb-4">
						<span class="inline-flex items-center gap-1.5 text-xs text-gray-500">
							<svg class="h-4 w-4 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
								<path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
							</svg>
							The theme cannot be updated, no support access
						</span>
					</div>
					<p class="text-sm text-gray-600 mb-4 max-w-3xl">
						Your <?php echo esc_html($theme_name_gbt_dash); ?> theme requires a valid license key. Activate your license to access built-in updates and priority support. <a href="#" class="show-license-types-help text-wp-blue hover:underline whitespace-nowrap">📋 Learn more</a>
					</p>
					<div class="flex flex-wrap items-center gap-4">
						<a href="<?php echo esc_url($license_page_url); ?>#license-area" class="inline-flex items-center gap-2 rounded-lg bg-[var(--color-wp-red)] px-5 py-2.5 text-sm font-semibold text-white shadow-sm hover:bg-[var(--color-wp-red)]/90 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-[var(--color-wp-red)] transition-colors">
							<svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
								<path stroke-linecap="round" stroke-linejoin="round" d="M15.75 5.25a3 3 0 013 3m3 0a6 6 0 01-7.029 5.912c-.563-.097-1.159.026-1.563.43L10.5 17.25H8.25v2.25H6v2.25H2.25v-2.818c0-.597.237-1.17.659-1.591l6.499-6.499c.404-.404.527-1 .43-1.563A6 6 0 1121.75 8.25z" />
							</svg>
							Activate License
						</a>
					</div>
				</div>
			</div>
		</div>
	</div>
	<?php
} elseif ($is_license_active && !$is_support_active) {
	// License active but support expired
	?>
	<div class="relative overflow-hidden bg-white border border-[var(--color-wp-red)] border-l-4 border-l-[var(--color-wp-red)] shadow-sm px-8 py-6 mb-8">
		<div class="w-full">
			<div class="flex items-start gap-6">
				<div class="flex-shrink-0">
					<div class="flex h-12 w-12 items-center justify-center rounded-full bg-[var(--color-wp-red)]/10">
						<svg class="h-7 w-7 text-[var(--color-wp-red)]" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
							<path stroke-linecap="round" stroke-linejoin="round" d="M12 9v3.75m9-.75a9 9 0 11-18 0 9 9 0 0118 0zm-9 3.75h.008v.008H12v-.008z" />
						</svg>
					</div>
				</div>
				<div class="flex-1 min-w-0">
					<div class="flex flex-wrap items-center gap-2 mb-2">
						<span class="inline-flex items-center rounded-md bg-[var(--color-wp-green)]/10 px-2.5 py-1 text-xs font-semibold text-[var(--color-wp-green)] ring-1 ring-inset ring-[var(--color-wp-green)]/20">
							License Active
						</span>
						<span class="inline-flex items-center rounded-md bg-[var(--color-wp-red)]/10 px-2.5 py-1 text-xs font-semibold text-[var(--color-wp-red)] ring-1 ring-inset ring-[var(--color-wp-red)]/20">
							Subscription Expired
						</span>
						<span class="text-gray-400">•</span>
						<span class="text-sm text-gray-600"><?php echo esc_html($theme_name_gbt_dash); ?> v<?php echo esc_html($theme_version_gbt_dash); ?></span>
					</div>
					<h3 class="text-xl font-semibold text-gray-900 mb-2">
						Your subscription for <?php echo esc_html($theme_name_gbt_dash); ?> theme has expired
					</h3>
					<div class="flex items-center gap-4 mb-4">
						<span class="inline-flex items-center gap-1.5 text-xs text-gray-500">
							<svg class="h-4 w-4 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
								<path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
							</svg>
							Built-in updates disabled, no support access
						</span>
					</div>
					<p class="text-sm text-gray-600 mb-4 max-w-3xl">
						Your subscription has ended. Renew now to restore access to built-in updates, security patches, and priority support for your <?php echo esc_html($theme_name_gbt_dash); ?> theme. <a href="#" class="show-license-types-help text-wp-blue hover:underline whitespace-nowrap">📋 Learn more</a>
					</p>
					<div class="flex flex-wrap items-center gap-4">
						<a href="<?php echo esc_url($license_page_url); ?>#license-options" class="inline-flex items-center gap-2 rounded-lg bg-[var(--color-wp-red)] px-5 py-2.5 text-sm font-semibold text-white shadow-sm hover:bg-[var(--color-wp-red)]/90 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-[var(--color-wp-red)] transition-colors">
							<svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
								<path stroke-linecap="round" stroke-linejoin="round" d="M16.023 9.348h4.992v-.001M2.985 19.644v-4.992m0 0h4.992m-4.993 0l3.181 3.183a8.25 8.25 0 0013.803-3.7M4.031 9.865a8.25 8.25 0 0113.803-3.7l3.181 3.182m0-4.991v4.99" />
							</svg>
							Renew Subscription
						</a>
					</div>
				</div>
			</div>
		</div>
	</div>
	<?php
} elseif ($is_license_active && $is_support_active && $license_manager->is_support_expiring_soon()) {
	// Support expiring soon
	$days_remaining = $license_manager->get_support_days_remaining();
	$support_expiration_date = $license_manager->get_support_expiration_date();
	$expiration_date = $support_expiration_date ? date('F j, Y', $support_expiration_date) : 'soon';
	?>
	<div class="relative overflow-hidden bg-white border border-[var(--color-wp-yellow)] border-l-4 border-l-[var(--color-wp-yellow)] shadow-sm px-8 py-6 mb-8">
		<div class="w-full">
			<div class="flex items-start gap-6">
				<div class="flex-shrink-0">
					<div class="flex h-12 w-12 items-center justify-center rounded-full bg-[var(--color-wp-yellow)]/10">
						<svg class="h-7 w-7 text-[var(--color-wp-yellow)]" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
							<path stroke-linecap="round" stroke-linejoin="round" d="M12 6v6h4.5m4.5 0a9 9 0 11-18 0 9 9 0 0118 0z" />
						</svg>
					</div>
				</div>
				<div class="flex-1 min-w-0">
					<div class="flex flex-wrap items-center gap-2 mb-2">
						<span class="inline-flex items-center rounded-md bg-[var(--color-wp-green)]/10 px-2.5 py-1 text-xs font-semibold text-[var(--color-wp-green)] ring-1 ring-inset ring-[var(--color-wp-green)]/20">
							License Active
						</span>
						<span class="inline-flex items-center rounded-md bg-[var(--color-wp-yellow)]/10 px-2.5 py-1 text-xs font-semibold text-[var(--color-wp-yellow)] ring-1 ring-inset ring-[var(--color-wp-yellow)]/20">
							<?php
							if ($days_remaining == 0) {
								echo 'Expires Today';
							} elseif ($days_remaining == 1) {
								echo 'Expires Tomorrow';
							} else {
								echo 'Expires in ' . esc_html($days_remaining) . ' Days';
							}
							?>
						</span>
						<span class="text-gray-400">•</span>
						<span class="text-sm text-gray-600"><?php echo esc_html($theme_name_gbt_dash); ?> v<?php echo esc_html($theme_version_gbt_dash); ?></span>
					</div>
					<h3 class="text-xl font-semibold text-gray-900 mb-2">
						Your subscription for <?php echo esc_html($theme_name_gbt_dash); ?> theme expiring soon
					</h3>
					<div class="flex items-center gap-4 mb-4">
						<span class="inline-flex items-center gap-1.5 text-xs text-[var(--color-wp-green)]">
							<svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
								<path stroke-linecap="round" stroke-linejoin="round" d="M9 12.75L11.25 15 15 9.75M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
							</svg>
							Built-in updates and support active until <?php echo esc_html($expiration_date); ?>
						</span>
					</div>
					<p class="text-sm text-gray-600 mb-4 max-w-3xl">
						Your subscription expires on <?php echo esc_html($expiration_date); ?>. 
						<?php
						if ($days_remaining == 0) {
							echo 'Renew today to maintain continuous access to built-in updates and support.';
						} elseif ($days_remaining == 1) {
							echo 'Renew before tomorrow to avoid losing access to built-in updates and support.';
						} else {
							echo 'Renew now to maintain uninterrupted access to built-in updates and support.';
						}
						?> <a href="#" class="show-license-types-help text-wp-blue hover:underline whitespace-nowrap">📋 Learn more</a>
					</p>
					<div class="flex flex-wrap items-center gap-4">
						<a href="<?php echo esc_url($license_page_url); ?>#license-options" class="inline-flex items-center gap-2 rounded-lg bg-[var(--color-wp-yellow)] px-5 py-2.5 text-sm font-semibold text-white shadow-sm hover:bg-[var(--color-wp-yellow)]/90 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-[var(--color-wp-yellow)] transition-colors">
							<svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke-width="2" stroke="currentColor">
								<path stroke-linecap="round" stroke-linejoin="round" d="M16.023 9.348h4.992v-.001M2.985 19.644v-4.992m0 0h4.992m-4.993 0l3.181 3.183a8.25 8.25 0 0013.803-3.7M4.031 9.865a8.25 8.25 0 0113.803-3.7l3.181 3.182m0-4.991v4.99" />
							</svg>
							Renew Now
						</a>
					</div>
				</div>
			</div>
		</div>
	</div>
	<?php
}
