<?php

if ( ! function_exists( 'marra_is_installed' ) ) {
	/**
	 * Function that checks if forward plugin installed
	 *
	 * @param string $plugin - plugin name
	 *
	 * @return bool
	 */
	function marra_is_installed( $plugin ) {
		
		switch ( $plugin ) {
			case 'framework';
				return class_exists( 'QodeFramework' );
				break;
			case 'core';
				return class_exists( 'MarraCore' );
				break;
			case 'woocommerce';
				return class_exists( 'WooCommerce' );
				break;
			case 'gutenberg-page';
				$current_screen = function_exists( 'get_current_screen' ) ? get_current_screen() : array();
				
				return method_exists( $current_screen, 'is_block_editor' ) && $current_screen->is_block_editor();
				break;
			case 'gutenberg-editor':
				return class_exists( 'WP_Block_Type' );
				break;
			default:
				return false;
		}
	}
}

if ( ! function_exists( 'marra_include_theme_is_installed' ) ) {
	/**
	 * Function that set case is installed element for framework functionality
	 *
	 * @param bool   $installed
	 * @param string $plugin - plugin name
	 *
	 * @return bool
	 */
	function marra_include_theme_is_installed( $installed, $plugin ) {
		
		if ( $plugin === 'theme' ) {
			return class_exists( 'MarraHandler' );
		}
		
		return $installed;
	}
	
	add_filter( 'qode_framework_filter_is_plugin_installed', 'marra_include_theme_is_installed', 10, 2 );
}

if ( ! function_exists( 'marra_template_part' ) ) {
	/**
	 * Function that echo module template part.
	 *
	 * @param string $module name of the module from inc folder
	 * @param string $template full path of the template to load
	 * @param string $slug
	 * @param array  $params array of parameters to pass to template
	 */
	function marra_template_part( $module, $template, $slug = '', $params = array() ) {
		echo marra_get_template_part( $module, $template, $slug, $params );
	}
}

if ( ! function_exists( 'marra_get_template_part' ) ) {
	/**
	 * Function that load module template part.
	 *
	 * @param string $module name of the module from inc folder
	 * @param string $template full path of the template to load
	 * @param string $slug
	 * @param array  $params array of parameters to pass to template
	 *
	 * @return string - string containing html of template
	 */
	function marra_get_template_part( $module, $template, $slug = '', $params = array() ) {
		//HTML Content from template
		$html          = '';
		$template_path = MARRA_INC_ROOT_DIR . '/' . $module;
		
		$temp = $template_path . '/' . $template;
		if ( is_array( $params ) && count( $params ) ) {
			extract( $params );
		}
		
		$template = '';
		
		if ( ! empty( $temp ) ) {
			if ( ! empty( $slug ) ) {
				$template = "{$temp}-{$slug}.php";
				
				if ( ! file_exists( $template ) ) {
					$template = $temp . '.php';
				}
			} else {
				$template = $temp . '.php';
			}
		}
		
		if ( $template ) {
			ob_start();
			include( $template );
			$html = ob_get_clean();
		}
		
		return $html;
	}
}

if ( ! function_exists( 'marra_get_page_id' ) ) {
	/**
	 * Function that returns current page id
	 * Additional conditional is to check if current page is any wp archive page (archive, category, tag, date etc.) and returns -1
	 *
	 * @return int
	 */
	function marra_get_page_id() {
		$page_id = get_queried_object_id();
		
		if ( marra_is_wp_template() ) {
			$page_id = - 1;
		}
		
		return apply_filters( 'marra_filter_page_id', $page_id );
	}
}

if ( ! function_exists( 'marra_is_wp_template' ) ) {
	/**
	 * Function that checks if current page default wp page
	 *
	 * @return bool
	 */
	function marra_is_wp_template() {
		return is_archive() || is_search() || is_404() || ( is_front_page() && is_home() );
	}
}

if ( ! function_exists( 'marra_get_ajax_status' ) ) {
	/**
	 * Function that return status from ajax functions
	 *
	 * @param string       $status - success or error
	 * @param string       $message - ajax message value
	 * @param string|array $data - returned value
	 * @param string       $redirect - url address
	 */
	function marra_get_ajax_status( $status, $message, $data = null, $redirect = '' ) {
		$response = array(
			'status'   => esc_attr( $status ),
			'message'  => esc_html( $message ),
			'data'     => $data,
			'redirect' => ! empty( $redirect ) ? esc_url( $redirect ) : '',
		);
		
		$output = json_encode( $response );
		
		exit( $output );
	}
}

if ( ! function_exists( 'marra_get_icon' ) ) {
	/**
	 * Function that return icon html
	 *
	 * @param string $icon - icon class name
	 * @param string $icon_pack - icon pack name
	 * @param string $backup_text - backup text label if framework is not installed
	 * @param array  $params - icon parameters
	 *
	 * @return string|mixed
	 */
	function marra_get_icon( $icon, $icon_pack, $backup_text, $params = array() ) {
		$value = marra_is_installed( 'framework' ) && marra_is_installed( 'core' ) ? qode_framework_icons()->render_icon( $icon, $icon_pack, $params ) : $backup_text;
		
		return $value;
	}
}

if ( ! function_exists( 'marra_render_icon' ) ) {
	/**
	 * Function that render icon html
	 *
	 * @param string $icon - icon class name
	 * @param string $icon_pack - icon pack name
	 * @param string $backup_text - backup text label if framework is not installed
	 * @param array  $params - icon parameters
	 */
	function marra_render_icon( $icon, $icon_pack, $backup_text, $params = array() ) {
		echo marra_get_icon( $icon, $icon_pack, $backup_text, $params );
	}
}

if ( ! function_exists( 'marra_get_button_element' ) ) {
	/**
	 * Function that returns button with provided params
	 *
	 * @param array $params - array of parameters
	 *
	 * @return string - string representing button html
	 */
	function marra_get_button_element( $params ) {
		if ( class_exists( 'MarraCoreButtonShortcode' ) ) {
			return MarraCoreButtonShortcode::call_shortcode( $params );
		} else {
			$link   = isset( $params['link'] ) ? $params['link'] : '#';
			$target = isset( $params['target'] ) ? $params['target'] : '_self';
			$text   = isset( $params['text'] ) ? $params['text'] : '';
			
			return '<a itemprop="url" class="qodef-theme-button" href="' . esc_url( $link ) . '" target="' . esc_attr( $target ) . '">' . esc_html( $text ) . '</a>';
		}
	}
}

if ( ! function_exists( 'marra_render_button_element' ) ) {
	/**
	 * Function that render button with provided params
	 *
	 * @param array $params - array of parameters
	 */
	function marra_render_button_element( $params ) {
		echo marra_get_button_element( $params );
	}
}

if ( ! function_exists( 'marra_class_attribute' ) ) {
	/**
	 * Function that render class attribute
	 *
	 * @param string|array $class
	 */
	function marra_class_attribute( $class ) {
		echo marra_get_class_attribute( $class );
	}
}

if ( ! function_exists( 'marra_get_class_attribute' ) ) {
	/**
	 * Function that return class attribute
	 *
	 * @param string|array $class
	 *
	 * @return string|mixed
	 */
	function marra_get_class_attribute( $class ) {
		$value = marra_is_installed( 'framework' ) ? qode_framework_get_class_attribute( $class ) : '';
		
		return $value;
	}
}

if ( ! function_exists( 'marra_get_post_value_through_levels' ) ) {
	/**
	 * Function that returns meta value if exists
	 *
	 * @param string $name name of option
	 * @param int    $post_id id of
	 *
	 * @return string value of option
	 */
	function marra_get_post_value_through_levels( $name, $post_id = null ) {
		return marra_is_installed( 'framework' ) && marra_is_installed( 'core' ) ? marra_core_get_post_value_through_levels( $name, $post_id ) : '';
	}
}

if ( ! function_exists( 'marra_get_space_value' ) ) {
	/**
	 * Function that returns spacing value based on selected option
	 *
	 * @param string $text_value - textual value of spacing
	 *
	 * @return int
	 */
	function marra_get_space_value( $text_value ) {
		return marra_is_installed( 'core' ) ? marra_core_get_space_value( $text_value ) : 0;
	}
}

if ( ! function_exists( 'marra_wp_kses_html' ) ) {
	/**
	 * Function that does escaping of specific html.
	 * It uses wp_kses function with predefined attributes array.
	 *
	 * @param string $type - type of html element
	 * @param string $content - string to escape
	 *
	 * @return string escaped output
	 * @see wp_kses()
	 *
	 */
	function marra_wp_kses_html( $type, $content ) {
		return marra_is_installed( 'framework' ) ? qode_framework_wp_kses_html( $type, $content ) : $content;
	}
}

if ( ! function_exists( 'marra_render_svg_icon' ) ) {
	/**
	 * Function that print svg html icon
	 *
	 * @param string $name - icon name
	 * @param string $class_name - custom html tag class name
	 */
	function marra_render_svg_icon( $name, $class_name = '' ) {
		echo marra_get_svg_icon( $name, $class_name );
	}
}

if ( ! function_exists( 'marra_get_svg_icon' ) ) {
	/**
	 * Returns svg html
	 *
	 * @param string $name - icon name
	 * @param string $class_name - custom html tag class name
	 *
	 * @return string|html
	 */
	function marra_get_svg_icon( $name, $class_name = '' ) {
		$html  = '';
		$class = isset( $class_name ) && ! empty( $class_name ) ? 'class="' . esc_attr( $class_name ) . '"' : '';
		
		switch ( $name ) {
			case 'menu':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="28px" height="17px" viewBox="0 0 28 17" enable-background="new 0 0 28 17" xml:space="preserve"><rect x="3.5" y="15" width="21" height="1"/><rect x="3.5" y="1" width="21" height="1"/><rect x="1" y="8" width="26" height="1"/></svg>';
				break;
			case 'search':
				$html = '<svg ' . $class . ' xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" width="20" height="20" viewBox="0 0 20 20"><path d="M18.869 19.162l-5.943-6.484c1.339-1.401 2.075-3.233 2.075-5.178 0-2.003-0.78-3.887-2.197-5.303s-3.3-2.197-5.303-2.197-3.887 0.78-5.303 2.197-2.197 3.3-2.197 5.303 0.78 3.887 2.197 5.303 3.3 2.197 5.303 2.197c1.726 0 3.362-0.579 4.688-1.645l5.943 6.483c0.099 0.108 0.233 0.162 0.369 0.162 0.121 0 0.242-0.043 0.338-0.131 0.204-0.187 0.217-0.503 0.031-0.706zM1 7.5c0-3.584 2.916-6.5 6.5-6.5s6.5 2.916 6.5 6.5-2.916 6.5-6.5 6.5-6.5-2.916-6.5-6.5z"></path></svg>';
				break;
		}
		
		return $html;
	}
}

if ( ! function_exists( 'marra_modify_wp_kses_svg_atts' ) ) {
	/**
	 * Function that modifies array of allowed attributes for svg content
	 *
	 * @param array $atts
	 *
	 * @return bool
	 */
	function marra_modify_wp_kses_svg_atts( $atts ) {
		
		$atts['polyline'] = array(
			'fill'              => true,
			'stroke'            => true,
			'stroke-miterlimit' => true,
			'points'            => true,
		
		);
		
		return $atts;
	}
	
	add_filter( 'qode_framework_filter_wp_kses_svg_atts', 'marra_modify_wp_kses_svg_atts' );
}

if ( ! function_exists( 'marra_get_post_image' ) ) {
	/**
	 * Function that generates thumbnail img tag for posts
	 *
	 * @param int $attachment_id
	 *
	 * @return string generated img tag
	 *
	 * @see qode_framework_generate_thumbnail()
	 */
	function marra_get_post_image( $attachment_id = 0 ) {

		if ( ! empty( $attachment_id ) ) {
			$html = wp_get_attachment_image( $attachment_id, 'full' );
		} else {
			$html = get_the_post_thumbnail( get_the_ID(), 'full' );
		}

		return apply_filters( 'marra_filter_post_image', $html, $attachment_id );
	}
}

if ( ! function_exists( 'marra_left_arrow_svg' ) ) {
	function marra_left_arrow_svg() {
		
		$html = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="38px" height="68px" viewBox="0 0 38 68" enable-background="new 0 0 38 68" xml:space="preserve">
					<polyline fill="none" stroke="#000000" stroke-miterlimit="10" points="35.5,1 2.5,34 35.5,67 "/>
				</svg>';
		
		return $html;
	}
}

if ( ! function_exists( 'marra_right_arrow_svg' ) ) {
	function marra_right_arrow_svg() {
		
		$html = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="38px" height="68px" viewBox="0 0 38 68" enable-background="new 0 0 38 68" xml:space="preserve">
					<polyline fill="none" stroke="#000000" stroke-miterlimit="10" points="2.5,1 35.5,34 2.5,67 "/>
				</svg>';
		
		return $html;
	}
}

if ( ! function_exists( 'marra_search_icon_svg' ) ) {
	function marra_search_icon_svg() {

		$html = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="20px" height="20px" viewBox="0 0 20 20" enable-background="new 0 0 20 20" xml:space="preserve">
					<g>
						<circle fill="none" stroke="#000000" stroke-width="1.2" stroke-miterlimit="10" cx="8.272" cy="8.814" r="6.87"/>
						<line fill="none" stroke="#000000" stroke-width="1.2" stroke-miterlimit="10" x1="18.402" y1="18.944" x2="13.13" y2="13.673"/>
					</g>
				</svg>';

		return $html;
	}
}

if ( ! function_exists( 'marra_quote_mark_svg' ) ) {
	function marra_quote_mark_svg() {

		$html = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="13px" height="11px" viewBox="0 0 13 11" enable-background="new 0 0 13 11" xml:space="preserve">
					<g>
						<path d="M2.797,10.192H0.911l2.116-9.384h3.45L2.797,10.192z M8.409,10.192H6.523l2.07-9.384h3.496L8.409,10.192z"/>
					</g>
				</svg>
';

		return $html;
	}
}

if ( ! function_exists( 'marra_link_mark_svg' ) ) {
	function marra_link_mark_svg() {

		$html = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" width="20px" height="26px" viewBox="0 0 20 26" enable-background="new 0 0 20 26" xml:space="preserve">
					<g>
						<path fill="#010101" d="M3.701,4.895c-0.456,0.456-0.684,1.009-0.684,1.66c0,0.651,0.228,1.205,0.684,1.66l8.398,8.398 c-1.758,0.456-3.256,0.049-4.492-1.221L2.09,9.875c-0.944-0.911-1.416-2.01-1.416-3.296c0-1.286,0.472-2.4,1.416-3.345l1.221-1.221 c0.911-0.911,2.018-1.367,3.32-1.367c1.302,0,2.408,0.456,3.32,1.367l5.518,5.518c1.27,1.27,1.66,2.783,1.172,4.541l-8.35-8.398 C7.835,3.218,7.281,2.99,6.631,2.99c-0.651,0-1.205,0.228-1.66,0.684L3.701,4.895z M4.531,18.469 c-1.27-1.27-1.66-2.783-1.172-4.541l8.35,8.398c0.455,0.456,1.009,0.684,1.66,0.684c0.65,0,1.204-0.228,1.66-0.684l1.27-1.221 c0.455-0.455,0.684-1.009,0.684-1.66c0-0.65-0.229-1.204-0.684-1.66L7.9,9.387c1.758-0.456,3.255-0.049,4.492,1.221l5.518,5.518 c0.943,0.912,1.416,2.01,1.416,3.296s-0.473,2.401-1.416,3.345l-1.221,1.221c-0.912,0.911-2.019,1.367-3.32,1.367 c-1.303,0-2.409-0.456-3.32-1.367L4.531,18.469z"/>
					</g>
				</svg>
';

		return $html;
	}
}

if ( ! function_exists( 'marra_cart_icon_svg' ) ) {
	function marra_cart_icon_svg() {
		
		$html = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
					 width="20px" height="20px" viewBox="0 0 20 20" enable-background="new 0 0 20 20" xml:space="preserve">
					<g>
						<rect x="1.281" y="9.014" fill="none" stroke="#FFFFFF" stroke-width="1.2" stroke-miterlimit="10" width="17.438" height="9.486"/>
						<path fill="none" stroke="#FFFFFF" stroke-width="1.2" stroke-miterlimit="10" d="M16.147,6.019C15.326,3.4,12.889,1.5,10,1.5
							S4.674,3.4,3.854,6.019"/>
					</g>
				</svg>';
		
		return $html;
	}
}
