<?php
/**
 * Utility Helper Class
 * 
 * Contains mathematical calculations and utility functions
 * 
 * @package Listeo_AI_Search
 * @since 1.0.5
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Search_Utility_Helper {
    
    /**
     * Transform similarity score to user-friendly percentage
     * 
     * @param float $raw_similarity Raw cosine similarity score
     * @return int User-friendly percentage
     */
    public static function transform_similarity_to_percentage($raw_similarity) {
        // Cosine similarity typically ranges 0.2-0.8 for text content
        // We want to map this to a more intuitive 0-100% scale
        
        // More generous mapping for user satisfaction
        // 0.3 = 50% (suggested match)
        // 0.5 = 75% (good match)  
        // 0.7 = 90% (excellent match)
        // 0.8+ = 95%+ (near perfect)
        
        if ($raw_similarity >= 0.75) {
            // Excellent matches: 90-100%
            return 90 + (($raw_similarity - 0.75) / 0.25) * 10;
        } elseif ($raw_similarity >= 0.6) {
            // Good matches: 75-90%
            return 75 + (($raw_similarity - 0.6) / 0.15) * 15;
        } elseif ($raw_similarity >= 0.45) {
            // Decent matches: 60-75%
            return 60 + (($raw_similarity - 0.45) / 0.15) * 15;
        } elseif ($raw_similarity >= 0.3) {
            // Suggested matches: 40-60%
            return 40 + (($raw_similarity - 0.3) / 0.15) * 20;
        } else {
            // Poor matches: 0-40%
            return max(0, ($raw_similarity / 0.3) * 40);
        }
    }
    
    /**
     * Calculate cosine similarity between two vectors
     * 
     * @param array $vector1 First vector
     * @param array $vector2 Second vector
     * @return float Cosine similarity score
     */
    public static function calculate_cosine_similarity($vector1, $vector2) {
        if (count($vector1) !== count($vector2)) {
            return 0;
        }
        
        $dot_product = 0;
        $magnitude1 = 0;
        $magnitude2 = 0;
        
        for ($i = 0; $i < count($vector1); $i++) {
            $dot_product += $vector1[$i] * $vector2[$i];
            $magnitude1 += $vector1[$i] * $vector1[$i];
            $magnitude2 += $vector2[$i] * $vector2[$i];
        }
        
        $magnitude1 = sqrt($magnitude1);
        $magnitude2 = sqrt($magnitude2);
        
        if ($magnitude1 == 0 || $magnitude2 == 0) {
            return 0;
        }
        
        return $dot_product / ($magnitude1 * $magnitude2);
    }
    
    /**
     * Generate search explanation based on query and results count
     * 
     * @param string $query Search query
     * @param int $count Number of results found
     * @return string Search explanation
     */
    public static function generate_search_explanation($query, $count) {
        if ($count === 0) {
            return sprintf(__('No listings found matching "%s"', 'listeo-ai-search'), $query);
        } elseif ($count === 1) {
            return sprintf(__('Top 1 listing matching "%s"', 'listeo-ai-search'), $query);
        } else {
            return sprintf(__('Top %d listings matching "%s"', 'listeo-ai-search'), $count, $query);
        }
    }
    
    /**
     * Custom debug logging to debug_search.log
     * 
     * @param string $message Log message
     * @param string $level Log level (info, error, warning, debug)
     */
    public static function debug_log($message, $level = 'info') {
        // Only log if debug mode is enabled
        if (!get_option('listeo_ai_search_debug_mode', false)) {
            return;
        }
        
        $log_file = WP_CONTENT_DIR . '/debug_search.log';
        $timestamp = date('Y-m-d H:i:s');
        $formatted_message = "[{$timestamp}] [{$level}] Listeo AI Search: {$message}" . PHP_EOL;
        
        // Ensure the log file is writable
        if (!file_exists($log_file)) {
            touch($log_file);
        }
        
        error_log($formatted_message, 3, $log_file);
    }
}
