<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class AIRS_Admin {

    private static $_instance = null;

    public static function instance() {
        if ( is_null( self::$_instance ) ) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    private function __construct() {
        add_action( 'admin_menu', [ $this, 'add_admin_menu' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
    }

    public function enqueue_scripts( $hook ) {
        // Only load on our plugin page
        if ( 'listing_page_ai-reviews-summary' !== $hook ) {
            return;
        }
        wp_enqueue_style( 'airs-admin-styles', AIRS_PLUGIN_URL . 'assets/css/airs-styles.css', [], AIRS_VERSION );
        wp_enqueue_script( 'airs-admin-js', AIRS_PLUGIN_URL . 'assets/js/airs-admin.js', [ 'jquery' ], AIRS_VERSION, true );
        wp_localize_script( 'airs-admin-js', 'airs_ajax', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'airs_ajax_nonce' ),
        ]);
    }

    public function add_admin_menu() {
        add_submenu_page(
            'edit.php?post_type=listing',
            __( 'AI Review Highlights', 'ai-review-summary' ),
            __( 'AI Review Highlights', 'ai-review-summary' ),
            'manage_options',
            'ai-reviews-summary',
            [ $this, 'render_admin_page' ]
        );
    }

    public function register_settings() {
        register_setting( 'airs_settings', 'airs_settings' );
    }

    public function render_admin_page() {
        $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'settings';
        ?>
        <div class="wrap airs-admin-wrap">
            <div class="airs-header">
                <div class="airs-header-content">
                    <div class="airs-header-icon">
                    ✨
                    </div>
                    <div class="airs-header-text">
                        <h1><?php _e( 'AI Review Highlights for Listeo', 'ai-review-summary' ); ?></h1>
                        <p><?php _e( 'Use AI to generate rating summaries and show them on individual listing pages', 'ai-review-summary' ); ?></p>
                    </div>
                </div>
            </div>

            <nav class="airs-nav-tab-wrapper nav-tab-wrapper">
                <a href="?post_type=listing&page=ai-reviews-summary&tab=settings" 
                   class="nav-tab <?php echo $active_tab == 'settings' ? 'nav-tab-active' : ''; ?>">
                    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 15.5A3.5 3.5 0 0 1 8.5 12A3.5 3.5 0 0 1 12 8.5a3.5 3.5 0 0 1 3.5 3.5a3.5 3.5 0 0 1-3.5 3.5m7.43-2.53c.04-.32.07-.64.07-.97c0-.33-.03-.66-.07-1l2.11-1.63c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.31-.61-.22l-2.49 1c-.52-.39-1.06-.73-1.69-.98l-.37-2.65A.506.506 0 0 0 14 2h-4c-.25 0-.46.18-.5.42l-.37 2.65c-.63.25-1.17.59-1.69.98l-2.49-1c-.22-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64L4.57 11c-.04.34-.07.67-.07 1c0 .33.03.65.07.97l-2.11 1.66c-.19.15-.25.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1.01c.52.4 1.06.74 1.69.99l.37 2.65c.04.24.25.42.5.42h4c.25 0 .46-.18.5-.42l.37-2.65c.63-.26 1.17-.59 1.69-.99l2.49 1.01c.22.08.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.66Z" fill="currentColor"/>
                    </svg>
                    <?php _e( 'Settings', 'ai-review-summary' ); ?>
                </a>
                <a href="?post_type=listing&page=ai-reviews-summary&tab=manage" 
                   class="nav-tab <?php echo $active_tab == 'manage' ? 'nav-tab-active' : ''; ?>">
                   ✨
                    <?php _e( 'Generate & Manage', 'ai-review-summary' ); ?>
                </a>
            </nav>

            <?php if ( $active_tab == 'settings' ) : ?>
                <div class="airs-tab-content airs-settings-tab">
                    <?php $this->render_settings_tab(); ?>
                </div>
            <?php else : ?>
                <div class="airs-tab-content airs-manage-tab">
                    <?php $this->render_manage_tab(); ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    private function render_settings_tab() {
        ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e( 'API Configuration', 'ai-review-summary' ); ?></h3>
                <p><?php _e( 'Configure your OpenAI API settings and summary preferences.', 'ai-review-summary' ); ?></p>
            </div>
            <div class="airs-card-body">
                <form method="post" action="options.php">
                    <?php
                    settings_fields( 'airs_settings' );
                    $options = get_option( 'airs_settings' );
                    $api_key = isset( $options['openai_api_key'] ) ? esc_attr( $options['openai_api_key'] ) : '';
                    ?>
                    
                    <div class="airs-form-group">
                        <label for="openai_api_key" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12 1L3 5V11C3 16.55 6.84 21.74 12 23C17.16 21.74 21 16.55 21 11V5L12 1M12 7C13.4 7 14.8 8.6 14.8 10.5V11C15.4 11 16 11.4 16 12V16C16 16.6 15.6 17 15 17H9C8.4 17 8 16.6 8 16V12C8 11.4 8.4 11 9 11V10.5C9 8.6 10.6 7 12 7M12 8.2C11.2 8.2 10.2 9.2 10.2 10.5V11H13.8V10.5C13.8 9.2 12.8 8.2 12 8.2Z" fill="currentColor"/>
                            </svg>
                            <?php _e( 'OpenAI API Key', 'ai-review-summary' ); ?>
                        </label>
                        <input type="password" id="openai_api_key" name="airs_settings[openai_api_key]" value="<?php echo $api_key; ?>" class="airs-input" placeholder="sk-..." />
                        <div class="airs-help-text">
                            <?php _e( 'Enter your OpenAI API key.', 'ai-review-summary' ); ?>
                            <br><a href="https://docs.purethemes.net/listeo/knowledge-base/how-to-create-open-ai-api-key/" target="_blank"><?php _e( 'Get your API key →', 'ai-review-summary' ); ?></a>
                        </div>
                    </div>

                    <div class="airs-form-group">
                        <label for="openai_model" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M9 2C10.1 2 11 2.9 11 4C11 5.1 10.1 6 9 6C7.9 6 7 5.1 7 4C7 2.9 7.9 2 9 2ZM13 12C14.1 12 15 12.9 15 14C15 15.1 14.1 16 13 16C11.9 16 11 15.1 11 14C11 12.9 11.9 12 13 12ZM5 6C6.1 6 7 6.9 7 8C7 9.1 6.1 10 5 10C3.9 10 3 9.1 3 8C3 6.9 3.9 6 5 6ZM9 10C10.1 10 11 10.9 11 12C11 13.1 10.1 14 9 14C7.9 14 7 13.1 7 12C7 10.9 7.9 10 9 10ZM19 10C20.1 10 21 10.9 21 12C21 13.1 20.1 14 19 14C17.9 14 17 13.1 17 12C17 10.9 17.9 10 19 10ZM15 6C16.1 6 17 6.9 17 8C17 9.1 16.1 10 15 10C13.9 10 13 9.1 13 8C13 6.9 13.9 6 15 6Z" fill="currentColor"/>
                            </svg>
                            <?php _e( 'AI Model', 'ai-review-summary' ); ?>
                        </label>
                        <select id="openai_model" name="airs_settings[openai_model]" class="airs-input">
                            <?php
                            $selected_model = isset($options['openai_model']) ? $options['openai_model'] : 'gpt-4o';
                            $models = [
                                'gpt-4o' => 'GPT-4o (Recommended) - Best quality, higher cost',
                                'gpt-4o-mini' => 'GPT-4o-mini - Good quality, lower cost', 
                                'gpt-4-turbo' => 'GPT-4 Turbo - High quality, moderate cost',
                                'gpt-3.5-turbo' => 'GPT-3.5 Turbo - Fast and cheap, basic quality'
                            ];
                            foreach ($models as $value => $label) {
                                printf(
                                    '<option value="%s" %s>%s</option>',
                                    esc_attr($value),
                                    selected($selected_model, $value, false),
                                    esc_html($label)
                                );
                            }
                            ?>
                        </select>
                        <div class="airs-help-text">
                            <?php _e( 'Choose the AI model for generating summaries. GPT-4o provides the best quality but costs more. GPT-4o-mini offers good balance of quality and cost.', 'ai-review-summary' ); ?>
                            <br><small><strong><?php _e( 'Cost per summary (approximate):', 'ai-review-summary' ); ?></strong> GPT-4o: $0.02, GPT-4o-mini: $0.002, GPT-4 Turbo: $0.015, GPT-3.5: $0.001</small>
                        </div>
                    </div>


                    <div class="airs-form-group">
                        <label for="summary_title" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M5 4v3h5.5v12h3V7H19V4z" fill="currentColor"/>
                            </svg>
                            <?php _e( 'Summary Section Title', 'ai-review-summary' ); ?>
                        </label>
                        <input type="text" id="summary_title" name="airs_settings[summary_title]" value="<?php echo isset($options['summary_title']) ? esc_attr($options['summary_title']) : '✨ AI Review Highlights'; ?>" class="airs-input" placeholder="✨ AI Review Highlights" />
                        <div class="airs-help-text"><?php _e( 'The title that appears above the summary on listing pages. You can include emojis and HTML entities.', 'ai-review-summary' ); ?></div>
                    </div>

                    <div class="airs-form-row">
                        <div class="airs-form-group">
                            <label for="total_points" class="airs-label">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM21 9V7L15 7C14.2 7 13.5 7.2 12.9 7.6L10.9 9.2C10.3 9.6 10 10.3 10 11V21C10 21.6 10.4 22 11 22H13C13.6 22 14 21.6 14 21V15H15L18.5 21C18.7 21.4 19.1 21.6 19.5 21.6C20.3 21.6 20.8 20.8 20.5 20.1L17.8 15.4C19.1 14.2 20 12.7 20 11C20 10.4 19.6 10 19 10H18V9H21ZM5.5 9C4.7 9 4 9.7 4 10.5S4.7 12 5.5 12S7 11.3 7 10.5S6.3 9 5.5 9Z" fill="currentColor"/>
                                </svg>
                                <?php _e( 'Max Total Summary Points', 'ai-review-summary' ); ?>
                            </label>
                            <input type="number" id="total_points" name="airs_settings[total_points]" value="<?php echo isset($options['total_points']) ? esc_attr($options['total_points']) : '6'; ?>" class="airs-input airs-input-small" min="2" max="9" />
                            <div class="airs-help-text"><?php _e( 'How many total summary points to generate (positive + negative). Default: 6. Min: 2, Max: 9.', 'ai-review-summary' ); ?></div>
                        </div>

                        <div class="airs-form-group">
                            <label for="hide_threshold" class="airs-label">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M12 9C10.9 9 10 9.9 10 11S10.9 13 12 13S14 12.1 14 11S13.1 9 12 9M12 17C10.9 17 10 17.9 10 19S10.9 21 12 21S14 20.1 14 19S13.1 17 12 17M12 1C10.9 1 10 1.9 10 3S10.9 5 12 5S14 4.1 14 3S13.1 1 12 1Z" fill="currentColor"/>
                                </svg>
                                <?php _e( 'Hide if More Than X Negative Points', 'ai-review-summary' ); ?>
                            </label>
                            <input type="number" id="hide_threshold" name="airs_settings[hide_threshold]" value="<?php echo isset($options['hide_threshold']) ? esc_attr($options['hide_threshold']) : '3'; ?>" class="airs-input airs-input-small" min="0" max="9" />
                            <div class="airs-help-text"><?php _e( 'Don\'t display highlights if there are more than this many negative points. Prevents overly negative summaries from showing. Default: 3. Set to 9 to always show.', 'ai-review-summary' ); ?></div>
                        </div>

                        <div class="airs-info-box" style="max-width: 660px; background: #e8f4fd; border: 1px solid #d1ecf1; padding: 15px; margin-bottom: 20px; border-radius: 4px;">
                            <h4 style="margin: 0 0 8px 0; color: #346fb3ff;">ℹ️ Google API Limitation</h4>
                            <p style="margin: 0; color: #346fb3ff; font-size: 14px;">
                                <?php _e( 'Google Places API only shows 5 recent reviews per location. This limit can\'t be changed. If there are no native Listeo reviews and the last 5 Google reviews are negative (even if the overall rating is high, like 4.8), the AI highlight may show mostly negative results.', 'ai-review-summary' ); ?>
                            </p>
                        </div>
                    </div>

                    <div class="airs-form-group">
                        <label class="airs-checkbox-label">
                            <input type="checkbox" name="airs_settings[show_description]" <?php checked( isset($options['show_description']), true ); ?> />
                            <span class="airs-checkbox-custom"></span>
                            <span class="airs-checkbox-text">
                                <?php _e( 'Display Point Descriptions', 'ai-review-summary' ); ?>
                                <small><?php _e( 'Show the description text below each summary title on the frontend.', 'ai-review-summary' ); ?></small>
                            </span>
                        </label>
                    </div>

                    <div class="airs-form-group">
                        <label for="min_reviews" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12 17.27L18.18 21L16.54 13.97L22 9.24L14.81 8.63L12 2L9.19 8.63L2 9.24L7.46 13.97L5.82 21L12 17.27Z" fill="currentColor"/>
                            </svg>
                            <?php _e( 'Minimum Reviews Required', 'ai-review-summary' ); ?>
                        </label>
                        <input type="number" id="min_reviews" name="airs_settings[min_reviews]" value="<?php echo isset($options['min_reviews']) ? esc_attr($options['min_reviews']) : '3'; ?>" class="airs-input airs-input-small" min="1" max="5" />
                        <div class="airs-help-text"><?php _e( 'Do not generate a summary if a listing has fewer than this many combined (Google + WordPress) reviews. Default: 3. Max: 5.', 'ai-review-summary' ); ?></div>
                    </div>

                    <div class="airs-form-group">
                        <label for="summary_language" class="airs-label">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M12.87 15.07l-2.54-2.51.03-.03c1.74-1.94 2.98-4.17 3.71-6.53H17V4h-7V2H8v2H1v1.99h11.17C11.5 7.92 10.44 9.75 9 11.35 8.07 10.32 7.3 9.19 6.69 8h-2c.73 1.63 1.73 3.17 2.98 4.56l-5.09 5.02L4 19l5-5 3.11 3.11.76-2.04zM18.5 10h-2L12 22h2l1.12-3h4.75L21 22h2l-4.5-12zm-2.62 7l1.62-4.33L19.12 17h-3.24z" fill="currentColor"/>
                            </svg>
                            <?php _e( 'Summary Language', 'ai-review-summary' ); ?>
                        </label>
                        <select id="summary_language" name="airs_settings[summary_language]" class="airs-input">
                            <?php
                            $site_language = get_locale(); // Get the site's current language
                            $selected_language = isset($options['summary_language']) ? $options['summary_language'] : $site_language;
                            $installed_languages = $this->get_installed_languages();
                            // Debug: Uncomment the next line to see values
                            //  echo "<!-- Site Language: $site_language | Selected: $selected_language -->";
                            foreach ($installed_languages as $code => $name) {
                                printf(
                                    '<option value="%s" %s>%s</option>',
                                    esc_attr($code),
                                    selected($selected_language, $code, false),
                                    esc_html($name)
                                );
                            }
                            ?>
                        </select>
                        <div class="airs-help-text">
                            <?php _e( 'The language for AI-generated summaries. Only installed WordPress language packs are shown.', 'ai-review-summary' ); ?>
                            <br><small><?php printf( __( 'Site language detected: %s', 'ai-review-summary' ), '<code>' . get_locale() . '</code>' ); ?></small>
                        </div>
                    </div>

                    <div class="airs-form-actions">
                        <?php submit_button( __( 'Save Settings', 'ai-review-summary' ), 'primary airs-button-primary', 'submit', false ); ?>
                    </div>
                </form>
            </div>
        </div>
        <?php
    }

    private function render_manage_tab() {
        ?>
        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e( 'Generate & Manage Summaries', 'ai-review-summary' ); ?></h3>
                <p><?php _e( 'Select listings to generate or delete summaries. Use the search to filter the list.', 'ai-review-summary' ); ?></p>
            </div>
            <div class="airs-card-body">
                <div id="airs-batch-form">
                    <div class="airs-batch-controls">
                        <div class="airs-search-wrapper">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M15.5 14h-.79l-.28-.27A6.471 6.471 0 0 0 16 9.5A6.5 6.5 0 1 0 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5S14 7.01 14 9.5S11.99 14 9.5 14z" fill="currentColor"/>
                            </svg>
                            <input type="search" id="airs-listing-search" placeholder="<?php _e('Search listings...', 'ai-review-summary'); ?>" class="airs-search-input">
                        </div>
                        <div class="airs-control-buttons">
                            <button id="airs-select-all" class="airs-button airs-button-secondary">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M9 16.17L4.83 12l-1.42 1.41L9 19L21 7l-1.41-1.41z" fill="currentColor"/>
                                </svg>
                                <?php _e('Select All Visible', 'ai-review-summary'); ?>
                            </button>
                            <button id="airs-deselect-all" class="airs-button airs-button-secondary">
                                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M19 6.41L17.59 5L12 10.59L6.41 5L5 6.41L10.59 12L5 17.59L6.41 19L12 13.41L17.59 19L19 17.59L13.41 12z" fill="currentColor"/>
                                </svg>
                                <?php _e('Deselect All', 'ai-review-summary'); ?>
                            </button>
                        </div>
                    </div>

                    <div class="airs-listing-list">
                        <?php
                        // Fetch all listing IDs that already have a summary for efficient checking.
                        global $wpdb;
                        $table_name = $wpdb->prefix . 'ai_review_summaries';
                        $existing_summaries = $wpdb->get_results("SELECT listing_id, summary_data FROM $table_name", OBJECT_K);
                        
                        // Get hide threshold setting
                        $options = get_option('airs_settings');
                        $hide_threshold = isset($options['hide_threshold']) ? intval($options['hide_threshold']) : 3;

                        $listings = new WP_Query([
                            'post_type' => 'listing',
                            'posts_per_page' => -1,
                            'post_status' => 'publish',
                            'orderby' => 'title',
                            'order'   => 'ASC',
                        ]);

                        if ($listings->have_posts()) : while ($listings->have_posts()) : $listings->the_post();
                            $listing_id = get_the_ID();
                            $has_summary = isset($existing_summaries[$listing_id]);
                            $is_hidden = false;
                            
                            // Check if summary would be hidden due to too many negative points
                            if ($has_summary) {
                                $summary_data = json_decode($existing_summaries[$listing_id]->summary_data, true);
                                if (isset($summary_data['summaries']) && is_array($summary_data['summaries'])) {
                                    $negative_count = 0;
                                    foreach ($summary_data['summaries'] as $item) {
                                        if (isset($item['sentiment']) && $item['sentiment'] === 'negative') {
                                            $negative_count++;
                                        }
                                    }
                                    $is_hidden = ($negative_count > $hide_threshold);
                                }
                            }
                            ?>
                            <label class="airs-listing-item">
                                <input type="checkbox" name="listing_ids[]" value="<?php echo $listing_id; ?>">
                                <span class="airs-checkbox-custom"></span>
                                <div class="airs-listing-info">
                                    <span class="airs-listing-title"><?php the_title(); ?></span>
                                    <span class="airs-listing-id">ID: <?php echo $listing_id; ?></span>
                                </div>
                                <?php if ($has_summary) : ?>
                                    <?php if ($is_hidden) : ?>
                                        <span class="airs-status-badge airs-status-hidden" data-listing-id="<?php echo $listing_id; ?>">
                                            <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M12 7c2.76 0 5 2.24 5 5c0 .65-.13 1.26-.36 1.83l2.92 2.92c1.51-1.26 2.7-2.89 3.43-4.75c-1.73-4.39-6-7.5-11-7.5c-1.4 0-2.74.25-3.98.7l2.16 2.16C10.74 7.13 11.35 7 12 7zM2 4.27l2.28 2.28l.46.46C3.08 8.3 1.78 10.02 1 12c1.73 4.39 6 7.5 11 7.5c1.55 0 3.03-.3 4.38-.84l.42.42L19.73 22L21 20.73L3.27 3L2 4.27zM7.53 9.8l1.55 1.55c-.05.21-.08.43-.08.65c0 1.66 1.34 3 3 3c.22 0 .44-.03.65-.08l1.55 1.55c-.67.33-1.41.53-2.2.53c-2.76 0-5-2.24-5-5c0-.79.2-1.53.53-2.2zm4.31-.78l3.15 3.15l.02-.16c0-1.66-1.34-3-3-3l-.17.01z" fill="currentColor"/>
                                            </svg>
                                            <?php _e('Hidden (too negative)', 'ai-review-summary'); ?>
                                        </span>
                                    <?php else : ?>
                                        <span class="airs-status-badge airs-status-generated" data-listing-id="<?php echo $listing_id; ?>">
                                            <svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M9 16.17L4.83 12l-1.42 1.41L9 19L21 7l-1.41-1.41z" fill="currentColor"/>
                                            </svg>
                                            <?php _e('Generated', 'ai-review-summary'); ?>
                                        </span>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </label>
                            <?php
                        endwhile; wp_reset_postdata(); else: ?>
                            <div class="airs-empty-state">
                                <svg width="48" height="48" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                    <path d="M19 3H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm-5 14H7v-2h7v2zm3-4H7v-2h10v2zm0-4H7V7h10v2z" fill="currentColor"/>
                                </svg>
                                <p><?php _e( 'No listings found.', 'ai-review-summary' ); ?></p>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="airs-batch-actions">
                        <button id="airs-start-batch" class="airs-button airs-button-primary">
                            <?php _e( '✨ Generate Summaries for Selected', 'ai-review-summary' ); ?>
                        </button>
                        <button id="airs-delete-selected" class="airs-button airs-button-danger">
                            <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                                <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z" fill="currentColor"/>
                            </svg>
                            <?php _e( 'Delete Summaries for Selected', 'ai-review-summary' ); ?>
                        </button>
                    </div>
                </div>

                <div id="airs-progress-container" class="airs-progress-container" style="display:none;">
                    <h4><?php _e( 'Processing...', 'ai-review-summary' ); ?></h4>
                    <div class="airs-progress-bar">
                        <div class="airs-progress-bar-inner"></div>
                    </div>
                    <div class="airs-progress-text">0%</div>
                    <div id="airs-log" class="airs-log"></div>
                </div>
            </div>
        </div>

        <div class="airs-card">
            <div class="airs-card-header">
                <h3><?php _e( 'Manual Testing & Debugging', 'ai-review-summary' ); ?></h3>
                <p><?php _e( 'Enter a single Listing ID to test the review gathering and AI generation process.', 'ai-review-summary' ); ?></p>
            </div>
            <div class="airs-card-body">
                <div class="airs-form-group">
                    <label for="airs-debug-listing-id" class="airs-label">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M9.5 3A6.5 6.5 0 0 1 16 9.5c0 1.61-.59 3.09-1.56 4.23l.27.27h.79l5 5-1.5 1.5-5-5v-.79l-.27-.27A6.516 6.516 0 0 1 9.5 16A6.5 6.5 0 0 1 3 9.5A6.5 6.5 0 0 1 9.5 3m0 2C7.01 5 5 7.01 5 9.5S7.01 14 9.5 14S14 11.99 14 9.5S11.99 5 9.5 5Z" fill="currentColor"/>
                        </svg>
                        <?php _e( 'Listing ID', 'ai-review-summary' ); ?>
                    </label>
                    <input type="number" id="airs-debug-listing-id" class="airs-input airs-input-small" placeholder="<?php _e('Enter listing ID', 'ai-review-summary'); ?>">
                </div>
                
                <div class="airs-form-actions">
                    <button id="airs-test-reviews" class="airs-button airs-button-secondary">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10s10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5l1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z" fill="currentColor"/>
                        </svg>
                        <?php _e( 'Test Review Fetching', 'ai-review-summary' ); ?>
                    </button>
                </div>
                
                <div id="airs-debug-output-wrapper" class="airs-debug-output-wrapper" style="display:none;">
                    <h4><?php _e( 'Debug Output:', 'ai-review-summary' ); ?></h4>
                    <pre id="airs-debug-output" class="airs-debug-output"></pre>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Get list of installed WordPress language packs
     * @return array Associative array of language code => language name
     */
    private function get_installed_languages() {
        $languages = array();
        
        // Get current site language first
        $site_language = get_locale();
        
        // Get available translations
        $available_languages = get_available_languages();
        
        // Add current site language first (even if it's en_US)
        if ($site_language === 'en_US') {
            $languages['en_US'] = __('English (United States)', 'ai-review-summary');
        } else {
            // Load WordPress language data to get proper names
            require_once ABSPATH . 'wp-admin/includes/translation-install.php';
            $translations = wp_get_available_translations();
            
            if (isset($translations[$site_language])) {
                $languages[$site_language] = $translations[$site_language]['native_name'];
            } else {
                $languages[$site_language] = $site_language;
            }
        }
        
        // Add English if it's not the site language
        if ($site_language !== 'en_US') {
            $languages['en_US'] = __('English (United States)', 'ai-review-summary');
        }
        
        if (!empty($available_languages)) {
            // Load WordPress language data if not already loaded
            if (!isset($translations)) {
                require_once ABSPATH . 'wp-admin/includes/translation-install.php';
                $translations = wp_get_available_translations();
            }
            
            foreach ($available_languages as $locale) {
                // Skip if already added (site language or en_US)
                if (isset($languages[$locale])) {
                    continue;
                }
                
                if (isset($translations[$locale])) {
                    $languages[$locale] = $translations[$locale]['native_name'];
                } else {
                    // Fallback for languages not in translations API
                    $languages[$locale] = $locale;
                }
            }
        }
        
        return $languages;
    }
}