<?php
/**
 * Plugin Name:       Listeo Data Scraper
 * Description:       Imports listings from Google Places. Use it responsibly - heavy or improper use could lead to API limits or account suspension. We do not take responsibility for any issues that come up.
 * Version:           2.2
 * License:           GPL-2.0+
 * Text Domain:       listeo-data-scraper
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * The main plugin class.
 *
 * This class is responsible for initializing the plugin, loading dependencies,
 * and setting up all hooks. It follows the Singleton pattern to ensure that
 * it is loaded only once.
 *
 * @since 1.0.0
 */
final class Listeo_Data_Scraper {

	/**
	 * The single instance of the class.
	 *
	 * @var Listeo_Data_Scraper|null
	 */
	private static $instance = null;

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	public $version = '2.2';

	/**
	 * Ensures only one instance of the class is loaded.
	 *
	 * @return Listeo_Data_Scraper - The single instance.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Private constructor.
	 *
	 * Sets up the plugin's core functionality.
	 */
	private function __construct() {
		$this->define_constants();
		$this->setup_autoloader();
		$this->init_hooks();
	}

	/**
	 * Define plugin constants.
	 */
	private function define_constants() {
		define( 'LDS_VERSION', $this->version );
		define( 'LDS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
		define( 'LDS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
	}

	/**
	 * Set up the autoloader for our classes.
	 *
	 * This function automatically loads class files when they are needed,
	 * following a specific naming convention: class-lds-*.php
	 */
	private function setup_autoloader() {
		spl_autoload_register( function ( $class_name ) {
			// Only autoload our plugin's classes.
			if ( strpos( $class_name, 'LDS_' ) !== 0 ) {
				return;
			}

			// Convert class name from CamelCase to kebab-case for the filename.
			// Example: LDS_Admin_Menu -> class-lds-admin-menu.php
			$file_name = 'class-' . str_replace( '_', '-', strtolower( $class_name ) ) . '.php';
			$file_path = LDS_PLUGIN_DIR . 'includes/' . $file_name;

			if ( file_exists( $file_path ) ) {
				require_once $file_path;
			}
		} );
	}

	/**
	 * Initialize WordPress hooks.
	 */
	private function init_hooks() {
		// Hook to initialize plugin classes.
		add_action( 'plugins_loaded', [ $this, 'init_plugin' ] );

		// Register activation and deactivation hooks.
		register_activation_hook( __FILE__, [ $this, 'activate' ] );
		register_deactivation_hook( __FILE__, [ $this, 'deactivate' ] );
	}

	/**
	 * Instantiate plugin classes.
	 *
	 * This runs on the 'plugins_loaded' hook to ensure all dependent plugins are loaded.
	 */
	public function init_plugin() {

		// The autoloader will find and include these class files automatically.
		new LDS_Admin_Menu();
		new LDS_Settings();
		new LDS_Ajax_Handler();
	}

	/**
	 * Plugin activation hook.
	 *
	 * This runs once when the plugin is activated.
	 */
	public function activate() {
		// Placeholder for activation tasks, e.g., flushing rewrite rules.
		// flush_rewrite_rules();
	}

	/**
	 * Plugin deactivation hook.
	 *
	 * This runs once when the plugin is deactivated.
	 */
	public function deactivate() {
		// Placeholder for deactivation tasks.
	}

/**
	 * Cloning is forbidden.
	 */
	private function __clone() {}

	/**
	 * Unserializing instances of this class is forbidden.
	 */
	public function __wakeup() {
		throw new Exception('Cannot unserialize a singleton.');
	}
}

/**
 * Begins execution of the plugin.
 *
 * Since everything within the plugin is registered via hooks,
 * kicking off the plugin from this point is all that is needed.
 *
 * @return Listeo_Data_Scraper The single instance of the plugin.
 */
function listeo_data_scraper_run() {
	return Listeo_Data_Scraper::instance();
}

// Let's get this party started!
listeo_data_scraper_run();

/**
 * Helper function for logging debug information to a file.
 * This function will only write to the log if Debug Mode is enabled in the plugin settings.
 *
 * @param mixed $data The data to log. Can be a string, array, or object.
 * @param string $context A descriptive label for the log entry.
 * @param string $level The severity level of the log.
 */
function lds_log($data, $context = 'GENERAL', $level = 'INFO') {
    // --- THE CORE OF THE UPGRADE IS HERE ---
    // First, check if our debug mode setting is enabled. If not, stop immediately.
    $debug_mode_enabled = (bool) get_option('lds_enable_debug_mode', 0);
    if (!$debug_mode_enabled) {
        return; // Exit the function silently.
    }
    // Also, respect the main WordPress debugging constant as a fallback.
    if (!defined('WP_DEBUG') || !WP_DEBUG) {
        return;
    }

    // --- The rest of the function is the same as before ---
    $log_entry = [
        'timestamp' => current_time('Y-m-d H:i:s'),
        'context' => $context,
        'level' => $level,
        'memory_usage' => size_format(memory_get_usage(true)),
        'data' => $data
    ];

    if (is_array($data) || is_object($data)) {
        $data_string = print_r($data, true);
    } else {
        $data_string = (string) $data;
    }

    $log_message = sprintf(
        "[%s] [%s] [%s] Memory: %s\n%s\n%s\n",
        $log_entry['timestamp'],
        $log_entry['level'],
        $log_entry['context'],
        $log_entry['memory_usage'],
        $data_string,
        str_repeat('-', 80)
    );

    // Write to the debug log file
    error_log($log_message, 3, WP_CONTENT_DIR . '/debug-lds.log');
}

/**
 * Get gallery data for a listing, preferring local gallery over Google
 * 
 * @param int $post_id The listing ID
 * @return array Gallery data with source information
 */
function lds_get_listing_gallery($post_id) {
    // First, check for standard WordPress gallery
    $gallery = get_post_meta($post_id, '_gallery', true);
    
    if (!empty($gallery) && is_array($gallery)) {
        // Return standard gallery data
        $gallery_data = [];
        foreach ($gallery as $attachment_id => $url) {
            // Use attachment ID to get proper image URL instead of stored URL
            $image_url = wp_get_attachment_image_url($attachment_id, 'listeo-gallery');
            if ($image_url) {
                $gallery_data[] = [
                    'url' => $image_url, // Use fresh URL from attachment ID
                    'id' => $attachment_id,
                    'source' => 'wordpress',
                    'attribution' => ''
                ];
            }
        }
        return $gallery_data;
    }
    
    // If no standard gallery, check for Google gallery
    $google_gallery = get_post_meta($post_id, '_gallery_google', true);
    
    if (!empty($google_gallery) && is_array($google_gallery)) {
        $gallery_data = [];
        foreach ($google_gallery as $photo) {
            $gallery_data[] = [
                'url' => $photo['url'],
                'id' => null,
                'source' => 'google',
                'attribution' => $photo['attribution']
            ];
        }
        return $gallery_data;
    }
    
    // No gallery found
    return [];
}

/**
 * Check if listing is using Google photos
 */
function lds_is_using_google_photos($post_id) {
    $gallery = get_post_meta($post_id, '_gallery', true);
    if (!empty($gallery)) {
        return false;
    }
    
    $google_gallery = get_post_meta($post_id, '_gallery_google', true);
    return !empty($google_gallery);
}