<?php
// /includes/class-lds-settings.php

class LDS_Settings {

    public function __construct() {
        add_action('admin_init', [ $this, 'register_settings' ]);
        
        // Convert old GPT-5 selections to GPT-5-mini for better performance
        $current_model = get_option('lds_gpt_model', 'gpt-4.1-mini');
        if ($current_model === 'gpt-5') {
            update_option('lds_gpt_model', 'gpt-4.1-mini');
        }
    }

    public function register_settings() {
        // Register the setting group
        register_setting('lds_settings_group', 'lds_google_api_key', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('lds_settings_group', 'lds_import_limit', ['sanitize_callback' => 'absint']);
        register_setting('lds_settings_group', 'lds_description_language', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('lds_settings_group', 'lds_enable_photo_import', ['sanitize_callback' => 'absint']);
        register_setting('lds_settings_group', 'lds_photo_import_limit', ['sanitize_callback' => 'absint']);
        register_setting('lds_settings_group', 'lds_photo_storage_method', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('lds_settings_group', 'lds_enable_ai_descriptions', ['sanitize_callback' => 'absint']);
        register_setting('lds_settings_group', 'lds_openai_api_key', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('lds_settings_group', 'lds_gpt_model', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('lds_settings_group', 'lds_description_word_length', ['sanitize_callback' => 'absint']);
        register_setting('lds_settings_group', 'lds_enable_debug_mode', ['sanitize_callback' => 'absint']);
        register_setting('lds_settings_group', 'lds_search_method', ['sanitize_callback' => 'sanitize_text_field']);
        // Image regeneration settings
        register_setting('lds_settings_group', 'lds_image_regeneration_method', ['sanitize_callback' => 'sanitize_text_field']);
        // Map settings
        register_setting('lds_settings_group', 'lds_enable_map_mode', ['sanitize_callback' => 'absint']);
        register_setting('lds_settings_group', 'lds_default_radius', ['sanitize_callback' => 'floatval']);
        register_setting('lds_settings_group', 'lds_default_map_center_lat', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('lds_settings_group', 'lds_default_map_center_lng', ['sanitize_callback' => 'sanitize_text_field']);
        register_setting('lds_settings_group', 'lds_map_zoom_level', ['sanitize_callback' => 'absint']);
        // Removed lds_default_listing_type setting

        // Add the settings section
        add_settings_section(
            'lds_main_section',
            'API and Import Settings',
            null, // Optional callback for section description
            'lds-settings-page' // Page slug where this section will be displayed
        );

        // Add the fields
        add_settings_field(
            'lds_google_api_key',
            'Google API Key',
            [ $this, 'render_api_key_field' ],
            'lds-settings-page',
            'lds_main_section'
        );
        add_settings_field(
            'lds_openai_api_key',
            'OpenAI API Key',
            [ $this, 'render_openai_api_key_field' ],
            'lds-settings-page',
            'lds_main_section'
        );
        add_settings_field(
            'lds_import_limit',
            'Listings to Import',
            [ $this, 'render_import_limit_field' ],
            'lds-settings-page',
            'lds_main_section'
        );

        // AI Description Toggle Field
        add_settings_field(
            'lds_enable_ai_descriptions',
            'Enable AI Descriptions',
            [ $this, 'render_ai_toggle_field' ],
            'lds-settings-page',
            'lds_main_section'
        );

        // GPT Model Selection Field
        add_settings_field(
            'lds_gpt_model',
            'GPT Model',
            [ $this, 'render_gpt_model_field' ],
            'lds-settings-page',
            'lds_main_section'
        );

        // Description Word Length Field
        add_settings_field(
            'lds_description_word_length',
            'Description Length',
            [ $this, 'render_description_length_field' ],
            'lds-settings-page',
            'lds_main_section'
        );

        add_settings_field(
            'lds_description_language',
            'Description Language',
            [ $this, 'render_language_field' ],
            'lds-settings-page',
            'lds_main_section'
        );

        add_settings_field(
            'lds_enable_photo_import',
            'Enable Photo Import',
            [ $this, 'render_photo_import_section' ],
            'lds-settings-page',
            'lds_main_section'
        );

        add_settings_field(
            'lds_enable_debug_mode',
            'Enable Debug Mode',
            [ $this, 'render_debug_mode_toggle_field' ],
            'lds-settings-page',
            'lds_main_section'
        );
        
        // Map mode is enabled by default - no UI toggle needed
        // add_settings_field(
        //     'lds_enable_map_mode',
        //     'Enable Map Mode',
        //     [ $this, 'render_map_mode_toggle_field' ],
        //     'lds-settings-page',
        //     'lds_main_section'
        // );
        
        // Map settings hidden - users can set defaults directly from the map interface
        // add_settings_field(
        //     'lds_map_settings',
        //     'Map Settings',
        //     [ $this, 'render_map_settings_section' ],
        //     'lds-settings-page',
        //     'lds_main_section'
        // );
        
        add_settings_field(
            'lds_image_regeneration',
            'Image Regeneration',
            [ $this, 'render_image_regeneration_section' ],
            'lds-settings-page',
            'lds_main_section'
        );
    }

    // Render AI toggle checkbox
    public function render_ai_toggle_field() {
        $value = get_option('lds_enable_ai_descriptions', 1); // Default: enabled
        $checked = checked(1, $value, false);
        
        echo "<label>";
        echo "<input type='checkbox' name='lds_enable_ai_descriptions' id='lds_enable_ai_descriptions' value='1' {$checked} /> ";
        echo "Generate AI descriptions for imported listings";
        echo "</label>";
        echo "<p class='description'>When disabled, a simple fallback description will be used instead.</p>";
        // echo "<p class='description'><strong>Note:</strong> Disabling this will make imports much faster but descriptions will be generic.</p>";
    }

    // Render GPT model selection dropdown
    public function render_gpt_model_field() {
        $value = get_option('lds_gpt_model', 'gpt-4.1-mini'); // Default: gpt-4.1-mini
        
        echo "<div id='lds_gpt_model_container' style='margin-top: 0;'>";
        echo "<select name='lds_gpt_model'>";
        echo "<option value='gpt-4.1-mini'" . selected($value, 'gpt-4.1-mini', false) . ">GPT-4.1-mini (Fast)</option>";
        echo "<option value='gpt-5-mini'" . selected($value, 'gpt-5-mini', false) . ">GPT-5-mini (Slower, reasoning)</option>";
        echo "</select>";
        echo "<p class='description'>Choose the GPT model for generating AI descriptions. GPT-4.1-mini uses traditional parameters, while GPT-5-mini uses the latest reasoning architecture.</p>";
        echo "</div>";
        
        // JavaScript to show/hide GPT model selection based on AI descriptions checkbox
        echo "<script>
        jQuery(document).ready(function($) {
            function toggleGptModelField() {
                if ($('#lds_enable_ai_descriptions').is(':checked')) {
                    $('#lds_gpt_model_container').closest('tr').show();
                } else {
                    $('#lds_gpt_model_container').closest('tr').hide();
                }
            }
            
            // Initial state
            toggleGptModelField();
            
            // Toggle on AI descriptions checkbox change
            $('#lds_enable_ai_descriptions').on('change', function() {
                toggleGptModelField();
            });
        });
        </script>";
    }

    // Render description word length field
    public function render_description_length_field() {
        $value = get_option('lds_description_word_length', 100); // Default: 100 words
        
        echo "<div id='lds_description_length_container' style='margin-top: 0;'>";
        echo "<input type='number' name='lds_description_word_length' value='" . esc_attr($value) . "' min='50' max='500' step='10' />";
        echo "<p class='description'>Approximate number of words for AI-generated descriptions (50-500 words). This helps control description length and API costs.</p>";
        echo "</div>";
        
        // JavaScript to show/hide description length field based on AI descriptions checkbox
        echo "<script>
        jQuery(document).ready(function($) {
            function toggleDescriptionLengthField() {
                if ($('#lds_enable_ai_descriptions').is(':checked')) {
                    $('#lds_description_length_container').closest('tr').show();
                } else {
                    $('#lds_description_length_container').closest('tr').hide();
                }
            }
            
            // Initial state
            toggleDescriptionLengthField();
            
            // Toggle on AI descriptions checkbox change
            $('#lds_enable_ai_descriptions').on('change', function() {
                toggleDescriptionLengthField();
            });
        });
        </script>";
    }

    public function render_ai_cleaner_url_field() {
        $value = get_option('lds_ai_cleaner_url', '');
        $default_url = plugin_dir_url(__FILE__) . '../ai-cleaner.php';
        
        echo "<input type='url' name='lds_ai_cleaner_url' value='" . esc_attr($value) . "' class='regular-text' placeholder='" . esc_attr($default_url) . "' />";
        echo "<p class='description'>Full URL to your ai-cleaner.php file. Leave blank to use default plugin location.</p>";
        echo "<p class='description'><strong>Current default:</strong> " . esc_html($default_url) . "</p>";
        echo "<p class='description'><em>Only used when AI descriptions are enabled above.</em></p>";
    }

    // OpenAI API key field
    public function render_openai_api_key_field() {
        $value = get_option('lds_openai_api_key', '');
        echo "<input type='text' name='lds_openai_api_key' value='" . esc_attr($value) . "' class='regular-text' />";
        echo "<p class='description'>Enter your OpenAI API key. This is required for AI descriptions.</p>";
        echo "<a class='instr-btn' target='blank' href='https://docs.purethemes.net/listeo/knowledge-base/how-to-create-open-ai-api-key/'>Instructions →</a>";
    }

    public function render_api_key_field() {
        $value = get_option('lds_google_api_key', '');
        
        // Input field and test button on the same row
        echo "<div style='display: flex; align-items: center; gap: 10px; margin-bottom: 8px;'>";
        echo "<input type='text' name='lds_google_api_key' id='lds_google_api_key' value='" . esc_attr($value) . "' class='regular-text' style='flex: 1;' />";
        echo "<button type='button' id='lds_test_api_key' class='button button-secondary' style='background: #dcf2dc; color: #148a15; border: none; white-space: nowrap;'>Test API Key</button>";
        echo "</div>";
        
        // Test result span on its own line
        echo "<div style='margin-bottom: 8px;'>";
        echo "<span id='lds_api_test_result' style='font-weight: bold;'></span>";
        echo "</div>";
        
        $server_ip = $this->get_server_ip();
        echo "<p class='description'>Enter your Google Places API key <strong style='color: #222;'>(restricted to your server IP: <code style='font-weight: bold; color: #d32f2f; background:transparent;'>{$server_ip}</code>)</strong>.</p>";
         // echo "<div style='background: #e8f4f8; border: 1px solid #b8dce8; border-radius: 4px; padding: 12px; margin: 8px 0; font-size: 13px;'>";
         // echo "<strong>🔧 Required API Key Configuration:</strong><br>";
         // echo "1. Go to <a href='https://console.cloud.google.com/apis/credentials' target='_blank'>Google Cloud Console → Credentials</a><br>";
         // echo "2. Click your API key → <strong>Application restrictions</strong> → Select <strong>\"IP addresses\"</strong><br>";
         // echo "3. Add this server IP: <code style='background: #fff; padding: 2px 6px; border-radius: 3px; font-weight: bold; color: #d32f2f;'>{$server_ip}</code><br>";
         //echo "4. Under <strong>API restrictions</strong> → Enable <strong>Places API</strong> and <strong>Maps JavaScript API</strong>";
         // echo "</div>";
        echo "<a class='instr-btn' target='_blank' href='https://docs.purethemes.net/listeo/knowledge-base/creating-google-maps-api-key/'>Instructions →</a>";
        
        
        // Security notice (blue/info)

        // Quota warning (red)
        echo "<p style='margin-top: 10px; border-radius: 5px; padding: 12px 14px; font-size: 14px; line-height: 20px;background: #e3f6ff; color: #24619f;'>
        <strong>Cost Example</strong>: Importing 100 businesses typically costs $2-4 in Google API fees. Start with small batches and monitor your Google Cloud billing closely.  </p> ";

        // Quota warning (yellow)
        echo "<p style='margin-top: 10px; border-radius: 5px; padding: 12px 14px; font-size: 14px; line-height: 20px; background: #ffe0e0; color: #a71717ff;'>
        <strong>Note:</strong> We do not take responsibility for any charges from Google related to your usage. Monitor your API usage and to avoid surprise charges and <strong>set up alerts in Google Clouds</strong>. </p>";

        // WARNING notice
        echo "<div style='background: #fff3de; color: #885c0e; padding: 15px; margin: 10px 0; border-radius: 4px;'>
<strong>⚠️ Heads up!</strong> It uses Google Places data and storing that data in WordPress might go against Google’s Terms.  Use it responsibly - heavy or improper use could lead to API limits or account suspension. We do not take responsibility for any issues that come up. :)</div>";

        // Add JavaScript for the test button
        echo "<script>
        jQuery(document).ready(function($) {
            $('#lds_test_api_key').on('click', function() {
                var apiKey = $('#lds_google_api_key').val().trim();
                var button = $(this);
                var resultSpan = $('#lds_api_test_result');
                
                if (!apiKey) {
                    resultSpan.html('<span style=\"color: #d63384;\">Please enter an API key first</span>');
                    return;
                }
                
                button.prop('disabled', true).text('Testing...');
                resultSpan.html('<span style=\"color: #0d6efd;\">Testing Places API...</span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'lds_test_api_key',
                        api_key: apiKey,
                        nonce: '" . wp_create_nonce('lds_test_api_key_nonce') . "'
                    },
                    success: function(response) {
                        if (response.success) {
                            resultSpan.html('<span style=\"color: #198754;\">' + response.data.message + '</span>');
                        } else {
                            resultSpan.html('<span style=\"color: #d63384;\">✗ ' + response.data.message + '</span>');
                        }
                    },
                    error: function() {
                        resultSpan.html('<span style=\"color: #d63384;\">✗ Network error occurred</span>');
                    },
                    complete: function() {
                        button.prop('disabled', false).text('Test API Key');
                    }
                });
            });
        });
        </script>";
    }

    public function render_import_limit_field() {
        $value = get_option('lds_import_limit', 20); 
        echo "<input type='number' name='lds_import_limit' value='" . esc_attr($value) . "' min='1' max='20' />"; 
        echo "<p class='description'>Maximum 20 listings per import. Higher values may timeout on shared hosting.</p>";
        echo "<p class='description' style='color: #666; font-size: 12px;'><strong>Note:</strong> Google returns the same results for repeated searches. After importing, use a different search query to find additional listings.</p>";
    }

    public function render_debug_mode_toggle_field() {
        $value = get_option('lds_enable_debug_mode', 0); 
        $checked = checked(1, $value, false);
        
        echo "<label>";
        echo "<input type='checkbox' name='lds_enable_debug_mode' value='1' {$checked} /> ";
        echo "Enable logging for debugging purposes";
        echo "</label>";
        echo "<p class='description'>When enabled, the plugin will write detailed information to a <code>debug-lds.log</code> file in your <code>/wp-content/</code> directory.</p>";
        echo "<p class='description'><strong>Note:</strong> This should be left disabled during normal use to improve performance and save disk space.</p>";
    }

    public function render_language_field() {
        $current_lang = get_option('lds_description_language', 'site-default');
        $site_locale = get_locale();
        $site_lang_name = \Locale::getDisplayLanguage($site_locale, 'en');
        $available_locales = get_available_languages();

        ?>
        <select name="lds_description_language">
            <option value="site-default" <?php selected($current_lang, 'site-default'); ?>>
                WordPress Site Language (Currently: <?php echo esc_html($site_lang_name); ?>)
            </option>
            
            <?php
            $language_names = [];
            if (!empty($available_locales)) {
                foreach ($available_locales as $locale) {
                    $language_names[] = \Locale::getDisplayLanguage($locale, 'en');
                }
            }
            
            if ($current_lang !== 'site-default' && !in_array($current_lang, $language_names)) {
                $language_names[] = $current_lang;
            }

            $language_names = array_unique($language_names);
            sort($language_names);

            foreach ($language_names as $lang_name) {
                echo '<option value="' . esc_attr($lang_name) . '" ' . selected($current_lang, $lang_name, false) . '>';
                echo esc_html($lang_name);
                echo '</option>';
            }
            ?>
        </select>
        <p class="description">Select the language for AI-generated descriptions and Google Places data (including reviews). This list is dynamically generated from the languages installed on your WordPress site.</p>
        <p class="description"><strong>Note:</strong> This setting affects the language of reviews fetched from Google Places API.</p>
        <?php
        
        if (!extension_loaded('intl')) {
            echo '<p style="color: #e74c3c;"><strong>Warning:</strong> The `intl` PHP extension is not enabled on your server. Language name display may be limited.</p>';
        }
    }

    // New method that combines photo settings
    public function render_photo_import_section() {
        $photo_enabled = get_option('lds_enable_photo_import', 0);
        $photo_limit = get_option('lds_photo_import_limit', 0);
        $storage_method = get_option('lds_photo_storage_method', 'google');
        
        $checked = checked(1, $photo_enabled, false);
        
        // Main checkbox
        echo "<label>";
        echo "<input type='checkbox' name='lds_enable_photo_import' id='lds_enable_photo_import' value='1' {$checked} /> ";
        echo "Import photos from Google Places";
        echo "</label>";
        // Quota warning (red)
        echo "<p style='margin-top: 10px; border-radius: 5px; padding: 12px 14px; font-size: 14px; line-height: 20px;background: #fff3de; color: #885c0e;'>
            Downloading photos may violate Google's Terms of Service. Hosting them from Google via API is allowed but will be costly (~$0.05 per image request and each page load triggers it). </p> ";

        // Photo settings container (hidden when checkbox is unchecked)
        echo "<div id='lds_photo_settings' style='margin-top: 15px; padding: 15px; border: 1px solid #ddd; border-radius: 5px; background: #f9f9f9;'>";
        
        // Number of photos field
        echo "<h4 style='margin-top: 0;'>Number of Photos to Import</h4>";
        echo "<input type='number' name='lds_photo_import_limit' value='" . esc_attr($photo_limit) . "' min='0' max='5' />"; 
        echo "<p class='description'>Max number of photos to import per listing (0-5).</p>";
        echo "<p class='description'><strong>Note:</strong> Photos will increase Google API usage and cost</p>";
        
        echo "<hr style='margin: 20px 0;'>";
        
        // Photo storage method field
        echo "<h4>Photo Storage Method</h4>";
        ?>
        <fieldset>
            <label>
                <input type="radio" name="lds_photo_storage_method" value="google" <?php checked($storage_method, 'google'); ?> />
                <span><?php _e('Display from Google servers (Costly)', 'listeo_core'); ?></span>
            </label>
            <p style="margin-top: 0px; border-radius: 5px; padding: 12px 14px; font-size: 14px; line-height: 20px;background: #d9effa; color: #24619f; } ">
            This is method compliant with Google's Terms of Service. Photos are loaded directly from Google using their official API. <br><br><strong>Unfortunately this is costly because will increase API usage, as each page load triggers image requests (~$0.05 per photo viewed).</strong></p>
            <br>
            <label>
                <input type="radio" name="lds_photo_storage_method" value="download" <?php checked($storage_method, 'download'); ?> id="download-method-radio" />
                <span><?php _e('Download to media library (Risk of TOS violation)', 'listeo_core'); ?></span>
                <p style='margin-top: 5px; border-radius: 5px; padding: 12px 14px; font-size: 14px; line-height: 20px; background: #d9effa; color: #24619f;'>Photos should not be downloaded and stored on external servers due to Google's Terms of Service.</p>
            </label>
        </fieldset>
        <?php
        
        echo "</div>"; // End photo settings container
        
        // JavaScript to show/hide photo settings
        echo "<script>
        jQuery(document).ready(function($) {
            function togglePhotoSettings() {
                if ($('#lds_enable_photo_import').is(':checked')) {
                    $('#lds_photo_settings').show();
                } else {
                    $('#lds_photo_settings').hide();
                }
            }
            
            // Initial state
            togglePhotoSettings();
            
            // Toggle on checkbox change
            $('#lds_enable_photo_import').on('change', function() {
                togglePhotoSettings();
            });
        });
        </script>";
    }

    public function render_photo_limit_field() {
        $value = get_option('lds_photo_import_limit', 0); 
        echo "<input type='number' name='lds_photo_import_limit' value='" . esc_attr($value) . "' min='0' max='5' />"; 
        echo "<p class='description'>Max number of photos to import per listing (0-5).</p>";
    }

    // Add this render method to the class:
    public function render_photo_storage_method_field() {
        $value = get_option('lds_photo_storage_method', 'google');
        ?>
        <fieldset>
            <label>
                <input type="radio" name="lds_photo_storage_method" value="google" <?php checked($value, 'google'); ?> />
                <span><?php _e('Display from Google servers', 'listeo_core'); ?></span>
            </label>
        <p style="margin-top: 0px; border-radius: 5px; padding: 12px 14px; font-size: 14px; line-height: 20px; background: #fff3de; color: #885c0e; ">
This is <strong>method compliant with Google’s Terms of Service</strong>. Photos are loaded directly from Google using their official API. <br><br><strong>This will increase API usage, as each page load triggers image requests (~$0.05 per photo viewed).</strong></p>
        </p>
            <br>
            <label>
                <input type="radio" name="lds_photo_storage_method" value="download" <?php checked($value, 'download'); ?> id="download-method-radio" />
                <span><?php _e('Download to media library (Risk of TOS violation)', 'listeo_core'); ?></span>
                <p style='margin-top: 5px; border-radius: 5px; padding: 12px 14px; font-size: 14px; line-height: 20px; background: #ffe0e0; color: #a71717ff;'>Photos should not be downloaded and stored on external servers due to Google's Terms of Service.</p>
            </label>
        </fieldset>
        
        <!-- <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#download-method-radio').on('click', function(e) {
                if (!confirm('Are you sure you want to enable downloading photos to your media library?\n\nThis method violates Google\'s Terms of Service and may result in:\n• API suspension\n• Legal issues\n• Account termination\n\nWe strongly recommend using the "Display from Google servers" option instead.')) {
                    e.preventDefault();
                    // Switch back to the Google method
                    $('input[name="lds_photo_storage_method"][value="google"]').prop('checked', true);
                    return false;
                }
            });
        });
        </script> -->
        <?php
    }

    /**
     * Render image regeneration section
     */
    public function render_image_regeneration_section() {
        $regeneration_method = get_option('lds_image_regeneration_method', 'refresh_api');
        
        // Toggle button with nice styling
        ?>
        <div style="margin-bottom: 15px;">
            <div id="lds_image_regen_toggle" style="
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                padding: 8px 16px;
                border-radius: 8px;
                cursor: pointer;
                display: inline-flex;
                align-items: center;
                gap: 10px;
                font-weight: 500;
                box-shadow: 0 2px 10px rgba(102, 126, 234, 0.3);
                transition: all 0.3s ease;
                user-select: none;
            ">
                <span>Image Regeneration Settings</span>
                <span id="lds_toggle_arrow" style="font-size: 12px; transition: transform 0.3s ease;">▼</span>
            </div>
        </div>
        <?php
        
        
        ?>
        <div id="lds_image_regeneration_content" style="background: #fff; display: none;">
            <!-- Info box with improved styling -->
            <div style="
                background: linear-gradient(135deg, #e8f4fd 0%, #f0f8ff 100%);
                color: #2c5aa0;
                padding: 15px;
                margin: 0 0 15px 0;
                border-radius: 5px;
            ">
                <p style="margin: 0; line-height: 1.5;">
                    Use these tools when Google-hosted images are no longer working (API key expired/changed) or when you want to clean up image storage for your listings.
                </p>
            </div>
            
            <h4 style="color: #333; margin: 20px 0 15px 0; font-size: 16px;">Choose Regeneration Method:</h4>
            
            <fieldset style="border: none; padding: 0;">
            <label style="
                display: block; 
                margin-bottom: 15px; 
                background: #f8f9fa; 
                padding: 15px; 
                border-radius: 6px; 
                border: 2px solid #e9ecef; 
                transition: all 0.3s ease;
                cursor: pointer;
            " onmouseover="this.style.borderColor='#667eea'; this.style.backgroundColor='#f4f6ff';" 
               onmouseout="this.style.borderColor='#e9ecef'; this.style.backgroundColor='#f8f9fa';">
                <div style="display: flex; align-items: flex-start; gap: 12px;">
                    <input type="radio" name="lds_image_regeneration_method" value="refresh_api" <?php checked($regeneration_method, 'refresh_api'); ?> 
                           style="margin-top: 2px;" />
                    <div>
                        <div style="font-weight: 500; color: #333; margin-bottom: 5px;">
                            <?php _e('Refresh Google-hosted images with new API key', 'listeo-data-scraper'); ?>
                        </div>
                        <p style="margin: 0; color: #6c757d; font-size: 13px; line-height: 1.4;">
                            Regenerate Google photo URLs using the current API key. Use this when your API key has expired or changed.
                        </p>
                    </div>
                </div>
            </label>
            
            <label style="
                display: block; 
                margin-bottom: 15px; 
                background: #f8f9fa; 
                padding: 15px; 
                border-radius: 6px; 
                border: 2px solid #e9ecef; 
                transition: all 0.3s ease;
                cursor: pointer;
            " onmouseover="this.style.borderColor='#667eea'; this.style.backgroundColor='#f4f6ff';" 
               onmouseout="this.style.borderColor='#e9ecef'; this.style.backgroundColor='#f8f9fa';">
                <div style="display: flex; align-items: flex-start; gap: 12px;">
                    <input type="radio" name="lds_image_regeneration_method" value="fallback_placeholder" <?php checked($regeneration_method, 'fallback_placeholder'); ?> 
                           style="margin-top: 2px;" />
                    <div>
                        <div style="font-weight: 500; color: #333; margin-bottom: 5px;">
                            <?php _e('Remove Google images and use theme fallback placeholder', 'listeo-data-scraper'); ?>
                        </div>
                        <p style="margin: 0; color: #6c757d; font-size: 13px; line-height: 1.4;">
                            Remove all Google-hosted images and let the theme display its default placeholder image.
                        </p>
                    </div>
                </div>
            </label>
        </fieldset>
        
        <div id="lds_regeneration_actions" style="
            margin-top: 5px; 
            padding: 20px; 
            background: #ffffff; 
            border-radius: 8px; 
            border: 1px solid #e0e0e0;
            box-shadow: 0 2px 4px rgba(0,0,0,0.05);
        ">
            <button type="button" id="lds_run_image_regeneration" class="button button-primary" style="
                background: linear-gradient(135deg, #28a745 0%, #20c997 100%); 
                border: none; 
                padding: 6px 14px; 
                font-size: 14px; 
                font-weight: 500;
                border-radius: 6px;
                box-shadow: 0 2px 8px rgba(40, 167, 69, 0.3);
                transition: all 0.3s ease;
            ">
                🚀 Run Image Regeneration
            </button>
            <span id="lds_regeneration_status" style="
                margin-left: 15px; 
                font-weight: 500;
                padding: 6px 12px;
                border-radius: 4px;
                display: inline-block;
            "></span>
            <div id="lds_regeneration_progress" style="display: none; margin-top: 20px;">
                <div style="
                    background: #f8f9fa; 
                    border-radius: 6px; 
                    padding: 4px;
                    border: 1px solid #e9ecef;
                ">
                    <div id="lds_progress_bar" style="
                        background: linear-gradient(90deg, #667eea 0%, #764ba2 100%); 
                        height: 24px; 
                        border-radius: 4px; 
                        width: 0%; 
                        transition: width 0.5s ease;
                        position: relative;
                        overflow: hidden;
                    ">
                        <div style="
                            position: absolute;
                            top: 0;
                            left: -100%;
                            width: 100%;
                            height: 100%;
                            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
                            animation: shimmer 2s infinite;
                        "></div>
                    </div>
                </div>
                <p id="lds_progress_text" style="
                    margin: 10px 0 0 0; 
                    font-size: 13px; 
                    color: #6c757d;
                    font-weight: 500;
                ">Preparing...</p>
            </div>
        </div>
        
        </div> <!-- Close content div -->
        
        <?php
        
        // JavaScript for the image regeneration functionality
        ?>
        <style>
        @keyframes shimmer {
            0% { left: -100%; }
            100% { left: 100%; }
        }
        
        #lds_image_regen_toggle:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4) !important;
        }
        
        #lds_run_image_regeneration:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(40, 167, 69, 0.4) !important;
        }
        </style>
        <script>
        jQuery(document).ready(function($) {
            // Toggle functionality with improved animations
            $('#lds_image_regen_toggle').on('click', function(e) {
                e.preventDefault();
                const content = $('#lds_image_regeneration_content');
                const arrow = $('#lds_toggle_arrow');
                const icon = $('#lds_toggle_icon');
                
                if (content.is(':visible')) {
                    content.slideUp(400, 'swing');
                    arrow.css('transform', 'rotate(0deg)');
                    icon.css('transform', 'scale(1)');
                } else {
                    content.slideDown(400, 'swing');
                    arrow.css('transform', 'rotate(180deg)');
                    icon.css('transform', 'scale(1.1)');
                }
            });
            
            // Handle regeneration button click
            $('#lds_run_image_regeneration').on('click', function() {
                const selectedMethod = $('input[name="lds_image_regeneration_method"]:checked').val();
                const button = $(this);
                const statusSpan = $('#lds_regeneration_status');
                const progressDiv = $('#lds_regeneration_progress');
                const progressBar = $('#lds_progress_bar');
                const progressText = $('#lds_progress_text');
                
                // Confirm action
                let confirmMessage = 'Are you sure you want to run image regeneration?\n\n';
                if (selectedMethod === 'refresh_api') {
                    confirmMessage += 'This will refresh all Google-hosted image URLs with the current API key. This may take some time and will use Google API quota.';
                } else if (selectedMethod === 'fallback_placeholder') {
                    confirmMessage += 'This will remove all Google-hosted images from your listings. They will show theme fallback placeholders instead. This action cannot be easily undone.';
                }
                
                if (!confirm(confirmMessage)) {
                    return;
                }
                
                // Start the process
                button.prop('disabled', true).text('Processing...');
                statusSpan.html('<span style="color: #0d6efd;">Initializing regeneration...</span>');
                progressDiv.show();
                progressBar.css('width', '0%');
                progressText.text('Starting image regeneration process...');
                
                // Run the regeneration
                runImageRegeneration(selectedMethod, button, statusSpan, progressBar, progressText);
            });
            
            function runImageRegeneration(method, button, statusSpan, progressBar, progressText) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'lds_run_image_regeneration',
                        method: method,
                        nonce: '<?php echo wp_create_nonce('lds_image_regeneration_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            // Start batch processing
                            processBatch(response.data.batch_id, method, button, statusSpan, progressBar, progressText);
                        } else {
                            statusSpan.html('<span style="color: #d63384;">✗ ' + response.data.message + '</span>');
                            button.prop('disabled', false).text('Run Image Regeneration');
                        }
                    },
                    error: function() {
                        statusSpan.html('<span style="color: #d63384;">✗ Network error occurred</span>');
                        button.prop('disabled', false).text('Run Image Regeneration');
                    }
                });
            }
            
            function processBatch(batchId, method, button, statusSpan, progressBar, progressText) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'lds_process_image_regeneration_batch',
                        batch_id: batchId,
                        method: method,
                        nonce: '<?php echo wp_create_nonce('lds_image_regeneration_nonce'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            const data = response.data;
                            
                            // Update progress
                            const progressPercent = (data.processed / data.total) * 100;
                            progressBar.css('width', progressPercent + '%');
                            progressText.text(`Processed ${data.processed} of ${data.total} listings...`);
                            
                            if (data.completed) {
                                // Process completed
                                progressBar.css('width', '100%');
                                progressText.text(`Completed! Processed ${data.total} listings.`);
                                statusSpan.html('<span style="color: #198754;">✓ Image regeneration completed successfully</span>');
                                button.prop('disabled', false).text('Run Image Regeneration');
                                
                                // Show summary
                                if (data.summary) {
                                    let summaryHtml = '<div style="background: #d4edda; color: #155724; padding: 10px; margin: 10px 0; border-radius: 4px;">';
                                    summaryHtml += '<strong>Regeneration Summary:</strong><br>';
                                    if (data.summary.updated) summaryHtml += '• Updated: ' + data.summary.updated + ' listings<br>';
                                    if (data.summary.skipped) summaryHtml += '• Skipped: ' + data.summary.skipped + ' listings<br>';
                                    if (data.summary.errors) summaryHtml += '• Errors: ' + data.summary.errors + ' listings<br>';
                                    summaryHtml += '</div>';
                                    
                                    progressText.html(summaryHtml);
                                }
                            } else {
                                // Continue processing
                                setTimeout(() => processBatch(batchId, method, button, statusSpan, progressBar, progressText), 1000);
                            }
                        } else {
                            statusSpan.html('<span style="color: #d63384;">✗ ' + response.data.message + '</span>');
                            button.prop('disabled', false).text('Run Image Regeneration');
                        }
                    },
                    error: function() {
                        statusSpan.html('<span style="color: #d63384;">✗ Network error occurred during processing</span>');
                        button.prop('disabled', false).text('Run Image Regeneration');
                    }
                });
            }
        });
        </script>
        <?php
    }

    // Render map mode toggle checkbox
    public function render_map_mode_toggle_field() {
        $value = get_option('lds_enable_map_mode', 1); // Default: enabled
        $checked = checked(1, $value, false);
        
        echo "<label>";
        echo "<input type='checkbox' name='lds_enable_map_mode' id='lds_enable_map_mode' value='1' {$checked} /> ";
        echo "Enable interactive map mode for location selection";
        echo "</label>";
        echo "<p class='description'>When enabled, users can toggle between text and map input modes for location selection.</p>";
    }

    // Render map settings section
    public function render_map_settings_section() {
        $default_radius = get_option('lds_default_radius', 2.0);
        $default_lat = get_option('lds_default_map_center_lat', '51.5074');
        $default_lng = get_option('lds_default_map_center_lng', '-0.1278');
        $zoom_level = get_option('lds_map_zoom_level', 12);
        
        echo "<div style='margin-top: 15px; padding: 15px; border: 1px solid #ddd; border-radius: 5px; background: #f9f9f9;'>";
        
        // Default radius
        echo "<div style='margin-bottom: 15px;'>";
        echo "<label for='lds_default_radius' style='display: block; margin-bottom: 5px;'><strong>Default Search Radius:</strong></label>";
        echo "<div style='display: flex; align-items: center; gap: 15px;'>";
        echo "<input type='range' name='lds_default_radius' id='lds_default_radius' min='0.5' max='50' step='0.5' value='" . esc_attr($default_radius) . "' style='flex: 1;' />";
        echo "<span id='lds_default_radius_display' style='min-width: 60px; text-align: center; background: #f8f9fa; padding: 6px 12px; border-radius: 4px; border: 1px solid #e9ecef; font-weight: 600;'>" . esc_html($default_radius) . " km</span>";
        echo "</div>";
        echo "</div>";
        
        echo "<script>
        document.addEventListener('DOMContentLoaded', function() {
            // Radius slider functionality
            const radiusSlider = document.getElementById('lds_default_radius');
            const radiusDisplay = document.getElementById('lds_default_radius_display');
            
            if (radiusSlider && radiusDisplay) {
                radiusSlider.addEventListener('input', function() {
                    radiusDisplay.textContent = this.value + ' km';
                });
            }
            
            // Geolocate functionality for settings page
            const geolocateBtn = document.getElementById('lds-settings-geolocate-btn');
            if (geolocateBtn) {
                geolocateBtn.addEventListener('click', function() {
                    const button = this;
                    const originalText = button.textContent;
                    
                    button.textContent = '⌛ Loading...';
                    button.disabled = true;
                    button.style.color = '#999';
                    
                    console.log('Settings: Geolocate button clicked');
                    
                    // Use the same IP geolocation as the main import page
                    getIPGeolocationForSettings()
                        .then(function(location) {
                            console.log('Settings: Got location:', location);
                            const latField = document.getElementById('lds_default_map_center_lat');
                            const lngField = document.getElementById('lds_default_map_center_lng');
                            
                            if (latField && lngField) {
                                latField.value = location.lat.toFixed(4);
                                lngField.value = location.lng.toFixed(4);
                                
                                // Highlight the updated fields briefly
                                latField.style.background = '#d4edda';
                                lngField.style.background = '#d4edda';
                                setTimeout(function() {
                                    latField.style.background = '';
                                    lngField.style.background = '';
                                }, 2000);
                            }
                            
                            button.textContent = originalText;
                            button.disabled = false;
                            button.style.color = '#0073aa';
                        })
                        .catch(function(error) {
                            console.log('Settings geolocation failed, trying GPS fallback:', error.message);
                            
                            // Fallback to GPS geolocation
                            if (navigator.geolocation) {
                                navigator.geolocation.getCurrentPosition(
                                    function(position) {
                                        console.log('Settings: GPS success:', position.coords);
                                        const latField = document.getElementById('lds_default_map_center_lat');
                                        const lngField = document.getElementById('lds_default_map_center_lng');
                                        
                                        if (latField && lngField) {
                                            latField.value = position.coords.latitude.toFixed(4);
                                            lngField.value = position.coords.longitude.toFixed(4);
                                            
                                            // Highlight the updated fields briefly
                                            latField.style.background = '#d4edda';
                                            lngField.style.background = '#d4edda';
                                            setTimeout(function() {
                                                latField.style.background = '';
                                                lngField.style.background = '';
                                            }, 2000);
                                        }
                                        
                                        button.textContent = originalText;
                                        button.disabled = false;
                                        button.style.color = '#0073aa';
                                    },
                                    function(gpsError) {
                                        console.error('Settings: GPS failed:', gpsError);
                                        alert('Unable to detect location: ' + gpsError.message);
                                        button.textContent = originalText;
                                        button.disabled = false;
                                        button.style.color = '#0073aa';
                                    }
                                );
                            } else {
                                alert('Geolocation is not supported by this browser');
                                button.textContent = originalText;
                                button.disabled = false;
                                button.style.color = '#0073aa';
                            }
                        });
                });
            } else {
                console.error('Settings: Geolocate button not found');
            }
            
            // Reload saved location functionality
            const reloadBtn = document.getElementById('lds-refresh-location-btn');
            if (reloadBtn) {
                reloadBtn.addEventListener('click', function() {
                    const button = this;
                    const originalText = button.textContent;
                    
                    button.textContent = '↻ Reloading...';
                    button.disabled = true;
                    
                    // Make AJAX request to get current saved values
                    fetch(lds_admin_vars.ajax_url, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: 'action=lds_get_current_location_settings&nonce=' + lds_admin_vars.nonce
                    })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            const latField = document.getElementById('lds_default_map_center_lat');
                            const lngField = document.getElementById('lds_default_map_center_lng');
                            
                            if (latField && lngField) {
                                latField.value = data.data.lat;
                                lngField.value = data.data.lng;
                                
                                // Highlight the updated fields briefly
                                latField.style.background = '#fff3cd';
                                lngField.style.background = '#fff3cd';
                                setTimeout(function() {
                                    latField.style.background = '';
                                    lngField.style.background = '';
                                }, 2000);
                            }
                        } else {
                            alert('Failed to reload location: ' + (data.data?.message || 'Unknown error'));
                        }
                        
                        button.textContent = originalText;
                        button.disabled = false;
                    })
                    .catch(error => {
                        console.error('Settings: Reload failed:', error);
                        alert('Failed to reload location settings');
                        button.textContent = originalText;
                        button.disabled = false;
                    });
                });
            }
        });
        
        // IP Geolocation function for settings page
        async function getIPGeolocationForSettings() {
            try {
                console.log('Settings: Attempting IP geolocation with ipapi.co...');
                
                const response1 = await fetch('https://ipapi.co/json/', {
                    method: 'GET',
                    headers: { 'Accept': 'application/json' }
                });
                
                if (response1.ok) {
                    const data = await response1.json();
                    console.log('Settings: ipapi.co response:', data);
                    
                    if (data.latitude && data.longitude && data.latitude !== 0 && data.longitude !== 0) {
                        return {
                            lat: parseFloat(data.latitude),
                            lng: parseFloat(data.longitude)
                        };
                    }
                }
            } catch (error) {
                console.log('Settings: ipapi.co failed:', error.message);
            }

            try {
                console.log('Settings: Fallback to ipinfo.io...');
                
                const ipResponse = await fetch('https://api.ipify.org?format=json');
                if (ipResponse.ok) {
                    const ipData = await ipResponse.json();
                    
                    const locationResponse = await fetch('https://ipinfo.io/' + ipData.ip + '/geo');
                    if (locationResponse.ok) {
                        const locationData = await locationResponse.json();
                        
                        if (locationData.loc) {
                            const [lat, lng] = locationData.loc.split(',');
                            if (lat && lng) {
                                return {
                                    lat: parseFloat(lat),
                                    lng: parseFloat(lng)
                                };
                            }
                        }
                    }
                }
            } catch (error) {
                console.log('Settings: Fallback service failed:', error.message);
            }

            throw new Error('All IP geolocation services failed');
        }
        </script>";
        
        // Default map center
        echo "<div style='margin-bottom: 15px;'>";
        echo "<label style='display: block; margin-bottom: 5px;'><strong>Default Map Center:</strong></label>";
        echo "<p style='font-size: 12px; color: #666; margin: 0 0 8px 0;'>This is the location where the map will center when users first load the import page. You can set this from the main import page using 'Save as Default Location' or manually enter coordinates here.</p>";
        echo "<div style='display: flex; gap: 10px; align-items: end;'>";
        echo "<div>";
        echo "<label for='lds_default_map_center_lat' style='font-size: 12px;'><strong>Latitude:</strong></label>";
        echo "<input type='text' name='lds_default_map_center_lat' id='lds_default_map_center_lat' value='" . esc_attr($default_lat) . "' style='width: 100px;' />";
        echo "</div>";
        echo "<div>";
        echo "<label for='lds_default_map_center_lng' style='font-size: 12px;'><strong>Longitude:</strong></label>";
        echo "<input type='text' name='lds_default_map_center_lng' id='lds_default_map_center_lng' value='" . esc_attr($default_lng) . "' style='width: 100px;' />";
        echo "</div>";
        echo "<div>";
        echo "<button type='button' id='lds-settings-geolocate-btn' style='background: none; border: none; color: #0073aa; text-decoration: underline; cursor: pointer; font-size: 14px; padding: 0; margin-top: 16px;'>";
        echo "Geolocate me";
        echo "</button>";
        echo "</div>";
        echo "<div>";
        echo "<button type='button' id='lds-refresh-location-btn' style='background: #f0f0f1; border: 1px solid #c3c4c7; color: #2c3338; cursor: pointer; font-size: 12px; padding: 4px 8px; margin-top: 16px; border-radius: 3px;'>";
        echo "Reload Saved";
        echo "</button>";
        echo "</div>";
        echo "</div>";
        echo "</div>";
        
        // Default zoom level
        echo "<div style='margin-bottom: 15px;'>";
        echo "<label for='lds_map_zoom_level' style='display: block; margin-bottom: 5px;'><strong>Default Zoom Level:</strong></label>";
        echo "<select name='lds_map_zoom_level' id='lds_map_zoom_level'>";
        for ($zoom = 8; $zoom <= 16; $zoom++) {
            echo "<option value='{$zoom}'" . selected($zoom_level, $zoom, false) . ">{$zoom}</option>";
        }
        echo "</select>";
        echo "<p class='description'>Map zoom level (<strong>8 = city level</strong>, <strong>12 = neighborhood level</strong>, <strong>16 = street level</strong>)</p>";
        echo "</div>";
        
        echo "</div>";
    }

    /**
     * Get the server's outbound IP address
     * 
     * @return string Server IP address or error message
     */
    private function get_server_ip() {
        // Check if we have a cached IP (valid for 24 hours)
        $cached_ip = get_transient('lds_server_ip');
        if ($cached_ip !== false) {
            return $cached_ip;
        }
        
        // Try multiple methods to get the server's outbound IP
        $ip = '';
        
        // Method 1: Use external service to detect outbound IP
        $services = [
            'https://api.ipify.org',
            'https://ipinfo.io/ip',
            'https://icanhazip.com',
            'https://ident.me'
        ];
        
        foreach ($services as $service) {
            $response = wp_remote_get($service, [
                'timeout' => 10,
                'headers' => [
                    'User-Agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url()
                ]
            ]);
            
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $ip = trim(wp_remote_retrieve_body($response));
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
                    // Cache the IP for 24 hours
                    set_transient('lds_server_ip', $ip, 24 * HOUR_IN_SECONDS);
                    return $ip;
                }
            }
        }
        
        // Method 2: Check server variables (may not be outbound IP)
        $server_vars = ['SERVER_ADDR', 'LOCAL_ADDR', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP'];
        foreach ($server_vars as $var) {
            if (!empty($_SERVER[$var])) {
                $potential_ip = $_SERVER[$var];
                // Handle comma-separated IPs (X-Forwarded-For)
                if (strpos($potential_ip, ',') !== false) {
                    $potential_ip = trim(explode(',', $potential_ip)[0]);
                }
                if (filter_var($potential_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
                    $result = $potential_ip . ' (detected from server)';
                    // Cache for shorter time since this might be internal IP
                    set_transient('lds_server_ip', $result, 6 * HOUR_IN_SECONDS);
                    return $result;
                }
            }
        }
        
        $error_msg = 'Unable to detect - Please check with your hosting provider';
        // Cache error for 1 hour to avoid repeated failed requests
        set_transient('lds_server_ip', $error_msg, HOUR_IN_SECONDS);
        return $error_msg;
    }
}