<?php
class LDS_Importer {

    /**
     * Create a new listing with support for region and custom listing type
     * 
     * @param array $data Listing data from Google Places
     * @param int $category_id Category term ID to assign
     * @param int $region_id Region term ID to assign (optional)
     * @param string $listing_type Listing type to assign (e.g., 'service', 'rental')
     * @param string $region_taxonomy The region taxonomy name to use
     * @return int|false Post ID on success, false on failure
     */
    public function create_listing($data, $category_id = 0, $region_id = 0, $listing_type = 'service', $region_taxonomy = 'listing_region') {
        lds_log($data, 'Final data received by create_listing() method');
        
        $existing_posts = get_posts([
            'post_type' => 'listing',
            'meta_key' => '_place_id',
            'meta_value' => $data['place_id'],
            'posts_per_page' => 1,
            'post_status' => 'any',
            'fields' => 'ids'
        ]);
        
        if (!empty($existing_posts)) {
            // This is a final safeguard, the main duplicate check is now in the AJAX handler.
            return false;
        }

        $post_data = [
            'post_title' => wp_strip_all_tags($data['name']),
            'post_content' => wp_kses_post($data['description']),
            'post_status' => 'publish',
            'post_author' => get_current_user_id(),
            'post_type' => 'listing'
        ];
        
        $post_id = wp_insert_post($post_data, true);
        if (is_wp_error($post_id)) {
            lds_log('Failed to create post. WP_Error: ' . $post_id->get_error_message(), 'POST CREATION FAILED');
            return false;
        }

        // --- Standard Meta Data ---
        update_post_meta($post_id, '_place_id', sanitize_text_field($data['place_id']));
        update_post_meta($post_id, '_address', sanitize_text_field($data['address']));
        update_post_meta($post_id, '_friendly_address', sanitize_text_field($data['address']));
        update_post_meta($post_id, '_geolocation_lat', sanitize_text_field($data['lat']));
        update_post_meta($post_id, '_geolocation_long', sanitize_text_field($data['lng']));
        update_post_meta($post_id, 'listeo-avg-rating', sanitize_text_field($data['rating']));
        
        // --- NEW: Use the provided listing type instead of hardcoded 'service' ---
        update_post_meta($post_id, '_listing_type', sanitize_text_field($listing_type));

        // --- Smart Social & Contact Info Sorting ---
        $website_url = $data['website'] ?? '';
        $phone_number = $data['phone_number'] ?? '';

        // Initialize all fields to empty strings
        update_post_meta($post_id, '_website', '');
        update_post_meta($post_id, '_facebook', '');
        update_post_meta($post_id, '_twitter', '');
        update_post_meta($post_id, '_youtube', '');
        update_post_meta($post_id, '_instagram', '');
        update_post_meta($post_id, '_whatsapp', '');
        update_post_meta($post_id, '_phone', '');

        // Sort the website URL
        if (!empty($website_url)) {
            if (strpos($website_url, 'facebook.com') !== false) {
                update_post_meta($post_id, '_facebook', esc_url_raw($website_url));
            } elseif (strpos($website_url, 'instagram.com') !== false) {
                update_post_meta($post_id, '_instagram', esc_url_raw($website_url));
            } elseif (strpos($website_url, 'twitter.com') !== false) {
                update_post_meta($post_id, '_twitter', esc_url_raw($website_url));
            } elseif (strpos($website_url, 'youtube.com') !== false) {
                update_post_meta($post_id, '_youtube', esc_url_raw($website_url));
            } else {
                // If it's not a known social media site, it's the main website
                update_post_meta($post_id, '_website', esc_url_raw($website_url));
            }
        }

        // Sort the phone number
        if (!empty($phone_number)) {
            if (strpos($phone_number, 'wa.me') !== false || strpos($phone_number, 'whatsapp') !== false) {
                update_post_meta($post_id, '_whatsapp', sanitize_text_field($phone_number));
            } else {
                // If it's not a WhatsApp link, it's a regular phone number
                update_post_meta($post_id, '_phone', sanitize_text_field($phone_number));
            }
        }

        // --- Opening Hours ---
        if (!empty($data['opening_hours']) && is_array($data['opening_hours'])) {
            update_post_meta($post_id, '_opening_hours_status', 'on');
            foreach ($data['opening_hours'] as $day => $hours) {
                if ($hours === 'closed' || (is_array($hours) && ($hours['opening'] === 'closed' || $hours['closing'] === 'closed'))) {
                    update_post_meta($post_id, "_{$day}_opening_hour", '');
                    update_post_meta($post_id, "_{$day}_closing_hour", '');
                } elseif (is_array($hours) && isset($hours['opening']) && isset($hours['closing'])) {
                    update_post_meta($post_id, "_{$day}_opening_hour", sanitize_text_field($hours['opening']));
                    update_post_meta($post_id, "_{$day}_closing_hour", sanitize_text_field($hours['closing']));
                }
            }
        }

        // --- Category Assignment ---
        if ($category_id > 0) {
            wp_set_post_terms($post_id, [$category_id], 'listing_category', false);
        }

        // --- NEW: Region Assignment with Debug ---
        if ($region_id > 0) {
            lds_log("Attempting to assign region ID {$region_id} to post {$post_id} using taxonomy {$region_taxonomy}", 'REGION_ASSIGNMENT');
            
            // First, verify the region taxonomy exists
            if (!taxonomy_exists($region_taxonomy)) {
                lds_log("Taxonomy {$region_taxonomy} does not exist!", 'REGION_ERROR');
                
                // Try alternative taxonomy names - prioritize region (confirmed from regions importer)
                $alternative_taxonomies = ['region', 'listing_region', 'regions', 'location', 'listing_location'];
                foreach ($alternative_taxonomies as $alt_tax) {
                    if (taxonomy_exists($alt_tax)) {
                        lds_log("Found alternative taxonomy: {$alt_tax}", 'REGION_ALTERNATIVE');
                        $region_taxonomy = $alt_tax;
                        break;
                    }
                }
                
                if (!taxonomy_exists($region_taxonomy)) {
                    lds_log("No valid region taxonomy found at all!", 'REGION_ERROR');
                    $region_taxonomy = null;
                }
            }
            
            if ($region_taxonomy) {
                // Verify the region term exists in this taxonomy
                $region_term = get_term($region_id, $region_taxonomy);
                if (is_wp_error($region_term)) {
                    lds_log("Region term error: " . $region_term->get_error_message(), 'REGION_ERROR');
                } else {
                    lds_log("Region term found: " . $region_term->name . " in taxonomy " . $region_taxonomy, 'REGION_FOUND');
                    $result = wp_set_post_terms($post_id, [$region_id], $region_taxonomy, false);
                    
                    if (is_wp_error($result)) {
                        lds_log("Region assignment failed: " . $result->get_error_message(), 'REGION_ERROR');
                    } else {
                        lds_log("Region assignment result: " . print_r($result, true), 'REGION_SUCCESS');
                        
                        // Verify the assignment worked
                        $assigned_regions = wp_get_post_terms($post_id, $region_taxonomy);
                        if (!is_wp_error($assigned_regions)) {
                            lds_log("Verified assigned regions: " . print_r($assigned_regions, true), 'REGION_VERIFY');
                        }
                    }
                }
            }
        } else {
            lds_log("No region ID provided (region_id = {$region_id})", 'REGION_SKIP');
        }

        // --- NEW: Cache Google Reviews in Listeo Format ---
        if (!empty($data['reviews']) && is_array($data['reviews'])) {
            // Create the Google Places API response format that Listeo expects
            $google_places_data = [
                'result' => [
                    'reviews' => $data['reviews'],
                    'rating' => $data['rating'] ?? 0,
                    'user_ratings_total' => $data['user_ratings_total'] ?? 0,
                    'place_id' => $data['place_id'],
                    'name' => $data['name']
                ]
            ];
            
            // Store in the transient format that your review highlights plugin expects
            $transient_name = 'listeo_reviews_' . $post_id;
            set_transient($transient_name, $google_places_data, 30 * DAY_IN_SECONDS); // Cache for 30 days
            
            lds_log('Cached ' . count($data['reviews']) . ' Google reviews for post ID ' . $post_id . ' in transient: ' . $transient_name, 'REVIEWS_CACHE');
        }

        // --- Photo Import ---
        $photo_import_enabled = get_option('lds_enable_photo_import', 0);
        
        // Only import photos if the feature is enabled
        if ($photo_import_enabled) {
            $storage_method = get_option('lds_photo_storage_method', 'google');

            if (!empty($data['photo_urls']) && is_array($data['photo_urls'])) {
                if ($storage_method === 'download') {
                // Original download method (for backwards compatibility)
                require_once(ABSPATH . 'wp-admin/includes/media.php');
                require_once(ABSPATH . 'wp-admin/includes/file.php');
                require_once(ABSPATH . 'wp-admin/includes/image.php');
                
                $gallery_data = [];
                $is_first_image = true;

                foreach ($data['photo_urls'] as $photo_url) {
                    // Download the image file first and check for errors
                    $temp_file = download_url($photo_url, 15);
                    
                    if (is_wp_error($temp_file)) {
                        lds_log('Failed to download image from URL: ' . $photo_url . ' Error: ' . $temp_file->get_error_message(), 'IMAGE DOWNLOAD FAILED');
                        continue; // Skip this image and continue with the next one
                    }
                    
                    $image_id = media_handle_sideload([
                        'tmp_name' => $temp_file,
                        'name' => 'import.jpg'
                    ], $post_id, $data['name']);

                    if (!is_wp_error($image_id)) {
                        $image_url = wp_get_attachment_url($image_id);
                        $gallery_data[$image_id] = $image_url;
                        if ($is_first_image) {
                            set_post_thumbnail($post_id, $image_id);
                            $is_first_image = false;
                        }
                    } else {
                        lds_log('Image import failed. URL: ' . $photo_url . ' Error: ' . $image_id->get_error_message(), 'IMAGE IMPORT FAILED');
                    }
                    
                    // Clean up the temporary file if it still exists
                    if (file_exists($temp_file)) {
                        @unlink($temp_file);
                    }
                }

                if (!empty($gallery_data)) {
                    update_post_meta($post_id, '_gallery', $gallery_data);
                }
            } else {
                // NEW: Google server method (TOS compliant)
                $google_gallery_data = [];
                
                foreach ($data['photo_urls'] as $index => $photo_url) {
                    $google_gallery_data[] = [
                        'url' => $photo_url,
                        'attribution' => isset($data['photo_attributions'][$index]) ? $data['photo_attributions'][$index] : []
                    ];
                }
                
                if (!empty($google_gallery_data)) {
                    update_post_meta($post_id, '_gallery_google', $google_gallery_data);
                    lds_log('Stored ' . count($google_gallery_data) . ' Google photo URLs for post ID ' . $post_id, 'GOOGLE_PHOTOS');
                }
            }
            } // Close the if (!empty($data['photo_urls']) && is_array($data['photo_urls'])) check
        } else {
            lds_log('Photo import disabled by user setting - skipping photo import', 'PHOTO_IMPORT');
        }

        lds_log('Successfully created listing with Post ID: ' . $post_id . ', Category: ' . $category_id . ', Region: ' . $region_id . ', Type: ' . $listing_type, 'POST CREATION SUCCESS');
        
        // Proactively fetch Google reviews if place_id exists and Google reviews are enabled
        if (!empty($data['place_id']) && get_option('listeo_google_reviews')) {
            $post = get_post($post_id);
            if ($post) {
                // Use the listeo_get_google_reviews function to fetch and cache reviews
                $reviews = listeo_get_google_reviews($post);
                
                // Log the proactive fetch for debugging
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    $has_reviews = !empty($reviews) && isset($reviews['result']['reviews']) && is_array($reviews['result']['reviews']);
                    $review_count = $has_reviews ? count($reviews['result']['reviews']) : 0;
                    error_log("Listeo: Proactively fetched Google reviews for imported listing {$post_id} with place_id: {$data['place_id']} (reviews fetched: {$review_count})");
                }
            }
        } elseif (empty($data['place_id'])) {
            lds_log('No place_id provided for listing, skipping Google reviews fetch', 'GOOGLE_REVIEWS_SKIP');
        } elseif (!get_option('listeo_google_reviews')) {
            lds_log('Google reviews disabled in settings, skipping fetch', 'GOOGLE_REVIEWS_DISABLED');
        }
        
        return $post_id;
    }
}