<?php
// --- START OF UPGRADED class-lds-ajax-handler.php ---
class LDS_Ajax_Handler {
    private $cache_expiry = 30 * DAY_IN_SECONDS;

    public function __construct() {
        add_action('wp_ajax_lds_get_place_ids', [ $this, 'get_place_ids' ]);
        add_action('wp_ajax_lds_fetch_place_details', [ $this, 'fetch_place_details' ]);
        add_action('wp_ajax_lds_process_batch_ai', [ $this, 'process_batch_ai' ]);
        add_action('wp_ajax_lds_process_single_job', [ $this, 'process_single_job' ]);
        add_action('wp_ajax_lds_check_duplicate', [ $this, 'check_duplicate' ]);
        add_action('wp_ajax_lds_test_api_key', [ $this, 'test_api_key' ]);
        add_action('wp_ajax_lds_run_image_regeneration', [ $this, 'run_image_regeneration' ]);
        add_action('wp_ajax_lds_process_image_regeneration_batch', [ $this, 'process_image_regeneration_batch' ]);
        add_action('wp_ajax_lds_save_default_location', [ $this, 'save_default_location' ]);
        add_action('wp_ajax_lds_get_current_location_settings', [ $this, 'get_current_location_settings' ]);
        add_action('admin_enqueue_scripts', [ $this, 'enqueue_scripts' ]);
        add_action('wp_ajax_lds_debug', [ $this, 'debug_ajax_request' ]);
    }

    public function enqueue_scripts($hook) {
        if (strpos($hook, 'listeo-data-scraper') === false && strpos($hook, 'lds-settings') === false) return;
    
        // Enqueue the main script
        wp_enqueue_script('lds-admin-js', LDS_PLUGIN_URL . 'assets/js/lds-admin-scripts.js', ['jquery'], LDS_VERSION, true);
    
        // --- NEW PART: Pass PHP settings to our JavaScript file ---
        $settings_for_js = [
            'import_limit' => (int) get_option('lds_import_limit', 20),
            'nonce' => wp_create_nonce('lds_import_nonce'),
            'ajax_url' => admin_url('admin-ajax.php')
        ];
        wp_localize_script('lds-admin-js', 'lds_admin_vars', $settings_for_js);
        // This creates a JavaScript object named 'lds_settings' that our script can access.
    
        // Enqueue the style
        wp_enqueue_style('lds-admin-css', LDS_PLUGIN_URL . 'assets/css/lds-admin-styles.css', [], LDS_VERSION);
    }
    

    public function get_place_ids() {
        check_ajax_referer('lds_import_nonce', 'nonce');
        
        $query = sanitize_text_field($_POST['query']);
        if (empty($query)) {
            wp_send_json_error(['message' => 'Search query cannot be empty.']);
        }
    
        $api_key = get_option('lds_google_api_key');
        if (empty($api_key)) {
            wp_send_json_error(['message' => 'Google API Key is not set in Settings.']);
        }

        // Check search mode (text or map)
        $search_mode = isset($_POST['lds_search_mode']) ? sanitize_text_field($_POST['lds_search_mode']) : 'text';
        
        // Get coordinates for map mode
        $lat = null;
        $lng = null;
        $radius_km = null;
        
        if ($search_mode === 'map') {
            $lat = isset($_POST['lds_lat']) ? floatval($_POST['lds_lat']) : null;
            $lng = isset($_POST['lds_lng']) ? floatval($_POST['lds_lng']) : null;
            $radius_km = isset($_POST['lds_radius']) ? floatval($_POST['lds_radius']) : 5.0;
            
            if (empty($lat) || empty($lng)) {
                wp_send_json_error(['message' => 'Map coordinates are required for map search mode.']);
            }
            
            lds_log("Map search: {$query} at ({$lat}, {$lng}) with {$radius_km}km radius (using Nearby Search)", 'MAP_SEARCH');
        }
    
        $user_limit = (int) get_option('lds_import_limit', 10);
        $limit = min($user_limit, 50); // Use user's setting, capped at 50 for safety.
        
        // Get language setting for Google API
        $lang_setting = get_option('lds_description_language', 'site-default');
        $language = ($lang_setting === 'site-default') ? get_locale() : $lang_setting;
        
        $google_api = new LDS_Google_API($api_key, $language);
        $found_new_place_ids = [];
        $next_page_token = null;
        $max_google_pages = 10;
        $current_page = 0;
    
        do {
            $current_page++;
            lds_log("Fetching Google page {$current_page} to find {$limit} new listings...", 'SMART_SEARCH');
    
            // Use appropriate search method based on mode
            if ($search_mode === 'map') {
                $radius_meters = $radius_km * 1000; // Convert km to meters
                $google_response = $google_api->fetch_places_by_coordinates($lat, $lng, $radius_meters, $query, $next_page_token);
            } else {
                $google_response = $google_api->fetch_place_ids_paginated($query, $next_page_token);
            }
    
            if (is_wp_error($google_response)) {
                // Enhanced API error handling - don't just break silently
                $error_message = $google_response->get_error_message();
                $api_key_related = false;
                
                // Check if this is likely an API key issue
                if (strpos($error_message, 'REQUEST_DENIED') !== false || 
                    strpos($error_message, 'API key not valid') !== false ||
                    strpos($error_message, 'restricted') !== false ||
                    strpos($error_message, 'OVER_QUERY_LIMIT') !== false ||
                    strpos($error_message, 'expired') !== false) {
                    $api_key_related = true;
                }
                
                if ($api_key_related) {
                    // Return specific API key error instead of generic "no results" message
                    wp_send_json_error([
                        'message' => 'Google API Key Error: ' . $error_message,
                        'type' => 'api_key_error',
                        'detailed_message' => $this->get_detailed_api_error_message($error_message)
                    ]);
                } else {
                    // For other errors, provide generic error but mention API
                    wp_send_json_error([
                        'message' => 'Google API Error: ' . $error_message,
                        'type' => 'api_error'
                    ]);
                }
            }
    
            $place_ids_from_google = $google_response['place_ids'];
            $next_page_token = $google_response['next_page_token'];
    
            if (empty($place_ids_from_google)) {
                break;
            }
    
            // --- THIS IS THE CORRECTED PART ---
            // Step A: Get the IDs of posts that match one of the Place IDs. This is very fast.
            $existing_post_ids = get_posts([
                'post_type'      => 'listing',
                'posts_per_page' => -1,
                'fields'         => 'ids', // This is the correct parameter to get just the post IDs.
                'meta_query'     => [
                    [
                        'key'     => '_place_id',
                        'value'   => $place_ids_from_google,
                        'compare' => 'IN',
                    ]
                ]
            ]);
            
            // Step B: Now, get the actual Place ID meta values for those posts.
            $existing_place_ids = [];
            if (!empty($existing_post_ids)) {
                foreach ($existing_post_ids as $post_id) {
                    // Get the single meta value (the string) for the _place_id key.
                    $place_id_meta = get_post_meta($post_id, '_place_id', true);
                    if ($place_id_meta) {
                        $existing_place_ids[] = $place_id_meta;
                    }
                }
            }
            
            // Step C: Now we can safely compare the two arrays of strings.
            $new_ids_in_batch = array_diff($place_ids_from_google, $existing_place_ids);
    
            if (!empty($new_ids_in_batch)) {
                foreach ($new_ids_in_batch as $new_id) {
                    if (count($found_new_place_ids) < $limit) {
                        $found_new_place_ids[] = $new_id;
                    } else {
                        break;
                    }
                }
            }
            
        } while (count($found_new_place_ids) < $limit && !empty($next_page_token) && $current_page < $max_google_pages);
    
        if (empty($found_new_place_ids)) {
            wp_send_json_error([
                'message' => 'No new listings found for this query. All results found already exist in your database or Google returned no results.',
                'type' => 'no_results',
                'suggestion' => 'If you expect results for this search, check: 1) Your Google API key is working (use "Test API Key" in Settings), 2) Try different search terms, 3) Check if Places API is enabled in Google Cloud Console.'
            ]);
        }
    
        lds_log("Smart search complete. Found " . count($found_new_place_ids) . " new listings to import.", 'SMART_SEARCH');
        wp_send_json_success(['place_ids' => $found_new_place_ids]);
    }

    /**
     * Get detailed user-friendly error message for Google API errors
     * 
     * @param string $error_message Raw error message from Google API
     * @return string User-friendly detailed message
     */
    private function get_detailed_api_error_message($error_message) {
        if (strpos($error_message, 'REQUEST_DENIED') !== false) {
            if (strpos($error_message, 'restricted') !== false) {
                return 'Your API key has restrictions that are blocking this request. Go to Google Cloud Console → APIs & Services → Credentials → Your API Key → Application restrictions / API restrictions and ensure "Places API" is enabled and HTTP referrer restrictions (if any) include your website domain.';
            } elseif (strpos($error_message, 'API key not valid') !== false) {
                return 'Your API key is invalid. Please check for typos in the Settings page, or generate a new API key in Google Cloud Console.';
            } elseif (strpos($error_message, 'expired') !== false) {
                return 'Your API key has expired. Generate a new API key in Google Cloud Console.';
            } elseif (strpos($error_message, 'not authorized to use this API key') !== false) {
                return 'API key authorization error. Go to Google Cloud Console → APIs & Services → Credentials → Your API Key and configure: 1) Set Application restrictions to "IP addresses (web servers, cron jobs, etc.)" and add your server IP address, 2) Under "API restrictions" ensure both "Places API" AND "Maps JavaScript API" are selected. Your server IP can be found in the plugin Settings page.';
            } else {
                return 'Google is denying requests from your API key. This usually means the key is invalid, expired, or has incorrect restrictions. Please check your API key settings in Google Cloud Console.';
            }
        } elseif (strpos($error_message, 'OVER_QUERY_LIMIT') !== false) {
            return 'You have exceeded your Google Places API quota. Check your Google Cloud Console billing and increase your quota limits, or wait until your quota resets.';
        } elseif (strpos($error_message, 'INVALID_REQUEST') !== false) {
            return 'The Google Places API is not properly enabled for your project. Go to Google Cloud Console → APIs & Services → Library → Search for "Places API" → Enable it.';
        } else {
            return 'Please check your Google API key configuration in Google Cloud Console and ensure the Places API is enabled.';
        }
    }

    public function fetch_place_details() {
        check_ajax_referer('lds_import_nonce', 'nonce');
        $place_id = sanitize_text_field($_POST['place_id']);
        if (empty($place_id)) { wp_send_json_error(['message' => 'Place ID cannot be empty.']); }
        $api_key = get_option('lds_google_api_key');
        
        // Check if photo import is enabled before setting photo limit
        $photo_import_enabled = get_option('lds_enable_photo_import', 0);
        $photo_limit = $photo_import_enabled ? (int) get_option('lds_photo_import_limit', 3) : 0;
        
        if (empty($api_key)) { wp_send_json_error(['message' => 'Google API Key is not set in Settings.']); }
        
        // Get language setting for Google API
        $lang_setting = get_option('lds_description_language', 'site-default');
        $language = ($lang_setting === 'site-default') ? get_locale() : $lang_setting;
        
        $google_api = new LDS_Google_API($api_key, $language);
        $place_data = $google_api->get_place_details($place_id, $photo_limit);
        if (is_wp_error($place_data) || empty($place_data)) { 
            $error_message = is_wp_error($place_data) ? $place_data->get_error_message() : 'Failed to get place details.';
            
            // Check if this is an API key related error
            $api_key_related = false;
            if (strpos($error_message, 'REQUEST_DENIED') !== false || 
                strpos($error_message, 'API key not valid') !== false ||
                strpos($error_message, 'restricted') !== false ||
                strpos($error_message, 'OVER_QUERY_LIMIT') !== false ||
                strpos($error_message, 'expired') !== false) {
                $api_key_related = true;
            }
            
            if ($api_key_related) {
                wp_send_json_error([
                    'message' => 'Google API Key Error: ' . $error_message,
                    'type' => 'api_key_error',
                    'detailed_message' => $this->get_detailed_api_error_message($error_message)
                ]);
            } else {
                wp_send_json_error(['message' => 'Could not fetch place details: ' . $error_message]);
            }
        }
        wp_send_json_success(['place_data' => $place_data, 'display_info' => ['name' => $place_data['name'] ?? 'Unknown Place', 'address' => $place_data['address'] ?? 'No address', 'rating' => $place_data['rating'] ?? 'No rating', 'phone' => $place_data['phone_number'] ?? 'No phone', 'website' => $place_data['website'] ?? 'No website']]);
    }

    public function process_batch_ai() {
        check_ajax_referer('lds_import_nonce', 'nonce');
        
        $places_data = stripslashes_deep($_POST['places_data']);
        if (empty($places_data) || !is_array($places_data)) {
            wp_send_json_error(['message' => 'Invalid places data received.']);
        }

        $ai_enabled = (bool) get_option('lds_enable_ai_descriptions', 1);

        $processed_listings = [];
        $places_needing_ai = [];
        $cache_hits = 0;

        if ($ai_enabled) {
            foreach ($places_data as $place_data) {
                $cache_key = $this->get_description_cache_key($place_data);
                $cached_description = get_transient($cache_key);

                if ($cached_description !== false) {
                    $listing_data = $this->process_data_locally($place_data, $cached_description);
                    $processed_listings[] = $listing_data;
                    $cache_hits++;
                } else {
                    // Pass the full place data, including new fields
                    $places_needing_ai[] = $place_data;
                }
            }

            lds_log("AI enabled. Cache performance: {$cache_hits} hits, " . count($places_needing_ai) . " need AI", 'CACHE_STATS');

            if (!empty($places_needing_ai)) {
                $ai_descriptions_result = $this->get_batch_descriptions_directly($places_needing_ai);
                
                if (!is_wp_error($ai_descriptions_result)) {
                    foreach ($places_needing_ai as $index => $place_data) {
                        $description = $ai_descriptions_result[$index] ?? $this->generate_fallback_description($place_data);
                        $cache_key = $this->get_description_cache_key($place_data);
                        set_transient($cache_key, $description, $this->cache_expiry);
                        $listing_data = $this->process_data_locally($place_data, $description);
                        $processed_listings[] = $listing_data;
                    }
                } else {
                    lds_log('AI batch processing failed: ' . $ai_descriptions_result->get_error_message(), 'AI_ERROR');
                    foreach ($places_needing_ai as $place_data) {
                        $description = $this->generate_fallback_description($place_data);
                        $listing_data = $this->process_data_locally($place_data, $description);
                        $processed_listings[] = $listing_data;
                    }
                }
            }
        } else {
            lds_log('AI descriptions disabled. Using fallback descriptions for all places.', 'AI_DISABLED');
            foreach ($places_data as $place_data) {
                $description = $this->generate_fallback_description($place_data);
                $listing_data = $this->process_data_locally($place_data, $description);
                $processed_listings[] = $listing_data;
            }
        }

        wp_send_json_success(['listings' => $processed_listings]);
    }

    public function check_duplicate() {
        check_ajax_referer('lds_import_nonce', 'nonce');
    
        $place_id = sanitize_text_field($_POST['place_id'] ?? '');
    
        if (empty($place_id)) {
            wp_send_json_error(['message' => 'No Place ID provided for duplicate check.']);
        }
    
        // This is the same query used by the importer. It's very efficient.
        $existing_posts = get_posts([
            'post_type'      => 'listing',
            'meta_key'       => '_place_id',
            'meta_value'     => $place_id,
            'posts_per_page' => 1,
            'post_status'    => 'any', // Check against all statuses (published, draft, etc.)
            'fields'         => 'ids', // We only need the ID, which is very fast.
        ]);
    
        if (!empty($existing_posts)) {
            // Found a duplicate
            wp_send_json_success(['duplicate' => true, 'post_id' => $existing_posts[0]]);
        } else {
            // Not a duplicate
            wp_send_json_success(['duplicate' => false]);
        }
    }

    private function get_batch_descriptions_directly($places_data) {
        $openai_api_key = get_option('lds_openai_api_key');
        if (empty($openai_api_key)) {
            return new WP_Error('ai_config_error', 'OpenAI API Key is not set in Listeo Scraper settings.');
        }

        // Get the selected GPT model from settings
        $selected_model = get_option('lds_gpt_model', 'gpt-4.1-mini');
        
        // Ensure we have a valid model (fallback safety)
        if (empty($selected_model) || !in_array($selected_model, ['gpt-4.1-mini', 'gpt-5-mini'])) {
            $selected_model = 'gpt-4.1-mini';
            // Update the option to prevent this issue in future
            update_option('lds_gpt_model', 'gpt-4.1-mini');
        }
        
        // Debug: Log the selected model
        if (get_option('lds_enable_debug_mode', 0)) {
            error_log("[LDS_DEBUG] Selected GPT model: " . $selected_model);
        }

        $lang_setting = get_option('lds_description_language', 'site-default');
        $description_language = ($lang_setting === 'site-default') ? \Locale::getDisplayLanguage(get_locale(), 'en') : $lang_setting;

        // Get description word length setting
        $word_length = get_option('lds_description_word_length', 100);

        $batch_size = count($places_data);
        
        // --- FINAL, UPGRADED SYSTEM PROMPT WITH HTML FORMATTING ---
        $system_prompt = "You are an expert local SEO and marketing copywriter creating content for a premium business directory. Your task is to write a compelling, HTML-formatted description for each business provided.

**Source Material You Will Be Given for each business:**
- `name`: The business name.
- `address`: The full address.
- `types`: Google's categories for the business (e.g., 'cafe', 'coffee_shop').
- `rating`: The overall star rating.
- `rating_count`: The total number of reviews.
- `review_snippets`: An array of short, relevant text from up to 3 real customer reviews.

**Your Writing & Formatting Rules:**
1.  **Synthesize, Don't Just List:** Weave the information together naturally. Instead of \"They have good coffee,\" write \"Customers consistently rave about the <strong>rich, aromatic coffee</strong>, calling it a must-try.\"
2.  **Incorporate Keywords:** Naturally include the business name, its type (e.g., \"cafe,\" \"restaurant\"), and its city/neighborhood.
3.  **Highlight Social Proof:** If the rating count is over 10, mention it (e.g., \"...trusted by over 500 reviewers\").
4.  **Use Review Content:** Use the `review_snippets` as the primary source for what makes the business special.
5.  **Professional Tone:** The description must be professional, inviting, and trustworthy.
6.  **DO NOT mention \"Google,\" \"reviews,\" or \"data.\"** and do not mention amount of reviews and average rating. Write as if you are a local expert recommending the place.
7.  **Language:** Write all descriptions in " . htmlspecialchars($description_language, ENT_QUOTES, 'UTF-8') . ".
8.  **Length:** Target approximately {$word_length} words for each description. Keep it concise and engaging.
9.  **Add paragraphs** Add <p> tags to make description easier to read, language should be casual not like ai generated and not clunky
10. **CRITICAL OUTPUT FORMAT:** For each business, you must generate a single string of HTML with the following structure:
    - A short, catchy, SEO-friendly headline enclosed in `<h2>` tags.
    - Following the headline, the main description paragraph.
    - Within the paragraph, strategically wrap 3-4 important keywords in `<strong>` tags. Good candidates are: the business name, the city, and key services (e.g., \"specialty coffee\", \"artisanal pastries\", \"cozy atmosphere\"). Do not overuse bolding.
    - Example: `<h2>A Coffee Lover's Paradise in Downtown</h2><p><strong>Example Cafe</strong> is a gem in the heart of <strong>Example City</strong>, beloved for its...`

10.  **Final JSON Structure:** Return ONLY a valid JSON object: {\"descriptions\": [\"<h2>...</h2><p>...</p>\", \"<h2>...</h2><p>...</p>\"]} in the exact same order as the input array.";

        // --- The rest of the function is the same ---
        $business_data_for_ai = [];
        foreach ($places_data as $listing) {
            $review_snippets = [];
            if (!empty($listing['reviews']) && is_array($listing['reviews'])) {
                $good_reviews = array_filter($listing['reviews'], function($review) {
                    return !empty($review['text']) && $review['rating'] >= 4;
                });
                usort($good_reviews, function($a, $b) {
                    return $b['rating'] <=> $a['rating'];
                });
                $top_reviews = array_slice($good_reviews, 0, 3);
                foreach ($top_reviews as $review) {
                    $review_snippets[] = $review['text'];
                }
            }

            $business_data_for_ai[] = [
                'name' => $listing['name'] ?? 'Unknown Business',
                'address' => $listing['address'] ?? '',
                'types' => $listing['types'] ?? [],
                'rating' => $listing['rating'] ?? 0,
                'rating_count' => $listing['user_ratings_total'] ?? 0,
                'review_snippets' => $review_snippets,
            ];
        }

        $user_prompt = "Generate HTML descriptions for these " . $batch_size . " businesses based on the provided data:\n\n" . json_encode($business_data_for_ai, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

        // Build payload with model-specific parameters
        $payload = [
            "model" => $selected_model,
            "messages" => [
                ["role" => "system", "content" => $system_prompt],
                ["role" => "user", "content" => $user_prompt]
            ],
            "response_format" => ["type" => "json_object"],
        ];

        // Add model-specific parameters
        if ($selected_model === 'gpt-4.1-mini') {
            // Traditional GPT-4 parameters
            $payload["max_tokens"] = 4000 * $batch_size;
            $payload["temperature"] = 0.7;
            $payload["top_p"] = 1.0;
        } else {
            // GPT-5 parameters (gpt-5-mini)
            $payload["max_completion_tokens"] = 8000 * $batch_size; // Higher limit for reasoning
            $payload["verbosity"] = "low"; // Reduce output length
        }

        // Debug: Log the payload model parameter
        if (get_option('lds_enable_debug_mode', 0)) {
            error_log("[LDS_DEBUG] API Payload model parameter: " . $payload["model"]);
        }

        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
            'method'    => 'POST', 'timeout'   => 90,
            'headers'   => ['Content-Type'  => 'application/json', 'Authorization' => 'Bearer ' . $openai_api_key,],
            'body'      => json_encode($payload), 'sslverify' => true,
        ]);

        if (is_wp_error($response)) { return new WP_Error('openai_connection_error', 'Failed to connect to OpenAI API: ' . $response->get_error_message()); }
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        // Debug logging for troubleshooting
        lds_log('OpenAI API Response Code: ' . $response_code, 'OPENAI_DEBUG');
        lds_log('OpenAI API Response Body: ' . substr($response_body, 0, 500), 'OPENAI_DEBUG');
        
        if ($response_code !== 200) { $error_message = $response_data['error']['message'] ?? 'Unknown API error'; return new WP_Error('openai_api_error', 'OpenAI API Error (HTTP ' . $response_code . '): ' . $error_message); }
        $ai_content = $response_data['choices'][0]['message']['content'] ?? null;
        if (!$ai_content) { 
            lds_log('Empty AI content. Full response: ' . print_r($response_data, true), 'OPENAI_DEBUG');
            return new WP_Error('openai_empty_response', 'Empty response content from OpenAI.'); 
        }
        $ai_response = json_decode($ai_content, true);
        if (json_last_error() !== JSON_ERROR_NONE || !isset($ai_response['descriptions'])) { return new WP_Error('openai_invalid_format', 'Invalid AI response format. AI returned: ' . $ai_content); }
        $descriptions = $ai_response['descriptions'];
        if (count($descriptions) !== $batch_size) { return new WP_Error('openai_count_mismatch', 'AI returned ' . count($descriptions) . ' descriptions, but ' . $batch_size . ' were expected.'); }
        return $descriptions;
    }
    
    private function get_description_cache_key($place_data) { 
        $name = sanitize_title($place_data['name'] ?? ''); 
        $address = sanitize_title($place_data['address'] ?? ''); 
        $lang = get_option('lds_description_language', 'site-default'); 
        return 'lds_desc_' . md5($name . '_' . $address . '_' . $lang); 
    }
    
    private function process_data_locally($place_data, $description) { 
        $processed_hours = $this->process_opening_hours_locally($place_data['opening_hours']); 
        return [ 
            'name' => $place_data['name'], 
            'place_id' => $place_data['place_id'], 
            'address' => $place_data['address'], 
            'lat' => $place_data['lat'], 
            'lng' => $place_data['lng'], 
            'website' => $place_data['website'], 
            'phone_number' => $place_data['phone_number'], 
            'rating' => $place_data['rating'], 
            'opening_hours' => $processed_hours, 
            'photo_urls' => $place_data['photo_urls'] ?? [], // Use empty array if photos disabled
            'description' => $description,
            'reviews' => $place_data['reviews'] ?? [], // NEW: Include reviews
            'user_ratings_total' => $place_data['user_ratings_total'] ?? 0 // NEW: Include total ratings
        ]; 
    }
    
    private function process_opening_hours_locally($opening_hours_array) { 
        if (empty($opening_hours_array) || !is_array($opening_hours_array)) { 
            return []; 
        } 
        $days_map = [
            // English
            'monday' => 'monday', 'tuesday' => 'tuesday', 'wednesday' => 'wednesday', 
            'thursday' => 'thursday', 'friday' => 'friday', 'saturday' => 'saturday', 'sunday' => 'sunday',
            
            // Polish
            'poniedziałek' => 'monday', 'wtorek' => 'tuesday', 'środa' => 'wednesday', 
            'czwartek' => 'thursday', 'piątek' => 'friday', 'sobota' => 'saturday', 'niedziela' => 'sunday',
            
            // Spanish
            'lunes' => 'monday', 'martes' => 'tuesday', 'miércoles' => 'wednesday', 
            'jueves' => 'thursday', 'viernes' => 'friday', 'sábado' => 'saturday', 'domingo' => 'sunday',
            
            // French
            'lundi' => 'monday', 'mardi' => 'tuesday', 'mercredi' => 'wednesday', 
            'jeudi' => 'thursday', 'vendredi' => 'friday', 'samedi' => 'saturday', 'dimanche' => 'sunday',
            
            // German
            'montag' => 'monday', 'dienstag' => 'tuesday', 'mittwoch' => 'wednesday', 
            'donnerstag' => 'thursday', 'freitag' => 'friday', 'samstag' => 'saturday', 'sonntag' => 'sunday',
            
            // Italian
            'lunedì' => 'monday', 'martedì' => 'tuesday', 'mercoledì' => 'wednesday', 
            'giovedì' => 'thursday', 'venerdì' => 'friday', 'sabato' => 'saturday', 'domenica' => 'sunday',
        ]; 
        $processed_hours = []; 
        foreach ($opening_hours_array as $day_text) { 
            lds_log("Processing opening hours text: '{$day_text}'", 'OPENING_HOURS_PARSE');
            
            $day_found = false;
            foreach ($days_map as $day_name => $day_key) { 
                if (stripos($day_text, $day_name) === 0) { 
                    $day_found = true;
                    lds_log("Matched day: '{$day_name}' -> '{$day_key}'", 'OPENING_HOURS_DAY_MATCH');
                    
                    if (stripos($day_text, 'closed') !== false || stripos($day_text, 'zamknięte') !== false || 
                        stripos($day_text, 'cerrado') !== false || stripos($day_text, 'fermé') !== false || 
                        stripos($day_text, 'geschlossen') !== false || stripos($day_text, 'chiuso') !== false) { 
                        $processed_hours[$day_key] = 'closed'; 
                        lds_log("Day marked as closed: {$day_key}", 'OPENING_HOURS_CLOSED');
                    } elseif (stripos($day_text, 'open 24 hours') !== false || stripos($day_text, '24 hours') !== false ||
                              stripos($day_text, 'otwarte 24 godziny') !== false || stripos($day_text, '24 godziny') !== false ||
                              stripos($day_text, 'abierto 24 horas') !== false || stripos($day_text, '24 horas') !== false ||
                              stripos($day_text, 'ouvert 24h') !== false || stripos($day_text, '24h/24') !== false ||
                              stripos($day_text, '24 stunden') !== false || stripos($day_text, 'rund um die uhr') !== false ||
                              stripos($day_text, 'aperto 24 ore') !== false || stripos($day_text, '24 ore') !== false) {
                        $processed_hours[$day_key] = ['opening' => '00:00', 'closing' => '23:59'];
                        lds_log("Day marked as 24 hours: {$day_key} -> 00:00-23:59", 'OPENING_HOURS_24H');
                    } else { 
                        $clean_text = preg_replace('/[\x{2013}\x{2014}\x{2212}]/u', '-', $day_text); 
                        $clean_text = preg_replace('/[\x{00A0}\x{2009}\x{202F}]/u', ' ', $clean_text); 
                        $clean_text = preg_replace('/\s+/', ' ', $clean_text); 
                        $clean_text = trim($clean_text); 
                        
                        lds_log("Cleaned text for parsing: '{$clean_text}'", 'OPENING_HOURS_CLEANED');
                        
                        // Enhanced regex pattern to handle multiple time formats
                        // Supports: 10:00 AM – 12:00 AM, 10:00 – 00:00, 10:00-00:00, etc.
                        $time_pattern = '/(\d{1,2}):(\d{2})(?:\s*(AM|PM))?\s*[-–—]\s*(\d{1,2}):(\d{2})(?:\s*(AM|PM))?/i'; 
                        
                        if (preg_match($time_pattern, $clean_text, $matches)) { 
                            $opening_hour = $matches[1]; 
                            $opening_minute = $matches[2]; 
                            $opening_ampm = $matches[3] ?? null; 
                            $closing_hour = $matches[4]; 
                            $closing_minute = $matches[5]; 
                            $closing_ampm = $matches[6] ?? null; 
                            
                            lds_log("Time matches: {$opening_hour}:{$opening_minute} {$opening_ampm} - {$closing_hour}:{$closing_minute} {$closing_ampm}", 'OPENING_HOURS_TIME_MATCH');
                            
                            // Always convert to 24-hour format for Listeo
                            $opening_time = $this->ensure_24h_format($opening_hour, $opening_minute, $opening_ampm); 
                            $closing_time = $this->ensure_24h_format($closing_hour, $closing_minute, $closing_ampm); 
                            
                            $processed_hours[$day_key] = ['opening' => $opening_time, 'closing' => $closing_time]; 
                            lds_log("Processed hours for {$day_key}: {$opening_time} - {$closing_time}", 'OPENING_HOURS_SUCCESS');
                        } else { 
                            lds_log("Failed to parse time from: '{$clean_text}'", 'OPENING_HOURS_PARSE_FAIL');
                            $processed_hours[$day_key] = 'closed'; 
                        } 
                    } 
                    break; // Found the day, no need to continue checking
                } 
            }
            
            if (!$day_found) {
                lds_log("No day match found for: '{$day_text}'", 'OPENING_HOURS_NO_MATCH');
            }
        } 
        
        lds_log("Final processed hours: " . print_r($processed_hours, true), 'OPENING_HOURS_FINAL'); 
        return $processed_hours; 
    }
    
    /**
     * Ensure time is always in 24-hour format for Listeo
     * Handles both 12-hour (with AM/PM) and 24-hour formats
     */
    private function ensure_24h_format($hour, $minute, $ampm = null) { 
        $hour = intval($hour); 
        $minute = intval($minute); 
        
        // If AM/PM is provided, convert from 12-hour to 24-hour
        if ($ampm) { 
            if (strtoupper($ampm) === 'PM' && $hour !== 12) { 
                $hour += 12; 
            } elseif (strtoupper($ampm) === 'AM' && $hour === 12) { 
                $hour = 0; 
            } 
        } else {
            // If no AM/PM provided, validate 24-hour format
            // Handle special cases like midnight represented as 24:00
            if ($hour === 24) {
                $hour = 0; // 24:00 becomes 00:00
            }
            
            // Ensure hour is within valid 24-hour range
            if ($hour > 23) {
                lds_log("Invalid hour value: {$hour}, setting to 23", 'OPENING_HOURS_HOUR_FIX');
                $hour = 23;
            }
        }
        
        // Ensure minute is within valid range
        if ($minute > 59) {
            lds_log("Invalid minute value: {$minute}, setting to 59", 'OPENING_HOURS_MINUTE_FIX');
            $minute = 59;
        }
        
        $formatted_time = sprintf('%02d:%02d', $hour, $minute);
        lds_log("Converted time: {$hour}:{$minute} {$ampm} -> {$formatted_time}", 'OPENING_HOURS_24H_CONVERT');
        
        return $formatted_time; 
    }

    /**
     * Generate an extremely simple, data-only, zero-translation fallback description.
     */
    private function generate_fallback_description($place_data) {
        // Start with the name and address.
        $name = $place_data['name'] ?? '';
        $address = $place_data['address'] ?? '';

        $description = '<p><strong>' . esc_html($name) . '</strong>';
        if ($address) {
            $description .= '<br>' . esc_html($address);
        }
        $description .= '</p>';
        
        // Create an array to hold the list items.
        $details_list = [];

        // 1. Add Rating (data only)
        if (!empty($place_data['rating']) && $place_data['rating'] > 0) {
            $details_list[] = '<li>⭐ ' . esc_html($place_data['rating']) . '</li>';
        }

        // 2. Add Phone Number (data only)
        if (!empty($place_data['phone_number'])) {
            $details_list[] = '<li>📞 ' . esc_html($place_data['phone_number']) . '</li>';
        }

        // 3. Add Website (data only, as a clickable link)
        if (!empty($place_data['website'])) {
            $details_list[] = '<li>🌐 <a rel="nofollow" href="' . esc_url($place_data['website']) . '" target="_blank" rel="noopener noreferrer">' . esc_html($place_data['website']) . '</a></li>';
        }

        // If we have any details, wrap them in a <ul> tag.
        if (!empty($details_list)) {
            $description .= '<ul>' . implode('', $details_list) . '</ul>';
        }
        
        return $description;
    }

    public function process_single_job() {
        check_ajax_referer('lds_import_nonce', 'nonce');
        
        $listing_data = stripslashes_deep($_POST['listing_data']);
        $category_id = isset($_POST['category_id']) ? absint($_POST['category_id']) : 0;
        $region_id = isset($_POST['region_id']) ? absint($_POST['region_id']) : 0; // NEW
        $region_taxonomy = isset($_POST['region_taxonomy']) ? sanitize_text_field($_POST['region_taxonomy']) : 'region'; // NEW - default back to region
        $listing_type = isset($_POST['listing_type']) ? sanitize_text_field($_POST['listing_type']) : 'service'; // NEW
        
        lds_log("Received job data - Region ID: {$region_id}, Region Taxonomy: {$region_taxonomy}, Listing Type: {$listing_type}", 'JOB_PROCESSING');
        
        if (empty($listing_data) || !is_array($listing_data)) {
            wp_send_json_error(['message' => 'Invalid listing data received.']);
        }
        
        $importer = new LDS_Importer();
        $post_id = $importer->create_listing($listing_data, $category_id, $region_id, $listing_type, $region_taxonomy);
        
        if ($post_id) {
            wp_send_json_success([
                'post_id' => $post_id, 
                'post_title' => get_the_title($post_id)
            ]);
        } else {
            wp_send_json_error([
                'message' => 'Failed to import listing: ' . ($listing_data['name'] ?? 'Unknown')
            ]);
        }
    }
     
    public function debug_ajax_request() { 
        lds_log($_POST, 'AJAX_POST_DATA', 'DEBUG'); 
        lds_log($_REQUEST, 'AJAX_REQUEST_DATA', 'DEBUG'); 
        $nonce_check = wp_verify_nonce($_POST['nonce'] ?? '', 'lds_import_nonce'); 
        lds_log($nonce_check, 'NONCE_CHECK_RESULT', 'DEBUG'); 
        $can_manage = current_user_can('manage_options'); 
        lds_log($can_manage, 'USER_CAN_MANAGE_OPTIONS', 'DEBUG'); 
        lds_log(getallheaders(), 'REQUEST_HEADERS', 'DEBUG'); 
        wp_send_json_success(['debug' => 'AJAX working', 'post_data' => $_POST]); 
    }

    /**
     * Test Google API key validity - Optimized to use only 1 API call
     */
    public function test_api_key() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'] ?? '', 'lds_test_api_key_nonce')) {
            wp_send_json_error(['message' => 'Security check failed.']);
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions.']);
        }

        $api_key = sanitize_text_field($_POST['api_key'] ?? '');
        
        if (empty($api_key)) {
            wp_send_json_error(['message' => 'API key is required.']);
        }

        // Use a well-known, stable Place ID for testing (Statue of Liberty, New York)
        // This tests the most important API (Place Details) that's used during imports
        $test_place_id = 'ChIJPTacEpBQwokRKwIlDXelxkA'; // Statue of Liberty - very stable landmark
        
        $details_url = 'https://maps.googleapis.com/maps/api/place/details/json?' . http_build_query([
            'place_id' => $test_place_id,
            'fields' => 'place_id,name,formatted_address,rating,reviews',
            'key' => $api_key
        ]);

        lds_log('Testing Place Details API with URL: ' . $details_url, 'API_KEY_TEST');

        $response = wp_remote_get($details_url, [
            'timeout' => 15,
            'user-agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url()
        ]);

        if (is_wp_error($response)) {
            lds_log('Place Details API test failed - WP Error: ' . $response->get_error_message(), 'API_KEY_TEST');
            wp_send_json_error(['message' => 'Failed to connect to Google Places API: ' . $response->get_error_message()]);
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);
        
        lds_log('Place Details API response: ' . print_r($data, true), 'API_KEY_TEST');

        if (!$data) {
            wp_send_json_error(['message' => 'Invalid response from Google Places API']);
        }

        // Handle API response with detailed error messages
        switch ($data['status']) {
            case 'OK':
                wp_send_json_success(['message' => "✓ API key is valid and working. Ready for imports!"]);
                break;
                
            case 'REQUEST_DENIED':
                $error_msg = isset($data['error_message']) ? $data['error_message'] : 'Request denied';
                
                if (strpos($error_msg, 'expired') !== false) {
                    wp_send_json_error(['message' => 'API key expired. Generate a new key in Google Cloud Console.']);
                } elseif (strpos($error_msg, 'API key not valid') !== false) {
                    wp_send_json_error(['message' => 'API key invalid. Check for typos or regenerate the key.']);
                } elseif (strpos($error_msg, 'restricted') !== false) {
                    wp_send_json_error(['message' => 'API key restrictions are blocking requests. Check HTTP referrer restrictions in Google Cloud Console.']);
                } else {
                    wp_send_json_error(['message' => 'Request denied: ' . $error_msg]);
                }
                break;
                
            case 'OVER_QUERY_LIMIT':
                wp_send_json_error(['message' => 'API quota exceeded. Check your Google Cloud billing and usage limits.']);
                break;
                
            case 'INVALID_REQUEST':
                wp_send_json_error(['message' => 'Invalid request. Ensure Places API is enabled in Google Cloud Console.']);
                break;
                
            case 'NOT_FOUND':
                // Fallback: If the specific place ID is not found, try a simple text search test
                $this->test_api_key_fallback($api_key);
                break;
                
            case 'ZERO_RESULTS':
                wp_send_json_success(['message' => '✓ API key is valid and working.']);
                break;
                
            default:
                wp_send_json_error(['message' => 'Unexpected API response: ' . $data['status']]);
                break;
        }
    }

    /**
     * Fallback API test using text search if place ID test fails
     */
    private function test_api_key_fallback($api_key) {
        lds_log('Using fallback text search test', 'API_KEY_TEST');
        
        $search_url = 'https://maps.googleapis.com/maps/api/place/textsearch/json?' . http_build_query([
            'query' => 'restaurant',
            'key' => $api_key
        ]);

        $response = wp_remote_get($search_url, [
            'timeout' => 15,
            'user-agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url()
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Failed to connect to Google API: ' . $response->get_error_message()]);
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if (!$data) {
            wp_send_json_error(['message' => 'Invalid response from Google API']);
        }

        switch ($data['status']) {
            case 'OK':
                wp_send_json_success(['message' => "✓ API key is valid and working. Ready for imports!"]);
                break;
            case 'ZERO_RESULTS':
                wp_send_json_success(['message' => '✓ API key is valid and working.']);
                break;
            default:
                $error_msg = isset($data['error_message']) ? $data['error_message'] : $data['status'];
                wp_send_json_error(['message' => 'API test failed: ' . $error_msg]);
                break;
        }
    }

    /**
     * Initialize image regeneration process
     */
    public function run_image_regeneration() {
        // Check nonce and permissions
        if (!wp_verify_nonce($_POST['nonce'], 'lds_image_regeneration_nonce') || !current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Security check failed']);
            return;
        }

        $method = sanitize_text_field($_POST['method']);
        
        if (!in_array($method, ['refresh_api', 'fallback_placeholder'])) {
            wp_send_json_error(['message' => 'Invalid regeneration method']);
            return;
        }

        // Get all listings that have Google photos
        $args = [
            'post_type' => 'listing',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'meta_query' => [
                'relation' => 'OR',
                [
                    'key' => '_gallery_google',
                    'compare' => 'EXISTS'
                ],
                [
                    'key' => '_gallery',
                    'compare' => 'EXISTS'
                ]
            ]
        ];

        $all_listing_ids = get_posts($args);
        
        // Filter to only include listings that actually have Google photos (not WordPress media library photos)
        $listing_ids = [];
        foreach ($all_listing_ids as $listing_id) {
            $google_gallery = get_post_meta($listing_id, '_gallery_google', true);
            $wp_gallery = get_post_meta($listing_id, '_gallery', true);
            
            // Include if it has _gallery_google OR if _gallery contains googleusercontent URLs
            if (!empty($google_gallery)) {
                $listing_ids[] = $listing_id;
                lds_log("Found listing {$listing_id} with _gallery_google: " . count($google_gallery) . " photos", 'IMAGE_REGENERATION');
            } elseif (!empty($wp_gallery) && is_array($wp_gallery)) {
                // Check if any URLs in the gallery are Google URLs
                foreach ($wp_gallery as $photo_id => $photo_url) {
                    if (strpos($photo_url, 'googleusercontent.com') !== false || strpos($photo_url, 'googleapis.com') !== false) {
                        $listing_ids[] = $listing_id;
                        lds_log("Found listing {$listing_id} with Google URLs in _gallery", 'IMAGE_REGENERATION');
                        break;
                    }
                }
            }
        }

        if (empty($listing_ids)) {
            wp_send_json_error(['message' => 'No listings with Google-hosted images found']);
            return;
        }

        // Create a unique batch ID for this regeneration process
        $batch_id = 'lds_regen_' . time() . '_' . wp_rand(1000, 9999);
        
        // Store batch data in transient
        $batch_data = [
            'listing_ids' => $listing_ids,
            'method' => $method,
            'total' => count($listing_ids),
            'processed' => 0,
            'current_index' => 0,
            'summary' => [
                'updated' => 0,
                'skipped' => 0,
                'errors' => 0
            ]
        ];
        
        set_transient($batch_id, $batch_data, 3600); // Store for 1 hour

        lds_log("Started image regeneration batch {$batch_id} with method {$method} for " . count($listing_ids) . " listings", 'IMAGE_REGENERATION');

        wp_send_json_success([
            'batch_id' => $batch_id,
            'total_listings' => count($listing_ids),
            'method' => $method
        ]);
    }

    /**
     * Process a batch of image regeneration
     */
    public function process_image_regeneration_batch() {
        // Check nonce and permissions
        if (!wp_verify_nonce($_POST['nonce'], 'lds_image_regeneration_nonce') || !current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Security check failed']);
            return;
        }

        $batch_id = sanitize_text_field($_POST['batch_id']);
        $method = sanitize_text_field($_POST['method']);
        
        // Get batch data
        $batch_data = get_transient($batch_id);
        if (!$batch_data) {
            wp_send_json_error(['message' => 'Batch data not found or expired']);
            return;
        }

        $batch_size = 5; // Process 5 listings at a time
        $current_index = $batch_data['current_index'];
        $listing_ids = $batch_data['listing_ids'];
        $total = $batch_data['total'];
        $processed = $batch_data['processed'];
        $summary = $batch_data['summary'];

        // Process this batch
        for ($i = 0; $i < $batch_size && $current_index < $total; $i++, $current_index++) {
            $listing_id = $listing_ids[$current_index];
            
            try {
                if ($method === 'refresh_api') {
                    $result = $this->regenerate_google_photos($listing_id);
                } else if ($method === 'fallback_placeholder') {
                    $result = $this->remove_google_photos($listing_id);
                }
                
                if ($result === 'updated') {
                    $summary['updated']++;
                } else if ($result === 'skipped') {
                    $summary['skipped']++;
                } else {
                    $summary['errors']++;
                }
                
            } catch (Exception $e) {
                lds_log("Error processing listing {$listing_id}: " . $e->getMessage(), 'IMAGE_REGENERATION_ERROR');
                $summary['errors']++;
            }
            
            $processed++;
        }

        // Update batch data
        $batch_data['current_index'] = $current_index;
        $batch_data['processed'] = $processed;
        $batch_data['summary'] = $summary;
        set_transient($batch_id, $batch_data, 3600);

        $completed = ($current_index >= $total);

        if ($completed) {
            // Clean up transient when completed
            delete_transient($batch_id);
            lds_log("Completed image regeneration batch {$batch_id}. Summary: " . json_encode($summary), 'IMAGE_REGENERATION');
        }

        wp_send_json_success([
            'processed' => $processed,
            'total' => $total,
            'completed' => $completed,
            'summary' => $completed ? $summary : null
        ]);
    }

    /**
     * Regenerate Google photos for a listing with new API key
     */
    private function regenerate_google_photos($listing_id) {
        $google_gallery = get_post_meta($listing_id, '_gallery_google', true);
        $wp_gallery = get_post_meta($listing_id, '_gallery', true);
        
        // Check if this listing has Google photos to regenerate
        $has_google_photos = false;
        $current_photo_count = 0;
        
        if (!empty($google_gallery) && is_array($google_gallery)) {
            $has_google_photos = true;
            $current_photo_count = count($google_gallery);
            lds_log("Listing {$listing_id} has _gallery_google with {$current_photo_count} photos", 'IMAGE_REGENERATION');
        } elseif (!empty($wp_gallery) && is_array($wp_gallery)) {
            // Check if the _gallery contains Google URLs
            foreach ($wp_gallery as $photo_id => $photo_url) {
                if (strpos($photo_url, 'googleusercontent.com') !== false || strpos($photo_url, 'googleapis.com') !== false) {
                    $has_google_photos = true;
                    $current_photo_count = count($wp_gallery);
                    lds_log("Listing {$listing_id} has Google URLs in _gallery with {$current_photo_count} photos", 'IMAGE_REGENERATION');
                    break;
                }
            }
        }
        
        if (!$has_google_photos) {
            lds_log("Listing {$listing_id} has no Google photos to regenerate", 'IMAGE_REGENERATION');
            return 'skipped';
        }

        $api_key = get_option('lds_google_api_key');
        if (empty($api_key)) {
            lds_log("No Google API key found for regenerating photos", 'IMAGE_REGENERATION_ERROR');
            return 'error';
        }

        // Get place ID to regenerate photos
        $place_id = get_post_meta($listing_id, '_place_id', true);
        if (empty($place_id)) {
            lds_log("No place ID found for listing {$listing_id}", 'IMAGE_REGENERATION_ERROR');
            return 'error';
        }

        try {
            // Get language setting
            $lang_setting = get_option('lds_description_language', 'site-default');
            $language = ($lang_setting === 'site-default') ? get_locale() : $lang_setting;
            
            // Initialize Google API
            $google_api = new LDS_Google_API($api_key, $language);
            
            // Use current photo count as limit, but minimum of 1
            $photo_limit = max(1, $current_photo_count);
            
            lds_log("Fetching fresh photos for place ID {$place_id}, photo limit: {$photo_limit}", 'IMAGE_REGENERATION');
            
            // Fetch fresh place details with photos
            $place_details = $google_api->get_place_details($place_id, $photo_limit);
            
            if (is_wp_error($place_details)) {
                lds_log("Failed to fetch place details for listing {$listing_id}: " . $place_details->get_error_message(), 'IMAGE_REGENERATION_ERROR');
                return 'error';
            }

            lds_log("Google API returned photo data: " . json_encode($place_details['photo_urls'] ?? []), 'IMAGE_REGENERATION');

            if (!empty($place_details['photo_urls'])) {
                // Remove old Google gallery data (both types)
                delete_post_meta($listing_id, '_gallery_google');
                if (!empty($wp_gallery)) {
                    // Only remove _gallery if it contained Google URLs
                    foreach ($wp_gallery as $photo_id => $photo_url) {
                        if (strpos($photo_url, 'googleusercontent.com') !== false || strpos($photo_url, 'googleapis.com') !== false) {
                            delete_post_meta($listing_id, '_gallery');
                            break;
                        }
                    }
                }
                
                // Create new Google gallery data
                $new_google_gallery = [];
                foreach ($place_details['photo_urls'] as $index => $photo_url) {
                    $new_google_gallery[] = [
                        'url' => $photo_url,
                        'attribution' => isset($place_details['photo_attributions'][$index]) ? $place_details['photo_attributions'][$index] : []
                    ];
                }
                
                // Store in the proper _gallery_google meta
                update_post_meta($listing_id, '_gallery_google', $new_google_gallery);
                lds_log("Regenerated " . count($new_google_gallery) . " Google photos for listing {$listing_id}. New URLs: " . json_encode(array_column($new_google_gallery, 'url')), 'IMAGE_REGENERATION');
                return 'updated';
            } else {
                lds_log("No photos available for place ID {$place_id} (listing {$listing_id})", 'IMAGE_REGENERATION');
                return 'skipped';
            }
            
        } catch (Exception $e) {
            lds_log("Exception regenerating photos for listing {$listing_id}: " . $e->getMessage(), 'IMAGE_REGENERATION_ERROR');
            return 'error';
        }
    }

    /**
     * Remove Google photos and use theme fallback
     */
    private function remove_google_photos($listing_id) {
        $google_gallery = get_post_meta($listing_id, '_gallery_google', true);
        $wp_gallery = get_post_meta($listing_id, '_gallery', true);
        
        $removed_count = 0;
        
        // Remove _gallery_google if it exists
        if (!empty($google_gallery)) {
            delete_post_meta($listing_id, '_gallery_google');
            $removed_count += count($google_gallery);
            lds_log("Removed _gallery_google with " . count($google_gallery) . " photos from listing {$listing_id}", 'IMAGE_REGENERATION');
        }
        
        // Remove _gallery if it contains Google URLs
        if (!empty($wp_gallery) && is_array($wp_gallery)) {
            $has_google_urls = false;
            foreach ($wp_gallery as $photo_id => $photo_url) {
                if (strpos($photo_url, 'googleusercontent.com') !== false || strpos($photo_url, 'googleapis.com') !== false) {
                    $has_google_urls = true;
                    break;
                }
            }
            
            if ($has_google_urls) {
                delete_post_meta($listing_id, '_gallery');
                $removed_count += count($wp_gallery);
                lds_log("Removed _gallery with Google URLs (" . count($wp_gallery) . " photos) from listing {$listing_id}", 'IMAGE_REGENERATION');
            }
        }
        
        // Also remove featured image if it was from Google
        $thumbnail_id = get_post_thumbnail_id($listing_id);
        if ($thumbnail_id) {
            $attachment_url = wp_get_attachment_url($thumbnail_id);
            if (strpos($attachment_url, 'googleapis.com') !== false || strpos($attachment_url, 'googleusercontent.com') !== false) {
                delete_post_thumbnail($listing_id);
                lds_log("Removed Google-hosted featured image from listing {$listing_id}", 'IMAGE_REGENERATION');
            }
        }
        
        if ($removed_count > 0) {
            lds_log("Successfully removed {$removed_count} Google photos from listing {$listing_id}", 'IMAGE_REGENERATION');
            return 'updated';
        } else {
            lds_log("No Google photos found to remove from listing {$listing_id}", 'IMAGE_REGENERATION');
            return 'skipped';
        }
    }

    /**
     * Save current location as default map center
     */
    public function save_default_location() {
        // Verify nonce
        if (!check_ajax_referer('lds_import_nonce', 'nonce', false)) {
            wp_send_json_error(['message' => 'Security check failed.']);
            return;
        }

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions.']);
            return;
        }

        // Get and validate coordinates
        $lat = sanitize_text_field($_POST['lat'] ?? '');
        $lng = sanitize_text_field($_POST['lng'] ?? '');
        $location_name = sanitize_text_field($_POST['location_name'] ?? '');

        if (empty($lat) || empty($lng)) {
            wp_send_json_error(['message' => 'Invalid coordinates provided.']);
            return;
        }

        // Validate coordinate format
        if (!is_numeric($lat) || !is_numeric($lng)) {
            wp_send_json_error(['message' => 'Coordinates must be numeric values.']);
            return;
        }

        // Validate coordinate ranges
        if ($lat < -90 || $lat > 90 || $lng < -180 || $lng > 180) {
            wp_send_json_error(['message' => 'Coordinates are out of valid range.']);
            return;
        }

        // Save to settings
        update_option('lds_default_map_center_lat', $lat);
        update_option('lds_default_map_center_lng', $lng);

        // Log the update
        lds_log("Default map center updated to: {$location_name} ({$lat}, {$lng})", 'SETTINGS');

        wp_send_json_success([
            'message' => 'Default location saved successfully!',
            'lat' => $lat,
            'lng' => $lng,
            'location_name' => $location_name
        ]);
    }

    /**
     * Get current saved location settings
     */
    public function get_current_location_settings() {
        // Verify nonce
        if (!check_ajax_referer('lds_import_nonce', 'nonce', false)) {
            wp_send_json_error(['message' => 'Security check failed.']);
            return;
        }

        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Insufficient permissions.']);
            return;
        }

        // Get current saved values
        $lat = get_option('lds_default_map_center_lat', '51.5074');
        $lng = get_option('lds_default_map_center_lng', '-0.1278');

        wp_send_json_success([
            'lat' => $lat,
            'lng' => $lng,
            'message' => 'Current location settings retrieved successfully.'
        ]);
    }
}