/**
 * Listeo Data Scraper - Map Functionality
 * Handles interactive Google Maps for location selection
 */

// Debug: Log when script loads
console.log('LDS Map script loaded');

// Global variables
let map;
let marker;
let radiusCircle;
let geocoder;
let mapInitialized = false;

/**
 * Initialize Google Map when needed (not on API load)
 */
function initLDSMap() {
    console.log('initLDSMap called');
    
    if (mapInitialized) {
        console.log('Map already initialized');
        return;
    }
    
    // Wait for DOM to be ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initLDSMap);
        return;
    }
    
    if (typeof ldsMapSettings === 'undefined') {
        console.error('LDS Map Settings not loaded');
        showMapError('Map settings not available');
        return;
    }

    if (typeof google === 'undefined' || !google.maps) {
        console.error('Google Maps API not loaded');
        showMapError('Google Maps API not available');
        return;
    }

    const mapContainer = document.getElementById('lds-google-map');
    if (!mapContainer) {
        console.error('Map container not found');
        return;
    }

    // Check if map container is visible (map mode is active)
    const mapMode = document.getElementById('lds-map-mode');
    if (mapMode && mapMode.style.display === 'none') {
        console.log('Map mode not active, skipping initialization');
        return;
    }

    // Initialize map
    const defaultZoom = parseInt(ldsMapSettings.default_zoom, 10);
    console.log('Default zoom value:', defaultZoom, 'Type:', typeof defaultZoom);
    
    const mapOptions = {
        center: {
            lat: parseFloat(ldsMapSettings.default_lat),
            lng: parseFloat(ldsMapSettings.default_lng)
        },
        zoom: defaultZoom || 12, // Fallback to 12 if conversion fails
        mapTypeControl: true,
        streetViewControl: false,
        fullscreenControl: true,
        gestureHandling: 'greedy', // Allow scrolling without Ctrl+ key
        scrollwheel: true // Enable mouse wheel zoom
    };

    try {
        map = new google.maps.Map(mapContainer, mapOptions);
        geocoder = new google.maps.Geocoder();
        mapInitialized = true;

        // Set up event listeners
        setupMapEventListeners();
        
        // Set default radius
        const radiusSelect = document.getElementById('lds-radius');
        if (radiusSelect) {
            const defaultRadius = parseFloat(ldsMapSettings.default_radius) || 2.0;
            radiusSelect.value = defaultRadius;
            // Update display
            const displayElement = document.getElementById('lds-radius-value-display');
            if (displayElement) {
                displayElement.textContent = defaultRadius;
            }
            // Update slider to match the default radius
            if (typeof window.updateSliderValue === 'function') {
                window.updateSliderValue(defaultRadius);
            }
        }

        // Place marker at default location from settings
        const defaultLocation = {
            lat: parseFloat(ldsMapSettings.default_lat),
            lng: parseFloat(ldsMapSettings.default_lng)
        };
        
        if (!isNaN(defaultLocation.lat) && !isNaN(defaultLocation.lng)) {
            const defaultLatLng = new google.maps.LatLng(defaultLocation.lat, defaultLocation.lng);
            placeMarker(defaultLatLng);
            initializeDefaultLocation(defaultLatLng); // Use special init function
        }

        console.log('LDS Map initialized successfully');
    } catch (error) {
        console.error('Error initializing map:', error);
        showMapError('Failed to initialize map: ' + error.message);
    }
}

// Make function available globally immediately
window.initLDSMap = initLDSMap;
console.log('initLDSMap function set on window object');

/**
 * Show map error message
 */
function showMapError(message) {
    const mapContainer = document.getElementById('lds-google-map');
    if (mapContainer) {
        mapContainer.innerHTML = '<div class="lds-map-error">' + 
            '<strong>Map Error:</strong> ' + message + 
            '</div>';
    }
}

/**
 * Set up map-related event listeners
 */
function setupMapEventListeners() {
    // Map click event
    map.addListener('click', function(event) {
        placeMarker(event.latLng);
        updateLocation(event.latLng);
    });
}

/**
 * Set up UI event listeners
 */
function setupUIEventListeners() {
    jQuery(document).ready(function($) {
        // Mode toggle buttons
        $('.lds-mode-btn').on('click', function() {
            const mode = $(this).data('mode');
            switchLocationMode(mode);
        });

        // Address search button
        $('#lds-search-btn').on('click', function() {
            searchAddress();
        });

        // Use current location button
        $('#lds-use-location-btn').on('click', function() {
            useCurrentLocation();
        });

        // Save as default location button
        $('#lds-save-default-location').on('click', function() {
            saveAsDefaultLocation();
        });

        // Address search on Enter key
        $('#lds-address-search').on('keypress', function(e) {
            if (e.which === 13) {
                e.preventDefault();
                searchAddress();
            }
        });

        // Radius change - now using custom slider
        initCustomSlider();
        
        // Initialize form validation
        initFormValidation();
        
        // Set default text input as required since text mode is default
        const textInput = document.getElementById('lds_location');
        if (textInput) {
            textInput.required = true;
        }
        
        // Note: Map initialization is no longer automatic since text mode is default
        // Map will be initialized when user switches to map mode
    });
}

// Initialize UI event listeners immediately
setupUIEventListeners();

/**
 * Initialize form validation for map mode
 */
function initFormValidation() {
    // Override form validation when in map mode
    jQuery('#lds-import-form').on('submit', function(e) {
        const searchMode = jQuery('#lds-search-mode').val();
        
        if (searchMode === 'map') {
            const lat = jQuery('#lds-lat').val();
            const lng = jQuery('#lds-lng').val();
            
            if (!lat || !lng) {
                e.preventDefault();
                alert('Please select a location on the map before searching.');
                return false;
            }
        }
    });
}

/**
 * Switch between text and map location input modes
 */
function switchLocationMode(mode) {
    const textMode = document.getElementById('lds-text-mode');
    const mapMode = document.getElementById('lds-map-mode');
    const searchModeInput = document.getElementById('lds-search-mode');
    const buttons = document.querySelectorAll('.lds-mode-btn');
    
    // Get info elements
    const textSearchInfo = document.getElementById('lds-text-search-info');
    const mapSearchInfo = document.getElementById('lds-map-search-info');

    // Update button states
    buttons.forEach(btn => {
        btn.classList.remove('active');
        if (btn.dataset.mode === mode) {
            btn.classList.add('active');
        }
    });
    
    // Update info display
    if (textSearchInfo && mapSearchInfo) {
        if (mode === 'text') {
            textSearchInfo.style.display = 'block';
            mapSearchInfo.style.display = 'none';
        } else {
            textSearchInfo.style.display = 'none';
            mapSearchInfo.style.display = 'block';
        }
    }

    // Show/hide modes
    if (mode === 'text') {
        textMode.style.display = 'block';
        mapMode.style.display = 'none';
        searchModeInput.value = 'text';
        
        // Make text input required
        document.getElementById('lds_location').required = true;
    } else {
        textMode.style.display = 'none';
        mapMode.style.display = 'block';
        searchModeInput.value = 'map';
        
        // Remove required from text input
        document.getElementById('lds_location').required = false;
        
        // Initialize map when switching to map mode
        if (!mapInitialized && typeof google !== 'undefined' && google.maps) {
            console.log('Initializing map on mode switch');
            setTimeout(() => {
                initLDSMap();
                // Resize map after showing
                if (map) {
                    google.maps.event.trigger(map, 'resize');
                    // Automatically detect user location using IP geolocation
                    autoDetectLocation();
                }
            }, 100);
        } else if (mapInitialized && map) {
            // Resize map if already initialized
            setTimeout(() => {
                google.maps.event.trigger(map, 'resize');
                // Automatically detect user location if not already set
                if (!marker) {
                    autoDetectLocation();
                }
            }, 100);
        }
    }
}

/**
 * Search for an address and center map on it
 */
function searchAddress() {
    const address = document.getElementById('lds-address-search').value.trim();
    
    if (!address) {
        alert('Please enter an address to search');
        return;
    }

    geocoder.geocode({ address: address }, function(results, status) {
        if (status === 'OK') {
            const location = results[0].geometry.location;
            
            // Center map on found location
            map.setCenter(location);
            map.setZoom(12); // Use a fixed zoom level for search results
            
            // Place marker at found location
            placeMarker(location);
            updateLocation(location, results[0].formatted_address);
        } else {
            alert('Address not found: ' + status);
        }
    });
}

/**
 * Place or update marker on map
 */
function placeMarker(location) {
    if (marker) {
        marker.setPosition(location);
    } else {
        marker = new google.maps.Marker({
            position: location,
            map: map,
            title: 'Selected Location',
            draggable: true
        });

        // Make marker draggable
        marker.addListener('dragend', function() {
            const newPosition = marker.getPosition();
            updateLocation(newPosition);
            updateRadiusCircle(); // Update radius circle when marker is dragged
        });
        
        // Update radius circle in real-time while dragging
        marker.addListener('drag', function() {
            updateRadiusCircle(); // Update radius circle during drag for smooth movement
        });
    }

    updateRadiusCircle();
}

/**
 * Update the radius circle around the marker
 */
function updateRadiusCircle() {
    if (!marker) return;

    const radiusKm = parseFloat(document.getElementById('lds-radius').value);
    const radiusMeters = radiusKm * 1000;

    // Remove existing circle
    if (radiusCircle) {
        radiusCircle.setMap(null);
    }

    // Create new circle
    radiusCircle = new google.maps.Circle({
        center: marker.getPosition(),
        radius: radiusMeters,
        fillColor: '#3498db',
        fillOpacity: 0.2,
        strokeColor: '#3498db',
        strokeOpacity: 0.5,
        strokeWeight: 2,
        map: map
    });
}

/**
 * Update location display and hidden fields
 */
function updateLocation(location, address = null) {
    const lat = location.lat();
    const lng = location.lng();

    // Update hidden fields
    document.getElementById('lds-lat').value = lat;
    document.getElementById('lds-lng').value = lng;
    updateHiddenFields();

    // Show save as default button
    const saveBtn = document.getElementById('lds-save-default-location');
    if (saveBtn) {
        saveBtn.style.display = 'inline-block';
    }

    // Update display text
    const locationText = document.getElementById('lds-location-text');
    if (address) {
        locationText.textContent = `${address} (${lat.toFixed(4)}, ${lng.toFixed(4)})`;
    } else {
        // Reverse geocode to get address
        geocoder.geocode({ location: location }, function(results, status) {
            if (status === 'OK' && results[0]) {
                locationText.textContent = `${results[0].formatted_address} (${lat.toFixed(4)}, ${lng.toFixed(4)})`;
            } else {
                locationText.textContent = `Location: ${lat.toFixed(4)}, ${lng.toFixed(4)}`;
            }
        });
    }
}

/**
 * Initialize location display for default location (without showing save button)
 */
function initializeDefaultLocation(location) {
    const lat = location.lat();
    const lng = location.lng();

    // Update hidden fields
    document.getElementById('lds-lat').value = lat;
    document.getElementById('lds-lng').value = lng;
    updateHiddenFields();

    // Don't show save as default button since this IS the default location

    // Update display text with reverse geocoding
    const locationText = document.getElementById('lds-location-text');
    geocoder.geocode({ location: location }, function(results, status) {
        if (status === 'OK' && results[0]) {
            locationText.textContent = `${results[0].formatted_address} (${lat.toFixed(4)}, ${lng.toFixed(4)})`;
        } else {
            locationText.textContent = `Default Location: ${lat.toFixed(4)}, ${lng.toFixed(4)}`;
        }
    });
}

/**
 * Update all hidden form fields
 */
function updateHiddenFields() {
    const radiusKm = parseFloat(document.getElementById('lds-radius').value);
    document.getElementById('lds-radius-value').value = radiusKm;
    
    // Update location text display with radius
    const locationTextEl = document.getElementById('lds-location-text');
    const currentText = locationTextEl.textContent;
    
    if (currentText && !currentText.includes('Click on map')) {
        // Add radius info if not already present
        if (!currentText.includes('km radius')) {
            const baseText = currentText.split(' - ')[0]; // Remove existing radius info
            locationTextEl.textContent = `${baseText} - ${radiusKm}km radius`;
        } else {
            // Update existing radius info
            const baseText = currentText.split(' - ')[0];
            locationTextEl.textContent = `${baseText} - ${radiusKm}km radius`;
        }
    }
}

/**
 * Save current location as default map center
 */
function saveAsDefaultLocation() {
    const lat = document.getElementById('lds-lat').value;
    const lng = document.getElementById('lds-lng').value;
    const locationText = document.getElementById('lds-location-text').textContent;
    
    if (!lat || !lng) {
        alert('No location selected to save.');
        return;
    }
    
    // Disable button and show loading state
    const saveBtn = document.getElementById('lds-save-default-location');
    const originalText = saveBtn.textContent;
    saveBtn.disabled = true;
    saveBtn.textContent = '💾 Saving...';
    
    // Prepare data for AJAX
    const formData = new FormData();
    formData.append('action', 'lds_save_default_location');
    formData.append('nonce', jQuery('#lds_nonce').val());
    formData.append('lat', lat);
    formData.append('lng', lng);
    formData.append('location_name', locationText);
    
    // Make AJAX request
    fetch(ajaxurl, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Show success message
            alert('✅ ' + data.data.message);
            console.log('Default location saved:', data.data);
        } else {
            // Show error message
            alert('❌ Error: ' + (data.data?.message || 'Failed to save default location'));
            console.error('Save default location error:', data);
        }
    })
    .catch(error => {
        console.error('AJAX error:', error);
        alert('❌ Network error occurred while saving default location');
    })
    .finally(() => {
        // Re-enable button
        saveBtn.disabled = false;
        saveBtn.textContent = originalText;
    });
}

/**
 * Get location using IP geolocation (automatic, no user permission needed)
 */
async function getIPGeolocation() {
    try {
        console.log('Attempting IP geolocation with ipapi.co...');
        
        // First try ipapi.co (1000 requests/day)
        const response1 = await fetch('https://ipapi.co/json/', {
            method: 'GET',
            headers: {
                'Accept': 'application/json'
            }
        });
        
        if (response1.ok) {
            const data = await response1.json();
            console.log('ipapi.co response:', data);
            
            if (data.latitude && data.longitude && data.latitude !== 0 && data.longitude !== 0) {
                return {
                    lat: parseFloat(data.latitude),
                    lng: parseFloat(data.longitude),
                    city: data.city || '',
                    country: data.country_name || '',
                    source: 'ipapi.co'
                };
            }
        }
    } catch (error) {
        console.log('ipapi.co failed:', error.message);
    }

    try {
        console.log('Fallback to ipify + ipinfo.io...');
        
        // Alternative approach: Get IP first, then get location data
        const ipResponse = await fetch('https://api.ipify.org?format=json');
        if (ipResponse.ok) {
            const ipData = await ipResponse.json();
            console.log('Got IP:', ipData.ip);
            
            // Use a free tier of ipinfo.io (50,000 requests/month)
            const locationResponse = await fetch(`https://ipinfo.io/${ipData.ip}/geo`);
            if (locationResponse.ok) {
                const locationData = await locationResponse.json();
                console.log('ipinfo.io response:', locationData);
                
                if (locationData.loc) {
                    const [lat, lng] = locationData.loc.split(',');
                    if (lat && lng) {
                        return {
                            lat: parseFloat(lat),
                            lng: parseFloat(lng),
                            city: locationData.city || '',
                            country: locationData.country || '',
                            source: 'ipinfo.io'
                        };
                    }
                }
            }
        }
    } catch (error) {
        console.log('Fallback service failed:', error.message);
    }

    throw new Error('All IP geolocation services failed');
}

/**
 * Automatically detect user location silently (used when switching to map mode)
 */
function autoDetectLocation() {
    console.log('Auto-detecting user location...');
    
    // Check if a custom default location has been set
    const defaultLat = parseFloat(ldsMapSettings.default_lat);
    const defaultLng = parseFloat(ldsMapSettings.default_lng);
    
    // Original default coordinates (London)
    const originalDefaultLat = 51.5074;
    const originalDefaultLng = -0.1278;
    
    // If the default location has been changed from the original, don't auto-geolocate
    if (Math.abs(defaultLat - originalDefaultLat) > 0.001 || Math.abs(defaultLng - originalDefaultLng) > 0.001) {
        console.log('Custom default location detected, skipping auto-geolocation');
        return;
    }
    
    getIPGeolocation()
        .then(function(ipLocation) {
            console.log('Auto IP geolocation successful:', ipLocation);
            
            const location = {
                lat: ipLocation.lat,
                lng: ipLocation.lng
            };
            
            let locationName = 'Your Location';
            if (ipLocation.city && ipLocation.country) {
                locationName = `${ipLocation.city}, ${ipLocation.country}`;
            } else if (ipLocation.city) {
                locationName = ipLocation.city;
            } else if (ipLocation.country) {
                locationName = ipLocation.country;
            }
            
            // Silently update map without changing zoom dramatically
            map.setCenter(location);
            map.setZoom(12); // Use a moderate zoom level for auto-detection
            placeMarker(new google.maps.LatLng(location.lat, location.lng));
            updateLocation(new google.maps.LatLng(location.lat, location.lng), locationName);
        })
        .catch(function(error) {
            console.log('Auto IP geolocation failed, staying at default location:', error.message);
            // Don't show error to user for automatic detection, just stay at default location
        });
}

/**
 * Get user's current location with IP geolocation as primary method
 */
function useCurrentLocation() {
    const button = document.getElementById('lds-use-location-btn');
    const originalText = button.textContent;
    button.textContent = 'Loading...';
    button.disabled = true;
    button.classList.add('loading');

    // Try IP geolocation first (automatic, no permission needed)
    getIPGeolocation()
        .then(function(ipLocation) {
            console.log('IP geolocation successful:', ipLocation);
            
            const location = {
                lat: ipLocation.lat,
                lng: ipLocation.lng
            };
            
            let locationName = 'Your Location';
            if (ipLocation.city && ipLocation.country) {
                locationName = `${ipLocation.city}, ${ipLocation.country}`;
            } else if (ipLocation.city) {
                locationName = ipLocation.city;
            } else if (ipLocation.country) {
                locationName = ipLocation.country;
            }
            
            map.setCenter(location);
            map.setZoom(14); // Use fixed zoom level for current location
            placeMarker(new google.maps.LatLng(location.lat, location.lng));
            updateLocation(new google.maps.LatLng(location.lat, location.lng), locationName);
            
            button.textContent = originalText;
            button.disabled = false;
            button.classList.remove('loading');
        })
        .catch(function(error) {
            console.log('IP geolocation failed, falling back to GPS:', error.message);
            
            // Fallback to GPS geolocation if IP geolocation fails
            if (!navigator.geolocation) {
                alert('Geolocation is not supported by this browser');
                button.textContent = originalText;
                button.disabled = false;
                button.classList.remove('loading');
                return;
            }

            navigator.geolocation.getCurrentPosition(
                function(position) {
                    const location = {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude
                    };
                    
                    map.setCenter(location);
                    map.setZoom(14);
                    placeMarker(new google.maps.LatLng(location.lat, location.lng));
                    updateLocation(new google.maps.LatLng(location.lat, location.lng), 'Your GPS Location');
                    
                    button.textContent = originalText;
                    button.disabled = false;
                    button.classList.remove('loading');
                }, 
                function(gpsError) {
                    let errorMessage = 'Location detection failed';
                    switch(gpsError.code) {
                        case gpsError.PERMISSION_DENIED:
                            errorMessage = 'Location access denied by user';
                            break;
                        case gpsError.POSITION_UNAVAILABLE:
                            errorMessage = 'Location information unavailable';
                            break;
                        case gpsError.TIMEOUT:
                            errorMessage = 'Location request timed out';
                            break;
                    }
                    alert(errorMessage);
                    button.textContent = originalText;
                    button.disabled = false;
                    button.classList.remove('loading');
                },
                {
                    enableHighAccuracy: true,
                    timeout: 10000,
                    maximumAge: 60000
                }
            );
        });
}

/**
 * Initialize custom slider functionality
 */
function initCustomSlider() {
    const slider = document.querySelector('.lds-custom-slider');
    const thumb = document.querySelector('.lds-slider-thumb');
    const progress = document.querySelector('.lds-slider-progress');
    const hiddenInput = document.getElementById('lds-radius');
    const display = document.getElementById('lds-radius-value-display');
    
    if (!slider || !thumb || !progress || !hiddenInput || !display) {
        console.error('Custom slider elements not found');
        return;
    }
    
    const min = 0.5;
    const max = 50;
    const step = 0.5;
    let value = parseFloat(ldsMapSettings?.default_radius) || 2.0;
    let isDragging = false;
    let sliderRect = null;
    let animationFrame = null;
    
    // Format value to remove floating point precision issues
    function formatValue(val) {
        // Convert to number first, then round to 1 decimal place and remove trailing zeros
        const numVal = parseFloat(val);
        if (isNaN(numVal)) {
            return 2.0; // Default fallback
        }
        return parseFloat(numVal.toFixed(1));
    }
    
    // Cache slider dimensions
    function updateSliderRect() {
        sliderRect = slider.getBoundingClientRect();
    }
    
    // Throttled update function using requestAnimationFrame
    function updateSlider(skipMapUpdate = false) {
        if (animationFrame && !isDragging) {
            cancelAnimationFrame(animationFrame);
        }
        
        const updateFn = () => {
            const percentage = ((value - min) / (max - min)) * 100;
            
            // For thumb positioning, we need to account for the thumb width
            // Use transform translateX instead of left for better performance
            thumb.style.left = percentage + '%';
            thumb.style.transform = 'translate(-50%, -50%)'; // Keep centered
            
            progress.style.width = percentage + '%';
            
            // Format the value to avoid floating point precision issues
            const formattedValue = formatValue(value);
            display.textContent = formattedValue;
            hiddenInput.value = formattedValue;
            
            // Only update map functions when not dragging for better performance
            if (!skipMapUpdate && !isDragging) {
                if (typeof updateRadiusCircle === 'function') {
                    updateRadiusCircle();
                }
                if (typeof updateHiddenFields === 'function') {
                    updateHiddenFields();
                }
            }
        };
        
        if (isDragging) {
            // Update immediately during drag for responsiveness
            updateFn();
        } else {
            // Use requestAnimationFrame for smooth updates when not dragging
            animationFrame = requestAnimationFrame(updateFn);
        }
    }
    
    // Get dynamic step size based on current value
    function getDynamicStep(currentValue) {
        if (currentValue <= 2) {
            return 0.1; // Very fine control for 0.5-2km
        } else if (currentValue <= 5) {
            return 0.25; // Fine control for 2-5km  
        } else if (currentValue <= 10) {
            return 0.5; // Normal control for 5-10km
        } else {
            return 1; // Coarser control for 10km+
        }
    }
    
    // Calculate value from mouse position with dynamic stepping
    function getValueFromPosition(clientX) {
        if (!sliderRect) updateSliderRect();
        
        const percentage = Math.max(0, Math.min(100, ((clientX - sliderRect.left) / sliderRect.width) * 100));
        const rawValue = min + (percentage / 100) * (max - min);
        
        // Find the appropriate step size for the raw value
        const dynamicStep = getDynamicStep(rawValue);
        
        // Round to the nearest dynamic step
        const steppedValue = Math.round(rawValue / dynamicStep) * dynamicStep;
        
        // Ensure we stay within bounds and format to avoid floating point issues
        const finalValue = Math.max(min, Math.min(max, steppedValue));
        return formatValue(finalValue);
    }
    
    // Mouse down on slider or thumb
    function handleMouseDown(e) {
        isDragging = true;
        slider.classList.add('dragging'); // Disable CSS transitions
        updateSliderRect(); // Update dimensions at start of drag
        
        document.addEventListener('mousemove', handleMouseMove, {passive: false});
        document.addEventListener('mouseup', handleMouseUp);
        
        value = getValueFromPosition(e.clientX);
        updateSlider(true); // Skip map updates during drag start
        
        e.preventDefault();
    }
    
    // Mouse move during drag (optimized)
    function handleMouseMove(e) {
        if (!isDragging) return;
        
        value = getValueFromPosition(e.clientX);
        updateSlider(true); // Skip heavy map updates during drag
        
        e.preventDefault();
    }
    
    // Mouse up - end drag
    function handleMouseUp() {
        isDragging = false;
        slider.classList.remove('dragging'); // Re-enable CSS transitions
        document.removeEventListener('mousemove', handleMouseMove);
        document.removeEventListener('mouseup', handleMouseUp);
        
        // Final update with map functions
        updateSlider(false);
    }
    
    // Touch events for mobile (optimized)
    function handleTouchStart(e) {
        const touch = e.touches[0];
        handleMouseDown({clientX: touch.clientX, preventDefault: () => e.preventDefault()});
    }
    
    function handleTouchMove(e) {
        if (!isDragging) return;
        const touch = e.touches[0];
        handleMouseMove({clientX: touch.clientX, preventDefault: () => e.preventDefault()});
    }
    
    function handleTouchEnd() {
        handleMouseUp();
    }
    
    // Event listeners
    slider.addEventListener('mousedown', handleMouseDown);
    thumb.addEventListener('mousedown', handleMouseDown);
    
    // Touch events
    slider.addEventListener('touchstart', handleTouchStart, {passive: false});
    thumb.addEventListener('touchstart', handleTouchStart, {passive: false});
    document.addEventListener('touchmove', handleTouchMove, {passive: false});
    document.addEventListener('touchend', handleTouchEnd);
    
    // Keyboard support with dynamic stepping
    let keyboardTimeout;
    slider.addEventListener('keydown', function(e) {
        let newValue = value;
        const currentStep = getDynamicStep(value);
        
        switch(e.key) {
            case 'ArrowLeft':
            case 'ArrowDown':
                newValue = Math.max(min, value - currentStep);
                break;
            case 'ArrowRight':
            case 'ArrowUp':
                newValue = Math.min(max, value + currentStep);
                break;
            case 'Home':
                newValue = min;
                break;
            case 'End':
                newValue = max;
                break;
            default:
                return;
        }
        
        value = formatValue(newValue);
        updateSlider();
        e.preventDefault();
    });
    
    // Window resize handler (debounced)
    let resizeTimeout;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(() => {
            updateSliderRect();
            updateSlider();
        }, 150);
    });
    
    // Make slider focusable
    slider.setAttribute('tabindex', '0');
    
    // Initial setup
    updateSliderRect();
    updateSlider();
    
    // Global function to update slider value
    window.updateSliderValue = function(newValue) {
        value = parseFloat(newValue) || value;
        updateSlider();
    };
    
    console.log('Optimized custom slider initialized');
}

/**
 * Update slider progress fill based on current value
 */
function updateSliderProgress(slider) {
    // This function is now handled by initCustomSlider
    console.log('updateSliderProgress called - handled by custom slider');
}

// Make functions available globally if needed
window.useCurrentLocation = useCurrentLocation;

// Fallback handler in case Google Maps API doesn't load
jQuery(document).ready(function($) {
    // Initialize custom slider immediately on page load
    initCustomSlider();
    
    // Sync slider with default radius if settings are available
    setTimeout(function() {
        if (typeof ldsMapSettings !== 'undefined' && ldsMapSettings.default_radius) {
            const defaultRadius = parseFloat(ldsMapSettings.default_radius) || 2.0;
            const radiusInput = document.getElementById('lds-radius');
            const radiusDisplay = document.getElementById('lds-radius-value-display');
            if (radiusInput) {
                radiusInput.value = defaultRadius;
            }
            if (radiusDisplay) {
                radiusDisplay.textContent = defaultRadius;
            }
            if (typeof window.updateSliderValue === 'function') {
                window.updateSliderValue(defaultRadius);
            }
        }
    }, 100);
    
    // Update slider on window resize
    $(window).on('resize', function() {
        // Re-initialize custom slider on resize
        setTimeout(initCustomSlider, 100);
    });
    
    // Backup initialization if callback doesn't work and user switches to map mode
    setTimeout(function() {
        const mapContainer = $('#lds-google-map');
        if (mapContainer.length && typeof google !== 'undefined' && google.maps && !mapInitialized) {
            // Don't auto-initialize, wait for user to switch to map mode
            console.log('Google Maps API loaded successfully');
        } else if (mapContainer.length && typeof google === 'undefined') {
            console.warn('Google Maps API failed to load');
        }
    }, 3000); // Wait 3 seconds for Google Maps to load
});
