jQuery(document).ready(function($) { 
    // --- Global variables for the import process ---
    let placeIdQueue = [];
    let googleDataQueue = [];
    let jobQueue = [];
    let totalPlaces = 0;
    let processedPlaces = 0;
    let totalJobs = 0;
    let processedJobs = 0;
    let $resultsDiv = $('#lds-import-results');
    let selectedCategoryId = '';
    let selectedRegionId = ''; // NEW
    let selectedRegionTaxonomy = ''; // NEW
    let selectedListingType = ''; // NEW
    let currentPhase = '';
    let manualSelectionEnabled = false; // NEW

    // Check if category select has valid options and remove required if not
    function handleCategoryValidation() {
        const categorySelect = $('#lds_category');
        if (categorySelect.length) {
            const optionCount = categorySelect.find('option').length;
            // If only has the default "-- Select a Category --" option or no options
            if (optionCount <= 1) {
                categorySelect.removeAttr('required');
                console.log('Removed required attribute from category select - no valid options available');
            }
        }
    }

    // Initialize category validation
    handleCategoryValidation();

    // --- Enhanced AJAX function with better error handling ---
    function makeAjaxRequest(action, data, successCallback, errorCallback) {
        const requestData = {
            action: action,
            nonce: $('#lds_nonce').val(),
            ...data
        };

        console.log('Making AJAX request:', action, requestData);

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: requestData,
            dataType: 'json',
            timeout: 120000, // 120 second timeout
            success: function(response) {
                console.log('AJAX Success:', action, response);
                if (successCallback) successCallback(response);
            },
            error: function(xhr, status, error) {
                console.error('AJAX Error:', {
                    action: action,
                    status: status,
                    error: error,
                    responseText: xhr.responseText,
                    statusCode: xhr.status
                });
                
                let errorMessage = 'Server error occurred';
                if (status === 'timeout') {
                    errorMessage = 'The request timed out. Please try again.';
                } else if (xhr.responseText) {
                    try {
                        const errorData = JSON.parse(xhr.responseText);
                        errorMessage = errorData.data?.message || errorData.error || errorMessage;
                    } catch (e) {
                        if (xhr.responseText.includes('Fatal error')) {
                           errorMessage = 'A fatal PHP error occurred on the server. Check the server error logs.';
                        } else {
                           errorMessage = xhr.responseText.substring(0, 200) + '...';
                        }
                    }
                }
                
                if (errorCallback) {
                    errorCallback(errorMessage, xhr);
                } else {
                    $('#lds-results-body').append('<div class="notice notice-error"><p>Error in ' + action + ': ' + errorMessage + '</p></div>');
                }
            }
        });
    }

    // --- Main form submission handler ---
    $('#lds-import-form').on('submit', function(e) {
        e.preventDefault();

        const $form = $(this);
        const $spinner = $form.find('.lds-spinner');
        const $submitButton = $form.find('.lds-submit-button');

        const businessType = $('#lds_query').val().trim();
        const location = $('#lds_location').val().trim();
        const categoryId = $('#lds_category').val();
        const regionId = $('#lds_region').val(); // NEW - can be empty
        const regionTaxonomy = $('#lds_region_taxonomy').val() || 'region'; // NEW - get detected taxonomy, default back to region
        const listingType = $('#lds_listing_type').val(); // NEW
        manualSelectionEnabled = $('#lds_manual_selection').is(':checked'); // NEW

        // Get search mode and map data
        const searchMode = $('#lds-search-mode').val() || 'text';
        const lat = $('#lds-lat').val();
        const lng = $('#lds-lng').val();
        const radius = $('#lds-radius-value').val();

        // Validation based on search mode
        if (searchMode === 'map') {
            if (!businessType || !listingType || !lat || !lng) {
                alert('Please fill out all required fields and select a location on the map.');
                return;
            }
            // Only check category if the select field exists and has options
            const categorySelect = $('#lds_category');
            if (categorySelect.length && categorySelect.find('option').length > 1 && !categoryId) {
                alert('Please select a category from the dropdown.');
                return;
            }
        } else {
            if (!businessType || !location || !listingType) {
                alert('Please fill out all required fields: Business Type, Location, and Listing Type.');
                return;
            }
            // Only check category if the select field exists and has options
            const categorySelect = $('#lds_category');
            if (categorySelect.length && categorySelect.find('option').length > 1 && !categoryId) {
                alert('Please select a category from the dropdown.');
                return;
            }
        }

        const combinedQuery = searchMode === 'map' 
            ? businessType + ' near ' + lat + ',' + lng 
            : businessType + ' in ' + location;
        
        // Reset state for a new import
        placeIdQueue = [], googleDataQueue = [], jobQueue = [], totalPlaces = 0, processedPlaces = 0, totalJobs = 0, processedJobs = 0;
        selectedCategoryId = categoryId;
        selectedRegionId = regionId; // NEW
        selectedRegionTaxonomy = regionTaxonomy; // NEW
        selectedListingType = listingType; // NEW

        $spinner.addClass('is-active');
        $submitButton.prop('disabled', true);
        
        // --- NEW: Create the main results card structure ---
        // Add manual-check class if manual selection is enabled
        const manualCheckClass = manualSelectionEnabled ? ' manual-check' : '';
        $resultsDiv.html(
            '<div class="lds-results-card' + manualCheckClass + '">' +
                '<h3 class="lds-results-header">📊 Import Progress</h3>' +
                '<div class="lds-card-body" id="lds-results-body">' +
                    '<h4 class="lds-step-header">Step 1/4: Searching for new listings of "' + escapeHtml(businessType) + '"...</h4>' +
                '</div>' +
            '</div>'
        );
        const $resultsBody = $('#lds-results-body');

        // AJAX Call 1: Get unique, new Place IDs
        const requestData = {
            query: combinedQuery,
            lds_search_mode: searchMode
        };

        // Add map-specific data if in map mode
        if (searchMode === 'map') {
            requestData.lds_lat = lat;
            requestData.lds_lng = lng;
            requestData.lds_radius = radius;
        }

        makeAjaxRequest('lds_get_place_ids', requestData, function(response) {
            if (response.success && response.data.place_ids && response.data.place_ids.length > 0) {
                placeIdQueue = response.data.place_ids;
                totalPlaces = placeIdQueue.length;
                
                $resultsBody.html(
                    '<h4 class="lds-step-header">Step 2/4: Found ' + totalPlaces + ' new listings. Fetching details...</h4>' +
                    '<div class="lds-progress-wrapper"><div id="lds-fetch-progress" style="background-color: #2196F3;">0%</div></div>' +
                    '<ul id="lds-google-list" class="lds-log-area" style="max-height: 250px;"></ul>'
                );
                
                processNextGoogleFetch();
            } else {
                let message = response.data?.message || 'No new places found. Try a different search term or location.';
                let errorType = response.data?.type || '';
                let detailedMessage = response.data?.detailed_message || '';
                
                // Enhanced error display for API key issues
                if (errorType === 'api_key_error') {
                    let errorHtml = '<div class="notice notice-error is-dismissible lds-api-key-error">';
                    errorHtml += '<h4>⚠️ Google API Key Problem Detected</h4>';
                    errorHtml += '<p style="margin: 0 0 10px 0;"><strong>Error:</strong> ' + message + '</p>';
                    if (detailedMessage) {
                        errorHtml += '<div class="lds-error-solution"><strong>How to fix:</strong> ' + detailedMessage + '</div>';
                    }
                    errorHtml += '<div class="lds-error-tip">💡 <strong>Tip:</strong> Use the "Test API Key" button in Settings to verify your key is working correctly.</div>';
                    errorHtml += '</div>';
                    $resultsBody.html(errorHtml);
                } else if (errorType === 'api_error') {
                    let errorHtml = '<div class="notice notice-error is-dismissible">';
                    errorHtml += '<p><strong>Google API Error:</strong> ' + message + '</p>';
                    errorHtml += '<p style="font-size: 13px; color: #666;">Check your API key settings or try again later.</p>';
                    errorHtml += '</div>';
                    $resultsBody.html(errorHtml);
                } else {
                    let warningHtml = '<div class="notice notice-warning is-dismissible">';
                    warningHtml += '<p>' + message + '</p>';
                    
                    // Add suggestion for no_results type
                    if (response.data?.type === 'no_results' && response.data?.suggestion) {
                        warningHtml += '<p style="font-size: 13px; color: #666; margin-top: 8px; padding: 8px; background: #f9f9f9; border-radius: 3px;">';
                        warningHtml += '<strong>💡 Troubleshooting tips:</strong> ' + response.data.suggestion;
                        warningHtml += '</p>';
                    }
                    
                    warningHtml += '</div>';
                    $resultsBody.html(warningHtml);
                }
                $spinner.removeClass('is-active');
                $submitButton.prop('disabled', false);
            }
        }, function(errorMessage, xhr) {
            // Check if this is an API key error for the initial search
            let errorHtml = '';
            let isApiKeyError = false;
            let apiErrorDetails = '';
            
            // Try to parse the response to check for API key error type
            if (xhr && xhr.responseText) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.data && response.data.type === 'api_key_error') {
                        isApiKeyError = true;
                        apiErrorDetails = response.data.detailed_message || '';
                        errorMessage = response.data.message || errorMessage;
                    }
                } catch (e) {
                    // If parsing fails, check the error message text for API key issues
                    if (errorMessage.includes('API key not valid') || 
                        errorMessage.includes('REQUEST_DENIED') || 
                        errorMessage.includes('restricted') ||
                        errorMessage.includes('OVER_QUERY_LIMIT') ||
                        errorMessage.includes('expired')) {
                        isApiKeyError = true;
                    }
                }
            }
            
            if (isApiKeyError) {
                errorHtml = '<div class="notice notice-error is-dismissible lds-api-key-error">';
                errorHtml += '<h4>⚠️ Google API Key Problem Detected</h4>';
                errorHtml += '<p style="margin: 0 0 10px 0;"><strong>Error:</strong> ' + errorMessage + '</p>';
                if (apiErrorDetails) {
                    errorHtml += '<div class="lds-error-solution"><strong>How to fix:</strong> ' + apiErrorDetails + '</div>';
                }
                errorHtml += '<div class="lds-error-tip">💡 <strong>Tip:</strong> Use the "Test API Key" button in Settings to verify your key is working correctly.</div>';
                errorHtml += '</div>';
            } else {
                errorHtml = '<div class="notice notice-error is-dismissible"><p>Step 1 failed: ' + errorMessage + '</p></div>';
            }
            
            $resultsBody.html(errorHtml);
            $spinner.removeClass('is-active');
            $submitButton.prop('disabled', false);
        });
    });

    function processNextGoogleFetch() {
        if (placeIdQueue.length === 0) {
            if (googleDataQueue.length === 0) {
                $('#lds-results-body').append('<div class="notice notice-warning is-dismissible"><p>No valid places were found after fetching from Google.</p></div>');
                $('.lds-spinner').removeClass('is-active'); $('.lds-submit-button').prop('disabled', false);
            } else {
                // NEW: Check if manual selection is enabled
                if (manualSelectionEnabled) {
                    showManualSelectionInterface();
                } else {
                    startProcessingPhase();
                }
            }
            return;
        }

        const placeId = placeIdQueue.shift();
        processedPlaces++;

        makeAjaxRequest('lds_fetch_place_details', {
            place_id: placeId
        }, function(response) {
            if (response.success && response.data.place_data) {
                googleDataQueue.push(response.data.place_data);
                const displayInfo = response.data.display_info;
                let itemContent = '<strong>' + escapeHtml(displayInfo.name) + '</strong><br>';
                itemContent += '<span style="color: #666; font-size: 12px;">' + escapeHtml(displayInfo.address) + '</span>';
                if (displayInfo.rating && displayInfo.rating > 0) { itemContent += ' <span style="color: #ff9800;">★ ' + displayInfo.rating + '</span>'; }
                if (displayInfo.phone) { itemContent += '<br><span style="color: #666; font-size: 11px;">📞 ' + escapeHtml(displayInfo.phone) + '</span>'; }
                if (displayInfo.website) { itemContent += ' <span style="color: #666; font-size: 11px;">🌐 Website</span>'; }
                $('#lds-google-list').append('<li>' + itemContent + '</li>');
            } else {
                $('#lds-google-list').append('<li style="background: #fff3cd; color: #856404;">⚠ SKIPPED: ' + (response.data?.message || 'Failed to fetch place') + '</li>');
            }
            updateAndContinueFetch();
        }, function(errorMessage, xhr) {
            // Check if this is an API key error based on the error message or response data
            let isApiKeyError = false;
            let apiErrorDetails = '';
            
            // Try to parse the response to check for API key error type
            if (xhr && xhr.responseText) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.data && response.data.type === 'api_key_error') {
                        isApiKeyError = true;
                        apiErrorDetails = response.data.detailed_message || '';
                        errorMessage = response.data.message || errorMessage;
                    }
                } catch (e) {
                    // If parsing fails, check the error message text
                    if (errorMessage.includes('API key not valid') || 
                        errorMessage.includes('REQUEST_DENIED') || 
                        errorMessage.includes('restricted') ||
                        errorMessage.includes('OVER_QUERY_LIMIT') ||
                        errorMessage.includes('expired')) {
                        isApiKeyError = true;
                    }
                }
            }
            
            if (isApiKeyError) {
                // Show API key error prominently and stop the import process
                let errorHtml = '<li class="lds-critical-error">';
                errorHtml += '<strong>🚨 CRITICAL: Google API Key Error Detected</strong><br>';
                errorHtml += '<span style="font-size: 12px;">' + escapeHtml(errorMessage) + '</span>';
                if (apiErrorDetails) {
                    errorHtml += '<br><br><strong>Solution:</strong> ' + escapeHtml(apiErrorDetails);
                }
                errorHtml += '<br><br><em>Import stopped. Please fix your API key and try again.</em>';
                errorHtml += '</li>';
                $('#lds-google-list').append(errorHtml);
                
                // Stop the import process
                $('.lds-spinner').removeClass('is-active');
                $('.lds-submit-button').prop('disabled', false);
                return; // Don't continue fetching
            } else {
                $('#lds-google-list').append('<li style="background: #f8d7da; color: #721c24;">✖ ERROR: ' + escapeHtml(errorMessage) + '</li>');
            }
            updateAndContinueFetch();
        });
    }

    // NEW: Function to show manual selection interface
    function showManualSelectionInterface() {
        const $resultsBody = $('#lds-results-body');
        
        let selectionHTML = '<h4 class="lds-step-header">Manual Selection: Choose which places to import</h4>';
        selectionHTML += '<div class="lds-selection-controls" style="margin: 15px 0;">';
        selectionHTML += '<button type="button" class="lds-selection-button lds-select-all" id="lds-select-all">Select All</button> ';
        selectionHTML += '<button type="button" class="lds-selection-button lds-deselect-all" id="lds-deselect-all">Deselect All</button>';
        selectionHTML += '</div>';
        selectionHTML += '<ul class="lds-selection-list" style="list-style: none; margin: 0; padding: 0; max-height: 400px; overflow-y: auto; border: 1px solid #e1e5e9; border-radius: 4px; background: #f8f9fa;">';
        
        googleDataQueue.forEach(function(place, index) {
            selectionHTML += '<li style="padding: 12px; border-bottom: 1px solid #e1e5e9; background: white; margin: 0;">';
            selectionHTML += '<label style="display: flex; align-items: flex-start; cursor: pointer;">';
            selectionHTML += '<input type="checkbox" class="lds-place-checkbox" data-index="' + index + '" checked style="margin-right: 10px; margin-top: 2px;">';
            selectionHTML += '<div style="flex: 1;">';
            selectionHTML += '<strong>' + escapeHtml(place.name) + '</strong><br>';
            selectionHTML += '<span style="color: #666; font-size: 12px;">' + escapeHtml(place.address) + '</span>';
            if (place.rating && place.rating > 0) {
                selectionHTML += ' <span style="color: #ff9800;">★ ' + place.rating + '</span>';
            }
            if (place.phone_number) {
                selectionHTML += '<br><span style="color: #666; font-size: 11px;">📞 ' + escapeHtml(place.phone_number) + '</span>';
            }
            if (place.website) {
                selectionHTML += ' <span style="color: #666; font-size: 11px;">🌐 Website</span>';
            }
            selectionHTML += '</div>';
            selectionHTML += '</label>';
            selectionHTML += '</li>';
        });
        
        selectionHTML += '</ul>';
        selectionHTML += '<div class="lds-submit-area" style="margin-top: 20px;">';
        selectionHTML += '<button type="button" class="lds-submit-button" id="lds-proceed-selected">';
        selectionHTML += '<span class="button-text">Proceed with Selected</span>';
        selectionHTML += '</button>';
        selectionHTML += '</div>';
        
        $resultsBody.html(selectionHTML);
        
        // Bind events for selection controls
        $('#lds-select-all').on('click', function() {
            $('.lds-place-checkbox').prop('checked', true);
        });
        
        $('#lds-deselect-all').on('click', function() {
            $('.lds-place-checkbox').prop('checked', false);
        });
        
        $('#lds-proceed-selected').on('click', function() {
            const selectedPlaces = [];
            $('.lds-place-checkbox:checked').each(function() {
                const index = $(this).data('index');
                selectedPlaces.push(googleDataQueue[index]);
            });
            
            if (selectedPlaces.length === 0) {
                alert('Please select at least one place to import.');
                return;
            }
            
            // Replace googleDataQueue with only selected places
            googleDataQueue = selectedPlaces;
            startProcessingPhase();
        });
    }

    function startProcessingPhase() {
        const batchSize = 5;
        const batches = [];
        for (let i = 0; i < googleDataQueue.length; i += batchSize) {
            batches.push(googleDataQueue.slice(i, i + batchSize));
        }
        $('#lds-results-body').append(
            '<h4 class="lds-step-header" id="lds-processing-text">Step 3/4: Processing ' + googleDataQueue.length + ' places with AI...</h4>' +
            '<div class="lds-progress-wrapper"><div id="lds-ai-progress" style="background-color: #FF9800;">0%</div></div>' +
            '<ul id="lds-ai-log" class="lds-log-area"></ul>'
        );
        processBatchSequentially(batches, 0);
    }

    function processBatchSequentially(batches, batchIndex) {
        if (batchIndex >= batches.length) {
            startImportPhase();
            return;
        }

        const currentBatch = batches[batchIndex];
        const cleanBatch = currentBatch.map(item => {
            return {
                name: item.name || '', place_id: item.place_id || '', address: item.address || '',
                lat: item.lat || 0, lng: item.lng || 0, website: item.website || '',
                phone_number: item.phone_number || '', rating: item.rating || 0, opening_hours: item.opening_hours || [],
                photo_urls: item.photo_urls || [], reviews: item.reviews || [], types: item.types || [], user_ratings_total: item.user_ratings_total || 0,
            };
        });

        const $aiProgress = $('#lds-ai-progress');
        const $processingText = $('#lds-processing-text');
        
        $processingText.html('🤖 <strong>Step 3/4:</strong> Contacting AI for batch ' + (batchIndex + 1) + ' of ' + batches.length + '. Please wait, this can take a minute...');
        $aiProgress.css('width', '100%').text('Processing...').addClass('lds-progress-pulsing');

        makeAjaxRequest('lds_process_batch_ai', {
            places_data: cleanBatch
        }, function(response) {
            $aiProgress.removeClass('lds-progress-pulsing');
            if (response.success && response.data.listings) {
                jobQueue = jobQueue.concat(response.data.listings);
                response.data.listings.forEach(function(listing) {
                    $('#lds-ai-log').prepend('<li><span style="color: green;">✔</span> PROCESSED: ' + escapeHtml(listing.name) + '</li>');
                });
            } else {
                $('#lds-ai-log').prepend('<li style="background: #f8d7da; color: #721c24;">✖ BATCH FAILED: ' + (response.data?.message || 'Unknown error') + '</li>');
            }
            const processingProgress = ((batchIndex + 1) / batches.length) * 100;
            $aiProgress.css('width', processingProgress + '%').text(Math.round(processingProgress) + '%');
            setTimeout(function() { processBatchSequentially(batches, batchIndex + 1); }, 500);
        }, function(errorMessage, xhr) {
            $aiProgress.removeClass('lds-progress-pulsing');
            $('#lds-ai-log').prepend('<li style="background: #f8d7da; color: #721c24;">✖ BATCH ERROR: ' + escapeHtml(errorMessage) + '</li>');
            const processingProgress = ((batchIndex + 1) / batches.length) * 100;
            $aiProgress.css('width', processingProgress + '%').text(Math.round(processingProgress) + '%');
            setTimeout(function() { processBatchSequentially(batches, batchIndex + 1); }, 500);
        });
    }

    function startImportPhase() {
        if (jobQueue.length === 0) {
            $('#lds-results-body').append('<div class="notice notice-error"><p>Processing complete, but no valid listings were generated for import.</p></div>');
            $('.lds-spinner').removeClass('is-active');
            $('.lds-submit-button').prop('disabled', false);
            return;
        }

        totalJobs = jobQueue.length;
        processedJobs = 0;
        
        $('#lds-results-body').append(
            '<h4 class="lds-step-header">Step 4/4: Importing ' + totalJobs + ' listings to WordPress...</h4>' +
            '<div class="lds-progress-wrapper"><div id="lds-import-progress" style="background-color: #4CAF50;">0%</div></div>' +
            '<ul id="lds-import-log" class="lds-log-area"></ul>'
        );
        
        processNextImportJob();
    }

    function processNextImportJob() {
        if (jobQueue.length === 0) {
            $('#lds-import-progress').css('width', '100%').text('100% - Complete!');
            $('#lds-import-log').append('<li style="font-weight: bold; color: #155724; background-color: #d4edda;">✔ Import finished! All new listings have been added.</li>');
            $('.lds-spinner').removeClass('is-active');
            $('.lds-submit-button').prop('disabled', false);
            return;
        }

        const job = jobQueue.shift();
        processedJobs++;

        // --- NEW: Include region and listing type in the AJAX request ---
        makeAjaxRequest('lds_process_single_job', {
            listing_data: job,
            category_id: selectedCategoryId,
            region_id: selectedRegionId, // NEW
            region_taxonomy: selectedRegionTaxonomy, // NEW
            listing_type: selectedListingType // NEW
        }, function(response) {
            if (response.success) {
                $('#lds-import-log').prepend('<li><span style="color: green;">✔</span> SUCCESS: Imported "' + escapeHtml(response.data.post_title) + '" (ID: ' + response.data.post_id + ')</li>');
            } else {
                $('#lds-import-log').prepend('<li style="background: #f8d7da; color: #721c24;">✖ FAILED: ' + escapeHtml(response.data.message) + '</li>');
            }
            const importProgress = (processedJobs / totalJobs) * 100;
            $('#lds-import-progress').css('width', importProgress + '%').text(Math.round(importProgress) + '%');
            setTimeout(processNextImportJob, 200);
        }, function(errorMessage, xhr) {
            $('#lds-import-log').prepend('<li style="background: #f8d7da; color: #721c24;">✖ ERROR: ' + escapeHtml(errorMessage) + '</li>');
            const importProgress = (processedJobs / totalJobs) * 100;
            $('#lds-import-progress').css('width', importProgress + '%').text(Math.round(importProgress) + '%');
            setTimeout(processNextImportJob, 200);
        });
    }

    function escapeHtml(text) {
        if (typeof text !== 'string') {
            return '';
        }
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    function updateAndContinueFetch() {
        const fetchProgress = (processedPlaces / totalPlaces) * 100;
        $('#lds-fetch-progress').css('width', fetchProgress + '%').text(Math.round(fetchProgress) + '%');
        const googleResults = $('#lds-google-list')[0];
        googleResults.scrollTop = googleResults.scrollHeight;
        setTimeout(processNextGoogleFetch, 300);
    }

    
});

class SearchableDropdown {
    constructor(selectElement, options = {}) {
        this.originalSelect = selectElement;
        this.options = {
            placeholder: options.placeholder || 'Search and select...',
            noResultsText: options.noResultsText || 'No results found',
            required: options.required || false,
            ...options
        };
        
        this.selectedValue = '';
        this.selectedText = '';
        this.isOpen = false;
        
        this.init();
    }
    
    init() {
        // Hide the original select
        this.originalSelect.style.display = 'none';
        
        // Create the searchable dropdown structure
        this.createDropdownHTML();
        this.bindEvents();
        
        // Set initial value if the original select has one
        const initialValue = this.originalSelect.value;
        if (initialValue) {
            const option = this.originalSelect.querySelector(`option[value="${initialValue}"]`);
            if (option) {
                this.selectOption(initialValue, option.textContent);
            }
        }
    }
    
    createDropdownHTML() {
        // Create wrapper
        this.wrapper = document.createElement('div');
        this.wrapper.className = 'lds-searchable-dropdown';
        
        // Create search input
        this.searchInput = document.createElement('input');
        this.searchInput.type = 'text';
        this.searchInput.className = 'lds-searchable-input';
        this.searchInput.placeholder = this.options.placeholder;
        this.searchInput.autocomplete = 'off';
        
        // Create dropdown arrow
        this.arrow = document.createElement('div');
        this.arrow.className = 'lds-dropdown-arrow';
        this.arrow.innerHTML = '▼';
        
        // Create input wrapper
        this.inputWrapper = document.createElement('div');
        this.inputWrapper.className = 'lds-input-wrapper';
        this.inputWrapper.appendChild(this.searchInput);
        this.inputWrapper.appendChild(this.arrow);
        
        // Create dropdown list
        this.dropdown = document.createElement('div');
        this.dropdown.className = 'lds-dropdown-list';
        
        // Create options list
        this.optionsList = document.createElement('ul');
        this.optionsList.className = 'lds-options-list';
        
        this.dropdown.appendChild(this.optionsList);
        
        // Assemble the component
        this.wrapper.appendChild(this.inputWrapper);
        this.wrapper.appendChild(this.dropdown);
        
        // Insert after the original select
        this.originalSelect.parentNode.insertBefore(this.wrapper, this.originalSelect.nextSibling);
        
        // Populate options
        this.populateOptions();
    }
    
    populateOptions() {
        this.optionsList.innerHTML = '';
        const options = Array.from(this.originalSelect.querySelectorAll('option'));
        
        options.forEach(option => {
            if (option.value === '') return; // Skip empty option
            
            const li = document.createElement('li');
            li.className = 'lds-option-item';
            li.dataset.value = option.value;
            
            // Handle indentation for hierarchical options
            const text = option.textContent;
            const indentLevel = (text.match(/^(\s|&nbsp;)*/)[0].length) / 4; // Count indentation
            const cleanText = text.replace(/^(\s|&nbsp;)*/, '').trim();
            
            li.innerHTML = `<span class="lds-option-text" style="padding-left: ${indentLevel * 10}px">${this.escapeHtml(cleanText)}</span>`;
            
            if (indentLevel > 0) {
                li.classList.add('lds-option-child');
            } else {
                li.classList.add('lds-option-parent');
            }
            
            this.optionsList.appendChild(li);
        });
    }
    
    bindEvents() {
        // Search input events
        this.searchInput.addEventListener('input', (e) => {
            this.filterOptions(e.target.value);
        });
        
        this.searchInput.addEventListener('focus', () => {
            this.openDropdown();
        });
        
        this.searchInput.addEventListener('keydown', (e) => {
            this.handleKeydown(e);
        });
        
        // Arrow click
        this.arrow.addEventListener('click', () => {
            if (this.isOpen) {
                this.closeDropdown();
            } else {
                this.openDropdown();
                this.searchInput.focus();
            }
        });
        
        // Option clicks
        this.optionsList.addEventListener('click', (e) => {
            const optionItem = e.target.closest('.lds-option-item');
            if (optionItem) {
                const value = optionItem.dataset.value;
                const text = optionItem.querySelector('.lds-option-text').textContent;
                this.selectOption(value, text);
            }
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', (e) => {
            if (!this.wrapper.contains(e.target)) {
                this.closeDropdown();
            }
        });
    }
    
    filterOptions(searchTerm) {
        const options = this.optionsList.querySelectorAll('.lds-option-item');
        let hasResults = false;
        
        options.forEach(option => {
            const text = option.querySelector('.lds-option-text').textContent.toLowerCase();
            const matches = text.includes(searchTerm.toLowerCase());
            
            if (matches || searchTerm === '') {
                option.style.display = 'block';
                hasResults = true;
            } else {
                option.style.display = 'none';
            }
        });
        
        // Show/hide no results message
        this.showNoResults(!hasResults && searchTerm !== '');
    }
    
    showNoResults(show) {
        let noResultsItem = this.optionsList.querySelector('.lds-no-results');
        
        if (show && !noResultsItem) {
            noResultsItem = document.createElement('li');
            noResultsItem.className = 'lds-no-results';
            noResultsItem.innerHTML = `<span class="lds-option-text">${this.options.noResultsText}</span>`;
            this.optionsList.appendChild(noResultsItem);
        } else if (!show && noResultsItem) {
            noResultsItem.remove();
        }
    }
    
    selectOption(value, text) {
        this.selectedValue = value;
        this.selectedText = text;
        
        // Update the original select
        this.originalSelect.value = value;
        
        // Update the search input
        this.searchInput.value = text;
        
        // Trigger change event on original select
        const changeEvent = new Event('change', { bubbles: true });
        this.originalSelect.dispatchEvent(changeEvent);
        
        this.closeDropdown();
    }
    
    openDropdown() {
        this.isOpen = true;
        this.dropdown.style.display = 'block';
        this.wrapper.classList.add('lds-dropdown-open');
        this.arrow.innerHTML = '▲';
        
        // Reset filter
        this.filterOptions('');
    }
    
    closeDropdown() {
        this.isOpen = false;
        this.dropdown.style.display = 'none';
        this.wrapper.classList.remove('lds-dropdown-open');
        this.arrow.innerHTML = '▼';
        
        // If nothing selected and input has text, clear it
        if (!this.selectedValue && this.searchInput.value) {
            this.searchInput.value = '';
        } else if (this.selectedValue) {
            this.searchInput.value = this.selectedText;
        }
    }
    
    handleKeydown(e) {
        switch (e.key) {
            case 'ArrowDown':
                e.preventDefault();
                this.navigateOptions('down');
                break;
            case 'ArrowUp':
                e.preventDefault();
                this.navigateOptions('up');
                break;
            case 'Enter':
                e.preventDefault();
                this.selectHighlightedOption();
                break;
            case 'Escape':
                this.closeDropdown();
                break;
        }
    }
    
    navigateOptions(direction) {
        const visibleOptions = Array.from(this.optionsList.querySelectorAll('.lds-option-item:not([style*="display: none"])'));
        const currentHighlighted = this.optionsList.querySelector('.lds-option-highlighted');
        
        let newIndex = 0;
        
        if (currentHighlighted) {
            const currentIndex = visibleOptions.indexOf(currentHighlighted);
            newIndex = direction === 'down' 
                ? Math.min(currentIndex + 1, visibleOptions.length - 1)
                : Math.max(currentIndex - 1, 0);
        }
        
        // Remove previous highlight
        if (currentHighlighted) {
            currentHighlighted.classList.remove('lds-option-highlighted');
        }
        
        // Add new highlight
        if (visibleOptions[newIndex]) {
            visibleOptions[newIndex].classList.add('lds-option-highlighted');
            visibleOptions[newIndex].scrollIntoView({ block: 'nearest' });
        }
    }
    
    selectHighlightedOption() {
        const highlighted = this.optionsList.querySelector('.lds-option-highlighted');
        if (highlighted) {
            const value = highlighted.dataset.value;
            const text = highlighted.querySelector('.lds-option-text').textContent;
            this.selectOption(value, text);
        }
    }
    
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
}
// Initialize searchable dropdowns when the document is ready
jQuery(document).ready(function($) {
    // Initialize category dropdown
    const categorySelect = document.getElementById('lds_category');
    if (categorySelect) {
        new SearchableDropdown(categorySelect, {
            placeholder: 'Search categories...',
            required: true
        });
    }
    
    // Initialize region dropdown
    const regionSelect = document.getElementById('lds_region');
    if (regionSelect) {
        new SearchableDropdown(regionSelect, {
            placeholder: 'Search regions... (optional)',
            required: false
        });
    }
    
    // Initialize listing type dropdown
    const listingTypeSelect = document.getElementById('lds_listing_type');
    if (listingTypeSelect) {
        new SearchableDropdown(listingTypeSelect, {
            placeholder: 'Select listing type...',
            required: true
        });
    }
});