<?php

namespace Arts\Optimizer;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use \Arts\Optimizer\Managers\HTML;
use \Arts\Optimizer\Managers\Preloads;
use \Arts\Optimizer\Managers\Scripts;
use \Arts\Utilities\Utilities;

/**
 * Class Plugin
 *
 * Main plugin class that initializes the plugin and adds managers.
 *
 * @package Arts\Optimizer
 */
class Plugin {
	/**
	 * The instance of this class.
	 *
	 * @var Plugin
	 */
	protected static $instance;

	/**
	 * Managers for the plugin.
	 *
	 * @var Object
	 */
	private $managers;

	/**
	 * Get the instance of this class.
	 *
	 * @return object The instance of this class.
	 */
	public static function instance( $args = array(), $strings = array() ) {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self( $args, $strings );
		}

		return self::$instance;
	}

	/**
	 * Constructor for Plugin class.
	 *
	 * @param array $args    Configuration arguments.
	 * @param array $strings Localization strings.
	 */
	private function __construct( $args = array(), $strings = array() ) {
		$this->managers = new \stdClass();
		$this->init( $args, $strings );
	}

	/**
	 * Proxy method to print the inline script to set the height of an element.
	 *
	 * @param array $args Arguments for the script.
	 *
	 * @return void
	 */
	public function print_inline_script_set_element_height( $args = array() ) {
		$args = apply_filters( 'arts/optimizer/inline_script_set_element_height/args', $args );
		$this->managers->scripts->print_inline_script_set_element_height( $args );
	}

	/**
	 * Proxy method to print the inline script to find and load the LCP image.
	 *
	 * @param array $args Arguments for the script.
	 *
	 * @return void
	 */
	public function print_inline_script_load_lcp_image( $args = array() ) {
		if ( self::is_inline_script_load_lcp_image_enabled() ) {
			$args = apply_filters( 'arts/optimizer/inline_script_load_lcp_image/args', $args );
			$this->managers->scripts->print_inline_script_load_lcp_image( $args );
		}
	}

	/**
	 * Proxy method to print the inline script to set the client height as a CSS variable.
	 *
	 * @param array $args Arguments for the script.
	 *
	 * @return void
	 */
	public function print_inline_script_set_client_height( $args = array() ) {
		if ( self::is_inline_script_set_client_height_enabled() ) {
			$args = apply_filters( 'arts/optimizer/inline_script_set_client_height/args', $args );
			$this->managers->scripts->print_inline_script_set_client_height( $args );
		}
	}

	/**
	 * Initializes the plugin by adding managers, filters, and actions.
	 *
	 * @param array $args Arguments to pass to the manager classes.
	 * @param array $strings Strings to pass to the manager classes.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function init( $args, $strings ) {
		$this
			->add_managers( $args, $strings )
			->init_managers()
			->add_filters( $args )
			->add_actions( $args );

		return $this;
	}

	/**
	 * Adds manager instances to the managers property.
	 *
	 * @param array $args Arguments to pass to the manager classes.
	 * @param array $strings Strings to pass to the manager classes.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function add_managers( $args, $strings ) {
		$args['enqueue_css_assets'] = self::is_css_enqueuing_enabled();

		$manager_classes = array(
			'html'     => HTML::class,
			'scripts'  => Scripts::class,
			'preloads' => Preloads::class,
		);
		foreach ( $manager_classes as $key => $class ) {
			$this->managers->$key = $this->get_manager_instance( $class, $args, $strings );
		}

		return $this;
	}

	/**
	 * Initialize all manager classes by calling their init method if it exists.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function init_managers() {
		$managers = $this->managers;

		foreach ( $managers as $manager ) {
			if ( method_exists( $manager, 'init' ) ) {
				$manager->init( $managers );
			}
		}

		return $this;
	}

	/**
	 * Helper method to instantiate a manager class.
	 *
	 * @param string $class The manager class to instantiate.
	 * @param array  $args Arguments to pass to the manager class.
	 * @param array  $strings Strings to pass to the manager class.
	 *
	 * @return object The instantiated manager class.
	 */
	private function get_manager_instance( $class, $args, $strings ) {
		try {
			$reflection = new \ReflectionClass( $class );
			return $reflection->newInstanceArgs( array( $args, $strings ) );
		} catch ( \ReflectionException $e ) {
			return new $class();
		}
	}

	/**
	 * Adds WordPress actions for the plugin.
	 *
	 * @param array $args Arguments to pass to the action hooks.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function add_actions( $args = array() ) {
		if ( ! self::should_halt_optimization() ) {
			if ( self::is_preloads_injection_enabled() || self::is_html_minification_enabled() ) {
				add_action( 'template_redirect', array( $this, 'start_unified_buffer' ), 10 );
				add_action( 'shutdown', array( $this, 'end_unified_buffer' ) );
			}
		}

		// Add inline script to set client height as a CSS variable on the <html> element
		add_action( 'wp_body_open', array( $this, 'print_inline_script_set_client_height' ) );

		// Add inline script to load the LCP image
		add_action( 'wp_footer', array( $this, 'print_inline_script_load_lcp_image' ), -1 );

		return $this;
	}

	/**
	 * Starts a single output buffer for all HTML transformations.
	 */
	public function start_unified_buffer() {
		ob_start( array( $this, 'process_html_output' ) );
	}

	/**
	 * Ends the output buffer safely.
	 */
	public function end_unified_buffer() {
		try {
			if ( ob_get_level() > 0 ) {
				ob_end_flush();
			}
		} catch ( \Exception $e ) {
			error_log( 'ArtsOptimizer: Error flushing output buffer - ' . $e->getMessage() );
			// Attempt recovery - get and clean the buffer without callback
			if ( ob_get_level() > 0 ) {
				$content = ob_get_clean();
				echo $content;
			}
		}
	}

	/**
	 * Process HTML output with all enabled transformations.
	 *
	 * @param string $buffer The HTML output to process.
	 * @return string The processed HTML output.
	 */
	public function process_html_output( $buffer ) {
		do_action( 'qm/start', 'arts_optimizer_process_html' );

		// Apply preloads if enabled
		if ( self::is_preloads_injection_enabled() ) {
			do_action( 'qm/start', 'arts_optimizer_preloads' );
			$buffer = $this->managers->preloads->inject_preload_links( $buffer );
			do_action( 'qm/stop', 'arts_optimizer_preloads' );
		}

		// Apply HTML minification if enabled
		if ( self::is_html_minification_enabled() ) {
			do_action( 'qm/start', 'arts_optimizer_minify_html' );
			$buffer = $this->managers->html->remove_white_space( $buffer );
			do_action( 'qm/stop', 'arts_optimizer_minify_html' );
		}

		do_action( 'qm/stop', 'arts_optimizer_process_html' );

		return $buffer;
	}

	/**
	 * Helper method to profile a callable with Query Monitor.
	 *
	 * @param string   $profile_name The name for the profiling section.
	 * @param callable $callback The callback to execute and profile.
	 * @return mixed The result of the callback.
	 */
	public static function profile( $profile_name, $callback ) {
		do_action( 'qm/start', 'arts_optimizer_' . $profile_name );
		$result = $callback();
		do_action( 'qm/stop', 'arts_optimizer_' . $profile_name );

		return $result;
	}

	/**
	 * Adds WordPress filters for the plugin.
	 *
	 * @return Plugin Returns the current instance for method chaining.
	 */
	private function add_filters( $args = array() ) {
		add_filter( 'autoptimize_filter_js_exclude', array( $this->managers->scripts, 'autoptimize_js_exclude' ) ); // phpcs:ignore WordPressVIPMinimum.Hooks.RestrictedHooks.

		return $this;
	}

	/**
	 * Determines whether the plugin should halt the optimization based on certain conditions.
	 *
	 * This method checks a set of default conditions to decide if the plugin should halt execution.
	 * These conditions can be customized using the `arts/optimizer/halt_optimization_conditions` filter.
	 *
	 * @return bool True if the plugin should halt execution, false otherwise.
	 */
	private static function should_halt_optimization() {
		$conditions = apply_filters(
			'arts/optimizer/halt_optimization_conditions',
			array(
				is_admin(), // Do not run in the admin area
				Utilities::is_elementor_editor_active(), // Do not run in Elementor editor
				defined( 'TEMPLATELY_URL' ), // Do not run if Templately is active due to output buffering conflict
				wp_doing_ajax(),
				defined( 'REST_REQUEST' ),
				( function_exists( 'wp_is_json_request' ) && wp_is_json_request() ),
			)
		);

		return in_array( true, $conditions, true );
	}

	/**
	 * Determines whether HTML minification is enabled.
	 *
	 * @return bool True if HTML minification is enabled, false otherwise. Default is false.
	 */
	private static function is_html_minification_enabled() {
		return apply_filters( 'arts/optimizer/html_minification_enabled', false );
	}

	/**
	 * Determines whether the printing of the inline script to set the client height is enabled.
	 *
	 * @return bool True if the inline script to set the client height is enabled, false otherwise. Default is true.
	 */
	private static function is_inline_script_set_client_height_enabled() {
		return apply_filters( 'arts/optimizer/inline_script_set_client_height_enabled', true );
	}

	/**
	 * Determines whether the printing of the inline script to load the LCP image is enabled.
	 *
	 * @return bool True if the inline script to load the LCP image is enabled, false otherwise. Default is false.
	 */
	private static function is_inline_script_load_lcp_image_enabled() {
		return apply_filters( 'arts/optimizer/inline_script_load_lcp_image_enabled', false );
	}

	/**
	 * Determines whether injection of preload tags is enabled.
	 *
	 * @return bool True if injection of preload tags is enabled, false otherwise. Default is true.
	 */
	private static function is_preloads_injection_enabled() {
		return apply_filters( 'arts/optimizer/preloads_injection_enabled', true );
	}

	/**
	 * Determines whether CSS assets should be enqueued instead of preloaded.
	 *
	 * @return bool True if CSS assets should be enqueued, false if they should be preloaded. Default is false.
	 */
	private static function is_css_enqueuing_enabled() {
		return apply_filters( 'arts/optimizer/preloads/enqueue_css_assets', false );
	}
}
