<?php

namespace Arts\Optimizer\Managers;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class Scripts
 *
 * Used to manage the scripts that are output to the page.
 *
 * @package Arts\Optimizer\Managers
 */
class Scripts extends BaseManager {
	/**
	 * Prints an inline script to set the client height as a CSS variable to minimize layout shift.
	 *
	 * This method outputs an inline JavaScript snippet that calculates the client's height
	 * and sets it as a CSS variable (`--client-height`) on the `document.documentElement`.
	 * It also sets a CSS variable (`--fix-bar-vh`) to fix the viewport height issue on mobile devices when hiding the address bar.
	 *
	 * @param array $args {
	 *   Optional. An array of arguments.
	 *
	 *   @type string $script_id The ID of the script element. Default 'client'.
	 *   @type string $css_var_client_height The CSS variable to set for client height. Default '--client-height'.
	 *   @type string $css_var_fix_bar The CSS variable to set for fixing the viewport height issue. Default '--fix-bar-vh'.
	 *   @type bool $minify Whether to minify the script. Default true.
	 * }
	 * @return void
	 */
	public function print_inline_script_set_client_height( $args = array() ) {
		$defaults = array(
			'script_id'             => 'set-client-height',
			'css_var_client_height' => '--client-height',
			'css_var_fix_bar'       => '--fix-bar-vh',
			'minify'                => true,
		);
		$args     = wp_parse_args( $args, $defaults );

		if ( empty( $args['css_var_client_height'] ) && empty( $args['css_var_fix_bar'] ) ) {
			return;
		}

		$script_id = self::get_script_id( $args['script_id'] );

		if ( isset( $args['minify'] ) && $args['minify'] ) {
			ob_start();
		}
		?>
		<script id="<?php echo esc_attr( $script_id ); ?>">
			(function() {
				let e = document.documentElement.clientHeight;
			<?php if ( $args['css_var_client_height'] ) : ?>
					document.documentElement.style.setProperty("<?php echo esc_js( $args['css_var_client_height'] ); ?>", e);
				<?php endif; ?>
			<?php if ( $args['css_var_fix_bar'] ) : ?>
					document.documentElement.style.setProperty("<?php echo esc_js( $args['css_var_fix_bar'] ); ?>", `${.01 * e}px`);
				<?php endif; ?>
			})();
		</script>
		<?php
		if ( isset( $args['minify'] ) && $args['minify'] ) {
			$script = ob_get_clean();
			echo self::minify_script( $script );
		}
	}

	/**
	 * Prints an inline script to set the height of a specified element as a CSS variable.
	 *
	 * This method outputs an inline JavaScript snippet that calculates the height of a specified element
	 * and sets it as a CSS variable on the `document.documentElement`. It also adds a class to the document
	 * to indicate that the height has been set.
	 *
	 * @param array $args {
	 *   Optional. An array of arguments.
	 *
	 *   @type string $script_id The ID of the script element. Default 'header-container'.
	 *   @type string $selector The CSS selector of the element. Default '#page-header .header__bar'.
	 *   @type string $css_var The CSS variable to set. Default '--header-height'.
	 *   @type string $set_document_class The optional class to add to the document. Default 'has-header-height'.
	 *   @type bool $minify Whether to minify the script. Default true.
	 * }
	 * @return void
	 */
	public function print_inline_script_set_element_height( $args = array() ) {
		$defaults = array(
			'script_id'          => 'set-header-container-height',
			'selector'           => '#page-header .header__bar',
			'css_var'            => '--header-height',
			'set_document_class' => 'has-header-height',
			'minify'             => true,
		);
		$args     = wp_parse_args( $args, $defaults );

		// Bail if required arguments are missing.
		if ( empty( $args['selector'] ) || empty( $args['css_var'] ) ) {
			return;
		}

		$script_id = self::get_script_id( $args['script_id'] );

		if ( isset( $args['minify'] ) && $args['minify'] ) {
			ob_start();
		}
		?>
		<script id="<?php echo esc_attr( $script_id ); ?>">
			(function() {
				let e = document.querySelector('<?php echo esc_js( $args['selector'] ); ?>');
				if (e) {
					let t = e.clientHeight;
					document.documentElement.style.setProperty('<?php echo esc_js( $args['css_var'] ); ?>', t + 'px');

				<?php if ( $args['set_document_class'] ) : ?>
						document.documentElement.classList.add('<?php echo esc_js( $args['set_document_class'] ); ?>');
					<?php endif; ?>
				}
			})();
		</script>
		<?php
		if ( isset( $args['minify'] ) && $args['minify'] ) {
			$script = ob_get_clean();
			echo self::minify_script( $script );
		}
	}

	/**
	 * Prints an inline script to load the Largest Contentful Paint (LCP) image.
	 *
	 * This method outputs an inline JavaScript snippet that finds the image with the highest fetch priority,
	 * sets its `src`, `srcset`, and `sizes` attributes, and adds a class to indicate that the image has been loaded.
	 *
	 * @param array $args {
	 *   Optional. An array of arguments.
	 *
	 *   @type string $script_id The ID of the script element. Default 'load-lcp-image'.
	 *   @type string $selector The CSS selector of the image element. Default 'img[fetchpriority="high"][data-src]'.
	 *   @type string $set_class The class to add to the image element. Default 'lazy-lcp'.
	 *   @type string $remove_class The class to remove from the image element. Default 'lazy'.
	 *   @type string $loaded_class The class to add to the image element once it is loaded. Default 'loaded'.
	 *   @type bool $minify Whether to minify the script. Default true.
	 * }
	 * @return void
	 */
	public function print_inline_script_load_lcp_image( $args = array() ) {
		$defaults = array(
			'script_id'    => 'load-lcp-image',
			'selector'     => 'img[fetchpriority="high"][data-src]',
			'set_class'    => 'lazy-lcp',
			'remove_class' => 'lazy',
			'loaded_class' => 'loaded',
			'minify'       => true,
		);
		$args     = wp_parse_args( $args, $defaults );

		// Bail if required arguments are missing.
		if ( empty( $args['selector'] ) ) {
			return;
		}

		$script_id = self::get_script_id( $args['script_id'] );

		if ( isset( $args['minify'] ) && $args['minify'] ) {
			ob_start();
		}
		?>
		<script id="<?php echo esc_attr( $script_id ); ?>">
			(function() {
				try {
					let t = document.querySelector('<?php echo str_replace( '&quot;', '"', esc_js( $args['selector'] ) ); ?>');
					if (t && t.hasAttribute("data-src")) {
						let e = t.getAttribute("data-src"),
							s = t.getAttribute("data-srcset"),
							a = t.getAttribute("data-sizes");
						<?php if ( $args['set_class'] ) : ?>
							t.classList.add('<?php echo esc_js( $args['set_class'] ); ?>');
						<?php endif; ?>
						<?php if ( $args['remove_class'] ) : ?>
							t.classList.remove('<?php echo esc_js( $args['remove_class'] ); ?>');
						<?php endif; ?>

						t.setAttribute("src", e);
						if (s) t.setAttribute("srcset", s);
						if (a) t.setAttribute("sizes", a);
						<?php if ( $args['loaded_class'] ) : ?>
							t.onload = () => {
								t.classList.add('<?php echo esc_js( $args['loaded_class'] ); ?>');
							};
						<?php endif; ?>
					}
				} catch (i) {
					console.warn(i);
				}
			})();
		</script>
		<?php
		if ( isset( $args['minify'] ) && $args['minify'] ) {
			$script = ob_get_clean();
			echo self::minify_script( $script );
		}
	}

	/**
	 * Generate a sanitized ID for printed inline scripts.
	 *
	 * @param string $id The input ID to be sanitized and used in the script ID.
	 * @return string The generated script ID.
	 */
	private static function get_script_id( $id ) {
		return self::get_script_prefix() . sanitize_html_class( $id ) . self::get_script_suffix();
	}

	/**
	 * Get the prefix used for script names.
	 *
	 * @return string The script prefix.
	 */
	private static function get_script_prefix() {
		return 'arts-optimizer-';
	}

	/**
	 * Get the suffix used for script names.
	 *
	 * @return string The script suffix.
	 */
	private static function get_script_suffix() {
		return '-js';
	}

	/**
	 * Excludes ArtsOptimizer scripts from Autoptimize processing.
	 *
	 * @param mixed $exclude Array or string of script IDs to exclude.
	 * @return mixed Modified exclude list.
	 */
	public function autoptimize_js_exclude( $exclude ) {
		// Determine the original type of $exclude
		$original_type = gettype( $exclude );

		// Convert string to array if necessary
		if ( is_string( $exclude ) ) {
			$exclude = array_filter( array_map( 'trim', explode( ',', $exclude ) ) );
		}

		// Add your exclusions to the existing list
		$exclude[] = self::get_script_prefix();

		// Convert back to the original type if necessary
		if ( $original_type === 'string' ) {
			$exclude = implode( ', ', $exclude );
		}

		return $exclude;
	}

	/**
	 * Minifies the given script by removing unnecessary whitespace.
	 *
	 * @param string $script The script to minify.
	 * @return string The minified script.
	 */
	private static function minify_script( $script ) {
		// Remove block comments
		$script = preg_replace( '#/\*.*?\*/#s', '', $script );
		// Remove line comments
		$script = preg_replace( '#//.*#', '', $script );

		// Remove whitespace around certain characters
		$script = preg_replace( '/\s*([{};,:])\s*/', '$1', $script );

		// Remove newlines, tabs, and extra spaces
		$script = preg_replace( '/[\r\n\t]+/', ' ', $script );
		$script = preg_replace( '/\s+/', ' ', $script );

		// Remove leading and trailing whitespace
		$script = trim( $script );

		return $script;
	}
}
